/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*-------------------------------------------------------------------------*/
/**
   @file    sph_dictionary.h
   @author  N. Devillard
   @date    Sep 2007
   @version $Revision: 1.12 $
   @brief   Implements a sph_dictionary for string variables.

   This module implements a simple sph_dictionary object, i.e. a list
   of string/string associations. This object is useful to store e.g.
   informations retrieved from a configuration file (ini files).
*/
/*--------------------------------------------------------------------------*/

/*
 $Id: sph_dictionary.h,v 1.12 2007-11-23 21:37:00 ndevilla Exp $
 $Author: ndevilla $
 $Date: 2007-11-23 21:37:00 $
 $Revision: 1.12 $
*/

#ifndef SPH_DICTIONARY_H_
#define SPH_DICTIONARY_H_

/*---------------------------------------------------------------------------
           Includes
 ---------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <cpl.h>

/*---------------------------------------------------------------------------
           New types
 ---------------------------------------------------------------------------*/


/*-------------------------------------------------------------------------*/
/**
  @brief Dictionary object

  This object contains a list of string/string associations. Each
  association is identified by a unique string key. Looking up values
  in the sph_dictionary is speeded up by the use of a (hopefully collision-free)
  hash function.
 */
/*-------------------------------------------------------------------------*/
typedef struct _sph_dictionary_ {
 int        n;    /** Number of entries in sph_dictionary */
 int        size; /** Storage size */
 char    ** val;  /** List of string values */
 char    ** key;  /** List of string keys */
 unsigned * hash; /** List of hash values for keys */
} sph_dictionary;


/*---------------------------------------------------------------------------
         Function prototypes
 ---------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/**
  @brief    Compute the hash key for a string.
  @param    key     Character string to use for key.
  @return   1 unsigned int on at least 32 bits.

  This hash function has been taken from an Article in Dr Dobbs Journal.
  This is normally a collision-free function, distributing keys evenly.
  The key is stored anyway in the struct so that collision can be avoided
  by comparing the key itself in last resort.
 */
/*--------------------------------------------------------------------------*/
unsigned sph_dictionary_hash(const char * key);

/*-------------------------------------------------------------------------*/
/**
  @brief    Create a new sph_dictionary object.
  @param    size    Optional initial size of the sph_dictionary.
  @return   1 newly allocated sph_dictionary objet.

  This function allocates a new sph_dictionary object of given size and returns
  it. If you do not know in advance (roughly) the number of entries in the
  sph_dictionary, give size=0.
 */
/*--------------------------------------------------------------------------*/
sph_dictionary * sph_dictionary_new(int size) CPL_ATTR_ALLOC;

/*-------------------------------------------------------------------------*/
/**
  @brief    Delete a sph_dictionary object
  @param    d   sph_dictionary object to deallocate.
  @return   void

  Deallocate a sph_dictionary object and all memory associated to it.
 */
/*--------------------------------------------------------------------------*/
void sph_dictionary_del(sph_dictionary * vd);

/*-------------------------------------------------------------------------*/
/**
  @brief    Get a value from a sph_dictionary.
  @param    d       sph_dictionary object to search.
  @param    key     Key to look for in the sph_dictionary.
  @param    def     Default value to return if key not found.
  @return   1 pointer to internally allocated character string.

  This function locates a key in a sph_dictionary and returns a pointer to its
  value, or the passed 'def' pointer if no such key can be found in
  sph_dictionary. The returned character pointer points to data internal to the
  sph_dictionary object, you should not try to free it or modify it.
 */
/*--------------------------------------------------------------------------*/
const char * sph_dictionary_get(const sph_dictionary * d,
                                const char * key,
                                const char * def)
    CPL_ATTR_NONNULL;

/*-------------------------------------------------------------------------*/
/**
  @brief    Set a value in a sph_dictionary.
  @param    d       sph_dictionary object to modify.
  @param    key     Key to modify or add.
  @param    val     Value to add.
  @return   int     0 if Ok, anything else otherwise

  If the given key is found in the sph_dictionary, the associated value is
  replaced by the provided one. If the key cannot be found in the
  sph_dictionary, it is added to it.

  It is Ok to provide a NULL value for val, but NULL values for the sph_dictionary
  or the key are considered as errors: the function will return immediately
  in such a case.

  Notice that if you sph_dictionary_set a variable to NULL, a call to
  sph_dictionary_get will return a NULL value: the variable will be found, and
  its value (NULL) is returned. In other words, setting the variable
  content to NULL is equivalent to deleting the variable from the
  sph_dictionary. It is not possible (in this implementation) to have a key in
  the sph_dictionary without value.

  This function returns non-zero in case of failure.
 */
/*--------------------------------------------------------------------------*/
int sph_dictionary_set(sph_dictionary * vd, const char * key, const char * val);

/*-------------------------------------------------------------------------*/
/**
  @brief    Delete a key in a sph_dictionary
  @param    d       sph_dictionary object to modify.
  @param    key     Key to remove.
  @return   void

  This function deletes a key in a sph_dictionary. Nothing is done if the
  key cannot be found.
 */
/*--------------------------------------------------------------------------*/
void sph_dictionary_unset(sph_dictionary * d, const char * key);

/*-------------------------------------------------------------------------*/
/**
  @brief    Dump a sph_dictionary to an opened file pointer.
  @param    d   Dictionary to dump
  @param    f   Opened file pointer.
  @return   void

  Dumps a sph_dictionary onto an opened file pointer. Key pairs are printed out
  as @c [Key]=[Value], one per line. It is Ok to provide stdout or stderr as
  output file pointers.
 */
/*--------------------------------------------------------------------------*/
void sph_dictionary_dump(const sph_dictionary * d, FILE * out);

#endif
