/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifndef SPH_FFT_H_
#define SPH_FFT_H_

#include "sph_error.h"

#include <cpl.h>

#include <gsl/gsl_fft.h>
#include <gsl/gsl_fft_complex.h>
#include <gsl/gsl_math.h>
#include <gsl/gsl_complex.h>
#include <gsl/gsl_fft_halfcomplex.h>
#include <gsl/gsl_complex_math.h>
#include <fftw3.h>

typedef enum _sph_fft_method_ {
    SPH_FFT_GSL_RADIX2,
    SPH_FFT_GSL_MIXEDRADIX,
    SPH_FFT_FFTW3_DP
} sph_fft_method;

typedef enum _sph_fft_fitler_method_ {
    SPH_FFT_FILTER_METHOD_NONE,
    SPH_FFT_FILTER_METHOD_TH,
    SPH_FFT_FILTER_METHOD_FERMI,
    SPH_FFT_FILTER_METHOD_BUTTER
} sph_fft_filter_method;

# define     SPH_FFT_FILTER_METHOD_DEFAULT     SPH_FFT_FILTER_METHOD_TH

typedef struct _sph_fft_ {
    int                        Np; // is N'
    int                        Npp; // is N''
    double                        angle; // rotangle in radians
    double                        scale; // scaling factor
    int                            nx;
    int                            ny;
    gsl_complex_packed_array    complexdata;
    gsl_fft_complex_wavetable*    wavetab;
    gsl_fft_complex_workspace*    workspace;
    double                    shiftx;
    double                    shifty;
    double                     top_hat_radius;
    sph_fft_method            method;

} sph_fft;

typedef struct _sph_fft_operand_ {
    cpl_image* image; /* Convolution operand */
    unsigned   nuse;  /* Use counter for diagnostics */

} sph_fft_operand;

sph_fft* sph_fft_new( sph_fft_method method) CPL_ATTR_ALLOC;

sph_fft_operand* sph_fft_operand_new(void) CPL_ATTR_ALLOC;
cpl_error_code sph_fft_smoothe_image(cpl_image*, sph_fft_operand*, double);
void sph_fft_operand_delete(sph_fft_operand*);

sph_error_code
sph_fft_set_shift(sph_fft* self, double dx, double dy);
sph_error_code
sph_fft_prepare( sph_fft* self, const cpl_image* im );
sph_error_code
sph_fft_forward( sph_fft* self );
sph_error_code
sph_fft_inverse( sph_fft* self );
gsl_complex
sph_fft_get_complex(sph_fft* self, int kx ,int ky );
sph_error_code
sph_fft_set_complex(sph_fft* self, int kx ,int ky, double real, double imag);
cpl_image*
sph_fft_complex2image_real( const sph_fft* self ) CPL_ATTR_ALLOC;
cpl_image*
sph_fft_complex2image_imag( const sph_fft* self ) CPL_ATTR_ALLOC;

sph_error_code
sph_fft_shift( sph_fft* self, double dx, double dy );
cpl_image*
sph_fft_shift_image( sph_fft* self, const cpl_image* im,double dx, double dy )
    CPL_ATTR_ALLOC;
fftw_complex*
sph_fft_pad_fftw_array_freq_space( sph_fft* self, const fftw_complex* in, int newn )
    CPL_ATTR_ALLOC;
fftw_complex*
sph_fft_crop_fftw_array( sph_fft* self, const fftw_complex* in,int newx, int shiftflag )
    CPL_ATTR_ALLOC;

sph_error_code
sph_fft_crop_complex_array( sph_fft* self, int newnx, int shiftflag );

sph_error_code
sph_fft_rotate_prepare( sph_fft* self, const cpl_image* im,
                        double degrees, double dx, double dy );
sph_error_code
sph_fft_rotate( sph_fft* self );
sph_error_code
sph_fft_skewx( sph_fft* self, const cpl_image* im, double degrees );
sph_error_code
sph_fft_skewy( sph_fft* self, const cpl_image* im, double degrees );
cpl_image*
sph_fft_rotate_complex2image( const sph_fft* self )
    CPL_ATTR_ALLOC;
cpl_image*
sph_fft_rotate_image( sph_fft* self, const cpl_image* image, double degrees )
    CPL_ATTR_ALLOC;
fftw_complex*
sph_fft_create_fftw_complex( sph_fft* self, const cpl_image* im,
                             int newx, int newy )
    CPL_ATTR_ALLOC;
cpl_image*
sph_fft_scale( sph_fft* self, const cpl_image* im, double scale )
    CPL_ATTR_ALLOC;
sph_error_code
sph_fft_pad_complex_array_freq_space( sph_fft* self, int newn );
sph_error_code
sph_fft_pad_complex_array_real_space( sph_fft* self, int newn );
void sph_fft_delete( sph_fft* self );
sph_error_code
sph_fft_filter_fftw_array( const sph_fft* self, fftw_complex* in,
                           sph_fft_filter_method method, double delta,
                           double param1, double param2 );
cpl_image*
sph_fft_freq_filter( sph_fft* self, const cpl_image* im,
        sph_fft_filter_method method, double radius,
        double param1, double param2 );
double
sph_fft_get_scale_vector(sph_fft* self, const cpl_image* im1, double* sigma );

cpl_error_code sph_fft_shift_triplet(const sph_fft*, cpl_image*, cpl_image*,
                                     cpl_image*, double, double);

#endif /* SPH_FFT_H_ */
