/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include "sph_fitting.h"

#include "sph_error.h"

#include <cpl.h>

#include <gsl/gsl_blas.h>
#include <gsl/gsl_multifit_nlin.h>
#include <gsl/gsl_sort.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_statistics.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_math.h>
#include <gsl/gsl_roots.h>
#include <gsl/gsl_version.h>
#include <string.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides ...
 */
/*----------------------------------------------------------------------------*/
/**@{*/

struct sph_fitting_lm_data__ {
      size_t n;
      double* r;
      double* y;
      double* sigma_y;
}; //private

static const double SPH_FITTING_ROBUST_ACCURACY = 0.00000001;

_sph_fitting_fit_robust_ws_* sph_fitting_fit_robust_ws = NULL;

static
int
sph_fitting_lm_dero_f( const gsl_vector * x, void *sph_fitting_lm_data, gsl_vector * f ) //private
{
    {
      size_t n = ((struct  sph_fitting_lm_data__ *) sph_fitting_lm_data)->n;
      double *r = ((struct  sph_fitting_lm_data__ *) sph_fitting_lm_data)->r;
      double *y = ((struct  sph_fitting_lm_data__ *) sph_fitting_lm_data)->y;
      double *sigma_y = ((struct  sph_fitting_lm_data__ *)  sph_fitting_lm_data)->sigma_y;

      double a = gsl_vector_get (x, 0);
      double b = gsl_vector_get (x, 1);
      double c = gsl_vector_get (x, 2);
      double d = gsl_vector_get (x, 3);
      double e = gsl_vector_get (x, 4);

      size_t i;

      for (i = 0; i < n; i++)
        {
          /* Model Yi = a*cos(r) + b*sin(r) + c*cos(2*r) +d*sin(2*r) + e  */
          //double r = i*2*PI/n;
          double Yi = a*cos(r[i]) + b*sin(r[i]) + c*cos(2*r[i]) + d*sin(2*r[i]) + e;
          gsl_vector_set (f, i, (Yi - y[i])/sigma_y[i]);
        }

      return GSL_SUCCESS;
    }
}

static
int
sph_fitting_lm_dero_df( const gsl_vector * x, void *sph_fitting_lm_data, gsl_matrix * J ) //private
{
  size_t n = ((struct sph_fitting_lm_data__ *)sph_fitting_lm_data)->n;
  double* r = ((struct sph_fitting_lm_data__ *)sph_fitting_lm_data)->r;
  double* sigma_y = ((struct sph_fitting_lm_data__ *) sph_fitting_lm_data)->sigma_y;

  size_t i;

  for (i = 0; i < n; i++)
    {
      /* Jacobian matrix J(i,j) = dfi / dxj, */
      /* where fi = (Yi - yi)/sigma_y[i],      */
      /*       Yi = a*cos(r) + b*sin(r) + c*cos(2*r) +d*sin(2*r) + e   */
      /* and the xj are the parameters (a, b, c, d, e) */
      gsl_matrix_set (J, i, 0, cos(r[i])/sigma_y[i]);
      gsl_matrix_set (J, i, 1, sin(r[i])/sigma_y[i]);
      gsl_matrix_set (J, i, 2, cos(2*r[i])/sigma_y[i]);
      gsl_matrix_set (J, i, 3, sin(2*r[i])/sigma_y[i]);
      gsl_matrix_set (J, i, 4, 1/sigma_y[i]);
    }
  return GSL_SUCCESS;
}

static
int
sph_fitting_lm_dero_fdf( const gsl_vector * x, void *sph_fitting_lm_data,
                             gsl_vector * f, gsl_matrix * J ) //private
{
  sph_fitting_lm_dero_f (x, sph_fitting_lm_data, f);
  sph_fitting_lm_dero_df (x, sph_fitting_lm_data, J);

  return GSL_SUCCESS;
}

static
void
sph_fitting_lm_dero_print_state (size_t iter, gsl_multifit_fdfsolver * s)
{
   sph_error_raise(SPH_ERROR_INFO,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "iter: %3u x = % 15.8f % 15.8f % 15.8f "
          "|f(x)| = %g\n",
          (unsigned int)iter,
          gsl_vector_get (s->x, 0),
          gsl_vector_get (s->x, 1),
          gsl_vector_get (s->x, 2),
          gsl_blas_dnrm2 (s->f));
}


int sph_fitting_lm_dero( size_t n, double* r, double* y, double* sigma_y,
                     double* init, cpl_vector* fit, cpl_vector* fiterr ) //public
{
  const gsl_multifit_fdfsolver_type *T;
  gsl_multifit_fdfsolver *s;
  gsl_multifit_function_fdf f;

  int status;
  unsigned int i, iter = 0;
  //const size_t n = N;
  const size_t p = 5;
  double chi;
  double dof;
  double c;
  gsl_matrix *covar = gsl_matrix_alloc (p, p);

  /*
  for (int i = 0; i < n; i++) {
         printf ("data: %u %g %g %g\n", i, r[i], y[i], sigma_y[i]);
  }
  */
  struct sph_fitting_lm_data__ d = { n, r, y, sigma_y};

  gsl_vector_view x = gsl_vector_view_array( init, p );

  f.f = &sph_fitting_lm_dero_f;
  f.df = &sph_fitting_lm_dero_df;
  f.fdf = &sph_fitting_lm_dero_fdf;
  f.n = n;
  f.p = p;
  f.params = &d;

  T = gsl_multifit_fdfsolver_lmsder;
  s = gsl_multifit_fdfsolver_alloc (T, n, p);
  gsl_multifit_fdfsolver_set (s, &f, &x.vector);

  sph_fitting_lm_dero_print_state (iter, s);
  /* Starting iterations... */
  do
    {
      iter++;
      status = gsl_multifit_fdfsolver_iterate (s);

         sph_error_raise(SPH_ERROR_INFO,
              __FILE__, __func__, __LINE__,
              SPH_ERROR_INFO, "status = %s\n", gsl_strerror (status));
      sph_fitting_lm_dero_print_state(iter, s);

      if ( status ) break;

      status = gsl_multifit_test_delta (s->dx, s->x,
                                        1.0e-3, 1.0e-3);
    }
  while (status == GSL_CONTINUE && iter < 1000);

#if defined GSL_MAJOR_VERSION && GSL_MAJOR_VERSION >= 2
  {
      gsl_matrix * J = gsl_matrix_alloc(n, p);
      gsl_multifit_fdfsolver_jac(s, J);
      gsl_multifit_covar (J, 0.0, covar);
      gsl_matrix_free (J);
  }
#else
  gsl_multifit_covar (s->J, 0.0, covar);
#endif


#define FIT(i) gsl_vector_get(s->x, i)
#define ERR(i) sqrt(gsl_matrix_get(covar,i,i))


    chi = gsl_blas_dnrm2(s->f);
    dof = n - p;
    c = GSL_MAX_DBL(1, chi / sqrt(dof));

    sph_error_raise(SPH_ERROR_INFO,
            __FILE__, __func__, __LINE__,
            SPH_ERROR_INFO, "status = %s\n, iters = %d, status %d", gsl_strerror (status), iter, status );
    sph_error_raise(SPH_ERROR_INFO,
             __FILE__, __func__, __LINE__,
             SPH_ERROR_INFO, "chisq/dof = %g\n",  pow(chi, 2.0) / dof);
    sph_error_raise(SPH_ERROR_INFO,
             __FILE__, __func__, __LINE__,
             SPH_ERROR_INFO, "a      = %.5f +/- %.5f\n"
                             "b      = %.5f +/- %.5f\n"
                             "c      = %.5f +/- %.5f\n"
                             "d      = %.5f +/- %.5f\n"
                            "e      = %.5f +/- %.5f\n",
                             FIT(0), c*ERR(0),
                             FIT(1), c*ERR(1),
                            FIT(2), c*ERR(2),
                            FIT(3), c*ERR(3),
                            FIT(4), c*ERR(4));


  // put the results in the output vectors
  //fit = cpl_vector_new( p + 1 );
  //fiterr = cpl_vector_new( p+ 1 );
  for ( i = 0; i < p; i++ ){
      cpl_vector_set( fit, i, FIT(i) );
      cpl_vector_set( fiterr, i, c*ERR(i) );
  }
  cpl_vector_set( fit, p, dof );
  cpl_vector_set( fiterr, p,  chi );

  gsl_multifit_fdfsolver_free (s);
  gsl_matrix_free (covar);
  if ( status == GSL_ETOLX ) {
    return GSL_SUCCESS;
  }
  return status;
}


static
double
sph_fitting_fit_robust_internal_median( double* ydata, int n) {
    double median = 0.0;
    gsl_sort(ydata,1,n);
    median = gsl_stats_median_from_sorted_data(ydata,1,n);
    return median;
}

static
double
sph_fitting_fit_robust_internal_func(double b, void* p) {
    struct sph_fitting_fit_robust_params__* params = (struct sph_fitting_fit_robust_params__*)p;
    int     ii  = 0;
    double      median = 0.0;
    double      y_minus_bx[params->n];
    double      y_minus_bxc[params->n];
    double      sum = 0.0;
    double      yy = 0.0;
    for (ii = 0; ii < params->n; ++ii) {
        y_minus_bx[ii] = params->ydata[ii] - b * params->xdata[ii];
        y_minus_bxc[ii] = y_minus_bx[ii];
    }

    median = sph_fitting_fit_robust_internal_median(y_minus_bxc,params->n);

    for (ii = 0; ii < params->n; ++ii) {
        yy = y_minus_bx[ii] - median;
        if ( yy*yy > SPH_FITTING_ROBUST_ACCURACY ) {
            sum = sum + params->xdata[ii] * copysign(1.0,yy);
        }
    }
    params->median = median;
    return sum;
}
static
void
sph_fitting_fit_robust_init( int n ) {
    const gsl_root_fsolver_type *T;

    sph_fitting_fit_robust_ws = cpl_calloc(1,sizeof(_sph_fitting_fit_robust_ws_));

    sph_fitting_fit_robust_ws->F.function = &sph_fitting_fit_robust_internal_func;
    sph_fitting_fit_robust_ws->F.params = &sph_fitting_fit_robust_ws->params;
    T = gsl_root_fsolver_brent;
    sph_fitting_fit_robust_ws->s = gsl_root_fsolver_alloc (T);
    sph_fitting_fit_robust_ws->params.n = n;
    sph_fitting_fit_robust_ws->params.xdata = cpl_calloc(n,sizeof(double));
    sph_fitting_fit_robust_ws->params.ydata = cpl_calloc(n,sizeof(double));
    sph_fitting_fit_robust_ws->params.median = 0.0;
}

static
void
sph_fitting_fit_robust_free(void) {
    cpl_free(sph_fitting_fit_robust_ws->params.xdata);
    cpl_free(sph_fitting_fit_robust_ws->params.ydata);
    gsl_root_fsolver_free(sph_fitting_fit_robust_ws->s);
    cpl_free(sph_fitting_fit_robust_ws);
    sph_fitting_fit_robust_ws = NULL;
}

static
double
sph_fitting_fit_robust_internal(
        double* xdata,
        double* ydata,
        int n,
        double x_low,
        double x_high,
        double* offset) {
    int status;
    int iter = 0, max_iter = 100;
    double r = 0;
    int ii = 0;
    double x_lo = x_low;
    double x_hi = x_high;
    gsl_error_handler_t*        gslerrhandler = NULL;
    if ( sph_fitting_fit_robust_ws == NULL ) {
        sph_fitting_fit_robust_init(n);
    }
    cpl_ensure(sph_fitting_fit_robust_ws, CPL_ERROR_NULL_INPUT,-1.0);
    for (ii = 0; ii < n; ++ii) {
        sph_fitting_fit_robust_ws->params.xdata[ii] = xdata[ii];
        sph_fitting_fit_robust_ws->params.ydata[ii] = ydata[ii];
    }
    //printf("Trying to fit between %f and %f:", x_lo, x_hi);


    gslerrhandler = gsl_set_error_handler_off();
    gsl_root_fsolver_set (sph_fitting_fit_robust_ws->s,
            &sph_fitting_fit_robust_ws->F, x_lo, x_hi);


    do
    {
        iter++;
        status = gsl_root_fsolver_iterate (sph_fitting_fit_robust_ws->s);
        r = gsl_root_fsolver_root (sph_fitting_fit_robust_ws->s);
        x_lo = gsl_root_fsolver_x_lower (sph_fitting_fit_robust_ws->s);
        x_hi = gsl_root_fsolver_x_upper (sph_fitting_fit_robust_ws->s);
        status = gsl_root_test_interval (x_lo, x_hi,
                0, 0.001);

    }
    while (status == GSL_CONTINUE && iter < max_iter);

    sph_fitting_fit_robust_ws->errest = x_hi - x_lo;
    if ( offset ) *offset = sph_fitting_fit_robust_ws->params.median;
    gsl_set_error_handler(gslerrhandler);
    return r;
}

double
sph_fitting_fit_robust(
        cpl_vector* xvals,
        cpl_vector* yvals,
        double x_lo,
        double x_hi,
        double* offset)
{
    double*     ydata = cpl_vector_get_data(yvals);
    double*     xdata = cpl_vector_get_data(xvals);
    double        result = 0.0;
    cpl_ensure(ydata,CPL_ERROR_ILLEGAL_INPUT,-1.0);
    cpl_ensure(xdata,CPL_ERROR_ILLEGAL_INPUT,-1.0);

    result = sph_fitting_fit_robust_internal(
            xdata,ydata,
            cpl_vector_get_size(xvals),
            x_lo,
            x_hi,
            offset);
    sph_fitting_fit_robust_free();
    return result;
}
cpl_image*
sph_fitting_fit_robust_imlist(
        cpl_imagelist* imlist,
        cpl_vector* xvals,
        double x_lo,
        double x_hi,
        cpl_image** offset,
        cpl_image* errimage) {
    double*     xdata0 = cpl_vector_get_data(xvals);
    double*     ydata0 = NULL;
    double*     ydata = NULL;
    double*     xdata = NULL;
    cpl_image*    result = NULL;
    double        lin = 0.0;
    double        offsetd = 0.0;
    int            xx        = 0;
    int            yy        = 0;
    int            nx        = 0;
    int            ny        = 0;
    int            bpix    = 0;
    int            cc    = 0;
    int            ii     = 0;


    cpl_ensure(xvals,CPL_ERROR_ILLEGAL_INPUT,NULL);
    cpl_ensure(cpl_vector_get_size(xvals) == cpl_imagelist_get_size(imlist),
            CPL_ERROR_ILLEGAL_INPUT,NULL);
    ydata0 = cpl_calloc(cpl_vector_get_size(xvals),sizeof(double));
    xdata = cpl_calloc(cpl_vector_get_size(xvals),sizeof(double));
    ydata = cpl_calloc(cpl_vector_get_size(xvals),sizeof(double));
    nx = cpl_image_get_size_x(cpl_imagelist_get_const(imlist,0));
    ny = cpl_image_get_size_y(cpl_imagelist_get_const(imlist,0));
    result = cpl_image_new(nx,ny,CPL_TYPE_DOUBLE);
    cpl_ensure(result,CPL_ERROR_ILLEGAL_OUTPUT,NULL);
    if ( offset ) {
        *offset = cpl_image_new(nx,ny,CPL_TYPE_DOUBLE);
        cpl_ensure(*offset,CPL_ERROR_ILLEGAL_OUTPUT,NULL);
    }
        for (yy = 0; yy < ny; ++yy) {
            for (xx = 0; xx < nx; ++xx) {
            cc = 0;
            for (ii = 0; ii < cpl_imagelist_get_size(imlist); ++ii) {
                ydata0[ii] = cpl_image_get(cpl_imagelist_get_const(imlist,ii),xx+1,yy+1,&bpix);
                if ( bpix == 0 ) {
                    xdata[cc] = xdata0[ii];
                    ydata[cc] = ydata0[ii];
                    cc++;
                }
            }
            lin = sph_fitting_fit_robust_internal(xdata,ydata,cc,x_lo,x_hi,&offsetd);
            cpl_image_set(result,xx+1,yy+1,lin);
            if ( offset ) {
                cpl_image_set(*offset,xx+1,yy+1,offsetd);
            }
            if ( errimage ) {
                cpl_image_set(errimage,xx+1,yy+1,sph_fitting_fit_robust_ws->errest);
            }
        }
    }
    sph_fitting_fit_robust_free();
    cpl_free(xdata);
    cpl_free(ydata);
    cpl_free(ydata0);
    return result;
}

cpl_polynomial*
sph_fitting_fit_poly1d( cpl_vector* xvals,
        cpl_vector* yvals,
        cpl_vector* err,
        int mindeg,
        int maxdeg,
        int allowshift, double guess, double* chisq) {
    cpl_matrix* samppos = cpl_matrix_new(1,cpl_vector_get_size(xvals));
    cpl_size maxd = maxdeg;
    cpl_size mind = mindeg;
    double minval = 0.0;
    int cc;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_polynomial* self = cpl_polynomial_new(1);
    cpl_polynomial* polyd = NULL;
    cpl_vector* v = NULL;
    for (cc = 0; cc < cpl_vector_get_size(xvals); ++cc) {
        cpl_matrix_set(samppos,0,cc,cpl_vector_get(xvals,cc));
    }
    rerr = cpl_polynomial_fit(self, samppos, NULL, yvals, NULL, CPL_FALSE, &mind, &maxd);
    if ( rerr != CPL_ERROR_NONE ) {
        cpl_polynomial_delete(self); self = NULL;
        cpl_matrix_delete(samppos);
        return NULL;
    }
    if ( chisq ) {
        v = cpl_vector_new(cpl_vector_get_size(yvals));
        cpl_vector_fill_polynomial_fit_residual(v,yvals,err,self,samppos,chisq);
        cpl_vector_delete(v); v = NULL;
    }

    if ( allowshift ) {
        polyd = cpl_polynomial_duplicate(self);
        rerr = cpl_polynomial_derivative(polyd,0);
        if ( rerr != CPL_ERROR_NONE ) {
            sph_error_raise( rerr,
                    __FILE__,
                    __func__,
                    __LINE__,SPH_ERROR_ERROR,
                    "Could not create derivative.");
            cpl_polynomial_delete(self); self = NULL;
            cpl_polynomial_delete(polyd); polyd = NULL;
            cpl_matrix_delete(samppos);
            return NULL;
        }
        rerr = cpl_polynomial_solve_1d( polyd,guess,&minval,1);
        if ( rerr != CPL_ERROR_NONE ) {
            sph_error_raise( rerr,
                    __FILE__,
                    __func__,
                    __LINE__,SPH_ERROR_ERROR,
                    "Could not solve to find minimum.");
            cpl_polynomial_delete(self); self = NULL;
            cpl_polynomial_delete(polyd); polyd = NULL;
            cpl_matrix_delete(samppos);
            return NULL;
        }
        rerr = cpl_polynomial_shift_1d( self,0,minval);
        if ( rerr != CPL_ERROR_NONE ) {
            sph_error_raise( rerr,
                    __FILE__,
                    __func__,
                    __LINE__,SPH_ERROR_ERROR,
                    "Could not shift to minimum.");
            cpl_polynomial_delete(self); self = NULL;
            cpl_polynomial_delete(polyd); polyd = NULL;
            cpl_matrix_delete(samppos);
            return NULL;
        }
        cpl_polynomial_delete(polyd); polyd = NULL;
    }
    cpl_matrix_delete(samppos);
    return self;

}

cpl_vector*
sph_fitting_estimate_error( cpl_polynomial* self,
        cpl_vector* xvals,
        cpl_vector* yvals,
        cpl_vector* err,
        cpl_vector** means_o)
{
    int             ntries  = 10;
    gsl_rng*        pRNG    = NULL;
    int             ii       = 0 ;
    int             jj       = 0 ;
    cpl_vector*     duplicat     = NULL;
    double          val = 0.0;
    double          sigma = 0.0;
    cpl_polynomial* polynew  = NULL;
    cpl_image*     coeffim  = NULL;
    cpl_vector*     means  = NULL;
    cpl_vector*     errs  = NULL;
    cpl_size        pows = 0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(xvals,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(yvals,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(err,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(cpl_polynomial_get_dimension(self) == 1,
            CPL_ERROR_ILLEGAL_INPUT,NULL);
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    coeffim = cpl_image_new(ntries,
            cpl_polynomial_get_degree(self) + 1,CPL_TYPE_DOUBLE);
    means = cpl_vector_new(cpl_polynomial_get_degree(self) + 1);
    errs = cpl_vector_new(cpl_polynomial_get_degree(self) + 1);
    cpl_vector_fill(means,0.0);
    cpl_vector_fill(errs,0.0);
    for (ii = 0; ii < ntries; ++ii) {
        duplicat = cpl_vector_duplicate(yvals);
        for (jj = 0; jj < cpl_vector_get_size(duplicat); ++jj) {
            val = cpl_vector_get(yvals,jj);
            sigma = cpl_vector_get(err,jj);
            cpl_vector_set(duplicat,jj,val + gsl_ran_gaussian(pRNG,sigma));
        }
        polynew = sph_fitting_fit_poly1d(xvals,duplicat,err,0,
                cpl_polynomial_get_degree(self),0,0.0,NULL);
        for (jj = 0; jj < cpl_polynomial_get_degree(polynew) + 1; ++jj) {
            pows = jj;
            val = cpl_polynomial_get_coeff(polynew,&pows);
            cpl_image_set(coeffim,ii+1,jj+1,val);
        }
        cpl_vector_delete(duplicat); duplicat = NULL;
        cpl_polynomial_delete(polynew); polynew = NULL;
    }

    for (jj = 0; jj < cpl_polynomial_get_degree(self) + 1; ++jj) {
        duplicat = cpl_vector_new_from_image_row(coeffim,jj + 1);
        val = cpl_vector_get_mean(duplicat);
        sigma = cpl_vector_get_stdev(duplicat);
        cpl_vector_set(means,jj,val);
        cpl_vector_set(errs,jj,sigma);
        cpl_vector_delete(duplicat); duplicat = NULL;
    }

    if ( means_o ) *means_o = means;
    else {
        cpl_vector_delete(means); means = NULL;
    }
    cpl_image_delete(coeffim);
    gsl_rng_free(pRNG);
    return errs;
}

cpl_polynomial*
sph_fitting_fit_poly2d(const cpl_vector* xvals, const cpl_vector* yvals,
                       const cpl_vector* zvals, double* chisq,
                       int maxdeg1, int maxdeg2)
{
    const cpl_size nsamp = cpl_vector_get_size(xvals);
    const cpl_size mind[2] = {0,       0};
    const cpl_size maxd[2] = {maxdeg1, maxdeg2};
    cpl_error_code rerr;
    cpl_polynomial* self;
    cpl_matrix* samppos;
    double* dsamppos;
    const double* dvals;

    cpl_ensure(cpl_vector_get_size(yvals) == nsamp,
               CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
    cpl_ensure(cpl_vector_get_size(zvals) == nsamp,
               CPL_ERROR_INCOMPATIBLE_INPUT, NULL);

    dsamppos = (double*)cpl_malloc(2 * (size_t)nsamp * sizeof(*dsamppos));

    dvals = cpl_vector_get_data_const(xvals);
    (void)memcpy(dsamppos, dvals, (size_t)nsamp * sizeof(*dsamppos));
    dvals = cpl_vector_get_data_const(yvals);
    (void)memcpy(dsamppos + (size_t)nsamp, dvals,
                 (size_t)nsamp * sizeof(*dsamppos));

    samppos = cpl_matrix_wrap(2, nsamp, dsamppos);

    self = cpl_polynomial_new(2);
    rerr = cpl_polynomial_fit(self, samppos, NULL, zvals, NULL, CPL_FALSE,
                              mind, maxd);

    if (rerr != CPL_ERROR_NONE) {
        (void)cpl_error_set_where(cpl_func);
        cpl_polynomial_delete(self);
        self = NULL;
    } else if (chisq != NULL) {
        cpl_vector* v = cpl_vector_new(nsamp);
        cpl_vector_fill_polynomial_fit_residual(v, zvals, NULL, self, samppos,
                                                chisq);
        cpl_vector_delete(v);
    }

    cpl_matrix_delete(samppos);
    return self;
}

cpl_error_code sph_fitting_fit_poly2d_1(cpl_polynomial* self,
                                        const cpl_matrix* xypos,
                                        const cpl_vector* zvals,
                                        double* pchisq,
                                        cpl_size mindeg,
                                        cpl_size maxdeg)
{
    const cpl_size nsamp = cpl_vector_get_size(zvals);
    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_error_code code;

    code = cpl_polynomial_fit(self, xypos, NULL, zvals, NULL, CPL_FALSE,
                              &mindeg, &maxdeg);

    if (code == CPL_ERROR_UNSUPPORTED_MODE && mindeg > 0) {
        cpl_msg_warning(cpl_func, "CPL 2D-polynomial fit with zero-valued "
                        "constant-term not supported, retry without");
        cpl_errorstate_dump(prestate, CPL_FALSE,
                            cpl_errorstate_dump_one_warning);
        cpl_errorstate_set(prestate);

        code = cpl_polynomial_fit(self, xypos, NULL, zvals, NULL, CPL_FALSE,
                                  NULL, &maxdeg);
    }

    if (code != CPL_ERROR_NONE) {
        return cpl_error_set_where(cpl_func);
    } else if (pchisq != NULL) {
        cpl_vector* v = cpl_vector_new(nsamp);
        code = cpl_vector_fill_polynomial_fit_residual(v, zvals, NULL, self,
                                                       xypos, pchisq);
        cpl_vector_delete(v);
        if (code != CPL_ERROR_NONE) {
            return cpl_error_set_where(cpl_func);
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Fit a 2D gaussian to an image
 * @param im            the image containing the gauss
 * @param imerr         the error image
 * @param xpos          the initial guess and output xposiion (CPL/FITS coords)
 * @param ypos          the initial guess and output yposiion (CPL/FITS coords)
 * @param xwidth        the initial guess and output width in x
 * @param ywidth        the initial guess and output width in y
 * @note If the underlying fit succeeds but is rejected, then failure is
         returned without setting an error code
 * @return CPL_ERROR_NONE on success or the relevant CPL error code
 * @see sph_fitting_fit_gauss2D_all()
 *
 */
/*----------------------------------------------------------------------------*/
int sph_fitting_fit_gauss2D(const cpl_image* im,
                            const cpl_image* im_err,
                            double * pxpos, double* pypos,
                            double *pxwidth, double * pywidth,
                            double boxfactor,
                            double boxborder)
{
    const cpl_size nx  = cpl_image_get_size_x(im);
    const cpl_size ny  = cpl_image_get_size_y(im);
    cpl_array    * out_pars;
    cpl_array    * fit_pars;
    cpl_error_code code;

    cpl_ensure_code(im       != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pxpos    != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pypos    != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pxwidth  != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pywidth  != NULL, CPL_ERROR_NULL_INPUT);

    cpl_ensure_code(*pxpos   >= 1.0,  CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(*pypos   >= 1.0,  CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(*pxpos   <= nx,   CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(*pypos   <= ny,   CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(*pxwidth >  0.0,  CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(*pywidth >  0.0,  CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(*pxwidth <= nx,   CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(*pywidth <= ny,   CPL_ERROR_ILLEGAL_INPUT);


    /* Elements not set are flagged as invalid and not fitted */
    fit_pars = cpl_array_new(7, CPL_TYPE_INT);
    out_pars = cpl_array_new(7, CPL_TYPE_DOUBLE);
    cpl_array_set(fit_pars, 2, 0);    /* Correlation is frozen */
    cpl_array_set(out_pars, 2, 0.0);  /* - at zero */
    cpl_array_set(out_pars, 3, *pxpos);
    cpl_array_set(out_pars, 4, *pypos);
    cpl_array_set(out_pars, 5, *pxwidth);
    cpl_array_set(out_pars, 6, *pywidth);

    code = sph_fitting_fit_gauss2D_all(im, im_err, out_pars, fit_pars,
                                       boxfactor, boxborder);

    if (code) {
        (void)cpl_error_set_where(cpl_func);
    } else {
        *pxpos   = cpl_array_get_double(out_pars, 3, NULL);
        *pypos   = cpl_array_get_double(out_pars, 4, NULL);
        *pxwidth = cpl_array_get_double(out_pars, 5, NULL);
        *pywidth = cpl_array_get_double(out_pars, 6, NULL);
    }

    cpl_array_delete(out_pars);
    cpl_array_delete(fit_pars);

    /* code may be set without having set a CPL error */
    return code;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Fit a 2D gaussian to an image
 * @param im       the image containing the gauss
 * @param imerr    the error image
 * @param ar_fit   The array of (seven) 2D Gaussian parameters to fit
 * @param do_fit   The matching array of flags, whether to fit
 * @note If the underlying fit succeeds but is rejected, then failure is
         returned without setting an error code
 * @return CPL_ERROR_NONE on success or the relevant CPL error code
 *
 * The returned gauss centre is in CPL (FITS) pixel coordinates.
 *
 * Note: This function does not work well for any gauss of FWHM < 2 pixels.
 *

 */
/*----------------------------------------------------------------------------*/
int sph_fitting_fit_gauss2D_all(const cpl_image* im,
                                const cpl_image* im_err,
                                cpl_array * ar_fit,
                                const cpl_array * do_fit,
                                double boxfactor,
                                double boxborder)
{
    const double   factorlim  =  5.0;
    const double   factor2lim = 10.0;
    cpl_error_code code = CPL_ERROR_ILLEGAL_OUTPUT; /* Failure has most modes */
    double         xpos, ypos, xwidth, ywidth;
    const double *ar_fitd = cpl_array_get_data_double_const(ar_fit);
    const double *pxpos   = ar_fitd + 3;
    const double *pypos   = ar_fitd + 4;
    const double *pxwidth = ar_fitd + 5;
    const double *pywidth = ar_fitd + 6;

    cpl_ensure_code(im       != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ar_fit   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(do_fit   != NULL, CPL_ERROR_NULL_INPUT);

    /* Copy input in case an error message must be generated */
    xpos   = *pxpos;
    ypos   = *pypos;
    xwidth = *pxwidth;
    ywidth = *pywidth;

    if (cpl_fit_image_gaussian(im, im_err, xpos, ypos,
                               (int)(xwidth) * boxfactor + boxborder,
                               (int)(ywidth) * boxfactor + boxborder,
                               ar_fit, NULL, do_fit, NULL, NULL,
                               NULL, NULL, NULL, NULL, NULL)) {
        code = cpl_error_set_where(cpl_func);
    } else {
        int invalid = 0;
        const double norm = cpl_array_get_double(ar_fit, 1, &invalid);

        if (invalid ||
            !cpl_array_is_valid(ar_fit, 3) ||
            !cpl_array_is_valid(ar_fit, 4) ||
            !cpl_array_is_valid(ar_fit, 5) ||
            !cpl_array_is_valid(ar_fit, 6)) {
            cpl_msg_warning(cpl_func, "Could not fit gaussian at (%g, %g) +/- "
                            "(%g, %g):", xpos, ypos, xwidth, ywidth);
        } else if (fabs(*pxpos - xpos) > xwidth * factorlim ||
                   fabs(*pypos - ypos) > ywidth * factorlim) {

            /* The above and below checks are really to cover an inadequacy in
               cpl_fit_image_gaussian():
               when the gauss is small (FWHM < 3 pixels) then the fitting
               routine does not produce an error but gives extremely wrong
               answers.
               This *should* catch these cases and turn them into an error.
            */

            cpl_msg_warning(cpl_func, "Discarding gaussian fit with non-"
                            "sensical location: (%g,%g) +/- (%g,%g) -> "
                            "(%g, %g) +/- (%g, %g)", xpos, ypos,
                            xwidth, ywidth, *pxpos, *pypos, *pxwidth, *pywidth);
        } else if (*pxwidth <= 0.0 || *pywidth <= 0.0) {

            cpl_msg_warning(cpl_func, "Discarding gaussian fit with negative"
                            "width: (%g,%g) +/- (%g,%g) -> "
                            "(%g, %g) +/- (%g, %g)", xpos, ypos,
                            xwidth, ywidth, *pxpos, *pypos, *pxwidth, *pywidth);
        } else if (xwidth   > *pxwidth * factor2lim ||
                   *pxwidth > xwidth   * factor2lim ||
                   ywidth   > *pywidth * factor2lim ||
                   *pywidth > ywidth   * factor2lim) {
            cpl_msg_warning(cpl_func, "Discarding gaussian fit with extreme "
                            "(%g) aspect ratio change: (%g,%g) +/- (%g,%g) -> "
                            "(%g, %g) +/- (%g, %g)", factor2lim, xpos, ypos,
                            xwidth, ywidth, *pxpos, *pypos, *pxwidth, *pywidth);
        } else if (norm <= 0.0) {
            cpl_msg_warning(cpl_func, "Discarding gaussian fit with non-"
                            "positive norm at (%g,%g) +/- (%g,%g): %g",
                            xpos, ypos, xwidth, ywidth, norm);
        } else {
            code = CPL_ERROR_NONE;
        }
    }

    return code;
}

/**@}*/
