/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include "sph_lenslet_descriptor.h"
#include "sph_error.h"
#include <math.h>
static
void
sph_lenslet_descriptor_fill_double( double* d, cpl_vector* vec ) {
    int        ii = 0;
    for (ii = 0; ii < cpl_vector_get_size(vec); ++ii) {
        d[ii] = cpl_vector_get(vec,ii);
    }
}
static
void
sph_lenslet_descriptor_fill_int( int* i, cpl_vector* vec ) {
    int        ii = 0;
    for (ii = 0; ii < cpl_vector_get_size(vec); ++ii) {
        i[ii] = cpl_vector_get(vec,ii);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_lenslet_descriptor Lenslet Descriptor Object
 *
 * @par Synopsis:
 * @code
 * typedef struct _sph_lenslet_descriptor
 * {
 *    short                   u;  // pixel u
 *    short                   v;  // pixel v
 *    short                   lensid; // lenslet id
 *    cpl_vector*             lambdas; //wavelength values
 *    double*                   values; //spectrum-values
 * } sph_lenslet_descriptor
 * @endcode
 * @par Descirption:
 * This module provides functions to define and operate on a lenslet
 * in the IFS wavelength calibration lenslet description table (LDT).
 *
 * In the present design, this module only contains a structure that is
 * directly accessed in recipes and other API functions.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
 Function definitions
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new lenslet descriptor thats empty.
 *
 * @param u         The u position (in hexagonal coordinate system)
 * @param v         The v position (in hexagonal coordinate system)
 * @param length    The length of the spectrum (number of points)
 * @note The allocated memory pointed by *lambdas will now have shared ownership
 * 
 * @return Pointer to new descriptor or null in case of error.
 *
 * A new lenslet descriptor table is created with length wavelength points.
 *
 */
/*----------------------------------------------------------------------------*/
sph_lenslet_descriptor*
sph_lenslet_descriptor_new( int id, int u, int v, cpl_vector* lambdas ) {
    sph_lenslet_descriptor*          self = NULL;

    if ( !lambdas ) return NULL;
    self = cpl_calloc( sizeof(sph_lenslet_descriptor),1);
    self->lambdas = lambdas;
    self->values = cpl_calloc(sizeof(double),cpl_vector_get_size(lambdas));
    self->bpix = cpl_calloc(sizeof(int),cpl_vector_get_size(lambdas));
    self->rms = cpl_calloc(sizeof(double),cpl_vector_get_size(lambdas));
    self->ncomb = cpl_calloc(sizeof(double),cpl_vector_get_size(lambdas));
    self->u = u;
    self->v = v;
    self->lensid = id;
    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return the u and v coordinates of the lenslet
 * @param self
 * @param u (out)
 * @param v (out)
 * @return error code
 *
 * Returns the u and v integer coordinates.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_lenslet_descriptor_get_uv(sph_lenslet_descriptor* self,int* u, int* v) {
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(u,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(v,CPL_ERROR_NULL_INPUT);
    *u = (int)self->u;
    *v = (int)self->v;
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Fill values from a vector
 * @param self the lenslet descriptor
 * @param vec the vector to use for values
 *
 * @return error code
 *
 * This function fills the descriptor with values from the
 * given vector. The type of the vector must be double and
 * the length must be the same as that of the wavelength
 * vector of the lenslet descriptor.
 * Error codes:
 * CPL_ERROR_INCOMPATIBLE_INPUT if vec has wrong size
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_lenslet_descriptor_fill_values( sph_lenslet_descriptor* self, cpl_vector* vec ) {
    if ( !self || !vec ) return CPL_ERROR_NULL_INPUT;
    if ( cpl_vector_get_size(vec) != cpl_vector_get_size(self->lambdas) ) {
        return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
    sph_lenslet_descriptor_fill_double( self->values, vec);
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Fill bpix from a vector
 * @param self the lenslet descriptor
 * @param vec the vector to use for values
 *
 * @return error code
 *
 * This function fills the descriptor with values from the
 * given vector. The type of the vector must be double and
 * the length must be the same as that of the wavelength
 * vector of the lenslet descriptor.
 * Error codes:
 * CPL_ERROR_INCOMPATIBLE_INPUT if vec has wrong size
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_lenslet_descriptor_fill_bpix( sph_lenslet_descriptor* self, cpl_vector* vec ) {
    if ( !self || !vec ) return CPL_ERROR_NULL_INPUT;
    if ( cpl_vector_get_size(vec) != cpl_vector_get_size(self->lambdas) ) {
        return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
    sph_lenslet_descriptor_fill_int( self->bpix, vec);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the whole lenslet as bad
 * @param self the lenslet
 *
 * @return error code
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_lenslet_descriptor_set_bad( sph_lenslet_descriptor* self ) {
    int            ii    = 0;
    for (ii = 0; ii < cpl_vector_get_size(self->lambdas); ++ii) {
        self->bpix[ii] = 1;
        self->values[ii] = 0.0;
        self->ncomb[ii] = 0.0;
        self->rms[ii] = 0.0;
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Fill rms from a vector
 * @param self the lenslet descriptor
 * @param vec the vector to use for values
 *
 * @return error code
 *
 * This function fills the descriptor with values from the
 * given vector. The type of the vector must be double and
 * the length must be the same as that of the wavelength
 * vector of the lenslet descriptor.
 * Error codes:
 * CPL_ERROR_INCOMPATIBLE_INPUT if vec has wrong size
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_lenslet_descriptor_fill_rms( sph_lenslet_descriptor* self, cpl_vector* vec ) {
    if ( !self || !vec ) return CPL_ERROR_NULL_INPUT;
    if ( cpl_vector_get_size(vec) != cpl_vector_get_size(self->lambdas) ) {
        return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
    sph_lenslet_descriptor_fill_double( self->rms, vec);
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Fill ncomb from a vector
 * @param self the lenslet descriptor
 * @param vec the vector to use for values
 *
 * @return error code
 *
 * This function fills the descriptor with values from the
 * given vector. The type of the vector must be double and
 * the length must be the same as that of the wavelength
 * vector of the lenslet descriptor.
 * Error codes:
 * CPL_ERROR_INCOMPATIBLE_INPUT if vec has wrong size
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_lenslet_descriptor_fill_ncomb( sph_lenslet_descriptor* self, cpl_vector* vec ) {
    if ( !self || !vec ) return CPL_ERROR_NULL_INPUT;
    if ( cpl_vector_get_size(vec) != cpl_vector_get_size(self->lambdas) ) {
        return CPL_ERROR_INCOMPATIBLE_INPUT;
    }
    sph_lenslet_descriptor_fill_double( self->ncomb, vec);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Fill data from a spectrum
 * @param self the lenslet descriptor
 * @param spec the spectrum to use
 *
 * @return error code
 *
 * This function fills the descriptor with values from the
 * given sph_spectrum.
 * Error codes:
 * CPL_ERROR_INCOMPATIBLE_INPUT if spec has wrong size
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_lenslet_descriptor_fill_spec( sph_lenslet_descriptor* self, sph_spectrum* spec ) {
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_spectrum*        nspec = NULL;
#ifdef SPH_LDT_REMOVE_STRIPES
    double                minlam = 0.0;
    double                maxlam = 0.0;
#endif

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE
    if ( !self || !spec ) return CPL_ERROR_NULL_INPUT;
#ifdef SPH_LDT_REMOVE_STRIPES
    minlam = cpl_vector_get_min( spec->wavelengths );
    maxlam = cpl_vector_get_max( spec->wavelengths );
#endif    
    nspec = sph_spectrum_new_interpolate(spec,self->lambdas);
    
    if ( !nspec ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    if ( nspec && (rerr = sph_lenslet_descriptor_fill_values( self, nspec->spec))
            == CPL_ERROR_NONE ) {
        if ( (rerr = sph_lenslet_descriptor_fill_bpix( self, nspec->bads))
                == CPL_ERROR_NONE ) {
            if ( (rerr = sph_lenslet_descriptor_fill_ncomb( self, nspec->ncomb))
                    == CPL_ERROR_NONE ) {
                rerr = sph_lenslet_descriptor_fill_rms( self, nspec->rms);
            }
        }
    }
    // The code below was uncommented to remove funny stripes
    // in the LDTs. Not sure why this code was there in the
    // first place. Remove completely later if other tests pass ok.
#ifdef SPH_LDT_REMOVE_STRIPES
    for (n = 0; n < cpl_vector_get_size(self->lambdas); ++n) {
        if ( cpl_vector_get(self->lambdas,n) < minlam || cpl_vector_get(self->lambdas,n) > maxlam ) {
            self->bpix[n] = 1;
            self->values[n] = 0.0;
            self->rms[n] = 0.0;
            self->ncomb[n] = 0.0;
        }
    }
#endif
    if ( rerr != CPL_ERROR_NONE ) sph_lenslet_descriptor_set_bad(self);
    if ( nspec ) sph_spectrum_delete(nspec);
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Add another lenslet descriptors
 * @param self   the descriptor to add to
 * @param add      the descriptor to add
 *
 * @return error code
 *
 * This function implicitly assumes (but does not check) that the
 * reference wavelengths for both descriptors are the same.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_lenslet_descriptor_add( sph_lenslet_descriptor* self,
        sph_lenslet_descriptor* add) {
    int ii = 0;

    if ( !self || !add ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    for (ii = 0; ii < cpl_vector_get_size(self->lambdas); ++ii) {
        self->bpix[ii] += add->bpix[ii];
        self->values[ii] += add->values[ii];
        self->rms[ii] = ( self->rms[ii] * self->ncomb[ii] ) *
                ( self->rms[ii] * self->ncomb[ii] ) +
                ( add->rms[ii] * add->ncomb[ii] ) *
                ( add->rms[ii] * add->ncomb[ii] );
        self->ncomb[ii] += add->ncomb[ii];
        self->rms[ii] = sqrt( self->rms[ii] ) / self->ncomb[ii];
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Subtract another lenslet descriptor
 * @param self   the descriptor to subtract from
 * @param add      the descriptor to subtract
 *
 * @return error code
 *
 * This function implicitly assumes (but does not check) that the
 * reference wavelengths for both descriptors are the same.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_lenslet_descriptor_subtract( sph_lenslet_descriptor* self,
        sph_lenslet_descriptor* add) {
    int ii = 0;

    if ( !self || !add ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    for (ii = 0; ii < cpl_vector_get_size(self->lambdas); ++ii) {
        self->bpix[ii] += add->bpix[ii];
        self->values[ii] -= add->values[ii];
        self->rms[ii] = ( self->rms[ii] * self->ncomb[ii] ) *
                ( self->rms[ii] * self->ncomb[ii] ) +
                ( add->rms[ii] * add->ncomb[ii] ) *
                ( add->rms[ii] * add->ncomb[ii] );
        self->ncomb[ii] += add->ncomb[ii];
        self->rms[ii] = sqrt( self->rms[ii] ) / self->ncomb[ii];
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Divide lenslet descriptor by scalar
 * @param self   the descriptor to divide
 * @param value      the value to divide by
 *
 * @return error code
 *
 * The value field of the lenslet descriptor is divided by value,
 * the rms is also modified accordingly.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_lenslet_descriptor_divide_scalar( sph_lenslet_descriptor* self,
        double value) {
    int ii = 0;

    if ( !self || value == 0 ) {
        return CPL_ERROR_NULL_INPUT;
    }
    for (ii = 0; ii < cpl_vector_get_size(self->lambdas); ++ii) {
        self->values[ii] /= value;
        self->rms[ii] /= value;
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Divide another lenslet descriptor
 * @param self   the descriptor to divide
 * @param add      the descriptor to divide by
 *
 * @return error code
 *
 * This function implicitly assumes (but does not check) that the
 * reference wavelengths for both descriptors are the same.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_lenslet_descriptor_divide( sph_lenslet_descriptor* self,
        sph_lenslet_descriptor* add) {
    int ii = 0;

    if ( !self || !add ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    for (ii = 0; ii < cpl_vector_get_size(self->lambdas); ++ii) {
        self->bpix[ii] += add->bpix[ii];
        if ( add->values[ii] != 0 ) {
            self->values[ii] /= add->values[ii];
            self->rms[ii] = ( self->rms[ii] * self->ncomb[ii] ) *
                    ( self->rms[ii] * self->ncomb[ii] ) +
                    ( add->rms[ii] * add->ncomb[ii] ) *
                    ( add->rms[ii] * add->ncomb[ii] );
            self->ncomb[ii] += add->ncomb[ii];
            self->rms[ii] = sqrt( self->rms[ii] ) / self->ncomb[ii];
        }
        else {
            self->bpix[ii]++;
        }
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Mutiply lenslet descriptors
 * @param self   the descriptor to multiply
 * @param add      the descriptor to multiply with
 *
 * @return error code
 *
 * This function implicitly assumes (but does not check) that the
 * reference wavelengths for both descriptors are the same.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_lenslet_descriptor_multiply( sph_lenslet_descriptor* self,
        sph_lenslet_descriptor* add) {
    int ii = 0;

    if ( !self || !add ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    for (ii = 0; ii < cpl_vector_get_size(self->lambdas); ++ii) {
        self->bpix[ii] += add->bpix[ii];
        self->values[ii] *= add->values[ii];
        self->rms[ii] = ( self->rms[ii] * self->ncomb[ii] ) *
                ( self->rms[ii] * self->ncomb[ii] ) +
                ( add->rms[ii] * add->ncomb[ii] ) *
                ( add->rms[ii] * add->ncomb[ii] );
        self->ncomb[ii] += add->ncomb[ii];
        self->rms[ii] = sqrt( self->rms[ii] ) / self->ncomb[ii];
    }
    return CPL_ERROR_NONE;
}
int
sph_lenslet_descriptor_get_overall_bad( sph_lenslet_descriptor* self ) {
    int ii = 0;

    for ( ii = 0; ii < cpl_vector_get_size(self->lambdas); ++ii )
    {
        if ( self->bpix[ii] == 0 ) return 0;
    }
    return 1;
}

double
sph_lenslet_descriptor_get_median( sph_lenslet_descriptor* self, double* rmsi ) {
    double      meda[cpl_vector_get_size(self->lambdas)];
    double      rms = 0.0;
    cpl_vector* v = NULL;
    int         cc = 0;
    double      median = 0.0;
    int ii = 0;
    for ( ii = 0; ii < cpl_vector_get_size(self->lambdas); ++ii )
    {
        if ( self->bpix[ii] == 0 ) {
            meda[cc] = self->values[ii];
            cc++;
        }
    }
    if ( cc < 1 ) return 0.0;
    v = cpl_vector_wrap( cc, meda );
    median = cpl_vector_get_median(v);
    if (rmsi) {
        cc = 0;
        for ( ii = 0; ii < cpl_vector_get_size(self->lambdas); ++ii )
        {
            if ( self->bpix[ii] == 0 ) {
                rms += (self->values[ii] - median) * (self->values[ii] - median);
                cc++;
            }
        }
        *rmsi = rms / (double)cc;
    }
    cpl_vector_unwrap(v); v = NULL;
    return median;
}

double
sph_lenslet_descriptor_get_median_rms( sph_lenslet_descriptor* self ) {
    double      meda[cpl_vector_get_size(self->lambdas)];
    cpl_vector* v = NULL;
    int         cc = 0;
    double      median = 0.0;
    int ii = 0;
    for ( ii = 0; ii < cpl_vector_get_size(self->lambdas); ++ii )
    {
        if ( self->bpix[ii] == 0 ) {
            meda[cc] = self->rms[ii];
            cc++;
        }
    }
    if ( cc < 1 ) return 0.0;
    v = cpl_vector_wrap( cc, meda );
    median = cpl_vector_get_median(v);
    cpl_vector_unwrap(v); v = NULL;
    return median;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief delete the lenslet descriptor
 * @param self
 *
 * Deletes the lenslet descriptor and frees up the memory.
 */
/*----------------------------------------------------------------------------*/
void
sph_lenslet_descriptor_delete( sph_lenslet_descriptor* self ) {
    if ( self->values ) {
        cpl_free(self->values);
        self->values = NULL;
    }
    if ( self->bpix ) {
        cpl_free(self->bpix);
        self->bpix = NULL;
    }
    if ( self->rms ) {
        cpl_free(self->rms);
        self->rms = NULL;
    }
    if ( self->ncomb ) {
        cpl_free(self->ncomb);
        self->ncomb = NULL;
    }
    cpl_free(self);
}

/**@}*/
