/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "sph_error.h"
#include "sph_loci.h"
#include <math.h>
#define SPH_LOCI_MAX_SUBSECS        5000
#define SPH_LOCI_MAX_RINGS        5000
#define SPH_LOCI_MAX_BYTES        1000000000
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

sph_loci*
sph_loci_new_empty(void) {
    sph_loci*       self = NULL;

    self = cpl_calloc(1,sizeof(sph_loci));
    self->subsections = cpl_calloc(SPH_LOCI_MAX_SUBSECS,sizeof(sph_loci_subsection*));
    self->loaded_subsect_maxindex = -1;
    self->loaded_subsect_minindex = 0;
    return self;
}

sph_loci*
sph_loci_new(double Na, double W, double ndelta,
        double cx, double cy, double g, double minR, double maxR)
{
    sph_loci*   self = NULL;
    self = sph_loci_new_empty();
    cpl_ensure(self,SPH_ERROR_MEMORY_FAIL,NULL);

    self->Na = Na;
    self->W = W;
    self->cx = cx;
    self->cy = cy;
    self->g = g;
    self->minR = minR;
    self->maxR = maxR;
    self->ndelta = ndelta;
    return self;
}
int sph_loci_get_number_sections( sph_loci* self ) {
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,-1);
    return self->nsubsections;
}

int sph_loci_load_new_subsection_set( sph_loci* self ) {
    int             ff      = 0;
    cpl_image*      dumim = NULL;
    int             nx = 0;
    int             ny = 0;
    cpl_frame*      aframe  = NULL;
    cpl_imagelist*  imlist   = NULL;
    cpl_imagelist*  blist   = NULL;
    int             subid       = 0;
    int             pp          = 0;
    int             bytes_loaded = 0;
    int             np = 0;
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->frameset,CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    for (subid = self->loaded_subsect_minindex;
            subid <= self->loaded_subsect_maxindex; ++subid) {
        if ( self->subsections[subid] ) {
            sph_loci_subsection_clear(self->subsections[subid]);
        }
        SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    }
    dumim = cpl_image_load(
            cpl_frame_get_filename(
                    cpl_frameset_get_first(self->frameset)),
            CPL_TYPE_INT,0,0);
    cpl_ensure_code(dumim,CPL_ERROR_NULL_INPUT);
    nx = cpl_image_get_size_x(dumim);
    ny = cpl_image_get_size_y(dumim);
    cpl_image_delete(dumim); dumim = NULL;
    for (subid = 0; subid < self->nsubsections; ++subid) {
        if ( self->subsections[subid]->bb_ll_x < 1 ||
                self->subsections[subid]->bb_ll_y < 1 ||
                self->subsections[subid]->bb_ur_x > nx ||
                self->subsections[subid]->bb_ur_y > nx) {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                    "The subsection %d has a boundary that is "
                    "outside the image size of %d,%d. "
                    "This is not allowed. Most likely the cause is that"
                    " the LOCI parameters are badly chosen: choose a smaller"
                    " maximum radius and/or a smaller N_a value.",subid,nx,ny);
            return CPL_ERROR_ILLEGAL_INPUT;
        }
    }

    subid = self->loaded_subsect_maxindex;
    self->loaded_subsect_minindex = subid + 1;
    while ( bytes_loaded < SPH_LOCI_MAX_BYTES && subid < self->nsubsections - 1) {
        subid++;
        sph_loci_subsection_clear(self->subsections[subid]);
        for (ff = 0; ff < cpl_frameset_get_size(self->frameset); ++ff) {
            aframe = cpl_frameset_get_position(self->frameset,ff);
            imlist = cpl_imagelist_load_window(cpl_frame_get_filename(aframe),
                    CPL_TYPE_DOUBLE,
                    0,
                    self->subsections[subid]->bb_ll_x,
                    self->subsections[subid]->bb_ll_y,
                    self->subsections[subid]->bb_ur_x,
                    self->subsections[subid]->bb_ur_y
                    );
            blist = cpl_imagelist_load_window(cpl_frame_get_filename(aframe),
                    CPL_TYPE_DOUBLE,
                    0,
                    self->subsections[subid]->bb_ll_x,
                    self->subsections[subid]->bb_ll_y,
                    self->subsections[subid]->bb_ur_x,
                    self->subsections[subid]->bb_ur_y
                    );
            if ( imlist && blist ) {
                bytes_loaded += ( sizeof(double) + sizeof(int) ) * cpl_imagelist_get_size(imlist) *
                        (self->subsections[subid]->bb_ur_x - self->subsections[subid]->bb_ll_x +1) *
                        (self->subsections[subid]->bb_ur_y - self->subsections[subid]->bb_ll_y +1);

                if ( self->subsections[subid]->imlist == NULL) {
                    self->subsections[subid]->imlist = cpl_imagelist_new();
                }
                if ( self->subsections[subid]->badlist == NULL) {
                    self->subsections[subid]->badlist = cpl_imagelist_new();
                }
                np = cpl_imagelist_get_size(imlist);
                for (pp = 0; pp < np; ++pp) {
                    cpl_imagelist_set(self->subsections[subid]->imlist,
                            cpl_imagelist_get(imlist,0),
                            cpl_imagelist_get_size(self->subsections[subid]->imlist));
                    cpl_imagelist_unset(imlist,0);
                    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
                }
                np = cpl_imagelist_get_size(blist);
                for (pp = 0; pp < np; ++pp) {
                    cpl_imagelist_set(self->subsections[subid]->badlist,
                            cpl_imagelist_get(blist,0),
                            cpl_imagelist_get_size(self->subsections[subid]->badlist));
                    cpl_imagelist_unset(blist,0);
                    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
                }
                cpl_imagelist_delete(imlist); imlist = NULL;
                cpl_imagelist_delete(blist); blist = NULL;
            }
        }
    }

    self->loaded_subsect_maxindex = subid;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

cpl_vector* sph_loci_calculate_weights( sph_loci* self,
        int subsection, int target_id,
        cpl_vector* angles_start,
        cpl_vector* angles_stop) {
    cpl_vector*     weights = NULL;
    cpl_matrix*     b_vector = NULL;
    cpl_matrix*     Amatrix = NULL;
    cpl_matrix*     solmatrix = NULL;
    int             ns = 0;
    int             row = 0;
    int             col = 0;
    cpl_image*      imrow = NULL;
    cpl_image*      imcol = NULL;
    cpl_vector*     ims_to_use = NULL;
    int             im_id = 0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(angles_start,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(angles_stop,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(subsection >= 0 && subsection < self->nsubsections,
            CPL_ERROR_NULL_INPUT,NULL);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    ims_to_use = sph_loci_get_valid_set(self,subsection,target_id,angles_start,angles_stop);
    cpl_ensure(ims_to_use,cpl_error_get_code(),NULL);
    ns = cpl_vector_get_size(ims_to_use);
    Amatrix = cpl_matrix_new(ns,ns);
    b_vector = cpl_matrix_new(ns,1);
    for (row = 0; row < ns; ++row) {
        for (col = 0; col < ns; ++col) {
            im_id = (int)cpl_vector_get(ims_to_use,row);
            imrow = sph_loci_subsection_get_opt_image(self->subsections[subsection],im_id);
            im_id = (int)cpl_vector_get(ims_to_use,col);
            imcol = sph_loci_subsection_get_opt_image(self->subsections[subsection],im_id);
            cpl_image_multiply(imrow,imcol);
            cpl_matrix_set(Amatrix,row,col,cpl_image_get_flux(imrow));
            cpl_image_delete(imrow); imrow = NULL;
            cpl_image_delete(imcol); imcol = NULL;
        }
    }
    imcol = sph_loci_subsection_get_opt_image(self->subsections[subsection],target_id);
    for (row = 0; row < ns; ++row) {
        im_id = (int)cpl_vector_get(ims_to_use,row);
        imrow = sph_loci_subsection_get_opt_image(self->subsections[subsection],im_id);
        cpl_image_multiply(imrow,imcol);
        cpl_matrix_set(b_vector,row,0,cpl_image_get_flux(imrow));
        cpl_image_delete(imrow); imrow = NULL;
    }
    cpl_image_delete(imcol); imcol = NULL;
    solmatrix = cpl_matrix_solve_normal(Amatrix,b_vector);
    if ( solmatrix ) {
        weights = cpl_vector_new(cpl_imagelist_get_size(self->subsections[subsection]->imlist));
        cpl_vector_fill(weights,0.0);
        for (row = 0; row < ns; ++row) {
            im_id = (int)cpl_vector_get(ims_to_use,row);
            cpl_vector_set(weights,im_id,cpl_matrix_get(solmatrix,row,0));
        }
    }
    else {
        cpl_matrix_delete(Amatrix); Amatrix = NULL;
        cpl_matrix_delete(b_vector); b_vector = NULL;
        cpl_matrix_delete(solmatrix); solmatrix = NULL;
        cpl_vector_delete(ims_to_use); ims_to_use = NULL;
        SPH_ERROR_CHECK_STATE_RETURN_NULL;
    }
    cpl_matrix_delete(Amatrix); Amatrix = NULL;
    cpl_matrix_delete(b_vector); b_vector = NULL;
    cpl_matrix_delete(solmatrix); solmatrix = NULL;
    cpl_vector_delete(ims_to_use); ims_to_use = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return weights;
}

cpl_image* sph_loci_calculate_ref_image( sph_loci* self,
        int subsection, cpl_vector* weights )
{
    cpl_image*      refimage = NULL;
    cpl_image*      dumimage = NULL;
    int             ii  = 0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(weights,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(subsection >= 0 && subsection < self->nsubsections,
            CPL_ERROR_NULL_INPUT,NULL);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if ( subsection < self->loaded_subsect_minindex ||
            subsection > self->loaded_subsect_maxindex ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "When calculating a reference image for "
                "subsection %d, need to have that "
                "subsection already loaded.",subsection);
        return NULL;
    }

    refimage = sph_loci_subsection_get_subtract_image(self->subsections[subsection],0);
    cpl_image_multiply_scalar(refimage,cpl_vector_get(weights,0));
    for (ii = 1; ii < cpl_vector_get_size(weights); ++ii) {
        dumimage = sph_loci_subsection_get_subtract_image(self->subsections[subsection],ii);
        cpl_image_multiply_scalar(dumimage,cpl_vector_get(weights,ii));
        cpl_image_add(refimage,dumimage);
        cpl_image_delete(dumimage);dumimage = NULL;
    }

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return refimage;
}
cpl_imagelist* sph_loci_calculate_imagelist( sph_loci* self,
        int subsection,
        cpl_vector* angles_start,
        cpl_vector* angles_stop)
{
    cpl_image*      refimage = NULL;
    cpl_image*      dumimage = NULL;
    int             ii  = 0;
    int             np  = 0;
    cpl_vector*     weights     = NULL;
    cpl_imagelist*  returnlist      = NULL;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(subsection >= 0 && subsection < self->nsubsections,
            CPL_ERROR_NULL_INPUT,NULL);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if ( subsection < self->loaded_subsect_minindex ||
            subsection > self->loaded_subsect_maxindex ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "When calculating a reference image for "
                "subsection %d, need to have that "
                "subsection already loaded.",subsection);
        return NULL;
    }
    returnlist = cpl_imagelist_new();

    np = cpl_imagelist_get_size(self->subsections[subsection]->imlist);
    for (ii = 0; ii < np; ++ii) {
        refimage = sph_loci_subsection_get_subtract_image(self->subsections[subsection],ii);
        weights = sph_loci_calculate_weights(self,subsection,ii,angles_start,angles_stop);
        dumimage = sph_loci_calculate_ref_image(self,subsection,weights);
        cpl_image_subtract(refimage,dumimage);
        cpl_imagelist_set(returnlist,refimage,ii);
        cpl_image_delete(dumimage); dumimage = NULL;
        cpl_vector_delete(weights); weights = NULL;
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return returnlist;
}

cpl_image*
sph_loci_process_single(sph_loci* self,
        int         targetid,
        int         nx,
        int         ny,
        cpl_vector* angles_start,
        cpl_vector* angles_stop)
{
    int         ss = 0;
    cpl_image*  parim   = NULL;
    cpl_image*  targetim  = NULL;
    cpl_image*  result_im = NULL;
    int         blx     = 0;
    int         bly     = 0;
    int         trx     = 0;
    int         try     = 0;
    cpl_vector*     weights = NULL;
    int         xx      = 0;
    int         yy      = 0;
    int         bpix        = 0;
    double      val     = 0.0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    result_im = cpl_image_new(nx,ny,CPL_TYPE_DOUBLE);

    self->loaded_subsect_maxindex = -1;
    self->loaded_subsect_minindex = 0;

    for (ss = 0; ss < self->nsubsections; ++ss) {
        if ( ss > self->loaded_subsect_maxindex ) {
            cpl_ensure(
                    sph_loci_load_new_subsection_set(self) == CPL_ERROR_NONE,
                    cpl_error_get_code(), NULL);
        }
        cpl_ensure( self->loaded_subsect_minindex >= 0, CPL_ERROR_DATA_NOT_FOUND, NULL);
        cpl_ensure( ss >= self->loaded_subsect_minindex, CPL_ERROR_DATA_NOT_FOUND, NULL);
        cpl_ensure( ss <= self->loaded_subsect_maxindex, CPL_ERROR_DATA_NOT_FOUND, NULL);
        blx = self->subsections[ss]->bb_ll_x;
        bly = self->subsections[ss]->bb_ll_y;
        trx = self->subsections[ss]->bb_ur_x;
        try = self->subsections[ss]->bb_ur_y;
        if ( blx < nx  && bly < ny  && trx >= 0 && try >= 0 &&
                trx < nx  && try < ny  && blx >= 0 && bly >= 0 ) {
            weights = sph_loci_calculate_weights(self,ss,targetid,angles_start,angles_stop);
            if ( weights ) {
                parim = sph_loci_calculate_ref_image(self,ss,weights);
                if ( !parim ) {
                    SPH_ERROR_RAISE_ERR(SPH_ERROR_GENERAL,"Could not get image for subsection %d", ss);
                    return NULL;
                }
                targetim = sph_loci_subsection_get_subtract_image(self->subsections[ss],targetid);
                cpl_image_subtract(targetim,parim);
                for (yy = bly; yy <= try; ++yy) {
                    for (xx = blx; xx <= trx; ++xx) {
                        val = cpl_image_get(targetim,xx+1-blx,yy+1-bly,&bpix);
                        if ( bpix == 0 ) {
                            val += cpl_image_get(result_im,xx+1,yy+1,&bpix);
                            cpl_image_set(result_im,xx+1,yy+1,val);
                        }

                    }
                }
                cpl_image_delete(parim);parim = NULL;
                cpl_image_delete(targetim);targetim = NULL;
                cpl_vector_delete(weights); weights = NULL;
            }
            else {
                SPH_ERROR_RAISE_WARNING(SPH_ERROR_GENERAL,"Could not get weights for subsection %d", ss);
                self->nbads++;
                cpl_error_reset();
            }
        }
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result_im;
}
cpl_vector*
sph_loci_get_valid_set(sph_loci* self,
        int subsectid,
        int targetid,
        cpl_vector* angles_start,
        cpl_vector* angles_stop) {
    cpl_vector*         result      = NULL;
    double              theta_min       = 0.0;
    double              theta_max       = 0.0;
    double              theta_target = 0.0;
    double              theta_target_min = 0.0;
    double              theta_target_max = 0.0;
    int                 ii          = 0;
    int                 np          = 0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(angles_start,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(angles_stop,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(subsectid >= 0 && subsectid < self->nsubsections,
            CPL_ERROR_ILLEGAL_INPUT,NULL);
    cpl_ensure(cpl_vector_get_size(angles_start) ==
            cpl_vector_get_size(angles_stop),
            CPL_ERROR_ILLEGAL_INPUT,NULL);
    cpl_ensure(self->ndelta * self->W < self->minR *
            CPL_MATH_RAD_DEG * 30.0,
            CPL_ERROR_ILLEGAL_INPUT,NULL);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if ( subsectid < self->loaded_subsect_minindex ||
            subsectid > self->loaded_subsect_maxindex ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "When calculating a reference image for "
                "subsection %d, need to have that "
                "subsection already loaded.",subsectid);
        return NULL;
    }
    if ( !self->subsections[subsectid] ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "The subsection %d does not exist", subsectid);
        return NULL;
    }
    if ( !self->subsections[subsectid]->imlist ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "The subsection %d does not seem to be loaded", subsectid);
        return NULL;
    }

    np =  cpl_imagelist_get_size(self->subsections[subsectid]->imlist);
    cpl_ensure(targetid >= 0 && targetid < np,CPL_ERROR_ILLEGAL_INPUT,NULL);

    if ( cpl_vector_get_size(angles_start) != np ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "The number of points in the angle vector (%d) "
                "are not the same as the number of points "
                "in the imagelist loaded for the subsection (%d).",
                (int)cpl_vector_get_size(angles_start),np);
    }
    theta_target_min = cpl_vector_get(angles_start,targetid);
    theta_target_max = cpl_vector_get(angles_stop,targetid);
    if ( theta_target_max < theta_target_min ) {
        theta_target_max += 360.0;
    }
    theta_target = 0.5 * ( theta_target_max + theta_target_min );
    theta_target_max = theta_target + self->W * self->ndelta;
    theta_target_min = theta_target - self->W * self->ndelta;
    if ( theta_target_min < 0.0 ) {
        theta_target_max += 360.0;
        theta_target_min += 360.0;
    }
    result = cpl_vector_new(1);
    for (ii = 0; ii < np; ++ii) {
        theta_min = cpl_vector_get(angles_start,ii);
        theta_max = cpl_vector_get(angles_stop,ii);
        if ( theta_max < theta_min ) {
            theta_max += 360.0;
        }
        if ( ii == targetid ) {

        }
        else if ( theta_min >= theta_target_min &&
                theta_min <= theta_target_max) {

        }
        else if ( theta_max >= theta_target_min &&
                theta_max <= theta_target_max) {

        }
        else if ( theta_min + 360.0 >= theta_target_min &&
                theta_min + 360.0 <= theta_target_max) {

        }
        else if ( theta_max + 360.0 >= theta_target_min &&
                theta_max + 360.0 <= theta_target_max) {

        }
        else {
            cpl_vector_set(result,cpl_vector_get_size(result)-1,ii);
            cpl_vector_set_size(result,cpl_vector_get_size(result)+1);
        }
    }

    cpl_vector_set_size(result,cpl_vector_get_size(result)-1);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

sph_error_code
sph_loci_set_frameset(sph_loci* self,cpl_frameset* frameset) {
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(frameset,CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    self->frameset = frameset;

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}




static
sph_loci_subsection*
sph_loci_create_subsection(sph_loci* self, double r,double phi,double dphi)
{
    sph_loci_subsection*        subs = NULL;
    double                      deltar = 0.0;
    double                      x[4];
    double                      y[4];
    int                         ii = 0;
    subs = sph_loci_subsection_new();
    subs->cx = self->cx;
    subs->cy = self->cy;
    subs->sub_minr = r;
    subs->sub_maxr = r + self->dr;
    subs->sub_mintheta = phi;
    subs->sub_maxtheta = phi + dphi;
    subs->opt_minr = r;
    deltar = sqrt(CPL_MATH_PI * self->g * self->Na) * self->W / 2.0;
    subs->opt_maxr = subs->opt_minr + deltar;
    subs->opt_mintheta = subs->sub_mintheta;
    subs->opt_maxtheta = subs->sub_maxtheta;

    x[0] = subs->opt_minr * cos(subs->opt_mintheta) + self->cx;
    x[1] = subs->opt_minr * cos(subs->opt_maxtheta) + self->cx;
    x[2] = subs->opt_maxr * cos(subs->opt_mintheta) + self->cx;
    x[3] = subs->opt_maxr * cos(subs->opt_maxtheta) + self->cx;
    y[0] = subs->opt_minr * sin(subs->opt_mintheta) + self->cy;
    y[1] = subs->opt_minr * sin(subs->opt_maxtheta) + self->cy;
    y[2] = subs->opt_maxr * sin(subs->opt_mintheta) + self->cy;
    y[3] = subs->opt_maxr * sin(subs->opt_maxtheta) + self->cy;

    subs->bb_ll_x = subs->bb_ur_x = x[0];
    subs->bb_ll_y = subs->bb_ur_y = y[0];
    for (ii = 0; ii < 4; ++ii) {
        if ( x[ii] < subs->bb_ll_x ) subs->bb_ll_x = x[ii];
        if ( y[ii] < subs->bb_ll_y ) subs->bb_ll_y = y[ii];
        if ( x[ii] > subs->bb_ur_x ) subs->bb_ur_x = x[ii];
        if ( y[ii] > subs->bb_ur_y ) subs->bb_ur_y = y[ii];
    }

    return subs;
}
sph_error_code
sph_loci_draw_subsections(sph_loci* self, cpl_image* image) {
    const cpl_size  nx = cpl_image_get_size_x(image);
    const cpl_size  ny = cpl_image_get_size_y(image);
    int         xx      = 0;
    int         yy      = 0;
    double      r       = 0.0;
    int         rr      = 0;
    int         rindex  = 0;
    int         ind     = 0;
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(image,CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    for (yy = 0; yy < ny; ++yy) {
        for (xx = 0; xx < nx; ++xx) {
            r = sqrt((xx - self->cx) * (xx - self->cx) +
                    (yy - self->cy)*(yy - self->cy));
            if ( r > self->minR && r < self->maxR ) {
                rindex = cpl_vector_find(self->rings_radii,r);
                for (rr = cpl_vector_get(self->rings_min_indices,rindex);
                        rr < cpl_vector_get(self->rings_max_indices,rindex);
                        ++rr)
                {
                    if ( sph_loci_subsection_test_inside_subtract_region(self->subsections[rr],xx,yy)) {
                        ind = rr+1;
                        cpl_image_set(image,xx,yy,ind);
                    }
                }
                if ( rindex > 0 ) {
                    for (rr = cpl_vector_get(self->rings_min_indices,rindex-1);
                            rr < cpl_vector_get(self->rings_max_indices,rindex-1);
                            ++rr)
                    {
                        if ( sph_loci_subsection_test_inside_subtract_region(self->subsections[rr],xx,yy)) {
                            ind = rr+1;
                            cpl_image_set(image,xx,yy,ind);
                        }
                    }
                }
            }
        }
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
sph_error_code
sph_loci_init_subsections(sph_loci* self,double dr,sph_loci_div_scheme scheme) {
    int         rr      = 0;
    double      r       = 0.0;
    double      dphi_arr[SPH_LOCI_MAX_RINGS];
    double      dphi    = 0.0;
    double      phi     = 0.0;
    int         ntotal_rings = 0;
    int         ntotal_segments = 0;
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(scheme == SPH_LOCI_FINE ||
            scheme == SPH_LOCI_NORMAL,
            CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(self->maxR > self->minR, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(self->minR > self->W * self->ndelta *2, CPL_ERROR_ILLEGAL_INPUT);
    self->dr = dr;
    for (r = self->minR; r < self->maxR ; r+=dr) {
        if (ntotal_rings < SPH_LOCI_MAX_RINGS) {
            dphi = dphi_arr[ntotal_rings] = 1.0 / ( self->g / 2.0 + 2 * r / self->W * sqrt(self->g/(CPL_MATH_1_PI*self->Na)));
            ntotal_rings++;
            for (phi = 0; phi < CPL_MATH_2PI; phi+=dphi) {
                ntotal_segments++;
            }
        }
    }

    if ( ntotal_rings >= SPH_LOCI_MAX_RINGS ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT, "Too many subsections.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    if ( ntotal_segments > SPH_LOCI_MAX_SUBSECS ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT, "Too many subsections.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    if ( self->rings_max_indices ) {
        cpl_vector_delete(self->rings_max_indices);
        self->rings_max_indices = NULL;
    }
    if ( self->rings_min_indices ) {
        cpl_vector_delete(self->rings_min_indices);
        self->rings_min_indices = NULL;
    }
    if ( self->rings_radii ) {
        cpl_vector_delete(self->rings_radii);
        self->rings_radii = NULL;
    }

    self->rings_max_indices = cpl_vector_new(ntotal_rings);
    self->rings_min_indices = cpl_vector_new(ntotal_rings);
    self->rings_radii = cpl_vector_new(ntotal_rings);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    for (r = self->minR; r < self->maxR ; r+=dr) {
        cpl_vector_set(self->rings_radii,rr,r);
        cpl_vector_set(self->rings_min_indices,rr,self->nsubsections);
        dphi = dphi_arr[rr];
        for (phi = 0; phi < CPL_MATH_2PI; phi+=dphi) {
            self->subsections[self->nsubsections] = sph_loci_create_subsection(self,r,phi,dphi);
            self->nsubsections++;
        }
        cpl_vector_set(self->rings_max_indices,rr,self->nsubsections);
        rr++;
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

void
sph_loci_delete(sph_loci* self) {
    int         ii;
    if ( self ) {
        if ( self->subsections ) {
            for (ii = 0; ii < self->nsubsections; ++ii) {
                if ( self->subsections[ii] ) {
                    sph_loci_subsection_delete(self->subsections[ii]);
                    self->subsections[ii] = NULL;
                }
            }
            cpl_free(self->subsections);self->subsections = NULL;
        }
        if ( self->rings_max_indices ) {
            cpl_vector_delete(self->rings_max_indices);
            self->rings_max_indices = NULL;
        }
        if ( self->rings_min_indices ) {
            cpl_vector_delete(self->rings_min_indices);
            self->rings_min_indices = NULL;
        }
        if ( self->rings_radii ) {
            cpl_vector_delete(self->rings_radii);
            self->rings_radii = NULL;
        }
        cpl_free(self);
    }
}

/**@}*/
