/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "sph_error.h"
#include "sph_loci_subsection.h"
#include <math.h>


/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

sph_loci_subsection*
sph_loci_subsection_new(void) {
    sph_loci_subsection*        self = NULL;

    self = cpl_calloc(1,sizeof(sph_loci_subsection));

    return self;
}

sph_error_code
sph_loci_subsection_load(
        sph_loci_subsection* self,
        cpl_frameset* frameset ) {
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(frameset,CPL_ERROR_NULL_INPUT);

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
static
int
sph_loci_subsection_get_size_x(sph_loci_subsection* self) {
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,-1);
    return self->bb_ur_x - self->bb_ll_x + 1;
}
static
int
sph_loci_subsection_get_size_y(sph_loci_subsection* self) {
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,-1);
    return self->bb_ur_y - self->bb_ll_y + 1;
}


int
sph_loci_subsection_test_inside_subtract_region(sph_loci_subsection* self, int xx, int yy) {
    double dxx = 0.0;
    double dyy = 0.0;
    double  drr = 0.0;
    double  theta = 0.0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,-1);

    dxx = (double)xx - self->cx;
    dyy = (double)yy - self->cy;

    drr = sqrt(dxx * dxx + dyy * dyy);

    theta = atan2(dyy,dxx);
    if ( theta < 0 ) theta = 2.0 * CPL_MATH_PI + theta;
    if ( drr >= self->sub_minr &&
            drr < self->sub_maxr &&
            theta >= self->sub_mintheta &&
            theta < self->sub_maxtheta )
    {
        return 1;
    }
    return 0;

}

int
sph_loci_subsection_test_inside_opt_region(sph_loci_subsection* self, int xx, int yy) {
    double dxx = 0.0;
    double dyy = 0.0;
    double  drr = 0.0;
    double  theta = 0.0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,-1);

    dxx = (double)xx - self->cx;
    dyy = (double)yy - self->cy;

    drr = sqrt(dxx * dxx + dyy * dyy);

    theta = atan2(dyy,dxx);
    if ( theta < 0 ) theta = 2.0 * CPL_MATH_PI + theta;
    if ( drr >= self->opt_minr &&
            drr < self->opt_maxr &&
            theta >= self->opt_mintheta &&
            theta < self->opt_maxtheta )
    {
        return 1;
    }
    return 0;
}

sph_error_code
sph_loci_subsection_calculate_subtract_mask(sph_loci_subsection* self ) {
    if (self == NULL) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        const int nx = sph_loci_subsection_get_size_x(self);
        const int ny = sph_loci_subsection_get_size_y(self);

        if (self->submask == NULL ||
            cpl_mask_get_size_x(self->submask) != nx ||
            cpl_mask_get_size_y(self->submask) != ny) {
            cpl_mask_delete(self->submask);
            self->submask = cpl_mask_new(nx, ny);
        }

        for (int yy = 0; yy < ny; ++yy) {
            for (int xx = 0; xx < nx; ++xx) {
                if (sph_loci_subsection_test_inside_subtract_region(self,
                                                                    xx + self->bb_ll_x,
                                                                    yy + self->bb_ll_y)) {
                    cpl_mask_set(self->submask,xx+1,yy+1,0);
                }
                else {
                    cpl_mask_set(self->submask,xx+1,yy+1,1);
                }
            }
        }

    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

sph_error_code
sph_loci_subsection_calculate_opt_mask(sph_loci_subsection* self ) {
    if (self == NULL) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        const int nx = sph_loci_subsection_get_size_x(self);
        const int ny = sph_loci_subsection_get_size_y(self);

        if (self->optmask == NULL ||
            cpl_mask_get_size_x(self->optmask) != nx ||
            cpl_mask_get_size_y(self->optmask) != ny) {
            cpl_mask_delete(self->optmask);
            self->optmask = cpl_mask_new(nx, ny);
        }

        for (int yy = 0; yy < ny; ++yy) {
            for (int xx = 0; xx < nx; ++xx) {
                if (sph_loci_subsection_test_inside_opt_region(self,
                                                               xx + self->bb_ll_x,
                                                               yy + self->bb_ll_y) ) {
                    cpl_mask_set(self->optmask,xx+1,yy+1,0);
                }
                else {
                    cpl_mask_set(self->optmask,xx+1,yy+1,1);
                }
            }
        }
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

cpl_image*
sph_loci_subsection_get_subtract_mask_image( sph_loci_subsection* self ) {
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);

    if ( !self->submask ) {
        sph_loci_subsection_calculate_subtract_mask(self);
    }

    cpl_ensure(self->submask,CPL_ERROR_NULL_INPUT,NULL);
    return cpl_image_new_from_mask(self->submask);
}

cpl_mask*
sph_loci_subsection_get_subtract_mask( sph_loci_subsection* self ) {
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);

    if ( !self->submask ) {
        sph_loci_subsection_calculate_subtract_mask(self);
    }

    cpl_ensure(self->submask,CPL_ERROR_NULL_INPUT,NULL);
    return self->submask;
}

cpl_image*
sph_loci_subsection_get_opt_mask_image( sph_loci_subsection* self ) {
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);

    if ( !self->optmask ) {
        sph_loci_subsection_calculate_opt_mask(self);
    }

    cpl_ensure(self->optmask,CPL_ERROR_NULL_INPUT,NULL);
    return cpl_image_new_from_mask(self->optmask);
}

cpl_mask*
sph_loci_subsection_get_opt_mask( sph_loci_subsection* self ) {
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);

    if ( !self->optmask ) {
        sph_loci_subsection_calculate_opt_mask(self);
    }

    cpl_ensure(self->optmask,CPL_ERROR_NULL_INPUT,NULL);
    return self->optmask;
}

cpl_image*
sph_loci_subsection_get_opt_image( sph_loci_subsection* self,int imid) {
    cpl_image*          im = NULL;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(self->imlist,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(imid < cpl_imagelist_get_size(self->imlist) && imid >=0,
            CPL_ERROR_ILLEGAL_INPUT,NULL);

    im = cpl_image_duplicate(cpl_imagelist_get(self->imlist,imid));
    cpl_image_reject_from_mask(im,sph_loci_subsection_get_opt_mask(self));
    cpl_image_fill_rejected(im,0.0);
    return im;
}

cpl_image*
sph_loci_subsection_get_subtract_image( sph_loci_subsection* self,int imid) {
    cpl_image*          im = NULL;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(self->imlist,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(imid < cpl_imagelist_get_size(self->imlist) && imid >=0,
            CPL_ERROR_ILLEGAL_INPUT,NULL);

    im = cpl_image_duplicate(cpl_imagelist_get(self->imlist,imid));
    cpl_image_reject_from_mask(im,sph_loci_subsection_get_subtract_mask(self));
    cpl_image_fill_rejected(im,0.0);
    return im;
}

sph_error_code
sph_loci_subsection_clear( sph_loci_subsection* self ) {
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    if ( self->imlist ) {
        cpl_imagelist_delete(self->imlist); self->imlist = NULL;
    }
    if ( self->badlist ) {
        cpl_imagelist_delete(self->badlist); self->badlist = NULL;
    }
    return CPL_ERROR_NONE;
}

void sph_loci_subsection_delete( sph_loci_subsection* self ) {
    if ( self ) {
        sph_loci_subsection_clear(self);
        if ( self->optmask ) {
            cpl_mask_delete(self->optmask); self->optmask = NULL;
        }
        if ( self->submask ) {
            cpl_mask_delete(self->submask); self->submask = NULL;
        }
        cpl_free(self);
    }
}


/**@}*/
