/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */
#include "sph_error.h"
#include "sph_overlap_calculator.h"

static const int SPH_OVERLAP_CALCULATOR_RES = 400;

/*----------------------------------------------------------------------------*/
/**
 @brief Initialises the overlap calculator

 @param polygon
 @param minx
 @param maxx
 @param miny
 @param maxy

 @return error code

 Description:
 Initialises the overlap calculator.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_overlap_calculator_init(
        sph_overlap_calculator* self,
        sph_polygon* poly,
        double border)
{
    double polright, poltop, polleft, polbottom;
    cpl_mask* bpm;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    if ( self->resolution < 1 )
        self->resolution = SPH_OVERLAP_CALCULATOR_RES + 1;

    if ( !poly )
        return SPH_ERROR_RAISE_ERR(CPL_ERROR_NULL_INPUT,"No polygon");

    if ( self->im ) cpl_image_delete( self->im );

    self->im = NULL;

    if ( self->poly ) sph_polygon_delete( self->poly );
    self->poly = sph_polygon_duplicate(poly);

    if ( self->stat_poly ) sph_polygon_delete( self->stat_poly );
    self->stat_poly = NULL;

    self->im = cpl_image_new(self->resolution,
                             self->resolution, CPL_TYPE_DOUBLE);

    /* Reject all pixels */
    bpm = cpl_image_get_bpm(self->im);
    cpl_mask_not(bpm);

    sph_polygon_get_extent(self->poly, &poltop, &polbottom, &polleft,
                           &polright);

    self->minx = 0.0 - border;
    self->miny = 0.0 - border;
    self->maxx = polright - polleft + border;
    self->maxy = poltop - polbottom + border;

    self->dx = ( self->maxx - self->minx ) / self->resolution;
    self->dy = ( self->maxy - self->miny ) / self->resolution;

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Create a new instance of an overlap calculator.

 @param poly        the polygon for overlap calculations
 @param minx        the minimum x offset
 @param miny        the minimum y offset
 @param minx        the maximum x offset
 @param miny        the maximum y offset
 @return new overlap calculator or NULL

 Description:
 Create a new instance of overlap calculator.
 */
/*----------------------------------------------------------------------------*/
sph_overlap_calculator*
sph_overlap_calculator_new(
        sph_polygon* poly,
        double border,
        int res)
{
    sph_overlap_calculator*        self = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    self = cpl_calloc( sizeof(sph_overlap_calculator), 1) ;
    self->resolution = res;
    if ( res < 0 ) self->resolution = SPH_OVERLAP_CALCULATOR_RES + 1;
    if ( sph_overlap_calculator_init(self,poly,border) != CPL_ERROR_NONE )  {
        sph_overlap_calculator_delete(self); self = NULL;
    }
    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Precalculate the overlap values
 * @param self          the overlap caclulater
 *
 * @return error code
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_overlap_calculator_precalc( sph_overlap_calculator* self ) {
    int             xx          = 0;
    int             yy          = 0;
    double          ddx         = 0.0;
    double          ddy         = 0.0;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    for (yy = -1; yy < self->resolution + 1; ++yy) {
        for (xx = -1; xx < self->resolution + 1; ++xx) {
            ddx = xx * self->dx + self->minx + self->dx / 2.0;
            ddy = yy * self->dy + self->miny + self->dy / 2.0;
            sph_overlap_calculator_get_overlap(self,ddx,ddy);
        }
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Return the overlap for a given offset.

 @param self
 @param offset_x relative to left edge of ref. polygon
 @param offset_y relative to bottom edge of ref. polygon

 @return the overlap

 Description:
 Returns the overlap for the given polygon with a pixel of size 1x1 that is
 located offset_x away from the left edge of the polygon and offset_y from the
 bottom edge of the polygon.
 */
/*----------------------------------------------------------------------------*/
double
sph_overlap_calculator_get_overlap(
        sph_overlap_calculator* self,
        double offset_x, double offset_y )
{
    int            xx = 0;
    int            yy = 0;
    double        result = 0.0;
    double        lx = 0.0;
    double         ly = 0.0;
    int            bp = 0;
    sph_polygon_workspace*    ws = NULL;

    if ( offset_x > self->maxx ) return 0.0;
    if ( offset_x < self->minx ) return 0.0;
    if ( offset_y > self->maxy ) return 0.0;
    if ( offset_y < self->miny ) return 0.0;
    xx = (offset_x - self->minx) / self->dx;
    yy = (offset_y - self->miny) / self->dy;

    ws = sph_polygon_workspace_new();
    result = cpl_image_get( self->im, xx + 1, yy + 1, &bp );
    if ( bp ) {
        lx = sph_polygon_get_left( self->poly );
        ly = sph_polygon_get_bottom( self->poly );
        if ( !self->stat_poly ) {
            self->stat_poly = sph_polygon_new();
            sph_polygon_add_point(self->stat_poly,lx + offset_x + 0.0, ly + offset_y + 0.0);
            sph_polygon_add_point(self->stat_poly,lx + offset_x + 1.0, ly + offset_y + 0.0);
            sph_polygon_add_point(self->stat_poly,lx + offset_x + 1.0, ly + offset_y + 1.0);
            sph_polygon_add_point(self->stat_poly,lx + offset_x + 0.0, ly + offset_y + 1.0);
        }
        else {
            self->stat_poly->points[0].x = lx + offset_x + 0.0;
            self->stat_poly->points[0].y = ly + offset_y + 0.0;
            self->stat_poly->points[1].x = lx + offset_x + 1.0;
            self->stat_poly->points[1].y = ly + offset_y + 0.0;
            self->stat_poly->points[2].x = lx + offset_x + 1.0;
            self->stat_poly->points[2].y = ly + offset_y + 1.0;
            self->stat_poly->points[3].x = lx + offset_x + 0.0;
            self->stat_poly->points[3].y = ly + offset_y + 1.0;
        }

        result = sph_polygon_calculate_overlap(self->poly,self->stat_poly,ws);
        cpl_image_accept(self->im, xx + 1, yy + 1 );
        cpl_image_set( self->im, xx + 1, yy + 1, result );
    }
    sph_polygon_workspace_delete(ws);
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Seleting an instance of sph_overlap_calculator

 @param self


 Description:
 Deletes instance of sph_overlap_calculator.
 */
/*----------------------------------------------------------------------------*/
void
sph_overlap_calculator_delete( sph_overlap_calculator* self )
{
    if ( self ) {
        if ( self->im ) cpl_image_delete( self->im );
        if ( self->poly ) sph_polygon_delete( self->poly );
        if ( self->stat_poly ) sph_polygon_delete( self->stat_poly );
        cpl_free( self );
    }
}
