/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_paint_polygon.h"

#include <math.h>

/**@{*/

/*-----------------------------------------------------------------------------
                        Static variables
 -----------------------------------------------------------------------------*/

static const double SPH_PAINT_POLYGON_MIN_OVERLAP_ = 0.0000000000001;

static const double SPH_PAINT_POLYGON_MAGIC_VAL_ = 0.0;

/*-----------------------------------------------------------------------------
                        Private function prototypes
 -----------------------------------------------------------------------------*/

static cpl_image*
sph_paint_polygon_calculate_overlappings_(const sph_master_frame* self,
                                          const sph_polygon* poly,
                                          sph_polygon_workspace* ws)
    CPL_ATTR_ALLOC;

static
sph_error_code
sph_paint_polygon_on_master_frame_(
        sph_master_frame* self,
        sph_polygon* poly,
        double imval,
        double bpixval,
        double rmsval,
        double weightval,
        cpl_image* overlapim,
        int overlap_always_full_area_flag);


/*-----------------------------------------------------------------------------
                              Function definitions
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/
sph_error_code
sph_paint_polygon_on_master(
        sph_master_frame* self,
        sph_polygon* poly,
        double imval,
        double bpixval,
        double rmsval,
        double weightval,
        cpl_image* overlaps )
{
    return sph_paint_polygon_on_master_frame_(self,
                poly,imval,bpixval,rmsval,
                weightval,overlaps,0);
}

sph_error_code
sph_paint_polygon_on_master_image(
        sph_master_frame* self,
        sph_polygon* poly,
        double val )
{
    if ( !self ) {
        SPH_NO_SELF;
        return cpl_error_get_code();
    }
    if ( !self->image || !poly ) {
        SPH_NULL_ERROR;
        return cpl_error_get_code();
    }
    return sph_paint_polygon_on_master(self,
            poly,
            val,
            SPH_PAINT_POLYGON_MAGIC_VAL_,
            SPH_PAINT_POLYGON_MAGIC_VAL_,
            SPH_PAINT_POLYGON_MAGIC_VAL_, NULL
            );
}
sph_error_code
sph_paint_polygon_on_master_image_avg(
        sph_master_frame* self,
        sph_polygon* poly,
        double val )
{
    if ( !self ) {
        SPH_NO_SELF;
        return cpl_error_get_code();
    }
    if ( !self->image || !poly ) {
        SPH_NULL_ERROR;
        return cpl_error_get_code();
    }
    return sph_paint_polygon_on_master_frame_(self,
            poly,
            val,
            SPH_PAINT_POLYGON_MAGIC_VAL_,
            SPH_PAINT_POLYGON_MAGIC_VAL_,
            SPH_PAINT_POLYGON_MAGIC_VAL_,
            NULL, 1
            );
}

sph_error_code
sph_paint_polygon_on_master_bpix(
        sph_master_frame* self,
        sph_polygon* poly,
        double val )
{
    if ( !self ) {
        SPH_NO_SELF;
        return cpl_error_get_code();
    }
    if ( !self->badpixelmap || !poly ) {
        SPH_NULL_ERROR;
        return cpl_error_get_code();
    }
    return sph_paint_polygon_on_master(self,
            poly,
            SPH_PAINT_POLYGON_MAGIC_VAL_,
            val,
            SPH_PAINT_POLYGON_MAGIC_VAL_,
            SPH_PAINT_POLYGON_MAGIC_VAL_, NULL
            );
}
sph_error_code
sph_paint_polygon_on_master_weightmap(
        sph_master_frame* self,
        sph_polygon* poly,
        double val )
{
    if ( !self ) {
        SPH_NO_SELF;
        return cpl_error_get_code();
    }
    if ( !self->ncombmap || !poly ) {
        SPH_NULL_ERROR;
        return cpl_error_get_code();
    }
    return sph_paint_polygon_on_master(self,
            poly,
            SPH_PAINT_POLYGON_MAGIC_VAL_,
            SPH_PAINT_POLYGON_MAGIC_VAL_,
            SPH_PAINT_POLYGON_MAGIC_VAL_,
            val, NULL
            );
}
sph_error_code
sph_paint_polygon_on_master_rmsmap(
        sph_master_frame* self,
        sph_polygon* poly,
        double val )
{
    if ( !self ) {
        SPH_NO_SELF;
        return cpl_error_get_code();
    }
    if ( !self->rmsmap || !poly ) {
        SPH_NULL_ERROR;
        return cpl_error_get_code();
    }
    return sph_paint_polygon_on_master(self,
            poly,
            SPH_PAINT_POLYGON_MAGIC_VAL_,
            SPH_PAINT_POLYGON_MAGIC_VAL_,
            val,
            SPH_PAINT_POLYGON_MAGIC_VAL_, NULL
            );
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Calculates an image giving pixel to polygon overlap areas
 * @param self     the master frame
 * @param poly        the polygon
 *
 * @return new CPL_TYPE_DOUBLE cpl_image without bpm or NULL on error;
 *
 * This calculates a new image of the same size as the polygon
 * boundaries on the master frame giving for each pixle the actual
 * overlap between the pixel and the polygon.
 * This function is used mainly to speed up painting of polygon
 * operations for the case when the geometrical image of each
 * polygon is the same for many projections but only the actual
 * values to "paint" change.
 * To speed up such a case, first call this function and then
 * use the result as the last argument to sph_master_frame_paint_polygon.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image* sph_paint_polygon_get_overlap_image(const sph_master_frame* self,
                                               const sph_polygon* poly,
                                               sph_polygon_workspace* ws)
{
    return sph_paint_polygon_calculate_overlappings_(self,poly,ws);
}


static cpl_image*
sph_paint_polygon_calculate_overlappings_(const sph_master_frame* self,
                                          const sph_polygon* poly,
                                          sph_polygon_workspace* ws)
{
    int         always_overlap = 0;
    int            nx        = 0;
    int            ny         = 0;
    int            overlapping = 0;
    cpl_image*    result = NULL;
    double polright, poltop, polleft, polbottom;
    int maxx, maxy, minx, miny;

    if ( cpl_error_get_code() ) {
        SPH_RAISE_CPL
        return NULL;
    }
    // The following line circumvents the
    // explicit testing of inside
    // before overlap calculation complete --
    // sharp polygon may not include a pixel
    // corner but still overlap with pixel.
    always_overlap = 1;

    nx = cpl_image_get_size_x(self->image);
    ny = cpl_image_get_size_y(self->image);

    sph_polygon_get_extent(poly, &poltop, &polbottom, &polleft, &polright);

    maxx = CPL_MIN((int)polright,  nx - 1);
    maxy = CPL_MIN((int)poltop,    ny - 1);
    minx = CPL_MAX((int)polleft,   0);
    miny = CPL_MAX((int)polbottom, 0);

    result = cpl_image_new(maxx-minx+1,maxy-miny+1,CPL_TYPE_DOUBLE);
    if ( !result ) {
        cpl_error_reset();
        return result;
    }
    for (int yy = miny; yy <= maxy; ++yy) {
        for (int xx = minx; xx <= maxx; ++xx) {
            overlapping = 0;
            if ( always_overlap ) overlapping = 1;
            else {
                if ( sph_polygon_test_inside(poly,xx+1.0,yy) > 0 ||
                        sph_polygon_test_inside(poly,xx+1.0,yy+1.0) > 0 ||
                        sph_polygon_test_inside(poly,xx,yy+1.0) > 0 ||
                        sph_polygon_test_inside(poly,xx,yy) > 0 ) {
                    overlapping = 1;
                }
            }
            if ( overlapping ) {
                double overlap;
                const sph_polygon* clipped;
                ws->poly2->npoints = 0; /* Reset temporary polygon */
                sph_polygon_add_point(ws->poly2, xx + 0.0, yy + 0.0);
                sph_polygon_add_point(ws->poly2, xx + 1.0, yy + 0.0);
                sph_polygon_add_point(ws->poly2, xx + 1.0, yy + 1.0);
                sph_polygon_add_point(ws->poly2, xx + 0.0, yy + 1.0);

                /* FIXME: poly2 is aliased */
                clipped = sph_polygon_clip_to_target(ws->poly2, poly, ws);

                overlap = sph_polygon_area(clipped);
                cpl_image_set(result, xx-minx+1, yy-miny+1, overlap);
            }
        }
    }
    SPH_RAISE_CPL
    return result;
}
static
sph_error_code
sph_paint_polygon_on_master_frame_(
        sph_master_frame* self,
        sph_polygon* poly,
        double imval,
        double bpixval,
        double rmsval,
        double weightval,
        cpl_image* overlapim,
        int overlap_always_full_area_flag)
{
    int            xx        = 0;
    int            yy        = 0;

    double polright, poltop, polleft, polbottom;
    int maxx, maxy, minx, miny;

    double        valold    = 0.0;
    double        valoldbp    = 0.0;
    double        valoldrms    = 0.0;
    double        valoldncomb    = 0.0;
    int            nx        = 0;
    int            ny         = 0;
    int            bp = 0;
    double        area    = 0.0;
    double         overlap = 0.0;
    int            internal_overlapim = 0;
    sph_polygon_workspace*  ws = NULL;
    SPH_RAISE_CPL
    if ( !self ) {
        SPH_NO_SELF;
        return cpl_error_get_code();
    }
    if ( !self->image || !self->badpixelmap ||
            !self->rmsmap || !self->ncombmap || !poly ) {
        SPH_NULL_ERROR;
        return cpl_error_get_code();
    }
    if ( !overlapim ) {
        internal_overlapim = 1;
        ws = sph_polygon_workspace_new();
        overlapim = sph_paint_polygon_calculate_overlappings_(self,poly,ws);
        sph_polygon_workspace_delete(ws); ws = NULL;
    }
    area = sph_polygon_area(poly);
    nx = cpl_image_get_size_x(self->image);
    ny = cpl_image_get_size_y(self->image);

    sph_polygon_get_extent(poly, &poltop, &polbottom, &polleft, &polright);

    maxx = CPL_MIN((int)polright,  nx - 1);
    maxy = CPL_MIN((int)poltop,    ny - 1);
    minx = CPL_MAX((int)polleft,   0);
    miny = CPL_MAX((int)polbottom, 0);

    for (yy = miny; yy <= maxy; ++yy) {
        for (xx = minx; xx <= maxx; ++xx) {
            overlap = cpl_image_get(overlapim,xx-minx+1,yy-miny+1,&bp);
            if ( overlap > SPH_PAINT_POLYGON_MIN_OVERLAP_ ) {
                valold = cpl_image_get( self->image,xx+1,yy+1,&bp);
                valoldbp = cpl_image_get( self->badpixelmap,xx+1,yy+1,&bp);
                valoldrms = cpl_image_get( self->rmsmap,xx+1,yy+1,&bp);
                valoldncomb = cpl_image_get( self->ncombmap,xx+1,yy+1,&bp);
                if ( overlap_always_full_area_flag ) overlap = area;
                if ( !bp ) {
                    if ( imval != SPH_PAINT_POLYGON_MAGIC_VAL_ ) {
                        cpl_image_set( self->image,xx+1,yy+1,
                                imval/area * overlap
                                +valold);
                    }
                    if ( rmsval != SPH_PAINT_POLYGON_MAGIC_VAL_ ) {
                        cpl_image_set( self->rmsmap,xx+1,yy+1,
                                rmsval/area * overlap
                                +valoldrms);
                    }
                    if ( weightval != SPH_PAINT_POLYGON_MAGIC_VAL_ ) {
                        cpl_image_set( self->ncombmap,xx+1,yy+1,
                                weightval/area * overlap
                                +valoldncomb);
                    }
                }
                if ( bpixval != SPH_PAINT_POLYGON_MAGIC_VAL_ ) {
                    cpl_image_set( self->badpixelmap,xx+1,yy+1,
                            bpixval
                            +valoldbp);
                }
            }
        }
    }
    SPH_RAISE_CPL;
    if ( internal_overlapim ) cpl_image_delete(overlapim);
    return cpl_error_get_code();
}


/**@}*/
