/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */


#ifndef SPH_POLYGON_H_
#define SPH_POLYGON_H_


/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "sph_error.h"
#include "sph_point.h"
#include "sph_line.h"

/*-----------------------------------------------------------------------------
                                   New types
 -----------------------------------------------------------------------------*/

typedef unsigned short polygon_sz; /* FIXME: Change ? */

typedef struct _sph_polygon_ {
    sph_point* points;
    size_t     nalloc; /* "private" member */
    polygon_sz npoints;
} sph_polygon;

typedef struct _sph_polygon_workspace_ {
    sph_point*      point1;
    sph_point*      point2;
    sph_point*      point3;
    sph_line*       line;
    sph_polygon*    poly1;
    sph_polygon*    poly2;
} sph_polygon_workspace;

/*-----------------------------------------------------------------------------
                              Function prototypes
 -----------------------------------------------------------------------------*/

cpl_size sph_polygon_get_size(const sph_polygon* self);

sph_point*
sph_polygon_get_point(
        sph_polygon* self,
        int p );

sph_polygon*
sph_polygon_new( void );

sph_error_code
sph_polygon_sort( sph_polygon* self );


void
sph_polygon_free_static( void );

sph_polygon*
sph_polygon_duplicate( sph_polygon* poly );


sph_error_code
sph_polygon_add_point(
        sph_polygon* self,
        double x, double y );

const sph_polygon* sph_polygon_clip_to_target(const sph_polygon*,
                                              const sph_polygon*,
                                              sph_polygon_workspace*);

double
sph_polygon_calculate_overlap(
        const sph_polygon* self,
        const sph_polygon* target,
        sph_polygon_workspace* ws );

short sph_polygon_test_inside(const sph_polygon* self, double x, double y);

void sph_polygon_get_extent(const sph_polygon* self,
                            double* ptop,
                            double* pbottom,
                            double* pleft,
                            double* pright) CPL_ATTR_NONNULL;

double sph_polygon_get_top(const sph_polygon* self) CPL_ATTR_PURE;

double sph_polygon_get_bottom(const sph_polygon* self) CPL_ATTR_PURE;

double sph_polygon_get_left(const sph_polygon* self) CPL_ATTR_PURE;

double sph_polygon_get_right(const sph_polygon* self) CPL_ATTR_PURE;

double sph_polygon_get_midx(const sph_polygon* self) CPL_ATTR_PURE;

double sph_polygon_get_midy(const sph_polygon* self) CPL_ATTR_PURE;

void sph_polygon_get_midxy_(const sph_polygon* self,
                            double* px,
                            double* py) CPL_ATTR_NONNULL;

cpl_error_code sph_polygon_get_mid_xy(sph_point *,
                                      const sph_polygon*);

cpl_boolean sph_polygon_same_side(const sph_polygon* self,
                                  int ee,
                                  const sph_point* p)
    CPL_ATTR_NONNULL;

sph_error_code
sph_polygon_rotate_around( sph_polygon* self,
                                          double x, double y,
                                          double cosangle,
                                          double sinangle,
                                          double dx, double dy );

sph_line* sph_polygon_get_edge(const sph_polygon* self, int edge)
    CPL_ATTR_NONNULL CPL_ATTR_ALLOC;

sph_error_code
sph_polygon_shift( sph_polygon* self,
                   double dx, double dy );

sph_error_code
sph_polygon_enlarge( sph_polygon* self, double fac );

double sph_polygon_area(const sph_polygon* self)
    CPL_ATTR_NONNULL;

void
sph_polygon_delete( sph_polygon* self );

void
sph_polygon_workspace_delete( sph_polygon_workspace* self );

sph_polygon_workspace*
sph_polygon_workspace_new( void );

void
sph_polygon_free_all( void );

#endif /*SPH_POLYGON_H_*/
