/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_fits.h"
#include "sph_time.h"
#include "sph_common_keywords.h"
#include "sph_zpl_detector_time_model.h"
#include "sph_skycalc.h"

#include <cpl.h>
#include <time.h>
#include <math.h>
#include <string.h>

/* Standard year-2000 form: CCYY-MM-DD[Thh:mm:ss[.sss...]] */
#define SPH_ISO8601_FORMAT "%4d-%2d-%2dT%2d:%2d:%lf"

static mytime t0[BENCH_NTIMERS];

static
cpl_error_code
sph_time_is_iso8601(
        int year, int month,
        int day, int hour,
        int minute, double second);

static mytime get_time(void)
{
     struct timeval tv;
     gettimeofday(&tv, 0);
     return tv;
}

static double elapsed(mytime t1, mytime t00)
{
     return (double)(t1.tv_sec - t00.tv_sec) +
      (double)(t1.tv_usec - t00.tv_usec) * 1.0E-6;
}

int sph_time_get_time_seed(void) {
    mytime t1;
    t1 = get_time();
    return t1.tv_sec * 1000000 + t1.tv_usec;
}


sph_error_code
sph_time_get_times( const cpl_frame* aframe,
        double* mjd_start, double* mjd_stop )
{
    cpl_propertylist* pl = NULL;

    cpl_ensure_code( aframe, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( mjd_start, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( mjd_stop, CPL_ERROR_NULL_INPUT);

    pl = cpl_propertylist_load(
            cpl_frame_get_filename(aframe),
            0);
    cpl_ensure_code( pl, cpl_error_get_code());

    if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_MJD_OBS) &&
    	 cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DET_FRAM_UTC) ){
    	*mjd_start = cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_MJD_OBS);
    	sph_time_mjd_from_string( mjd_stop, cpl_propertylist_get_string(pl, SPH_COMMON_KEYWORD_DET_FRAM_UTC));
    }
    else {
        if (!cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_MJD_OBS) ){
            SPH_ERROR_RAISE_ERR( SPH_ERROR_GENERAL,
                    "Mandatory Keyword:  %s doesn't exist in the header!",
                    SPH_COMMON_KEYWORD_MJD_OBS);
        }
        if (!cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DET_FRAM_UTC) ){
            SPH_ERROR_RAISE_ERR( SPH_ERROR_GENERAL,
                    "Mandatory Keyword:  %s doesn't exist in the header!",
                    SPH_COMMON_KEYWORD_DET_FRAM_UTC);
        }
        cpl_propertylist_delete(pl); pl = NULL;
        return CPL_ERROR_DATA_NOT_FOUND;
    }
    cpl_propertylist_delete(pl); pl = NULL;

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

sph_error_code
sph_time_get_time_ut( const cpl_frame* aframe, char* time_ut )
{
    cpl_propertylist* pl = NULL;

    cpl_ensure_code( aframe, CPL_ERROR_NULL_INPUT);

    pl = cpl_propertylist_load(
            cpl_frame_get_filename(aframe),
            0);
    cpl_ensure_code( pl, cpl_error_get_code());

    if ( cpl_propertylist_has(pl, SPH_COMMON_KEYWORD_DET_FRAM_UTC) ) {
        /* FIXME: Potential buffer overflow */
        (void)strcpy(time_ut,cpl_propertylist_get_string(pl,SPH_COMMON_KEYWORD_DET_FRAM_UTC));
    }
    else {
        cpl_propertylist_delete(pl); pl = NULL;
        return CPL_ERROR_DATA_NOT_FOUND;
    }
    cpl_propertylist_delete(pl); pl = NULL;

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Return estimates times of all planes for a data cube
 * @param aframe       input data cube
 * @param zplmodel     zimpol detector time model
 *
 *
 * @return new cpl_vector of all times of the cubes as estimated.
 *
 */
/*----------------------------------------------------------------------------*/

cpl_vector*
sph_time_extract_times_zpl( const cpl_frame* inframe ){
    cpl_vector*                     result        = NULL;
    sph_zpl_detector_time_model*    zplmodel      = NULL;
    cpl_propertylist*               pl            = NULL;
    double                          mjd_start_ut  = 0.0;
    double                          dit_days      = 0.0;
    int                             nplanes       = 0;
    const char*                     date_obs_string    = NULL;
    int  year, mounth, day, hour, minute;
    double second;


    cpl_ensure( inframe, CPL_ERROR_NULL_INPUT, NULL);


    //first extract utc and dit keywords, converting it into mjd
    pl = cpl_propertylist_load( cpl_frame_get_filename(inframe), 0 );
    cpl_ensure( pl, cpl_error_get_code(), NULL);

    if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DIT ) &&
         cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_MJD_OBS ) &&
         cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DATE_OBS ) &&
         cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_UTC ) &&
         cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_READ_CURID )){

        date_obs_string = cpl_propertylist_get_string( pl, SPH_COMMON_KEYWORD_DATE_OBS);
        sph_time_iso8601_from_string(&year, &mounth, &day, &hour, &minute, &second, date_obs_string);
        sph_time_mjd_from_iso8601( &mjd_start_ut, year, mounth, day, 0, 0, 0.0);
        mjd_start_ut = mjd_start_ut + cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_UTC )/60./60./24.;

        dit_days = cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_DIT)/60./60./24.;
        zplmodel = sph_zpl_detector_time_model_new_days( cpl_propertylist_get_int(pl, SPH_COMMON_KEYWORD_READ_CURID) );
    }
    else {
        if (!cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DIT) ){
            SPH_ERROR_RAISE_ERR( SPH_ERROR_GENERAL,
                    "Mandatory Keyword:  %s doesn't exist in the header!",
                    SPH_COMMON_KEYWORD_DIT);
        }
        if (!cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_UTC) ){
            SPH_ERROR_RAISE_ERR( SPH_ERROR_GENERAL,
                    "Mandatory Keyword:  %s doesn't exist in the header!",
                    SPH_COMMON_KEYWORD_UTC);
        }
        if (!cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_MJD_OBS) ){
            SPH_ERROR_RAISE_ERR( SPH_ERROR_GENERAL,
                    "Mandatory Keyword:  %s doesn't exist in the header!",
                    SPH_COMMON_KEYWORD_MJD_OBS);
        }
        if (!cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DATE_OBS) ){
            SPH_ERROR_RAISE_ERR( SPH_ERROR_GENERAL,
                    "Mandatory Keyword:  %s doesn't exist in the header!",
                    SPH_COMMON_KEYWORD_DATE_OBS);
        }
        if (!cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_READ_CURID) ){
            SPH_ERROR_RAISE_ERR( SPH_ERROR_GENERAL,
                    "Mandatory Keyword:  %s doesn't exist in the header!",
                    SPH_COMMON_KEYWORD_READ_CURID);
        }

        cpl_propertylist_delete(pl); pl = NULL;
        return NULL;
    }
    cpl_propertylist_delete(pl); pl = NULL;

    nplanes = sph_fits_get_nplanes(cpl_frame_get_filename(inframe), 0 );

    cpl_ensure( nplanes > 0, CPL_ERROR_ILLEGAL_INPUT, NULL );
    result = cpl_vector_new(nplanes);

    for (int ii = 0; ii < nplanes; ++ii){
        cpl_vector_set( result, ii,
                mjd_start_ut + zplmodel->start_time_delay + dit_days/2. + ii*(dit_days + zplmodel->dit_gap) );
    }

    sph_zpl_detector_time_model_delete( zplmodel );
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL
    return result;

}


/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/


int sph_time_start_timer(int n)
{
     cpl_ensure_code(n >= 0 && n < BENCH_NTIMERS,CPL_ERROR_ILLEGAL_INPUT);
     t0[n] = get_time();
     return CPL_ERROR_NONE;
}

double sph_time_stop_timer(int n)
{
     mytime t1;
     double result = 0.0;
     cpl_ensure(n >= 0 && n < BENCH_NTIMERS,CPL_ERROR_ILLEGAL_INPUT,-1.0);
     t1 = get_time();
     result = elapsed(t1, t0[n]);
     return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return estimates times of all planes for a data cube
 * @param inframe       input data cube
 * @param special_flag  currently unused
 *
 * @return new cpl_vector of all times of the cubes as estimated.
 *
 *
 */
/*----------------------------------------------------------------------------*/
cpl_vector*
sph_time_extract_times_from_frame(
        const cpl_frame* inframe)
{
    cpl_error_code  rerr = CPL_ERROR_NONE;
    cpl_vector*     result = NULL;
    int             nplanes = 0;
    double          start_time = 0.0 ;
    double          end_time = 0.0 ;
    int             pp          = 0;
    double          dt          = 0.0;

    cpl_ensure( inframe, CPL_ERROR_NULL_INPUT, NULL);


    rerr = sph_time_get_times(inframe,&start_time,&end_time);
    if ( rerr != CPL_ERROR_NONE ) return result;

    SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
            "Start_Time = %f, End_Time = %f, nplanes = %d",
             start_time, end_time, nplanes );

    if (end_time <= start_time) {
        SPH_ERR("The end time <= start time, time interpolation is not possible !");
        return result;
    }

    nplanes = sph_fits_get_nplanes(
            cpl_frame_get_filename(inframe), 0 );
    result = cpl_vector_new(nplanes);

    dt = (end_time - start_time) / (double)nplanes;

    for (pp = 0; pp < nplanes; ++pp) {
        cpl_vector_set( result, pp, start_time + pp * dt + dt/2.0);
    }

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief   Convert a date from ISO-8601 to Modified Julian Date (MJD)
  @param   pmjd   On success, the MJD
  @param   year   The ISO-8601 Year
  @param   month  The ISO-8601 Month  (1 for first)
  @param   day    The ISO-8601 Day    (1 for first)
  @param   hour   The ISO-8601 Hour   (0 for first)
  @param   minute The ISO-8601 Minute (0 for first)
  @param   second The ISO-8601 Second (0 for first)
  @return  CPL_ERROR_NONE on success, otherwise the error
  @see The conversion code in wcslib version 4.4.4

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_time_mjd_from_iso8601(double * pmjd, int year, int month,
                                           int day, int hour, int minute,
                                           double second)
{

    cpl_ensure_code(pmjd  != NULL,        CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(!sph_time_is_iso8601(year, month, day, hour, minute,
                                           second), cpl_error_get_code());

    /* Compute MJD. */
    *pmjd = (double)((1461*(year - (12-month)/10 + 4712))/4
                     + (306*((month+9)%12) + 5)/10
                     - (3*((year - (12-month)/10 + 4900)/100))/4
                     + day - 2399904)
        + (hour + (minute + second/60.0)/60.0)/24.0;

    return CPL_ERROR_NONE;

}


/*----------------------------------------------------------------------------*/
/**
  @brief   Extract an ISO-8601 date from a string
  @param   pyear   The ISO-8601 Year
  @param   pmonth  The ISO-8601 Month  (1 for first)
  @param   pday    The ISO-8601 Day    (1 for first)
  @param   phour   The ISO-8601 Hour   (0 for first)
  @param   pminute The ISO-8601 Minute (0 for first)
  @param   psecond The ISO-8601 Second (0 for first)
  @param   iso8601 The ISO-8601 formatted string
  @return  CPL_ERROR_NONE on success, otherwise the error
  @see irplib_wcs_mjd_from_iso8601()
  @note The format must be the
        standard year-2000 form: CCYY-MM-DD[Thh:mm:ss[.sss...]]

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_time_iso8601_from_string(int * pyear, int * pmonth,
                                              int * pday, int * phour,
                                              int * pminute, double * psecond,
                                              const char * iso8601)
{

    int nret;

    cpl_ensure_code(pyear   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pmonth  != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pday    != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(phour   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pminute != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(psecond != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(iso8601 != NULL, CPL_ERROR_NULL_INPUT);

    nret = sscanf(iso8601, SPH_ISO8601_FORMAT, pyear, pmonth,
                  pday, phour, pminute, psecond);

    if (nret != 6) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Parsed"
                                     " %d != 6: input %s is not in format %s",
                                     nret, iso8601, SPH_ISO8601_FORMAT);
    }

    return sph_time_is_iso8601(*pyear, *pmonth, *pday, *phour, *pminute,
                                 *psecond)
        ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

cpl_error_code
sph_time_mjd_to_iso8601string(
        double mjd,
        char * iso8601)
{
    double sec = 0.0;
    int    year = 0;
    int    month = 0;
    int    day = 0;
    int    hour = 0;
    int    minute = 0;

    cpl_ensure_code( iso8601, CPL_ERROR_NULL_INPUT );

    cpl_ensure_code(
            sph_time_iso8601_from_mjd(
                    &year,
                    &month,
                    &day,
                    &hour,
                    &minute,
                    &sec,
                    mjd ) == CPL_ERROR_NONE,
                    CPL_ERROR_NULL_INPUT);

    /* Standard year-2000 form: CCYY-MM-DD[Thh:mm:ss[.sss...]] */
    sprintf(iso8601,"%4d-%02d-%02dT%02d:%02d:%lf", year, month, day, hour,
            minute, sec);

    return CPL_ERROR_NONE;
}


cpl_error_code sph_radec_rad_to_iso8601string(double radec,
                                              char * iso8601,
                                              int    is_ra )
{
    const cpl_error_code code =
        sph_radec_deg_to_iso8601string(radec * CPL_MATH_DEG_RAD,
                                       iso8601, is_ra);
    return code ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

cpl_error_code sph_radec_deg_to_iso8601string(double radec,
                                              char * iso8601,
                                              int    is_ra )
{
    double sec;
    double hour;
    double minute;
    double mfloat;

    cpl_ensure_code(iso8601 != NULL, CPL_ERROR_NULL_INPUT);

    if (is_ra) {
    	cpl_ensure_code(radec >=   0.0, CPL_ERROR_ILLEGAL_INPUT);
    	cpl_ensure_code(radec <= 360.0, CPL_ERROR_ILLEGAL_INPUT);
    	radec /= 15.0;
    } else {
    	cpl_ensure_code(radec >= -90.0, CPL_ERROR_ILLEGAL_INPUT);
    	cpl_ensure_code(radec <=  90.0, CPL_ERROR_ILLEGAL_INPUT);
    }

    if (radec >= 0.0) {
        hour   = floor(radec);
        mfloat = (radec - hour) * 60.0;
    } else {
        hour   = ceil(radec);
        mfloat = (hour - radec) * 60.0;
    }
    minute = floor(mfloat);
    sec    = (mfloat - minute) * 60.0;

    sprintf(iso8601, is_ra ? "%02d:%02d:%05.2f" : "%02d:%02d:%04.1f",
            (int)hour, (int)minute, sec);

    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
  @brief   Convert a date from a ISO-8601 string to Modified Julian Date (MJD)
  @param   pmjd    On success, the MJD
  @param   iso8601 The ISO-8601 formatted string
  @return  CPL_ERROR_NONE on success, otherwise the error
  @see irplib_wcs_iso8601_from_string(), irplib_wcs_mjd_from_iso8601()

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_time_mjd_from_string(double * pmjd, const char * iso8601)
{


    int year, day, month, hour, minute;
    double second;

    return sph_time_iso8601_from_string(&year, &month, &day, &hour,
                                          &minute, &second, iso8601)
        || sph_time_mjd_from_iso8601(pmjd, year, month, day, hour, minute,
                                       second)
        ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}



/*----------------------------------------------------------------------------*/
/**
  @brief   Convert a date from Modified Julian Date (MJD) to ISO-8601
  @param   pyear   The ISO-8601 Year
  @param   pmonth  The ISO-8601 Month  (1 for first)
  @param   pday    The ISO-8601 Day    (1 for first)
  @param   phour   The ISO-8601 Hour   (0 for first)
  @param   pminute The ISO-8601 Minute (0 for first)
  @param   psecond The ISO-8601 Second (0 for first)
  @param   mjd     The MJD
  @return  CPL_ERROR_NONE on success, otherwise the error
  @see irplib_wcs_mjd_from_iso8601()

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_time_iso8601_from_mjd(int * pyear, int * pmonth,
                                           int * pday, int * phour,
                                           int * pminute, double * psecond,
                                           double mjd)
{

    int jd, n4, dd;
    double t;

    cpl_ensure_code(pyear   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pmonth  != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pday    != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(phour   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pminute != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(psecond != NULL, CPL_ERROR_NULL_INPUT);

    /* Copied from datfix() in wcslib (v. 4.4.4) */

    jd = 2400001 + (int)mjd;

    n4 =  4*(jd + ((2*((4*jd - 17918)/146097)*3)/4 + 1)/2 - 37);
    dd = 10*(((n4-237)%1461)/4) + 5;

    *pyear  = n4/1461 - 4712;
    *pmonth = (2 + dd/306)%12 + 1;
    *pday   = (dd%306)/10 + 1;

    t = mjd - (int)mjd; /* t is now days */

    t *= 24.0; /* t is now hours */
    *phour = (int)t;
    t = 60.0 * (t - *phour); /* t is now minutes */
    *pminute = (int)t;
    *psecond = 60.0 * (t - *pminute);

    /* A failure here implies that this code has a bug */
    cpl_ensure_code(!sph_time_is_iso8601(*pyear, *pmonth, *pday, *phour,
                                           *pminute, *psecond),
                    CPL_ERROR_UNSPECIFIED);

    return CPL_ERROR_NONE;
}


/**@}*/

/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief   Verify that the six numbers comprise a valid ISO-8601 date
  @param   year   The Year
  @param   month  The Month  (1 for first)
  @param   day    The Day    (1 for first)
  @param   hour   The Hour   (0 for first)
  @param   minute The Minute (0 for first)
  @param   second The Second (0 for first)
  @return  CPL_ERROR_NONE on valid input, otherwise CPL_ERROR_ILLEGAL_INPUT
 */
/*----------------------------------------------------------------------------*/
static
cpl_error_code
sph_time_is_iso8601(
        int year, int month,
        int day, int hour,
        int minute, double second)
{

    const cpl_boolean is_leap = year % 4 ? CPL_FALSE : CPL_TRUE;
    const int mlen[] = {0, 31, is_leap ? 29 : 28, 31, 30, 31, 30, 31, 31, 30,
                        31, 30, 31};

    cpl_ensure_code(month > 0,            CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(month <= 12,          CPL_ERROR_ILLEGAL_INPUT);

    cpl_ensure_code(day   > 0,            CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(day   <= mlen[month], CPL_ERROR_ILLEGAL_INPUT);

    cpl_ensure_code(minute  < 60,         CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(minute  >= 0,         CPL_ERROR_ILLEGAL_INPUT);

    cpl_ensure_code(second  < 60.0,       CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(second  >= 0.0,       CPL_ERROR_ILLEGAL_INPUT);

    cpl_ensure_code(hour  >= 0,           CPL_ERROR_ILLEGAL_INPUT);
    /* 24:00:00 is valid ISO-8601 */
    cpl_ensure_code(hour  <= (minute > 0 || second > 0.0 ? 23 : 24),
                    CPL_ERROR_ILLEGAL_INPUT);

    return CPL_ERROR_NONE;
}



/**@}*/
