/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include <math.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_spline.h>
#include "sph_ifs_keywords.h"
#include "sph_error.h"
#include "sph_ifs_tags.h"
#include "sph_common_keywords.h"
#include "sph_test_ifs_instrument_model.h"
#include "sph_ldt.h"
#include "sph_ifs_lenslet_model.h"
#include "sph_pixel_description_table.h"
#include "sph_overlap_calculator.h"
#include "sph_paint_polygon.h"
static
double
sph_test_ifs_instrument_model_get_flux_in_poly__(
        sph_test_ifs_instrument_model* self, cpl_image* image,
        sph_polygon* poly, sph_overlap_calculator* oc);

static sph_error_code
sph_polygon_shrink_x(sph_polygon* self, double fac);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE Testing Utils Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides a very simple IFS instrument simulator. Together
 * with the sph_test_ngc_ir_simulator this can be used to create simulated
 * raw IFS instrument data.
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new sph_test_ifs_instrument_model
 * @param detector_pix_size
 * @param lenslets_per_side
 *
 * @return new sph_test_ifs_instrument_model
 *
 * Creates a new test ifs instrument model from the given lenslet model.
 * Do not delete the instrument model with sph_ifs_lenslet_model_delete since
 * this is done by the call to sph_test_ifs_instrument_model_delete call!
 *
 */
/*----------------------------------------------------------------------------*/
sph_test_ifs_instrument_model*
sph_test_ifs_instrument_model_new(int detector_pix_size,
        double lens_size_microns, int lenslets_per_side) {
    sph_test_ifs_instrument_model* result = NULL;
    int vv = 0;

    result = cpl_calloc(1, sizeof(sph_test_ifs_instrument_model));
    if (result) {

        result->lensmodel = sph_ifs_lenslet_model_new();
        result->lensmodel->detsize_pixels = detector_pix_size;
        result->lensmodel->detsize_microns = detector_pix_size
                * SPH_IFS_LENSLET_MODEL_PIX_SIZE_MICRONS;
        result->lensmodel->lensize_microns = lens_size_microns;
        result->lensmodel->lenslets_per_side = lenslets_per_side;

        result->flatvals = cpl_vector_new(
                sph_test_ifs_instrument_model_get_nlens(result));

        for (vv = 0; vv < cpl_vector_get_size(result->flatvals); ++vv) {
            cpl_vector_set(result->flatvals, vv, 1.0);
        }
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new sph_test_ifs_instrument_model for a given lenslet model
 * @param self  the lenslet model to use
 *
 * @return new sph_test_ifs_instrument_model
 *
 * Creates a new test ifs instrument model from the given lenslet model.
 * Do not delete the instrument model with sph_ifs_lenslet_model_delete since
 * this is done by the call to sph_test_ifs_instrument_model_delete call!
 *
 */
/*----------------------------------------------------------------------------*/
sph_test_ifs_instrument_model*
sph_test_ifs_instrument_model_new_from_lenslet_model(
        sph_ifs_lenslet_model* model) {
    sph_test_ifs_instrument_model* result = NULL;

    cpl_ensure(model, CPL_ERROR_NULL_INPUT, NULL);
    cpl_test_error(CPL_ERROR_NONE);

    result = sph_test_ifs_instrument_model_new(model->detsize_pixels,
            model->lensize_microns, model->lenslets_per_side);

    if (result) {
        sph_ifs_lenslet_model_delete(result->lensmodel);
        result->lensmodel = model;
    }
    cpl_test_error(CPL_ERROR_NONE);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return a lenslet hexagon
 * @param self
 * @param int    u
 * @param int   v
 *
 * @return polygon pointer or NULL
 *
 * Creates and returns a new lenslet hexagon as polygon.
 * The current rotaton angle of the lenslet array is taken
 * into account.
 *
 */
/*----------------------------------------------------------------------------*/
sph_polygon*
sph_test_ifs_instrument_model_get_lenslet_hexagon(
        sph_test_ifs_instrument_model* self, int u, int v) {
    sph_polygon* hex = NULL;

    cpl_ensure( self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( self->lensmodel, CPL_ERROR_ILLEGAL_INPUT, NULL);

    hex = sph_ifs_lenslet_model_get_poly(self->lensmodel, u, v);

    return hex;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Get the flat field value for a lenslet
 * @param self     the instrument model
 * @param uu        the uu of lenslet
 * @param vv        the vv of lenslet
 *
 * @return flat field value
 *
 * This function returns the current flat field value for the given lenslet
 */
/*----------------------------------------------------------------------------*/
double sph_test_ifs_instrument_model_get_flat_value(
        sph_test_ifs_instrument_model* self, int uu, int vv) {
    int index = 0;
    cpl_ensure(self, CPL_ERROR_NULL_INPUT, 0.0);
    cpl_ensure(self->lensmodel, CPL_ERROR_NULL_INPUT, 0.0);
    cpl_ensure(self->flatvals, CPL_ERROR_NULL_INPUT, 0.0);
    cpl_ensure(
            cpl_vector_get_size(self->flatvals) == sph_test_ifs_instrument_model_get_nlens(self),
            CPL_ERROR_NULL_INPUT, 0.0);
    cpl_ensure(cpl_error_get_code() == CPL_ERROR_NONE, cpl_error_get_code(),
            0.0);

    index = sph_ifs_lenslet_model_get_index(self->lensmodel, uu, vv) - 1;

    if (index >= 0 && index < sph_test_ifs_instrument_model_get_nlens(self)) {
        return cpl_vector_get(self->flatvals, index);
    }

    SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_OUTPUT, "Bad lenslet index.");
    return 0.0;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the flat field value for a lenslet
 * @param self     the instrument model
 * @param uu        the uu of lenslet
 * @param vv        the vv of lenslet
 * @param value        the flat field value
 *
 * @return error code of operation
 *
 * This function sets the flat field value of a lenslet. When calculating
 * a processed frame using one of the sph_test_ifs_instrument_model_disperse
 *  or sph_ifs_instrument_model_digest functions the values for that
 *  lenslets spectral region are multiplied with the value set here.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_test_ifs_instrument_model_set_flat_value(
        sph_test_ifs_instrument_model* self, int uu, int vv, double value) {
    int index = 0;
    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
    cpl_ensure(self->lensmodel, CPL_ERROR_NULL_INPUT, 0.0);
    cpl_test_error(CPL_ERROR_NONE);
    index = sph_ifs_lenslet_model_get_index(self->lensmodel, uu, vv);
    if (index >= 0) {
        cpl_vector_set(self->flatvals, index, value);
    } else {
        return CPL_ERROR_ILLEGAL_OUTPUT;
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return the total number of lenslets

 @param self

 @return the number of lenslets or -1 on error.

 Description: Returns the total number of lenslets

 */
/*----------------------------------------------------------------------------*/
int sph_test_ifs_instrument_model_get_nlens(sph_test_ifs_instrument_model* self) {

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, -1);
    cpl_ensure(self->lensmodel, CPL_ERROR_NULL_INPUT, -1);
    return sph_ifs_lenslet_model_get_nlens(self->lensmodel);
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Obtain flux within a lenslet given an image
 * @param self
 * @param uu        the u coordinate of lenslet
 * @param vv        the v coordinate of lenslet
 * @param image        the image to take flux from
 *
 * @return the measured flux
 *
 */
/*----------------------------------------------------------------------------*/
double sph_test_ifs_instrument_model_get_flux_in_hexagon(
        sph_test_ifs_instrument_model* self, int uu, int vv,
        sph_overlap_calculator* oc, cpl_image* image) {
    double val = 0.0;
    double pixsize = 0.0;
    int nn = 0;
    int nx = cpl_image_get_size_x(image);
    int ny = cpl_image_get_size_y(image);
    sph_polygon* hexpol = NULL;

    hexpol = sph_test_ifs_instrument_model_get_lenslet_hexagon(self, uu, vv);
    pixsize = 4.0 * self->lensmodel->lensize_microns
            * self->lensmodel->lenslets_per_side / nx;

    for (nn = 0; nn < hexpol->npoints; ++nn) {
        hexpol->points[nn].x /= pixsize;
        hexpol->points[nn].y /= pixsize;
        hexpol->points[nn].x += nx / 2.0;
        hexpol->points[nn].y += ny / 2.0;
    }

    val = sph_test_ifs_instrument_model_get_flux_in_poly__(self, image, hexpol,
            oc);
    sph_polygon_delete(hexpol);
    return val;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Return polygon of spectral area

 @param self            the lenslet description table
 @param u                the u-position of the hex to obtain
 @param v                the v-position of the hex to obtain.

 @return a new polygon representation of the hexagon lenslet spectra projection area

 Description: Returns the polygon (rectangle) of the projected area on the detector
 of the chosen hexagon at u and v. Coordinates
 are such that the central lenslet has u=0, v=0 and dither position 0 is assumed
 for the detector dither position.

 */
/*----------------------------------------------------------------------------*/
sph_polygon*
sph_test_ifs_instrument_model_get_spec_region(
        sph_test_ifs_instrument_model* self, int uu, int vv) {
    sph_polygon* specpoly = NULL;

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);

    specpoly = sph_polygon_duplicate(
            sph_ifs_lenslet_model_get_spec_region_pixel_coords(self->lensmodel,
                    uu, vv, self->dither_x, self->dither_y));

    return specpoly;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set flux on master frame within a spectral region
 * @param self
 * @param uu        the u coordinate of lenslet
 * @param vv        the v coordinate of lenslet
 * @param flux        the flux to set
 * @param mframe    the master frame
 *
 * Set the flux for the spectral region.
 *
 */
/*----------------------------------------------------------------------------*/
static
sph_error_code sph_test_ifs_instrument_model_set_flux_specregion(
        sph_test_ifs_instrument_model* self, int uu, int vv, double flux,
        sph_master_frame* mframe) {
    sph_polygon* poly = NULL;
    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->lensmodel, CPL_ERROR_NULL_INPUT);

    poly = sph_test_ifs_instrument_model_get_spec_region(self, uu, vv);

    sph_polygon_enlarge(poly, 0.95);
    sph_polygon_shrink_x(poly, 0.6);

    sph_paint_polygon_on_master_image(mframe, poly, flux);

    sph_polygon_delete(poly);
    poly = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Take an image (before lenslet array) through the lenslet array
 * @param self  the instrument model
 * @param inim    the input image
 * @param dx    the dithering position
 * @param dy    the dithering position
 *
 * @return new image after lenslet array
 *
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_ifs_instrument_model_disperse(sph_test_ifs_instrument_model* self,
        cpl_image* inim, double dx, double dy) {
    double val = 0.0;
    int uu = 0;
    int vv = 0;
    double pixsize = 0.0;
    int nn = 0;
    cpl_image* result = NULL;
    sph_master_frame* mframe = NULL;
    sph_overlap_calculator* overlapcalc = NULL;
    sph_polygon* hexpol = NULL;
    int nx = cpl_image_get_size_x(inim);
    int ny = cpl_image_get_size_y(inim);
    double dxsav = 0.0;
    double dysav = 0.0;

    /* TODO: make this function a lot faster....
     *
     */
    cpl_ensure( self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( self->lensmodel, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( inim, CPL_ERROR_NULL_INPUT, NULL);

    mframe = sph_master_frame_new(self->lensmodel->detsize_pixels,
            self->lensmodel->detsize_pixels);

    hexpol = sph_test_ifs_instrument_model_get_lenslet_hexagon(self, 0, 0);
    pixsize = 4.0 * self->lensmodel->lensize_microns
            * self->lensmodel->lenslets_per_side / nx;

    for (nn = 0; nn < hexpol->npoints; ++nn) {
        hexpol->points[nn].x /= pixsize;
        hexpol->points[nn].y /= pixsize;
        hexpol->points[nn].x += nx / 2.0;
        hexpol->points[nn].y += ny / 2.0;
    }

    dxsav = self->dither_x;
    dysav = self->dither_y;
    self->dither_x = dx;
    self->dither_y = dy;
    // The value of 1000 as resolution gives roughly 0.1% accuracy
    overlapcalc = sph_overlap_calculator_new(hexpol, 2.0, 1000);
    sph_overlap_calculator_precalc(overlapcalc);
    cpl_test_error(CPL_ERROR_NONE);
    {
        for (vv = -self->lensmodel->lenslets_per_side;
                vv <= self->lensmodel->lenslets_per_side; ++vv) {
            if (vv >= 0) {
                for (uu = -self->lensmodel->lenslets_per_side;
                        uu <= self->lensmodel->lenslets_per_side - vv; ++uu) {
                    cpl_test_error(CPL_ERROR_NONE);
                    val = sph_test_ifs_instrument_model_get_flux_in_hexagon(
                            self, uu, vv, overlapcalc, inim);
                    cpl_test_error(CPL_ERROR_NONE);
                    val = val
                            * sph_test_ifs_instrument_model_get_flat_value(self,
                                    uu, vv);
                    cpl_test_error(CPL_ERROR_NONE);
                    if (fabs(val) > 0.0000001)
                        sph_test_ifs_instrument_model_set_flux_specregion(self,
                                uu, vv, val, mframe);
                    cpl_test_error(CPL_ERROR_NONE);
                }
            } else {
                for (uu = 0 - (self->lensmodel->lenslets_per_side + vv);
                        uu <= self->lensmodel->lenslets_per_side; ++uu) {
                    cpl_test_error(CPL_ERROR_NONE);
                    val = sph_test_ifs_instrument_model_get_flux_in_hexagon(
                            self, uu, vv, overlapcalc, inim);
                    cpl_test_error(CPL_ERROR_NONE);
                    val = val
                            * sph_test_ifs_instrument_model_get_flat_value(self,
                                    uu, vv);
                    cpl_test_error(CPL_ERROR_NONE);
                    if (fabs(val) > 0.0000001)
                        sph_test_ifs_instrument_model_set_flux_specregion(self,
                                uu, vv, val, mframe);
                    cpl_test_error(CPL_ERROR_NONE);
                }
            }
        }
    }

    result = cpl_image_duplicate(mframe->image);
    sph_polygon_delete(hexpol);
    hexpol = NULL;
    sph_overlap_calculator_delete(overlapcalc);
    overlapcalc = NULL;
    sph_master_frame_delete(mframe);
    self->dither_x = dxsav;
    self->dither_y = dysav;
    cpl_test_error(CPL_ERROR_NONE);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return a propertylist describing the instrument model
 * @param self          the instrument model
 *
 * @return new propertylist or NULL on error
 *
 */
/*----------------------------------------------------------------------------*/
cpl_propertylist*
sph_test_ifs_instrument_model_get_proplist(sph_test_ifs_instrument_model* self) {
    cpl_propertylist* plist = NULL;

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);
    plist = cpl_propertylist_new();
    if (plist) {
        cpl_propertylist_update_int(plist, SPH_COMMON_KEYWORD_CALIB_PRISM_ID,
                self->prism_id);
    }
    return plist;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the instrument model
 * @param self
 *
 * @return nothing
 *
 */
/*----------------------------------------------------------------------------*/
void sph_test_ifs_instrument_model_delete(sph_test_ifs_instrument_model* self) {
    if (self) {
        if (self->lensmodel) {
            sph_ifs_lenslet_model_delete(self->lensmodel);
            self->lensmodel = NULL;
        }
        if (self->filter_lambdas) {
            cpl_vector_delete(self->filter_lambdas);
            self->filter_lambdas = NULL;
        }
        if (self->nd_filter_lambdas) {
            cpl_vector_delete(self->nd_filter_lambdas);
            self->nd_filter_lambdas = NULL;
        }
        if (self->filter_transmission) {
            cpl_vector_delete(self->filter_transmission);
            self->filter_transmission = NULL;
        }
        if (self->nd_filter_attenuation) {
            cpl_vector_delete(self->nd_filter_attenuation);
            self->nd_filter_attenuation = NULL;
        }
        if (self->flatvals) {
            cpl_vector_delete(self->flatvals);
            self->flatvals = NULL;
        }
        cpl_free(self);
    }
}

/* ------------------------------------------------------------------------*/
/*                      INTERNALS                                          */
/*-------------------------------------------------------------------------*/

static
double sph_test_ifs_instrument_model_get_flux_in_poly__(
        sph_test_ifs_instrument_model* self, cpl_image* image,
        sph_polygon* poly, sph_overlap_calculator* oc) {
    int xx = 0;
    int yy = 0;
    int maxx = 0;
    int minx = 0;
    int maxy = 0;
    int miny = 0;
    double valold = 0.0;
    double val = 0.0;
    int nx = 0;
    int ny = 0;
    double dx = 0.0;
    double dy = 0.0;
    int bp = 0;

    if (!self || !poly || !image) {
        SPH_NULL_ERROR;
        return cpl_error_get_code();
    }
    nx = cpl_image_get_size_x(image);
    ny = cpl_image_get_size_y(image);
    maxx = sph_polygon_get_right(poly);
    if (maxx >= nx)
        maxx = nx - 1;
    maxy = sph_polygon_get_top(poly);
    if (maxy >= ny)
        maxy = ny - 1;
    minx = sph_polygon_get_left(poly);
    if (minx < 0)
        minx = 0;
    miny = sph_polygon_get_bottom(poly);
    if (miny < 0)
        miny = 0;
    for (xx = minx; xx <= maxx; ++xx) {
        for (yy = miny; yy <= maxy; ++yy) {
            valold = cpl_image_get(image, xx + 1, yy + 1, &bp);
            if (!bp) {
                dx = (double) xx - sph_polygon_get_left(poly);
                dy = (double) yy - sph_polygon_get_bottom(poly);
                val += valold * sph_overlap_calculator_get_overlap(oc, dx, dy);
            }
        }
    }

    return val;
}

static sph_error_code sph_polygon_shrink_x(sph_polygon* self, double fac) {
    double x = 0.0;
    double y = 0.0;
    int ii = 0;
    if (!self) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    if (fac <= DBL_MIN) {
        SPH_NULL_ERROR
        return CPL_ERROR_NULL_INPUT;
    }
    for (ii = 0; ii < self->npoints; ++ii) {
        x += self->points[ii].x;
        y += self->points[ii].y;
    }
    if (self->npoints > 0) {
        x /= self->npoints;
        y /= self->npoints;
    }
    for (ii = 0; ii < self->npoints; ++ii) {
        self->points[ii].x = x + fac * (self->points[ii].x - x);
    }
    return CPL_ERROR_NONE;
}

/**@}*/
