/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */
#include "sph_test_ngc_ccd_simulator.h"
#include "sph_test_image_tools.h"
#include "sph_ird_tags.h"
#include "sph_ird_keywords.h"

#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_master_frame.h"
#include "sph_framecombination.h"
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>

static
void sph_test_ngc_ccd_simulator_saveimage(sph_test_ngc_ccd_simulator* self,
        cpl_image** pwindow_image, cpl_image* animage, char* filename,
        cpl_propertylist* pl) {
    FILE* fp = NULL;
    if (*pwindow_image) {
        if (self->dont_reset) {
            cpl_image_add(*pwindow_image, animage);
        } else {
            cpl_image_delete(*pwindow_image);
            *pwindow_image = NULL;
        }
    }
    if (*pwindow_image == NULL) {
        *pwindow_image = cpl_image_duplicate(animage);
    }
    if (self->singlecube) {
        fp = fopen(filename, "r");
        if (fp == NULL) {
            cpl_image_save(*pwindow_image, filename, CPL_BPP_32_SIGNED, pl,
                    CPL_IO_DEFAULT);
        } else {
            fclose(fp);
            cpl_image_save(*pwindow_image, filename, CPL_BPP_32_SIGNED, pl,
                    CPL_IO_EXTEND);
        }
    } else {
        cpl_image_save(*pwindow_image, filename, CPL_BPP_32_SIGNED, pl,
                CPL_IO_CREATE);
    }
}

static
void sph_test_ngc_ccd_simulator_image_add_bias_and_dark(
        sph_test_ngc_ccd_simulator* self, cpl_image* animage, double exptime) {
    cpl_image_add_scalar(animage, self->mean_bias + self->mean_dark * exptime);
}

static
void sph_test_ngc_ccd_simulator_image_add_noise(
        sph_test_ngc_ccd_simulator* self, cpl_image* animage) {
    if (self->poisson) {
        sph_test_image_tools_apply_poisson_noise(animage, self->pRNG);
    }
    if (self->add_ron_noise) {
        sph_test_image_tools_add_noise(animage, self->sigma_dark, self->pRNG);
        if (self->rom == SPH_TEST_NGC_ROM_CCD_SINGLE) {
            sph_test_image_tools_add_noise(animage, self->sigma_bias,
                    self->pRNG);
        } else {
            ; // If its not single readout, then bias noise wont matter.
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Creates a new ngc_simulator_ccd factory
 *
 * @return the new framecrator factory
 *
 * The mode is set to a default value. All other values/flags are set to zero/
 * false. It also initializes a random number generator from gsl as gsl_rng_taus
 * with seed 1.
 *
 * NOTE: the seed is always the same !!
 * Change it with gsl_rng_set( result->pRNG, seed)
 */
/*----------------------------------------------------------------------------*/
sph_test_ngc_ccd_simulator*
sph_test_ngc_ccd_simulator_new(void) {
    sph_test_ngc_ccd_simulator* result = NULL;

    result = cpl_calloc(1, sizeof(sph_test_ngc_ccd_simulator));
    result->add_ron_noise = 0;
    result->pRNG = gsl_rng_alloc(gsl_rng_taus);
    gsl_rng_set(result->pRNG, 1);
    sph_test_ngc_ccd_simulator_set_detector_size(result, 256, 256);
    result->det_linear_factor = 1.0;
    result->lambda_ramp_linear = 1.0;
    result->mean_dark = 1.0;
    result->mean_bias = 50.0;
    return result;
}

void sph_test_ngc_ccd_simulator_set_detector_size(
        sph_test_ngc_ccd_simulator* self, int nx, int ny) {
    self->det_size_x = nx;
    self->det_size_y = ny;
    self->adu1_nx = nx / 2;
    self->adu2_nx = nx / 2;
    self->adu1_ny = ny;
    self->adu2_ny = ny;
}

void sph_test_ngc_ccd_simulator_set_overscans(sph_test_ngc_ccd_simulator* self,
        int adu1_ovrsx, int adu1_ovrsy, int adu2_ovrsx, int adu2_ovrsy) {
    self->adu1_ovrsx = adu1_ovrsx;
    self->adu1_ovrsy = adu1_ovrsy;
    self->adu2_ovrsx = adu2_ovrsx;
    self->adu2_ovrsy = adu2_ovrsy;
}

void sph_test_ngc_ccd_simulator_set_prescans(sph_test_ngc_ccd_simulator* self,
        int adu1_presx, int adu1_presy, int adu2_presx, int adu2_presy) {
    self->adu1_presx = adu1_presx;
    self->adu1_presy = adu1_presy;
    self->adu2_presx = adu2_presx;
    self->adu2_presy = adu2_presy;
}

void sph_test_ngc_ccd_simulator_set_bias(sph_test_ngc_ccd_simulator* self,
        double mean, double sigma) {
    self->mean_bias = mean;
    self->sigma_bias = sigma;
}
void sph_test_ngc_ccd_simulator_set_dark(sph_test_ngc_ccd_simulator* self,
        double mean, double sigma) {
    self->mean_dark = mean;
    self->sigma_dark = sigma;
}

void sph_test_ngc_ccd_simulator_set_readout_mode(
        sph_test_ngc_ccd_simulator* self, sph_test_rom_ccd mode) {
    self->rom = mode;
}
void sph_test_ngc_ccd_simulator_set_readout_noise(
        sph_test_ngc_ccd_simulator* self, double ron) {
    self->ron = ron;
}
void sph_test_ngc_ccd_simulator_set_num_deadpixels(
        sph_test_ngc_ccd_simulator* self, int n) {
    self->ndead = n;
}
void sph_test_ngc_ccd_simulator_set_num_hotpixels(
        sph_test_ngc_ccd_simulator* self, int n) {
    self->nhot = n;
}
void sph_test_ngc_ccd_simulator_set_num_crazypixels(
        sph_test_ngc_ccd_simulator* self, int n) {
    self->ncrazy = n;
}
void sph_test_ngc_ccd_simulator_set_flat_linear_coeff(
        sph_test_ngc_ccd_simulator* self, double c) {
    self->det_linear_factor = c;
}
void sph_test_ngc_ccd_simulator_set_flat_quadratic_coeff(
        sph_test_ngc_ccd_simulator* self, double c) {
    self->det_qaudratic_factor = c;
}
void sph_test_ngc_ccd_simulator_set_dont_reset(sph_test_ngc_ccd_simulator* self,
        int yesorno) {
    self->dont_reset = yesorno;
}
void sph_test_ngc_ccd_simulator_set_save_singlecube(
        sph_test_ngc_ccd_simulator* self, int yesorno) {
    self->singlecube = yesorno;
}
void sph_test_ngc_ccd_simulator_set_add_noise(sph_test_ngc_ccd_simulator* self,
        int addnoise, int addpoisson) {
    self->add_ron_noise = addnoise;
    self->poisson = addpoisson;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Crete the dark as an image
 * @param self the ngc_simulator_ccd
 * @param ndit the number of dits
 * @param exptime the exposure time in seconds
 *
 * @return image or NULL
 *
 * Returns the dark current per second as an image. The ndit and exptime
 * parameters are currently ignored.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_ngc_ccd_simulator_create_dark_image(sph_test_ngc_ccd_simulator* self,
        int ndit, double exptime) {
    cpl_image* image = cpl_image_new(self->det_size_x, self->det_size_y,
            CPL_TYPE_FLOAT);

    if (ndit > 0 && exptime > 0.0) {
        int xx = 0;
        int yy = 0;

        for (xx = 0; xx < self->det_size_x; ++xx) {
            for (yy = 0; yy < self->det_size_y; ++yy) {
                cpl_image_set(image, xx + 1, yy + 1, self->mean_dark);
            }
        }
    }

    return image;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Crete the bias as an image
 * @param self the ngc_simulator_ccd
 * @param ndit the number of dits
 * @param exptime the exposure time in seconds
 *
 * @return image or NULL
 *
 * Returns the bias as an image. The ndit and exptime
 * parameters are currently ignored.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_ngc_ccd_simulator_create_bias_image(sph_test_ngc_ccd_simulator* self,
        int ndit, double exptime) {
    cpl_image* image = cpl_image_new(self->det_size_x, self->det_size_y,
            CPL_TYPE_FLOAT);

    if (ndit > 0 && exptime > 0.0) {
        int xx = 0;
        int yy = 0;

        for (xx = 0; xx < self->det_size_x; ++xx) {
            for (yy = 0; yy < self->det_size_y; ++yy) {
                cpl_image_set(image, xx + 1, yy + 1, self->mean_bias);
            }
        }
    }

    return image;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Crete the linear response (flat) as an image
 * @param self the ngc_simulator_ccd
 * @param ndit the number of dits
 * @param exptime the exposure time in seconds
 *
 * @return image or NULL
 *
 * Returns the detector flat as an image. Specifically, this
 * means that the linear coefficient of the response function
 * is returned as a function of pixel coordinate (i.e. as an image)
 * The ndit and exptime parameters are currently ignored.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_ngc_ccd_simulator_create_detector_flat_image(
        sph_test_ngc_ccd_simulator* self) {
    cpl_image* image = NULL;
    int xx = 0;
    int yy = 0;

    if (self->flatim) {
        image = cpl_image_duplicate(self->flatim);
    } else {
        image = cpl_image_new(self->det_size_x, self->det_size_y,
                CPL_TYPE_FLOAT);
        for (xx = 0; xx < self->det_size_x; ++xx) {
            for (yy = 0; yy < self->det_size_y; ++yy) {
                cpl_image_set(image, xx + 1, yy + 1, self->det_linear_factor);
            }
        }
    }
    return image;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Set the flatfield to a gradient
 * @param self pointer to the ngc simulator
 * @param minval    the minimum value (in lower left corner)
 * @param maxval    the maximum value (in upper left corner)
 *
 * @return error code
 *
 */
/*----------------------------------------------------------------------------*/
void sph_test_ngc_ccd_simulator_set_dff_gradient(
        sph_test_ngc_ccd_simulator* self, double min, double max) {
    cpl_image* image = NULL;
    int xx = 0;
    int yy = 0;
    double step = 0.0;

    if (self->flatim) {
        cpl_image_delete(self->flatim);
        self->flatim = NULL;
    }
    image = cpl_image_new(self->det_size_x, self->det_size_y, CPL_TYPE_FLOAT);
    if (!image) {
        return;
    }

    step = (max - min) / (self->det_size_x + self->det_size_y);

    for (xx = 0; xx < self->det_size_x; ++xx) {
        for (yy = 0; yy < self->det_size_y; ++yy) {
            cpl_image_set(image, xx + 1, yy + 1, min + step * (xx + yy));
        }
    }
    self->flatim = image;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the standard dither pattern for the test ngc_simulator_ccd functions
 * @param self
 * @param number of dither steps -- after that cyclic repeat
 *
 * @return
 *
 * Set the standard dither pattern.
 *
 *
 */
/*----------------------------------------------------------------------------*/
void sph_test_ngc_ccd_simulator_set_standard_ditherpattern(
        sph_test_ngc_ccd_simulator* self, int ndithers) {
    int ii = 0;
    double dx = 0.0;
    double dy = 0.0;
    double stepx = 1.0;
    double stepy = 1.0;
    if (!self) {
        SPH_NO_SELF
        return;
    }
    if (ndithers < 1) {
        return;
    }
    if (self->ditherpattern_x) {
        cpl_vector_delete(self->ditherpattern_x);
        self->ditherpattern_x = NULL;
    }
    if (self->ditherpattern_y) {
        cpl_vector_delete(self->ditherpattern_y);
        self->ditherpattern_y = NULL;
    }

    self->ditherpattern_x = cpl_vector_new(ndithers);
    self->ditherpattern_y = cpl_vector_new(ndithers);
    for (ii = 0; ii < ndithers; ++ii) {
        dx = dx + stepx;
        dy = dy + stepy;
        cpl_vector_set(self->ditherpattern_x, ii, dx);
        cpl_vector_set(self->ditherpattern_x, ii, dy);
    }
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Split the input image to two ADUs
 * @param self  the ngc_simulator_ccd
 * @param the input image
 * @param phase    1 or 0.
 *
 * @return pointer to a new image
 *
 * Places ADUs on detector. The resulting image has the size of the detector
 * plus the overscan and prescan regions. In case that the phase is 1,
 * the ADUs are started one pixel higher and the overscan in y is one less.
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_image*
sph_test_ngc_ccd_simulator_split_adus(sph_test_ngc_ccd_simulator* self,
        cpl_image* inimage, int phase) {
    cpl_image* result = NULL;
    int inx = 0;
    int iny = 0;
    cpl_image* leftin = NULL;
    cpl_image* rightin = NULL;
    int nnx = 0;
    int nny = 0;
    sph_error_code rerr = CPL_ERROR_NONE;
    if (!inimage) {
        SPH_NULL_ERROR
        return NULL;
    }
    if (phase != 0 && phase != 1) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Phase has to be 1 or zero!");
        return NULL;
    }

    inx = cpl_image_get_size_x(inimage);
    iny = cpl_image_get_size_y(inimage);

    /* Check that sizes work out */
    if (self->det_size_x != inx || self->det_size_y != iny) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "The size of the input pupil image must be "
                        "equal to the detector size of %d, %d pixels. ",
                self->det_size_x, self->det_size_y);
        return NULL;
    }

    /* Split the input image in two in the middle */
    leftin = cpl_image_extract(inimage, 1, 1, inx / 2, iny);
    rightin = cpl_image_extract(inimage, inx / 2 + 1, 1, inx, iny);
    if (!leftin || !rightin) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not split input image");
        return NULL;
    }

    nnx = self->adu1_presx + self->adu1_ovrsx + self->det_size_x
            + self->adu2_presx + self->adu2_ovrsx;
    nny = self->adu1_presy + self->adu1_ovrsy + self->det_size_y
            + self->adu2_presy + self->adu2_ovrsy;

    result = cpl_image_new(nnx, nny, CPL_TYPE_DOUBLE);
    if (result) {
        rerr = cpl_image_copy(result, leftin, self->adu1_presx + 1,
                self->adu1_presy + 1 + phase);
        if (rerr == CPL_ERROR_NONE) {
            rerr = cpl_image_copy(result, rightin, self->adu2_presx + 1,
                    self->adu2_presy + 1 + phase);
        }
    }
    if (!result || rerr != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not place ADUs on image.");
        if (result)
            cpl_image_delete(result);
        result = NULL;
    }

    /* Clean up */
    cpl_image_delete(leftin);
    leftin = NULL;
    cpl_image_delete(rightin);
    rightin = NULL;
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Mask odd rows in image
 * @param self    the ngc simulator
 * @param the input image to mask
 *
 * @return error code
 *
 * Masks every odd row to value zero.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_test_ngc_ccd_simulator_apply_mask(
        sph_test_ngc_ccd_simulator* self, cpl_image* inim) {
    sph_error_code rerr = CPL_ERROR_NONE;
    int xx = 0;
    int yy = 0;
    int nx = 0;
    int ny = 0;
    cpl_mask* amask = NULL;

    if (!self) {
        SPH_NO_SELF
        return CPL_ERROR_NULL_INPUT;
    }
    if (!inim) {
        return CPL_ERROR_NULL_INPUT;
    }
    nx = cpl_image_get_size_x(inim);
    ny = cpl_image_get_size_y(inim);

    amask = cpl_mask_new(nx, ny);
    if (!amask) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not create mask");
        return SPH_ERROR_GENERAL;
    }
    for (xx = 0; xx < nx; ++xx) {
        for (yy = 0; yy < ny; ++yy) {
            if (yy % 2) {
                rerr = cpl_mask_set(amask, xx + 1, yy + 1, 0);
            } else {
                rerr = cpl_mask_set(amask, xx + 1, yy + 1, 0);
            }
            if (rerr) {
                break;
            }
        }
        if (rerr) {
            break;
        }
    }
    if (rerr) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not set mask values");
        cpl_mask_delete(amask);
        return SPH_ERROR_GENERAL;
    }
    rerr = cpl_image_reject_from_mask(inim, amask);
    cpl_mask_delete(amask);
    amask = NULL;
    return rerr;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new set of raw dark frames
 * @param self  the ngc_simulator_ccd
 * @param nframes    the number of frames
 * @param exptime    the exposure time
 *
 * @return pointer to a new frameset
 *
 * This creates a set of raw dark frames, taken with the given exposure time.
 * The function creates darks for all the currently set read out mode.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_frameset*
sph_test_ngc_ccd_simulator_create_raw_frameset(sph_test_ngc_ccd_simulator* self,
        cpl_frame* inframeA, cpl_frame* inframeB, double exptime,
        const char* base_filename, cpl_propertylist* pl, const char* tag,
        int ndit) {
    cpl_frameset* result = cpl_frameset_new();
    cpl_image* animage = NULL;
    cpl_image* aimage = NULL;
    cpl_frame* aframe = NULL;
    cpl_image* window_image = NULL;
    cpl_image* pupilim = NULL;
    cpl_propertylist* pl2 = NULL;
    int current_frame_number = 0;
    char filename[256];
    int dn = 0;
    double dx = 0.0;
    double dy = 0.0;
    sph_framecombination* framecomb = NULL;

    if (!pl) {
        pl = cpl_propertylist_new();
    }

    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_EXPTIME, exptime);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NREADS, 1);

    if (self->singlecube) {
        sprintf(filename, "%s.fits", base_filename);
        unlink(filename);
    }
    for (current_frame_number = 0; current_frame_number < ndit;
            ++current_frame_number) {
        pl2 = cpl_propertylist_load(cpl_frame_get_filename(inframeA), 0);
        cpl_propertylist_append(pl2, pl);
        if (!self->singlecube) {
            sprintf(filename, "%s_DIT_%d.fits", base_filename,
                    current_frame_number);
        }
        pupilim = cpl_image_load(cpl_frame_get_filename(inframeA),
                CPL_TYPE_DOUBLE, 0, 0);
        aimage = sph_test_ngc_ccd_simulator_split_adus(self, pupilim, 0);
        if (aimage) {
            if (self->ditherpattern_x) {
                if (cpl_vector_get_size(self->ditherpattern_x) > 0
                        && cpl_vector_get_size(self->ditherpattern_x)
                                == cpl_vector_get_size(self->ditherpattern_y)) {
                    framecomb = sph_framecombination_new();
                    sph_framecombination_set_method_gslfft(framecomb, 0.0);
                    dn = current_frame_number;
                    while (dn >= cpl_vector_get_size(self->ditherpattern_x)) {
                        dn = dn - cpl_vector_get_size(self->ditherpattern_x);
                    }
                    dx = cpl_vector_get(self->ditherpattern_x, dn);
                    dy = cpl_vector_get(self->ditherpattern_x, dn);
                    animage = sph_framecombination_shift_image(framecomb,
                            aimage, dx, dy);
                    cpl_propertylist_append_double(pl2,
                            SPH_COMMON_KEYWORD_SPH_DITHERX, dx);
                    cpl_propertylist_append_double(pl2,
                            SPH_COMMON_KEYWORD_SPH_DITHERY, dy);
                    sph_framecombination_delete(framecomb);
                } else {
                    animage = cpl_image_duplicate(aimage);
                }
                cpl_image_delete(pupilim);
                pupilim = NULL;
            } else {
                animage = cpl_image_duplicate(aimage);
            }
            cpl_image_delete(aimage);
        }
        if (animage) {
            sph_test_ngc_ccd_simulator_image_add_bias_and_dark(self, animage,
                    exptime);
            sph_test_ngc_ccd_simulator_image_add_noise(self, animage);
            sph_test_ngc_ccd_simulator_saveimage(self, &window_image, animage,
                    filename, pl2);
        }
        if (!self->singlecube) {
            aframe = cpl_frame_new();
            cpl_frame_set_filename(aframe, filename);
            cpl_frame_set_group(aframe, CPL_FRAME_GROUP_RAW);
            cpl_frame_set_tag(aframe, tag);
            cpl_frameset_insert(result, aframe);
        }
        cpl_propertylist_delete(pl2);
        pl2 = NULL;
        if (animage) {
            cpl_image_delete(animage);
            animage = NULL;
        }
        pupilim = cpl_image_load(cpl_frame_get_filename(inframeB),
                CPL_TYPE_DOUBLE, 0, 0);
        aimage = sph_test_ngc_ccd_simulator_split_adus(self, pupilim, 1);
        if (aimage) {
            if (self->ditherpattern_x) {
                if (cpl_vector_get_size(self->ditherpattern_x) > 0
                        && cpl_vector_get_size(self->ditherpattern_x)
                                == cpl_vector_get_size(self->ditherpattern_y)) {
                    framecomb = sph_framecombination_new();
                    sph_framecombination_set_method_gslfft(framecomb, 0.0);
                    dn = current_frame_number;
                    while (dn >= cpl_vector_get_size(self->ditherpattern_x)) {
                        dn = dn - cpl_vector_get_size(self->ditherpattern_x);
                    }
                    dx = cpl_vector_get(self->ditherpattern_x, dn);
                    dy = cpl_vector_get(self->ditherpattern_x, dn);
                    animage = sph_framecombination_shift_image(framecomb,
                            aimage, dx, dy);
                    cpl_propertylist_append_double(pl2,
                            SPH_COMMON_KEYWORD_SPH_DITHERX, dx);
                    cpl_propertylist_append_double(pl2,
                            SPH_COMMON_KEYWORD_SPH_DITHERY, dy);
                    sph_framecombination_delete(framecomb);
                } else {
                    animage = cpl_image_duplicate(aimage);
                }
                cpl_image_delete(pupilim);
                pupilim = NULL;
            } else {
                animage = cpl_image_duplicate(aimage);
            }
            cpl_image_delete(aimage);
        }
        if (animage) {
            sph_test_ngc_ccd_simulator_image_add_bias_and_dark(self, animage,
                    exptime);
            sph_test_ngc_ccd_simulator_image_add_noise(self, animage);
            sph_test_ngc_ccd_simulator_saveimage(self, &window_image, animage,
                    filename, pl2);
        }
        if (!self->singlecube) {
            aframe = cpl_frame_new();
            cpl_frame_set_filename(aframe, filename);
            cpl_frame_set_group(aframe, CPL_FRAME_GROUP_RAW);
            cpl_frame_set_tag(aframe, tag);
            cpl_frameset_insert(result, aframe);
        }
        cpl_propertylist_delete(pl2);
        pl2 = NULL;
        if (animage) {
            cpl_image_delete(animage);
            animage = NULL;
        }
    }
    if (self->singlecube) {
        aframe = cpl_frame_new();
        cpl_frame_set_filename(aframe, filename);
        cpl_frame_set_group(aframe, CPL_FRAME_GROUP_RAW);
        cpl_frame_set_tag(aframe, SPH_IRD_TAG_DARK_RAW);
        cpl_frameset_insert(result, aframe);
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new set of raw dark frames
 * @param self  the ngc_simulator_ccd
 * @param nframes    the number of frames
 * @param exptime    the exposure time
 *
 * @return pointer to a new frameset
 *
 * This creates a set of raw dark frames, taken with the given exposure time.
 * The function creates darks for all the currently set read out mode.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_frameset*
sph_test_ngc_ccd_simulator_create_raw_frameset_many(
        sph_test_ngc_ccd_simulator* self, cpl_frameset* inframes,
        double exptime, const char* base_filename, cpl_propertylist* pl,
        const char* tag) {
    cpl_frameset* result = cpl_frameset_new();
    cpl_image* animage = NULL;
    cpl_image* aimage = NULL;
    cpl_frame* aframe = NULL;
    cpl_frame* inframe = NULL;
    cpl_image* window_image = NULL;
    cpl_image* pupilim = NULL;
    cpl_propertylist* pl2 = NULL;
    int current_frame_number = 0;
    char filename[256];
    int ndit = 0;
    int dn = 0;
    double dx = 0.0;
    double dy = 0.0;
    sph_framecombination* framecomb = NULL;

    if (!pl) {
        pl = cpl_propertylist_new();
    }

    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_EXPTIME, exptime);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NREADS, 1);

    ndit = cpl_frameset_get_size(inframes);
    if (self->singlecube) {
        sprintf(filename, "%s.fits", base_filename);
        unlink(filename);
    }
    for (current_frame_number = 0; current_frame_number < ndit;
            ++current_frame_number) {
        inframe = cpl_frameset_get_position(inframes, current_frame_number);
        pl2 = cpl_propertylist_load(cpl_frame_get_filename(inframe), 0);
        cpl_propertylist_append(pl2, pl);
        if (!self->singlecube) {
            sprintf(filename, "%s_DIT_%d.fits", base_filename,
                    current_frame_number);
        }
        pupilim = cpl_image_load(cpl_frame_get_filename(inframe),
                CPL_TYPE_DOUBLE, 0, 0);
        if (current_frame_number % 2) {
            aimage = sph_test_ngc_ccd_simulator_split_adus(self, pupilim, 0);
        } else {
            aimage = sph_test_ngc_ccd_simulator_split_adus(self, pupilim, 1);
        }
        if (aimage) {
            if (self->ditherpattern_x) {
                if (cpl_vector_get_size(self->ditherpattern_x) > 0
                        && cpl_vector_get_size(self->ditherpattern_x)
                                == cpl_vector_get_size(self->ditherpattern_y)) {
                    framecomb = sph_framecombination_new();
                    sph_framecombination_set_method_gslfft(framecomb, 0.0);
                    dn = current_frame_number;
                    while (dn >= cpl_vector_get_size(self->ditherpattern_x)) {
                        dn = dn - cpl_vector_get_size(self->ditherpattern_x);
                    }
                    dx = cpl_vector_get(self->ditherpattern_x, dn);
                    dy = cpl_vector_get(self->ditherpattern_x, dn);
                    animage = sph_framecombination_shift_image(framecomb,
                            aimage, dx, dy);
                    cpl_propertylist_append_double(pl2,
                            SPH_COMMON_KEYWORD_SPH_DITHERX, dx);
                    cpl_propertylist_append_double(pl2,
                            SPH_COMMON_KEYWORD_SPH_DITHERY, dy);
                    sph_framecombination_delete(framecomb);
                } else {
                    animage = cpl_image_duplicate(aimage);
                }
                cpl_image_delete(pupilim);
                pupilim = NULL;
            } else {
                animage = cpl_image_duplicate(aimage);
            }
            cpl_image_delete(aimage);
        }
        if (animage) {
            sph_test_ngc_ccd_simulator_image_add_bias_and_dark(self, animage,
                    exptime);
            sph_test_ngc_ccd_simulator_image_add_noise(self, animage);
            sph_test_ngc_ccd_simulator_saveimage(self, &window_image, animage,
                    filename, pl2);
        }
        if (!self->singlecube) {
            aframe = cpl_frame_new();
            cpl_frame_set_filename(aframe, filename);
            cpl_frame_set_group(aframe, CPL_FRAME_GROUP_RAW);
            cpl_frame_set_tag(aframe, tag);
            cpl_frameset_insert(result, aframe);
        }
        cpl_propertylist_delete(pl2);
        pl2 = NULL;
        if (animage) {
            cpl_image_delete(animage);
            animage = NULL;
        }
        cpl_image_delete(pupilim);
        pupilim = NULL;
    }
    if (self->singlecube) {
        aframe = cpl_frame_new();
        cpl_frame_set_filename(aframe, filename);
        cpl_frame_set_group(aframe, CPL_FRAME_GROUP_RAW);
        cpl_frame_set_tag(aframe, SPH_IRD_TAG_DARK_RAW);
        cpl_frameset_insert(result, aframe);
    }
    cpl_image_delete(window_image);
    window_image = NULL;
    cpl_propertylist_delete(pl);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the ngc_simulator_ccd
 * @param self pointer to the ngc_simulator_ccd to delete
 *
 */
/*----------------------------------------------------------------------------*/
void sph_test_ngc_ccd_simulator_delete(sph_test_ngc_ccd_simulator* self) {
    if (self) {
        if (self->pRNG) {
            gsl_rng_free(self->pRNG);
        }
        if (self->ditherpattern_x) {
            cpl_vector_delete(self->ditherpattern_x);
        }
        if (self->ditherpattern_y) {
            cpl_vector_delete(self->ditherpattern_y);
        }
        cpl_free(self);
    }
}

