/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include <math.h>
#include "sph_error.h"
#include "sph_test_validations.h"
#include "sph_fits.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides ...
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Validate that a frame has the expected structure (extensions, planes,..)
 * @param inframe    the frame
 * @param minext    the minimum extensions
 * @param maxext    the maximum number of extensions ( -1 for infinite )
 * @param minplanes    the minimum number of planes per image extension
 * @param maxplanes    the maximum number of planes per image extension
 * @param ntabs        the number of table extensions
 *
 * @return
 *
 */
/*----------------------------------------------------------------------------*/
sph_test_validation_result sph_test_validation_framestructure(
        cpl_frame* inframe, int minext, int maxext, int minplanes,
        int maxplanes, int ntabs) {
    sph_test_validation_result result = SPH_TEST_VALIDATION_RESULT_VALID;
    int ii = 0;
    int ntab = 0;
    if (!inframe) {
        return SPH_TEST_VALIDATION_RESULT_ERROR;
    }
    if (cpl_frame_get_nextensions(inframe) < minext
            || (cpl_frame_get_nextensions(inframe) > maxext && maxext >= 0)) {
        result = SPH_TEST_VALIDATION_RESULT_INVALID;
    } else {
        for (ii = 0; ii < cpl_frame_get_nextensions(inframe); ++ii) {
            if (minplanes < 2
                    && sph_fits_get_nplanes(cpl_frame_get_filename(inframe), ii)
                            < 2) {
                result = SPH_TEST_VALIDATION_RESULT_VALID;
            } else if (minplanes > 1
                    && sph_fits_get_nplanes(cpl_frame_get_filename(inframe), ii)
                            < 1) {
                cpl_error_reset();
                result = SPH_TEST_VALIDATION_RESULT_INVALID;
            } else if (sph_fits_get_nplanes(cpl_frame_get_filename(inframe), 0)
                    < minplanes
                    || (sph_fits_get_nplanes(cpl_frame_get_filename(inframe), 0)
                            > maxplanes && maxplanes >= 0)) {
                result = SPH_TEST_VALIDATION_RESULT_INVALID;
            }
            if (cpl_table_load(cpl_frame_get_filename(inframe), ii, 0)) {
                ntab++;
            } else {
                cpl_error_reset();
            }
        }
    }
    if (ntabs != ntab)
        result = SPH_TEST_VALIDATION_RESULT_INVALID;
    return result;
}

/**@}*/
