/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_ird_fixtures.h"
#include "sph_test_image_tools.h"
#include "sph_ird_tags.h"
#include "sph_ird_instrument_flat.h"
#include "sph_ird_instrument_flat.c"
#include "sph_ird_instrument_flat_run.c"
#include "sph_fits.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/
static cpl_frameset* cutest_sph_ird_instrument_flat_fixture_ideal_raws(
        int nraws, int nplanes, int nx, int ny) {
    cpl_frameset* result = NULL;
    int pp = 0;
    int ff = 0;
    cpl_image* im = NULL;
    double val = 0.0;
    cpl_frame* aframe = NULL;
    cpl_propertylist* pl = NULL;
    result = cpl_frameset_new();
    pl = cpl_propertylist_new();
    for (ff = 0; ff < nraws; ++ff) {
        val = ff * 1000.0 + 500.0;
        cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_SEQ1DIT,
                ff * 1.2 + 1.2);
        cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_SEQ1DIT,
                ff * 1.2 + 1.2);
        aframe = sph_filemanager_create_temp_frame("IRD_FLAT_TEST",
                SPH_IRD_TAG_FLAT_RAW, CPL_FRAME_GROUP_RAW);
        for (pp = 0; pp < nplanes; ++pp) {
            im = sph_test_image_tools_create_flat_image_double(nx, ny, val);
            cpl_test_nonnull( im );
            sph_test_image_tools_apply_poisson_noise(im, NULL);
            if (pp == 0) {
                cpl_test_eq(
                        sph_test_image_tools_save_image_as_new_cube(im, cpl_frame_get_filename(aframe), CPL_TYPE_DOUBLE,pl,CPL_IO_CREATE),
                        CPL_ERROR_NONE);
            } else {
                cpl_test_eq(
                        sph_fits_append_image( cpl_frame_get_filename(aframe),im,0),
                        CPL_ERROR_NONE);
            }
            cpl_image_delete(im);
            im = NULL;
        }cpl_test_error(CPL_ERROR_NONE);
        cpl_frameset_insert(result, aframe);
    }

    cpl_propertylist_delete(pl);
    return result;
}

static cpl_frameset* cutest_sph_ird_instrument_flat_fixture_gradient_raws(
        int nraws, int nplanes, int nx, int ny) {
    cpl_frameset* result = NULL;
    int pp = 0;
    int ff = 0;
    cpl_image* im = NULL;
    double val = 0.0;
    cpl_frame* aframe = NULL;
    cpl_propertylist* pl = NULL;
    result = cpl_frameset_new();
    pl = cpl_propertylist_new();
    for (ff = 0; ff < nraws; ++ff) {
        val = ff * 1000.0 + 500.0;
        cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_SEQ1DIT,
                ff * 1.2 + 1.2);
        cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_SEQ1DIT,
                ff * 1.2 + 1.2);
        aframe = sph_filemanager_create_temp_frame("IRD_FLAT_TEST",
                SPH_IRD_TAG_FLAT_RAW, CPL_FRAME_GROUP_RAW);
        for (pp = 0; pp < nplanes; ++pp) {
            im = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
            sph_test_image_tools_add_gradient_x(im, 0.5 * val, 1.5 * val);
            cpl_test_nonnull( im );
            sph_test_image_tools_apply_poisson_noise(im, NULL);
            if (pp == 0) {
                cpl_test_eq(
                        sph_test_image_tools_save_image_as_new_cube(im, cpl_frame_get_filename(aframe), CPL_TYPE_DOUBLE,pl,CPL_IO_CREATE),
                        CPL_ERROR_NONE);
            } else {
                cpl_test_eq(
                        sph_fits_append_image( cpl_frame_get_filename(aframe),im,0),
                        CPL_ERROR_NONE);
            }
            cpl_image_delete(im);
            im = NULL;
        }cpl_test_error(CPL_ERROR_NONE);
        cpl_frameset_insert(result, aframe);
    }

    cpl_propertylist_delete(pl);
    return result;
}

static
void cutest_sph_ird_instrument_flat_fixture_const_dark(cpl_frameset* inframes,
        int nx, int ny, double val, double dit) {
    cpl_image* im = NULL;
    cpl_frame* frame = NULL;
    cpl_propertylist* pl = NULL;

    pl = cpl_propertylist_new();
    im = sph_test_image_tools_create_flat_image_double(nx, ny, val);
    cpl_test_nonnull( im );
    sph_test_image_tools_apply_poisson_noise(im, NULL);
    cpl_frameset_insert(
            inframes,
            frame = sph_filemanager_create_temp_frame("test_dark.fits",
                    SPH_IRD_TAG_DARK_RAW, CPL_FRAME_GROUP_RAW));
    cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_SEQ1DIT, dit);
    cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_SEQ1DIT, dit);
    SPH_RAISE_CPL;
    cpl_image_save(im, cpl_frame_get_filename(frame), CPL_BPP_32_SIGNED, pl,
            CPL_IO_CREATE);
    SPH_RAISE_CPL;cpl_test_error(CPL_ERROR_NONE);
    cpl_propertylist_delete(pl);
    cpl_image_delete(im);
    im = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

static sph_master_frame*
cutest_sph_ird_instrument_flat_run__(cpl_frameset* inframes) {
    sph_ird_instrument_flat* insflat = NULL;
    cpl_parameterlist* params = NULL;
    sph_master_frame* flat = NULL;
    params = sph_ird_instrument_flat_create_paramlist();

    insflat = sph_ird_instrument_flat_new(inframes, params);
    cpl_test_nonnull( insflat );

    cpl_test_eq_error(sph_ird_instrument_flat_run(insflat), CPL_ERROR_NONE);

    flat = sph_master_frame_load(insflat->outfilename, 0);
    cpl_test_nonnull( flat );

    sph_ird_instrument_flat_delete(insflat);
    insflat = NULL;
    cpl_parameterlist_delete(params);
    params = NULL;
    return flat;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ird_instrument_flat_ideal_data(void) {
    cpl_frameset* inframes = NULL;
    int nx = 512;
    int ny = 256;
    int nfiles = 3;
    int nplanes = 3;
    sph_master_frame* flat = NULL;

    inframes = cutest_sph_ird_instrument_flat_fixture_ideal_raws(nfiles,
            nplanes, nx, ny);

    cpl_frameset_insert(inframes,
            sph_test_ird_fixtures_instrument_model_frame(nx, ny));

    cutest_sph_ird_instrument_flat_fixture_const_dark(inframes, nx, ny, 0.0,
            1.0);
    
    sph_error_dump(SPH_ERROR_ERROR);
    flat = cutest_sph_ird_instrument_flat_run__(inframes);
    cpl_test_nonnull( flat );
    cpl_test_zero(sph_master_frame_get_nbads(flat));
    cpl_test_abs(sph_master_frame_get_mean(flat, NULL), 1.0, 0.00001);
    cpl_test_abs(cpl_image_get_max(flat->image), 1.0, 0.1);
    cpl_test_abs(cpl_image_get_min(flat->image), 1.0, 0.1);
    sph_master_frame_delete(flat);
    flat = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ird_instrument_flat_ideal_data_bg(void) {
    cpl_frameset* inframes = NULL;
    int nx = 512;
    int ny = 256;
    int nfiles = 3;
    int nplanes = 3;
    sph_master_frame* flat = NULL;
    cpl_propertylist* pl = NULL;

    inframes = cutest_sph_ird_instrument_flat_fixture_ideal_raws(nfiles,
            nplanes, nx, ny);

    cpl_frameset_insert(inframes,
            sph_test_ird_fixtures_instrument_model_frame(nx, ny));

    cutest_sph_ird_instrument_flat_fixture_const_dark(inframes, nx, ny, 0.0,
            1.0);
    cutest_sph_ird_instrument_flat_fixture_const_dark(inframes, nx, ny, 0.0,
            2.0);
    cutest_sph_ird_instrument_flat_fixture_const_dark(inframes, nx, ny, 0.0,
            3.7);

    flat = cutest_sph_ird_instrument_flat_run__(inframes);
    cpl_test_nonnull( flat );
    cpl_test_zero(sph_master_frame_get_nbads(flat));
    cpl_test_abs(sph_master_frame_get_mean(flat, NULL), 1.0, 0.00001);
    cpl_test_abs(cpl_image_get_max(flat->image), 1.0, 0.1);
    cpl_test_abs(cpl_image_get_min(flat->image), 1.0, 0.1);

    pl = cpl_propertylist_load("irdis_flat.fits", 0);
    cpl_test_nonnull( pl );

    cpl_test( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_FLAT_RMS ));
    cpl_test_abs(
            cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_FLAT_RMS ),
            0.016, 0.002);
    cpl_test( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_FLAT_FPN ));
    cpl_test_abs(
            cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_FLAT_FPN ),
            cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_FLAT_RMS ),
            0.01);
    cpl_test( cpl_propertylist_has( pl, SPH_IRD_KEYWORD_FLAT_FPN_LEFT ));
    cpl_test_abs(
            cpl_propertylist_get_double( pl, SPH_IRD_KEYWORD_FLAT_FPN_LEFT ),
            cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_FLAT_RMS ),
            0.01);
    cpl_test( cpl_propertylist_has( pl, SPH_IRD_KEYWORD_FLAT_FPN_RIGHT ));
    cpl_test_abs(
            cpl_propertylist_get_double( pl, SPH_IRD_KEYWORD_FLAT_FPN_RIGHT ),
            cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_FLAT_RMS ),
            0.01);

    cpl_propertylist_delete(pl);
    pl = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ird_instrument_flat_ideal_data_gradient(void) {
    cpl_frameset* inframes = NULL;
    int nx = 512;
    int ny = 256;
    int nfiles = 3;
    int nplanes = 3;
    sph_master_frame* flat = NULL;
    cpl_propertylist* pl = NULL;

    inframes = cutest_sph_ird_instrument_flat_fixture_gradient_raws(nfiles,
            nplanes, nx, ny);

    cpl_frameset_insert(inframes,
            sph_test_ird_fixtures_instrument_model_frame(nx, ny));

    cutest_sph_ird_instrument_flat_fixture_const_dark(inframes, nx, ny, 0.0,
            1.0);
    cutest_sph_ird_instrument_flat_fixture_const_dark(inframes, nx, ny, 0.0,
            2.0);
    cutest_sph_ird_instrument_flat_fixture_const_dark(inframes, nx, ny, 0.0,
            3.7);

    flat = cutest_sph_ird_instrument_flat_run__(inframes);
    cpl_test_nonnull( flat );
    cpl_test_zero(sph_master_frame_get_nbads(flat));
    cpl_test_abs(sph_master_frame_get_mean(flat, NULL), 1.0, 0.00001);

    pl = cpl_propertylist_load("irdis_flat.fits", 0);
    cpl_test_nonnull( pl );

    cpl_test( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_FLAT_RMS ));
    cpl_test_abs(
            cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_FLAT_RMS ), 0.1,
            0.1);
    cpl_test( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_FLAT_FPN ));
    cpl_test_abs(
            cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_FLAT_FPN ),
            0.016, 0.01);
    cpl_test( cpl_propertylist_has( pl, SPH_IRD_KEYWORD_FLAT_FPN_LEFT ));
    cpl_test_abs(
            cpl_propertylist_get_double( pl, SPH_IRD_KEYWORD_FLAT_FPN_LEFT ),
            0.016, 0.01);
    cpl_test( cpl_propertylist_has( pl, SPH_IRD_KEYWORD_FLAT_FPN_RIGHT ));
    cpl_test_abs(
            cpl_propertylist_get_double( pl, SPH_IRD_KEYWORD_FLAT_FPN_RIGHT ),
            0.016, 0.01);
    cpl_propertylist_delete(pl);
    pl = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing sph_ird_instrument_flat_run",
            cutest_init_testsuite, cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing simple standard case w. ideal data",
                    cutest_sph_ird_instrument_flat_ideal_data)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "Testing simple standard case w. ideal data multiple bg",
                    cutest_sph_ird_instrument_flat_ideal_data_bg)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "Testing simple standard case w. ideal data gradient",
                    cutest_sph_ird_instrument_flat_ideal_data_gradient)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
