/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "cutest_zpl_modem_efficiency.h"
#include "sph_zpl_modem_efficiency.h"
#include "sph_zpl_utils.h"
#include "sph_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_test_zpl_simulator.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"

#include <cpl.h>

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>

/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/
#define SPH_BASE "cutest_zpl_modem_efficiency"
#define SPH_CFG  SPH_BASE ".cfg"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

#ifdef SPH_TEST_MODEM_EFFICIENCY
static
void cutest_zpl_modem_create_head_file_hwpz0(const char* infile) {
    FILE* fp;
    char outfilename[256];

    sprintf(outfilename, "%s.head", infile);
    fp = fopen(outfilename, "w");
    if (fp != NULL) {
        fprintf(fp, "ESO DRS ZPL HWPZ SETTING = 0.0\n");
        fclose(fp);
    }

}

static
void cutest_zpl_modem_create_head_file_hwpz45(const char* infile) {
    FILE* fp;
    char outfilename[256];

    sprintf(outfilename, "%s.head", infile);
    fp = fopen(outfilename, "w");
    if (fp != NULL) {
        fprintf(fp, "ESO DRS ZPL HWPZ SETTING = 45.0\n");
        fclose(fp);
    }

}

static sph_keyword_manager* cutest_zpl_modem_create_cfg_file(void) {
    FILE* fp;
    sph_keyword_manager* km = NULL;

    fp = fopen(SPH_CFG, "w");
    fprintf(fp,
            "SPH.KEYWORDS.ESO_DRS_ZPL_HWPZ_SETTING.NAME = \"ESO DRS ZPL HWPZ SETTING\"\n");
    fprintf(fp, "SPH.KEYWORDS.ESO_DRS_ZPL_HWPZ_SETTING.TYPE = \"DOUBLE\"\n");
    fprintf(fp, "SPH.KEYWORDS.ESO_DRS_ZPL_HWPZ_SETTING.VALUE = \"0.00\"\n");
    fclose(fp);

    km = sph_keyword_manager_new();
    cpl_test_nonnull(km);

    return km;
}
#endif

static int cutest_init_modem_efficiency_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     *--------------------------------------------------------------------*/
    sph_test_nop_code();
    return CPL_ERROR_NONE;
}

static int cutest_clean_modem_efficiency_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_modem_efficiency only with preproc frames
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_modem_efficiency_run_1(void) {
    sph_zpl_modem_efficiency* zpl_modem_efficiency = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;
    //int                         nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    //int                         ny                  = 100; // -//-

    double dV = 1.0;
    //double              dS						= 0.0;

    double zero_odd = 169.0;
    double zero_even = 29.0;
    double pi_odd = 29.0;
    double pi_even = 169.0;
    cpl_propertylist* pl = NULL;
    char qkeyname[1024];

    inparams = sph_zpl_modem_efficiency_create_paramlist();
    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_set_qplus(&zexpcube,
            "test_modem_eff_preproc_qplus.fits.tmp", zero_odd, zero_even,
            pi_odd, pi_even, dV, nframes);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_MODEM_EFF_PREPROC_CAM1);
    cpl_frame_set_group(zexpcube, CPL_FRAME_GROUP_RAW);
    rerr = cpl_frameset_insert(inframes, zexpcube);

    zero_odd = 49.0;
    zero_even = 149.0;
    pi_odd = 149.0;
    pi_even = 49.0;
    sph_test_simulator_create_zplpreproc_cube_set_qminus(&zexpcube1,
            "test_modem_eff_preproc_qminus.fits.tmp", zero_odd, zero_even,
            pi_odd, pi_even, dV, nframes);
    cpl_frame_set_tag(zexpcube1, SPH_ZPL_TAG_MODEM_EFF_PREPROC_CAM1);
    cpl_frame_set_group(zexpcube1, CPL_FRAME_GROUP_RAW);
    rerr = cpl_frameset_insert(inframes, zexpcube1);

    zpl_modem_efficiency = sph_zpl_modem_efficiency_new(inframes, inparams);
    cpl_test_nonnull( zpl_modem_efficiency );

    rerr = sph_zpl_modem_efficiency_run(zpl_modem_efficiency);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    pl = cpl_propertylist_load(
            zpl_modem_efficiency->modem_efficiency_outfilename_cam1, 0);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME),
            0.6, 1e-10);
    cpl_propertylist_delete(pl);
    pl = NULL;

    pl = cpl_propertylist_load(
            zpl_modem_efficiency->modem_efficiency_plus_outfilename_cam1, 0);
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME, SPH_ZPL_KEYWORD_STOKES_PARAM_QPLUS);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, qkeyname),
            400.0, 1e-10);
    strcpy(qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME, SPH_ZPL_KEYWORD_STOKES_PARAM_QPLUS);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, qkeyname),
            280.0, 1e-10);
    cpl_propertylist_delete(pl);
    pl = NULL;

    pl = cpl_propertylist_load(
            zpl_modem_efficiency->modem_efficiency_minus_outfilename_cam1, 0);
    strcpy(qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME, SPH_ZPL_KEYWORD_STOKES_PARAM_QMINUS);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, qkeyname),
            400.0, 1e-10);
    strcpy(qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME, SPH_ZPL_KEYWORD_STOKES_PARAM_QMINUS);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, qkeyname),
            -200.0, 1e-10);
    cpl_propertylist_delete(pl);
    pl = NULL;

    rerr = sph_zpl_modem_efficiency_delete(zpl_modem_efficiency);
    cpl_test_error(CPL_ERROR_NONE);
    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_modem_efficiency only with preproc frames
 *      but 2 cameras
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_modem_efficiency_run_1_1(void) {
    sph_zpl_modem_efficiency* zpl_modem_efficiency = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;
    cpl_frame* zexpcube2 = NULL;
    cpl_frame* zexpcube3 = NULL;

    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;
    //int                         nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    //int                         ny                  = 100; // -//-

    double dV = 1.0;
    //double              dS						= 0.0;

    double zero_odd = 169.0;
    double zero_even = 29.0;
    double pi_odd = 29.0;
    double pi_even = 169.0;
    cpl_propertylist* pl = NULL;
    char qkeyname[1024];

    inparams = sph_zpl_modem_efficiency_create_paramlist();
    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_set_qplus(&zexpcube,
            "test_modem_eff_preproc_qplus.fits.tmp", zero_odd, zero_even,
            pi_odd, pi_even, dV, nframes);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_MODEM_EFF_PREPROC_CAM1);
    cpl_frame_set_group(zexpcube, CPL_FRAME_GROUP_RAW);
    rerr = cpl_frameset_insert(inframes, zexpcube);

    zero_odd = 49.0;
    zero_even = 149.0;
    pi_odd = 149.0;
    pi_even = 49.0;
    sph_test_simulator_create_zplpreproc_cube_set_qminus(&zexpcube1,
            "test_modem_eff_preproc_qminus.fits.tmp", zero_odd, zero_even,
            pi_odd, pi_even, dV, nframes);
    cpl_frame_set_tag(zexpcube1, SPH_ZPL_TAG_MODEM_EFF_PREPROC_CAM1);
    cpl_frame_set_group(zexpcube1, CPL_FRAME_GROUP_RAW);
    rerr = cpl_frameset_insert(inframes, zexpcube1);

    zero_odd = 169.0;
    zero_even = 29.0;
    pi_odd = 29.0;
    pi_even = 169.0;

    sph_test_simulator_create_zplpreproc_cube_set_qplus(&zexpcube2,
            "test_modem_eff_preproc_qplus2.fits.tmp", zero_odd, zero_even,
            pi_odd, pi_even, dV, nframes);
    cpl_frame_set_tag(zexpcube2, SPH_ZPL_TAG_MODEM_EFF_PREPROC_CAM2);
    cpl_frame_set_group(zexpcube2, CPL_FRAME_GROUP_RAW);
    rerr = cpl_frameset_insert(inframes, zexpcube2);

    zero_odd = 49.0;
    zero_even = 149.0;
    pi_odd = 149.0;
    pi_even = 49.0;
    sph_test_simulator_create_zplpreproc_cube_set_qminus(&zexpcube3,
            "test_modem_eff_preproc_qminus2.fits.tmp", zero_odd, zero_even,
            pi_odd, pi_even, dV, nframes);
    cpl_frame_set_tag(zexpcube3, SPH_ZPL_TAG_MODEM_EFF_PREPROC_CAM2);
    cpl_frame_set_group(zexpcube3, CPL_FRAME_GROUP_RAW);
    rerr = cpl_frameset_insert(inframes, zexpcube3);

    zpl_modem_efficiency = sph_zpl_modem_efficiency_new(inframes, inparams);
    cpl_test_nonnull( zpl_modem_efficiency );

    rerr = sph_zpl_modem_efficiency_run(zpl_modem_efficiency);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    pl = cpl_propertylist_load(
            zpl_modem_efficiency->modem_efficiency_outfilename_cam1, 0);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME),
            0.6, 1e-10);
    cpl_propertylist_delete(pl);
    pl = NULL;

    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME, SPH_ZPL_KEYWORD_STOKES_PARAM_QPLUS);
    pl = cpl_propertylist_load(
            zpl_modem_efficiency->modem_efficiency_plus_outfilename_cam1, 0);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, qkeyname),
            400.0, 1e-10);
    strcpy( qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME, SPH_ZPL_KEYWORD_STOKES_PARAM_QPLUS);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, qkeyname),
            280.0, 1e-10);
    cpl_propertylist_delete(pl);
    pl = NULL;

    pl = cpl_propertylist_load(
            zpl_modem_efficiency->modem_efficiency_minus_outfilename_cam1, 0);
    strcpy( qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME, SPH_ZPL_KEYWORD_STOKES_PARAM_QMINUS);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, qkeyname),
            400.0, 1e-10);
    strcpy( qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME, SPH_ZPL_KEYWORD_STOKES_PARAM_QMINUS);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, qkeyname),
            -200.0, 1e-10);
    cpl_propertylist_delete(pl);
    pl = NULL;

    pl = cpl_propertylist_load(
            zpl_modem_efficiency->modem_efficiency_outfilename_cam2, 0);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME),
            0.6, 1e-10);
    cpl_propertylist_delete(pl);
    pl = NULL;

    pl = cpl_propertylist_load(
            zpl_modem_efficiency->modem_efficiency_plus_outfilename_cam2, 0);
    strcpy( qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME, SPH_ZPL_KEYWORD_STOKES_PARAM_QPLUS);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, qkeyname),
            400.0, 1e-10);
    strcpy( qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME, SPH_ZPL_KEYWORD_STOKES_PARAM_QPLUS);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, qkeyname),
            280.0, 1e-10);
    cpl_propertylist_delete(pl);
    pl = NULL;

    pl = cpl_propertylist_load(
            zpl_modem_efficiency->modem_efficiency_minus_outfilename_cam2, 0);
    strcpy( qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME, SPH_ZPL_KEYWORD_STOKES_PARAM_QMINUS);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, qkeyname),
            400.0, 1e-10);
    strcpy( qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME, SPH_ZPL_KEYWORD_STOKES_PARAM_QMINUS);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, qkeyname),
            -200.0, 1e-10);
    cpl_propertylist_delete(pl);
    pl = NULL;

    rerr = sph_zpl_modem_efficiency_delete(zpl_modem_efficiency);
    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);

    cpl_test_error(CPL_ERROR_NONE);

    return;
}

#ifdef SPH_TEST_MODEM_EFFICIENCY
/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_modem_efficiency with +Q rawframes: hwpz = 0.0
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_modem_efficiency_run_2(void) {
    sph_zpl_modem_efficiency* zpl_modem_efficiency = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;
    //int                         nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    //int                         ny                  = 100; // -//-

    double dV = 1.0;
    //double              dS						= 0.0;

    double zero_odd = 169.0;
    double zero_even = 29.0;
    double pi_odd = 29.0;
    double pi_even = 169.0;
    cpl_propertylist* pl = NULL;

    inparams = sph_zpl_modem_efficiency_create_paramlist();
    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_set_hwpz( &zexpcube, "test_modem_eff_raw_hwpz_0.fits.tmp", zero_odd, zero_even, pi_odd, pi_even, dV, 0.0, nframes );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube );

    zero_odd = 172.0; zero_even = 32.0;
    pi_odd = 32.0; pi_even = 172.0;
    sph_test_simulator_create_zplpreproc_cube_set_hwpz( &zexpcube1, "test_modem_eff_raw_hwpz_45.fits.tmp", zero_odd, zero_even, pi_odd, pi_even, dV, 0.0, nframes );
    cpl_frame_set_tag( zexpcube1, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube1, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube1 );

    zpl_modem_efficiency = sph_zpl_modem_efficiency_new( inframes, inparams);
    cpl_test_nonnull( zpl_modem_efficiency );

    rerr = sph_zpl_modem_efficiency_run( zpl_modem_efficiency );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_outfilename, 0);

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME), 0.6896551724, 1e-10);
    cpl_propertylist_delete( pl ); pl = NULL;

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_plus_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), 406.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), 280.0, 1e-10);

    rerr = sph_zpl_modem_efficiency_delete( zpl_modem_efficiency );
    cpl_test_error(CPL_ERROR_NONE );

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_modem_efficiency with -Q rawframes: 45.0
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_modem_efficiency_run_3(void) {
    sph_zpl_modem_efficiency* zpl_modem_efficiency = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;
    //int                         nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    //int                         ny                  = 100; // -//-

    double dV = 1.0;
    //double              dS						= 0.0;

    double zero_odd = 169.0;
    double zero_even = 29.0;
    double pi_odd = 29.0;
    double pi_even = 169.0;
    cpl_propertylist* pl = NULL;

    inparams = sph_zpl_modem_efficiency_create_paramlist();
    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_set_hwpz( &zexpcube, "test_modem_eff_raw_hwpz_0.fits.tmp", zero_odd, zero_even, pi_odd, pi_even, dV, 45.0, nframes );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube );

    zero_odd = 172.0; zero_even = 32.0;
    pi_odd = 32.0; pi_even = 172.0;
    sph_test_simulator_create_zplpreproc_cube_set_hwpz( &zexpcube1, "test_modem_eff_raw_hwpz_45.fits.tmp", zero_odd, zero_even, pi_odd, pi_even, dV, 45.0, nframes );
    cpl_frame_set_tag( zexpcube1, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube1, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube1 );

    zpl_modem_efficiency = sph_zpl_modem_efficiency_new( inframes, inparams);
    cpl_test_nonnull( zpl_modem_efficiency );

    rerr = sph_zpl_modem_efficiency_run( zpl_modem_efficiency );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_outfilename, 0);

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME), 0.6896551724, 1e-10);

    rerr = sph_zpl_modem_efficiency_delete( zpl_modem_efficiency );
    cpl_test_error(CPL_ERROR_NONE );

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_modem_efficiency without hwpz keywords
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_modem_efficiency_run_4(void) {
    sph_zpl_modem_efficiency* zpl_modem_efficiency = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;
    //int                         nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    //int                         ny                  = 100; // -//-

    double dV = 1.0;
    //double              dS						= 0.0;

    double zero_odd = 169.0;
    double zero_even = 29.0;
    double pi_odd = 29.0;
    double pi_even = 169.0;
    cpl_propertylist* pl = NULL;

    inparams = sph_zpl_modem_efficiency_create_paramlist();
    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_set_filename( &zexpcube, "test_modem_eff_raw_1.fits.tmp", zero_odd, zero_even, pi_odd, pi_even, dV, nframes );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube );

    zero_odd = 172.0; zero_even = 32.0;
    pi_odd = 32.0; pi_even = 172.0;
    sph_test_simulator_create_zplpreproc_cube_set_filename( &zexpcube1, "test_modem_eff_raw_2.fits.tmp", zero_odd, zero_even, pi_odd, pi_even, dV, nframes );
    cpl_frame_set_tag( zexpcube1, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube1, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube1 );

    zpl_modem_efficiency = sph_zpl_modem_efficiency_new( inframes, inparams);
    cpl_test_nonnull( zpl_modem_efficiency );

    rerr = sph_zpl_modem_efficiency_run( zpl_modem_efficiency );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_outfilename, 0);

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME), 0.6896551724, 1e-10);

    rerr = sph_zpl_modem_efficiency_delete( zpl_modem_efficiency );
    cpl_test_error(CPL_ERROR_NONE );

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_modem_efficiency with rawframes & bias, dark, iff, pff,  calibrations
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_modem_efficiency_run_5(void) {
    sph_zpl_modem_efficiency* zpl_modem_efficiency = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100;// -//-

    double dV = 1.0;
    //double              dS						= 0.0;

    double zero_odd = 169.0;
    double zero_even = 29.0;
    double pi_odd = 29.0;
    double pi_even = 169.0;
    cpl_propertylist* pl = NULL;
    cpl_frame* master_bias_frame = NULL;
    cpl_frame* master_dark_frame = NULL;
    cpl_frame* intensity_flat_frame = NULL;
    cpl_frame* polarization_flat_frame = NULL;

    inparams = sph_zpl_modem_efficiency_create_paramlist();
    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_set_hwpz( &zexpcube, "modem_eff_raw_hwpz_0.fits.tmp", zero_odd, zero_even, pi_odd, pi_even, dV, 0.0, nframes );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube );

    zero_odd = 49.0; zero_even = 149.0;
    pi_odd = 149.0; pi_even = 49.0;
    sph_test_simulator_create_zplpreproc_cube_set_hwpz( &zexpcube1, "modem_eff_raw_hwpz_45.fits.tmp", zero_odd, zero_even, pi_odd, pi_even, dV, 45.0, nframes );
    cpl_frame_set_tag( zexpcube1, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube1, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube1 );

    //simualate master bias
    zero_odd = 10; zero_even = 11; pi_odd = 12; pi_even = 14;
    master_bias_frame = sph_test_zpl_simulator_create_quad_image_frame("test_master_bias_calib.fits.tmp", nx, ny, zero_odd, zero_even, pi_odd, pi_even);
    cpl_frame_set_tag( master_bias_frame, SPH_ZPL_TAG_BIAS_CALIB );
    cpl_frame_set_group( master_bias_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, master_bias_frame );

    //simualate master dark
    zero_odd = 1; zero_even = 2; pi_odd = 3; pi_even = 5;
    master_dark_frame = sph_test_zpl_simulator_create_quad_image_frame("test_master_dark_calib.fits.tmp", nx, ny, zero_odd, zero_even, pi_odd, pi_even);
    cpl_frame_set_tag( master_dark_frame, SPH_ZPL_TAG_DARK_CALIB );
    cpl_frame_set_group( master_dark_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, master_dark_frame );

    //simualate intensity flat
    zero_odd = 2; zero_even = 2; pi_odd = 2; pi_even = 2;
    intensity_flat_frame = sph_test_zpl_simulator_create_quad_image_frame("test_master_intensity_flat_calib.fits.tmp", nx, ny, zero_odd, zero_even, pi_odd, pi_even);
    cpl_frame_set_tag( intensity_flat_frame, SPH_ZPL_TAG_IFF_CALIB );
    cpl_frame_set_group( intensity_flat_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, intensity_flat_frame );

    //simualate polarization flat
    polarization_flat_frame = sph_test_zpl_simulator_create_double_image_frame("test_master_polarization_flat_calib.fits.tmp", nx, ny, 200.0, 100.0 );
    cpl_frame_set_tag( polarization_flat_frame, SPH_ZPL_TAG_PFF_CALIB );
    cpl_frame_set_group( polarization_flat_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, polarization_flat_frame );

    zpl_modem_efficiency = sph_zpl_modem_efficiency_new( inframes, inparams);
    cpl_test_nonnull( zpl_modem_efficiency );

    rerr = sph_zpl_modem_efficiency_run( zpl_modem_efficiency );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME), 0.701754386, 1e-10);
    cpl_propertylist_delete( pl ); pl = NULL;

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_plus_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), 171.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), 53.5, 1e-10);
    cpl_propertylist_delete( pl ); pl = NULL;

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_minus_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), 171.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), -186.5, 1e-10);

    rerr = sph_zpl_modem_efficiency_delete( zpl_modem_efficiency );
    cpl_test_error(CPL_ERROR_NONE );

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_modem_efficiency without hwpz keywords but with config and head-files
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_modem_efficiency_run_6(void) {
    sph_zpl_modem_efficiency* zpl_modem_efficiency = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;
    //int                         nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    //int                         ny                  = 100; // -//-

    double dV = 1.0;
    //double              dS						= 0.0;

    double zero_odd = 169.0;
    double zero_even = 29.0;
    double pi_odd = 29.0;
    double pi_even = 169.0;
    cpl_propertylist* pl = NULL;

    sph_keyword_manager* keyman = NULL;

    cpl_error_reset();

    //create config
    keyman = cutest_zpl_modem_create_cfg_file();
    cpl_test_nonnull( keyman );

    inparams = sph_zpl_modem_efficiency_create_paramlist();
    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_set_filename( &zexpcube, "test_modem_eff_raw_1.fits", zero_odd, zero_even, pi_odd, pi_even, dV, nframes );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube );

    //create .head file
    cutest_zpl_modem_create_head_file_hwpz0("test_modem_eff_raw_1");

    zero_odd = 49.0; zero_even = 149.0;
    pi_odd = 149.0; pi_even = 49.0;
    sph_test_simulator_create_zplpreproc_cube_set_filename( &zexpcube1, "test_modem_eff_raw_2.fits", zero_odd, zero_even, pi_odd, pi_even, dV, nframes );
    cpl_frame_set_tag( zexpcube1, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube1, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube1 );
    //create .head file
    cutest_zpl_modem_create_head_file_hwpz45("test_modem_eff_raw_2");

    zpl_modem_efficiency = sph_zpl_modem_efficiency_new( inframes, inparams);
    cpl_test_nonnull( zpl_modem_efficiency );

    rerr = sph_zpl_modem_efficiency_run( zpl_modem_efficiency );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME), 0.6, 1e-10);
    cpl_propertylist_delete( pl ); pl = NULL;

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_plus_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), 400.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), 280.0, 1e-10);
    cpl_propertylist_delete( pl ); pl = NULL;

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_minus_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), 400.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), -200.0, 1e-10);

    rerr = sph_zpl_modem_efficiency_delete( zpl_modem_efficiency );
    cpl_test_error(CPL_ERROR_NONE );
    sph_keyword_manager_delete();

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_modem_efficiency
 @note It is fed by raw pre-processed frame with overscan table with hwpz=0 (+Q) keywords
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_modem_efficiency_run_7(void) {
    sph_zpl_modem_efficiency* zpl_modem_efficiency = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube_minus = NULL;
    cpl_frame* zexpcube_plus = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nz = 3;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100;// -//-
    int zero_odd = 169;
    int zero_even = 29;
    int pi_odd = 29;
    int pi_even = 169;
    double ovsc[8] = {5.0, 1.0, 6.0, 2.0, 7.0, 3.0, 8.0, 4.0};
    //double 					ovsc[8]		= {0.0, 1.0, 0.0, 2.0, 0.0, 3.0, 0.0, 4.0};

    int zero_odd_1 = 172;
    int zero_even_1 = 32;
    int pi_odd_1 = 32;
    int pi_even_1 = 172;
    double ovsc_1[8] = {15.0, 1.0, 16.0, 2.0, 17.0, 3.0, 18.0, 4.0};
    //double 					ovsc_1[8]		= {0.0, 1.0, 0.0, 2.0, 0.0, 3.0, 0.0, 4.0};
    double adu1_zero_odd_plus = 0;
    double adu2_zero_odd_plus = 0;
    double adu1_zero_even_plus = 0;
    double adu2_zero_even_plus = 0;
    double adu1_pi_odd_plus = 0;
    double adu2_pi_odd_plus = 0;
    double adu1_pi_even_plus = 0;
    double adu2_pi_even_plus = 0;

    double adu1_zero_odd_minus = 0;
    double adu2_zero_odd_minus = 0;
    double adu1_zero_even_minus = 0;
    double adu2_zero_even_minus = 0;
    double adu1_pi_odd_minus = 0;
    double adu2_pi_odd_minus = 0;
    double adu1_pi_even_minus = 0;
    double adu2_pi_even_minus = 0;
    double iframe = 0.0;
    double pframe = 0.0;
    double iframe_adu1 = 0.0;
    double pframe_adu1 = 0.0;
    double iframe_adu2 = 0.0;
    double pframe_adu2 = 0.0;

    double pframe_over_iframe = 0.0;

    inparams = sph_zpl_modem_efficiency_create_paramlist();
    inframes = cpl_frameset_new();

    // zpl frame cube with overscan table
    zexpcube_plus = sph_test_simulator_create_zpl_exposure_overscans_cube_set_hwpz( nx, ny, nz,
            "test_modem_efficiency_zplexp_ovsc_cube", zero_odd, zero_even, pi_odd, pi_even, ovsc,
            1, 1, 0.0);
    cpl_frame_set_tag( zexpcube_plus, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW);
    rerr = cpl_frameset_insert( inframes, zexpcube_plus );

    // zpl frame cube with overscan table
    zexpcube_minus = sph_test_simulator_create_zpl_exposure_overscans_cube_set_hwpz( nx, ny, nz,
            "test_modem_efficiency_zplexp_ovsc_cube_1", zero_odd_1, zero_even_1, pi_odd_1,
            pi_even_1, ovsc_1, 1, 0, 0.0 );
    cpl_frame_set_tag( zexpcube_minus, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW);
    rerr = cpl_frameset_insert( inframes, zexpcube_minus );

    zpl_modem_efficiency = sph_zpl_modem_efficiency_new( inframes, inparams );
    cpl_test_nonnull( zpl_modem_efficiency  );

    rerr = sph_zpl_modem_efficiency_run( zpl_modem_efficiency );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_outfilename, 0);

    //calculate expected values
    adu1_zero_odd_plus = (169.-ovsc[0] +170.-(ovsc[0]+1)+171.-(ovsc[0]+2));
    adu2_zero_odd_plus = (169.-ovsc[2] +170.-(ovsc[2]+1)+171.-(ovsc[2]+2));

    adu1_zero_even_plus = (29.-ovsc[0] +30.-(ovsc[0]+1)+31.-(ovsc[0]+2));
    adu2_zero_even_plus = (29.-ovsc[2] +30.-(ovsc[2]+1)+31.-(ovsc[2]+2));

    adu1_pi_odd_plus = (29.-ovsc[4] +30.-(ovsc[4]+1)+31.-(ovsc[4]+2));
    adu2_pi_odd_plus = (29.-ovsc[6] +30.-(ovsc[6]+1)+31.-(ovsc[6]+2));

    adu1_pi_even_plus = (169.-ovsc[4] +170.-(ovsc[4]+1)+171.-(ovsc[4]+2));
    adu2_pi_even_plus = (169.-ovsc[6] +170.-(ovsc[6]+1)+171.-(ovsc[6]+2));

    adu1_zero_odd_minus = (172.-ovsc_1[0]+173.-ovsc_1[0]+174.-ovsc_1[0]);
    adu2_zero_odd_minus = (172.-ovsc_1[2]+173.-ovsc_1[2]+174.-ovsc_1[2]);

    adu1_zero_even_minus = (32.-ovsc_1[0]+33.-ovsc_1[0]+34.-ovsc_1[0]);
    adu2_zero_even_minus = (32.-ovsc_1[2]+33.-ovsc_1[2]+34.-ovsc_1[2]);

    adu1_pi_odd_minus = (32.-ovsc_1[4]+33.-ovsc_1[4]+34.-ovsc_1[4]);
    adu2_pi_odd_minus = (32.-ovsc_1[6]+33.-ovsc_1[6]+34.-ovsc_1[6]);

    adu1_pi_even_minus = (172.-ovsc_1[4]+173.-ovsc_1[4]+174.-ovsc_1[4]);
    adu2_pi_even_minus = (172.-ovsc_1[6]+173.-ovsc_1[6]+174.-ovsc_1[6]);

    iframe_adu1 = (adu1_zero_odd_plus + adu1_zero_odd_minus)/6. + (adu1_zero_even_plus + adu1_zero_even_minus)/6. +
    (adu1_pi_odd_plus + adu1_pi_odd_minus)/6. + (adu1_pi_even_plus + adu1_pi_even_minus)/6.;

    iframe_adu2 = (adu2_zero_odd_plus + adu2_zero_odd_minus)/6. + (adu2_zero_even_plus + adu2_zero_even_minus)/6. +
    (adu2_pi_odd_plus + adu2_pi_odd_minus)/6. + (adu2_pi_even_plus + adu2_pi_even_minus)/6.;

    pframe_adu1 = (adu1_zero_odd_plus + adu1_zero_odd_minus)/6. - (adu1_zero_even_plus + adu1_zero_even_minus)/6. -
    (adu1_pi_odd_plus + adu1_pi_odd_minus)/6. + (adu1_pi_even_plus + adu1_pi_even_minus)/6.;

    pframe_adu2 = (adu2_zero_odd_plus + adu2_zero_odd_minus)/6. - (adu2_zero_even_plus + adu2_zero_even_minus)/6. -
    (adu2_pi_odd_plus + adu2_pi_odd_minus)/6. + (adu2_pi_even_plus + adu2_pi_even_minus)/6.;

    // both input cubes were simulated with the same hwpz = 0, i.e. they are both plus Q
    iframe = (iframe_adu1 + iframe_adu2)/2.;
    pframe = (pframe_adu1 + pframe_adu2)/2.;
    pframe_over_iframe = (pframe_adu1/iframe_adu1 + pframe_adu2/iframe_adu2)/2.;

    //printf("iframe = %f\n", iframe );
    //printf("pframe = %f\n", pframe );
    //printf("pframe/iframe = %f\n", pframe/iframe );
    //printf("PFRAME/IFRAME = %f\n", pframe_over_iframe );

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME), pframe_over_iframe, 1e-10);
    //printf("SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME = %f\n", cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME));

    cpl_propertylist_delete( pl ); pl = NULL;

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_plus_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), iframe, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), pframe, 1e-10);

    rerr = sph_zpl_modem_efficiency_delete( zpl_modem_efficiency );
    cpl_test_error(CPL_ERROR_NONE );

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete( inframes );
    cpl_parameterlist_delete(inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_modem_efficiency
 @note It is fed by raw pre-processed frame with overscan table with hwpz=0(+Q) and hwpz=45(-Q) keywords
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_modem_efficiency_run_8(void) {
    sph_zpl_modem_efficiency* zpl_modem_efficiency = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube_minus = NULL;
    cpl_frame* zexpcube_plus = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nz = 3;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100;// -//-
    int zero_odd = 169;
    int zero_even = 29;
    int pi_odd = 29;
    int pi_even = 169;
    double ovsc[8] = {5.0, 1.0, 6.0, 2.0, 7.0, 3.0, 8.0, 4.0};
    //double 					ovsc[8]		= {0.0, 1.0, 0.0, 2.0, 0.0, 3.0, 0.0, 4.0};

    int zero_odd_1 = 172;
    int zero_even_1 = 32;
    int pi_odd_1 = 32;
    int pi_even_1 = 172;
    double ovsc_1[8] = {15.0, 1.0, 16.0, 2.0, 17.0, 3.0, 18.0, 4.0};
    //double 					ovsc_1[8]		= {0.0, 1.0, 0.0, 2.0, 0.0, 3.0, 0.0, 4.0};
    double adu1_zero_odd_plus = 0;
    double adu2_zero_odd_plus = 0;
    double adu1_zero_even_plus = 0;
    double adu2_zero_even_plus = 0;
    double adu1_pi_odd_plus = 0;
    double adu2_pi_odd_plus = 0;
    double adu1_pi_even_plus = 0;
    double adu2_pi_even_plus = 0;
    double iframe_plus = 0.0;
    double pframe_plus = 0.0;

    double adu1_zero_odd_minus = 0;
    double adu2_zero_odd_minus = 0;
    double adu1_zero_even_minus = 0;
    double adu2_zero_even_minus = 0;
    double adu1_pi_odd_minus = 0;
    double adu2_pi_odd_minus = 0;
    double adu1_pi_even_minus = 0;
    double adu2_pi_even_minus = 0;
    double iframe_minus = 0.0;
    double pframe_minus = 0.0;

    double iframe_adu1_plus = 0.0;
    double pframe_adu1_plus = 0.0;
    double iframe_adu2_plus = 0.0;
    double pframe_adu2_plus = 0.0;

    double iframe_adu1_minus = 0.0;
    double pframe_adu1_minus = 0.0;
    double iframe_adu2_minus = 0.0;
    double pframe_adu2_minus = 0.0;

    double pframe_over_iframe = 0.0;
    double pframe_over_iframe_plus = 0.0;
    double pframe_over_iframe_minus = 0.0;

    inparams = sph_zpl_modem_efficiency_create_paramlist();
    inframes = cpl_frameset_new();

    // zpl frame cube with overscan table
    zexpcube_plus = sph_test_simulator_create_zpl_exposure_overscans_cube_set_hwpz( nx, ny, nz,
            "test_modem_efficiency_zplexp_ovsc_cube", zero_odd, zero_even, pi_odd, pi_even, ovsc,
            1, 1, 0.0);
    cpl_frame_set_tag( zexpcube_plus, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW);
    rerr = cpl_frameset_insert( inframes, zexpcube_plus );

    // zpl frame cube with overscan table
    zexpcube_minus = sph_test_simulator_create_zpl_exposure_overscans_cube_set_hwpz( nx, ny, nz,
            "test_modem_efficiency_zplexp_ovsc_cube_1", zero_odd_1, zero_even_1, pi_odd_1,
            pi_even_1, ovsc_1, 1, 0, 45.0 );
    cpl_frame_set_tag( zexpcube_minus, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW);
    rerr = cpl_frameset_insert( inframes, zexpcube_minus );

    zpl_modem_efficiency = sph_zpl_modem_efficiency_new( inframes, inparams );
    cpl_test_nonnull( zpl_modem_efficiency  );

    rerr = sph_zpl_modem_efficiency_run( zpl_modem_efficiency );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_outfilename, 0);

    //calculate expected values
    adu1_zero_odd_plus = (169.-ovsc[0] +170.-(ovsc[0]+1)+171.-(ovsc[0]+2));
    adu2_zero_odd_plus = (169.-ovsc[2] +170.-(ovsc[2]+1)+171.-(ovsc[2]+2));

    adu1_zero_even_plus = (29.-ovsc[0] +30.-(ovsc[0]+1)+31.-(ovsc[0]+2));
    adu2_zero_even_plus = (29.-ovsc[2] +30.-(ovsc[2]+1)+31.-(ovsc[2]+2));

    adu1_pi_odd_plus = (29.-ovsc[4] +30.-(ovsc[4]+1)+31.-(ovsc[4]+2));
    adu2_pi_odd_plus = (29.-ovsc[6] +30.-(ovsc[6]+1)+31.-(ovsc[6]+2));

    adu1_pi_even_plus = (169.-ovsc[4] +170.-(ovsc[4]+1)+171.-(ovsc[4]+2));
    adu2_pi_even_plus = (169.-ovsc[6] +170.-(ovsc[6]+1)+171.-(ovsc[6]+2));

    adu1_zero_odd_minus = (172.-ovsc_1[0]+173.-ovsc_1[0]+174.-ovsc_1[0]);
    adu2_zero_odd_minus = (172.-ovsc_1[2]+173.-ovsc_1[2]+174.-ovsc_1[2]);

    adu1_zero_even_minus = (32.-ovsc_1[0]+33.-ovsc_1[0]+34.-ovsc_1[0]);
    adu2_zero_even_minus = (32.-ovsc_1[2]+33.-ovsc_1[2]+34.-ovsc_1[2]);

    adu1_pi_odd_minus = (32.-ovsc_1[4]+33.-ovsc_1[4]+34.-ovsc_1[4]);
    adu2_pi_odd_minus = (32.-ovsc_1[6]+33.-ovsc_1[6]+34.-ovsc_1[6]);

    adu1_pi_even_minus = (172.-ovsc_1[4]+173.-ovsc_1[4]+174.-ovsc_1[4]);
    adu2_pi_even_minus = (172.-ovsc_1[6]+173.-ovsc_1[6]+174.-ovsc_1[6]);

    iframe_adu1_plus = (adu1_zero_odd_plus)/3. + (adu1_zero_even_plus)/3. +
    (adu1_pi_odd_plus)/3. + (adu1_pi_even_plus)/3.;

    iframe_adu2_plus = (adu2_zero_odd_plus)/3. + (adu2_zero_even_plus)/3. +
    (adu2_pi_odd_plus)/3. + (adu2_pi_even_plus)/3.;

    pframe_adu1_plus = (adu1_zero_odd_plus)/3. - (adu1_zero_even_plus)/3. -
    (adu1_pi_odd_plus)/3. + (adu1_pi_even_plus)/3.;

    pframe_adu2_plus = (adu2_zero_odd_plus)/3. - (adu2_zero_even_plus)/3. -
    (adu2_pi_odd_plus)/3. + (adu2_pi_even_plus)/3.;

    iframe_adu1_minus = (adu1_zero_odd_minus)/3. + (adu1_zero_even_minus)/3. +
    (adu1_pi_odd_minus)/3. + (adu1_pi_even_minus)/3.;

    iframe_adu2_minus = (adu2_zero_odd_minus)/3. + (adu2_zero_even_minus)/3. +
    (adu2_pi_odd_minus)/3. + (adu2_pi_even_minus)/3.;

    pframe_adu1_minus = (adu1_zero_odd_minus)/3. - (adu1_zero_even_minus)/3. -
    (adu1_pi_odd_minus)/3. + (adu1_pi_even_minus)/3.;

    pframe_adu2_minus = (adu2_zero_odd_minus)/3. - (adu2_zero_even_minus)/3. -
    (adu2_pi_odd_minus)/3. + (adu2_pi_even_minus)/3.;

    iframe_plus = (iframe_adu1_plus + iframe_adu2_plus)/2.;
    pframe_plus = (pframe_adu1_plus + pframe_adu2_plus)/2.;
    iframe_minus = (iframe_adu1_minus + iframe_adu2_minus)/2.;
    pframe_minus = (pframe_adu1_minus + pframe_adu2_minus)/2.;

    pframe_over_iframe_plus = (pframe_adu1_plus/iframe_adu1_plus + pframe_adu2_plus/iframe_adu2_plus)/2.;
    pframe_over_iframe_minus = (pframe_adu1_minus/iframe_adu1_minus + pframe_adu2_minus/iframe_adu2_minus)/2.;
    pframe_over_iframe = (pframe_over_iframe_plus - pframe_over_iframe_minus)/2.;

    //printf("iframe = %f\n", iframe );
    //printf("pframe = %f\n", pframe );
    //printf("pframe/iframe = %f\n", pframe/iframe );
    //printf("PFRAME/IFRAME = %f\n", pframe_over_iframe );

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME), pframe_over_iframe, 1e-10);
    //printf("SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME = %f\n", cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME));

    cpl_propertylist_delete( pl ); pl = NULL;

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_plus_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), iframe_plus, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), pframe_plus, 1e-10);

    cpl_propertylist_delete( pl ); pl = NULL;

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_minus_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), iframe_minus, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), pframe_minus, 1e-10);

    cpl_propertylist_delete( pl ); pl = NULL;

    rerr = sph_zpl_modem_efficiency_delete( zpl_modem_efficiency );
    cpl_test_error(CPL_ERROR_NONE );

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete( inframes );
    cpl_parameterlist_delete(inparams);

    return;

}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_modem_efficiency with rawframes & bias, dark, iff as a master frame, pff,  calibrations
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_modem_efficiency_run_9(void) {
    sph_zpl_modem_efficiency* zpl_modem_efficiency = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100;// -//-

    double dV = 1.0;
    //double              dS						= 0.0;

    double zero_odd = 169.0;
    double zero_even = 29.0;
    double pi_odd = 29.0;
    double pi_even = 169.0;
    cpl_propertylist* pl = NULL;
    cpl_frame* master_bias_frame = NULL;
    cpl_frame* master_dark_frame = NULL;
    cpl_frame* intensity_flat_frame = NULL;
    cpl_frame* polarization_flat_frame = NULL;

    inparams = sph_zpl_modem_efficiency_create_paramlist();
    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_set_hwpz( &zexpcube, "modem_eff_raw_hwpz_0.fits.tmp", zero_odd, zero_even, pi_odd, pi_even, dV, 0.0, nframes );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube );

    zero_odd = 49.0; zero_even = 149.0;
    pi_odd = 149.0; pi_even = 49.0;
    sph_test_simulator_create_zplpreproc_cube_set_hwpz( &zexpcube1, "modem_eff_raw_hwpz_45.fits.tmp", zero_odd, zero_even, pi_odd, pi_even, dV, 45.0, nframes );
    cpl_frame_set_tag( zexpcube1, SPH_ZPL_TAG_MODEM_EFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube1, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube1 );

    //simualate master bias
    zero_odd = 10; zero_even = 11; pi_odd = 12; pi_even = 14;
    master_bias_frame = sph_test_zpl_simulator_create_quad_image_frame("test_master_bias_calib.fits.tmp", nx, ny, zero_odd, zero_even, pi_odd, pi_even);
    cpl_frame_set_tag( master_bias_frame, SPH_ZPL_TAG_BIAS_CALIB );
    cpl_frame_set_group( master_bias_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, master_bias_frame );

    //simualate master dark
    zero_odd = 1; zero_even = 2; pi_odd = 3; pi_even = 5;
    master_dark_frame = sph_test_zpl_simulator_create_quad_image_frame("test_master_dark_calib.fits.tmp", nx, ny, zero_odd, zero_even, pi_odd, pi_even);
    cpl_frame_set_tag( master_dark_frame, SPH_ZPL_TAG_DARK_CALIB );
    cpl_frame_set_group( master_dark_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, master_dark_frame );

    //simualate intensity flat as a masterframe
    //zero_odd = 2; zero_even = 2; pi_odd = 2; pi_even = 2;
    //intensity_flat_frame = sph_test_zpl_simulator_create_quad_image_frame("test_master_intensity_flat_calib.fits.tmp", nx, ny, zero_odd, zero_even, pi_odd, pi_even);
    intensity_flat_frame = sph_test_zpl_simulator_create_masterframe_frame("test_master_intensity_flat_masterframe_calib.fits.tmp", nx, ny, 2.0);
    cpl_frame_set_tag( intensity_flat_frame, SPH_ZPL_TAG_IFFM_CALIB );
    cpl_frame_set_group( intensity_flat_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, intensity_flat_frame );

    //simualate polarization flat
    polarization_flat_frame = sph_test_zpl_simulator_create_double_image_frame("test_master_polarization_flat_calib.fits.tmp", nx, ny, 200.0, 100.0 );
    cpl_frame_set_tag( polarization_flat_frame, SPH_ZPL_TAG_PFF_CALIB );
    cpl_frame_set_group( polarization_flat_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, polarization_flat_frame );

    zpl_modem_efficiency = sph_zpl_modem_efficiency_new( inframes, inparams);
    cpl_test_nonnull( zpl_modem_efficiency );

    rerr = sph_zpl_modem_efficiency_run( zpl_modem_efficiency );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME), 0.701754386, 1e-10);
    cpl_propertylist_delete( pl ); pl = NULL;

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_plus_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), 171.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), 53.5, 1e-10);
    cpl_propertylist_delete( pl ); pl = NULL;

    pl = cpl_propertylist_load( zpl_modem_efficiency->modem_efficiency_minus_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), 171.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), -186.5, 1e-10);

    rerr = sph_zpl_modem_efficiency_delete( zpl_modem_efficiency );
    cpl_test_error(CPL_ERROR_NONE );

    return;
}
#endif

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_modem_efficiency recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    (void)sph_keyword_manager_set_cfg(SPH_CFG);
    (void)sph_test_init();

    pSuite = sph_add_suite("modem_efficiency_test",
            cutest_init_modem_efficiency_testsuite,
            cutest_clean_modem_efficiency_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_modem_efficiency_run_1: 2 cube-preprocs with +Q[Qplus] and -Q[Qminus]",
                    cutest_zpl_modem_efficiency_run_1)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_modem_efficiency_run_1_1: 4 cube-preprocs(2 for Cam1, 2 for Cam2) with +Q[Qplus] and -Q[Qminus]",
                    cutest_zpl_modem_efficiency_run_1_1)) {
        return sph_test_get_error();
    }

#ifdef SPH_TEST_MODEM_EFFICIENCY
     if ( NULL == sph_test_do(pSuite, "sph_modem_efficiency_run_2: 2 cube-rawframes with +Q[hwpz=0.0] and +Q[hwpz=0.0]",
     cutest_zpl_modem_efficiency_run_2 ) )
     {
     return sph_test_get_error();
     }

     if ( NULL == sph_test_do(pSuite, "sph_modem_efficiency_run_3: 2 cube-rawframes with -Q[hwpz=45.0] and -Q[hwpz=45.0]",
     cutest_zpl_modem_efficiency_run_3 ) )
     {
     return sph_test_get_error();
     }

     if ( NULL == sph_test_do(pSuite, "sph_modem_efficiency_run_4: 2 cube-rawframes without hwpz keywords",
     cutest_zpl_modem_efficiency_run_4 ) )
     {
     return sph_test_get_error();
     }

     if ( NULL == sph_test_do(pSuite, "sph_modem_efficiency_run_5: 2 cube-rawframes +Q and -Q  with bias-, dark-, iff-as a quad image, pff- calibrations",
     cutest_zpl_modem_efficiency_run_5 ) )
     {
     return sph_test_get_error();
     }

     if ( NULL == sph_test_do(pSuite, "sph_modem_efficiency_run_6: 2 cube-rawframes without hwpz keywords, with config and head-files",
     cutest_zpl_modem_efficiency_run_6 ) )
     {
     return sph_test_get_error();
     }
     if ( NULL == sph_test_do(pSuite, "sph_modem_efficiency_run_7: 2 cube-rawframes with the same hwpz-keywords +Q[0.0, 0.0] and overscan tables",
     cutest_zpl_modem_efficiency_run_7 ) )
     {
     return sph_test_get_error();
     }

     if ( NULL == sph_test_do(pSuite, "sph_modem_efficiency_run_8: 2 cube-rawframes with the hwpz-keywords +Q[0.0], -Q[45.0] and overscan tables",
     cutest_zpl_modem_efficiency_run_8 ) )
     {
     return sph_test_get_error();
     }

     if ( NULL == sph_test_do(pSuite, "sph_modem_efficiency_run_9: 2 cube-rawframes +Q and -Q  with bias-, dark-, iff-as a masterframe, pff- calibrations",
     cutest_zpl_modem_efficiency_run_9 ) )
     {
     return sph_test_get_error();
     }
#endif

    return sph_test_end();
}

/**@}*/
