/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_apertures.h"
#include "sph_test_image_tools.h"
#include "sph_utils.h"
#include "sph_common_keywords.h"
#include "sph_background.h"

#include <math.h>

/*----------------------------------------------------------------------------*/
/*-                            Declarations of private functions              */
/*----------------------------------------------------------------------------*/

static
void cutest_sph_background_verify_constant_cubes__(cpl_frameset*, double);

static cpl_frameset*
cutest_sph_background_fixture_raw_cubes__(int, int, int, cpl_frameset*);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                      */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_sph_background_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_sph_background_clean_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_background_apply_to_raw_cubes(void) {
    cpl_frameset* raw_frames = NULL;
    cpl_frameset* flat_frames = NULL;
    cpl_frameset* dark_frames = NULL;
    cpl_frameset* outframes = NULL;

    raw_frames = cutest_sph_background_fixture_raw_cubes__(10, 256, 256,
            NULL);

    flat_frames = sph_utils_extract_frames(raw_frames, "FTAG");
    dark_frames = sph_utils_extract_frames(raw_frames, "DTAG");

    cpl_test_error(CPL_ERROR_NONE);

    outframes = sph_background_apply_to_raw_cubes(flat_frames, dark_frames);

    cpl_test_nonnull( outframes );

    cutest_sph_background_verify_constant_cubes__(outframes, 0.0);

    cpl_test_eq(cpl_frameset_get_size(outframes), 10);

    cpl_test_error(CPL_ERROR_NONE);

    cpl_frameset_delete(raw_frames);
    cpl_frameset_delete(dark_frames);
    cpl_frameset_delete(flat_frames);
    cpl_frameset_delete(outframes);
}

static cpl_frameset*
cutest_sph_background_fixture_raw_cubes__(int ncubes, int nx,
        int ny, cpl_frameset* inframes) {
    cpl_frameset* outframes = NULL;
    int ff = 0;
    cpl_image* im = NULL;
    cpl_propertylist* pl_flat = NULL;
    cpl_propertylist* pl_dark = NULL;
    cpl_frame* dark_frame = NULL;
    cpl_frame* flat_frame = NULL;
    char fname[256];

    if (inframes) {
        outframes = inframes;
    } else {
        outframes = cpl_frameset_new();
    }

    pl_flat = cpl_propertylist_new();
    pl_dark = cpl_propertylist_new();
    cpl_propertylist_update_string(pl_dark, SPH_COMMON_KEYWORD_DPR_TYPE,
            "DARK");
    cpl_propertylist_update_string(pl_flat, SPH_COMMON_KEYWORD_DPR_TYPE,
            "FLAT, LAMP");

    for (ff = 0; ff < ncubes; ++ff) {
        dark_frame = cpl_frame_new();
        cpl_frame_set_tag(dark_frame, "DTAG");
        cpl_frame_set_group(dark_frame, CPL_FRAME_GROUP_RAW);
        flat_frame = cpl_frame_new();
        cpl_frame_set_tag(flat_frame, "FTAG");
        cpl_frame_set_group(flat_frame, CPL_FRAME_GROUP_RAW);
        im = sph_test_image_tools_create_flat_image_int(nx, ny, ff * 10);

        cpl_propertylist_update_double(pl_flat, SPH_COMMON_KEYWORD_DIT,
                ff * 1.2 + 1.2);
        cpl_propertylist_update_double(pl_flat, SPH_COMMON_KEYWORD_SEQ1DIT,
                ff * 1.2 + 1.2);
        sprintf(fname, "test_background__flat_%03d.fits", ff);
        cpl_image_save(im, fname, CPL_TYPE_INT, pl_flat, CPL_IO_CREATE);
        sph_filemanager_add_tmp_file(fname);
        cpl_frame_set_filename(flat_frame, fname);

        cpl_propertylist_update_double(pl_dark, SPH_COMMON_KEYWORD_DIT,
                ff * 1.2 + 1.2);
        cpl_propertylist_update_double(pl_dark, SPH_COMMON_KEYWORD_SEQ1DIT,
                ff * 1.2 + 1.2);
        sprintf(fname, "test_background__dark_%03d.fits", ff);
        cpl_image_save(im, fname, CPL_TYPE_INT, pl_dark, CPL_IO_CREATE);
        cpl_image_delete(im);
        im = NULL;
        sph_filemanager_add_tmp_file(fname);
        cpl_frame_set_filename(dark_frame, fname);
        cpl_frameset_insert(outframes, flat_frame);
        cpl_frameset_insert(outframes, dark_frame);
    }

    cpl_propertylist_delete(pl_dark);
    pl_dark = NULL;
    cpl_propertylist_delete(pl_flat);
    pl_flat = NULL;
    return outframes;
}

static
void cutest_sph_background_verify_constant_cubes__(cpl_frameset* inframes,
        double val) {
    int ff = 0;
    int pp = 0;
    int np = 0;
    cpl_image* im = NULL;
    cpl_propertylist* pl = NULL;

    for (ff = 0; ff < cpl_frameset_get_size(inframes); ++ff) {
        pl = cpl_propertylist_load(
                cpl_frame_get_filename(cpl_frameset_get_position(inframes, ff)),
                0);
        if (cpl_propertylist_has(pl, "NAXIS3")) {
            np = cpl_propertylist_get_int(pl, "NAXIS3");
        } else
            np = 1;

        for (pp = 0; pp < np; ++pp) {
            im = cpl_image_load(
                    cpl_frame_get_filename(
                            cpl_frameset_get_position(inframes, ff)),
                    CPL_TYPE_INT, 0, pp);

            cpl_test_abs( cpl_image_get_absflux(im), val, 0.00000001);
            cpl_test_abs( cpl_image_get_mean(im), val, 0.00000001);
            cpl_image_delete(im);
            im = NULL;
        }
        cpl_propertylist_delete(pl);
        pl = NULL;
    }
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing sph_background",
            cutest_sph_background_init_testsuite,
            cutest_sph_background_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_background_apply",
                    cutest_sph_background_apply_to_raw_cubes)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
