/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include <cpl.h>
#include <cpl_image_io.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_common_science.h"
#include "sph_zpl_exposure.h"
#include "sph_test_zpl_simulator.h"
#include "sph_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
static cpl_frame*
cutest_sph_common_science_create_cubes__(void) {
    cpl_image* badpix = NULL;
    cpl_image* rms = NULL;
    cpl_image* image = NULL;
    cpl_frame* aframe = NULL;
    cpl_imagelist* imlist = NULL;
    cpl_imagelist* bpixlist = NULL;
    cpl_imagelist* rmslist = NULL;
    int pp = 0;
    int nplanes = 99;

    imlist = cpl_imagelist_new();
    rmslist = cpl_imagelist_new();
    bpixlist = cpl_imagelist_new();
    for (pp = 0; pp < nplanes; ++pp) {
        image = sph_test_image_tools_create_flat_image_double(512, 512, 1.0);
        cpl_imagelist_set(imlist, image, pp);
        rms = sph_test_image_tools_create_flat_image_double(512, 512, 0.1);
        cpl_imagelist_set(rmslist, rms, pp);
        badpix = sph_test_image_tools_create_flat_image_int(512, 512, 0);
        cpl_imagelist_set(bpixlist, badpix, pp);
    }
    aframe = sph_filemanager_create_temp_frame("TEST_FULLSET", "RAWSCI",
            CPL_FRAME_GROUP_RAW);
    cpl_imagelist_save(imlist, cpl_frame_get_filename(aframe), CPL_TYPE_DOUBLE,
            NULL, CPL_IO_CREATE);
    SPH_RAISE_CPL;
    cpl_imagelist_save(bpixlist, cpl_frame_get_filename(aframe), CPL_TYPE_INT,
            NULL, CPL_IO_EXTEND);
    SPH_RAISE_CPL;
    cpl_imagelist_save(rmslist, cpl_frame_get_filename(aframe), CPL_TYPE_DOUBLE,
            NULL, CPL_IO_EXTEND);
    SPH_RAISE_CPL;

    cpl_imagelist_delete(bpixlist);
    bpixlist = NULL;
    cpl_imagelist_delete(rmslist);
    rmslist = NULL;
    cpl_imagelist_delete(imlist);
    imlist = NULL;
    return aframe;
}
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}
#ifdef SPH_TEST_COMMON_SCIENCE
static
int cutest_sph_common_science_fixture(void) {
    return 0;
}
#endif

static sph_distortion_model*
cutest_sph_common_science_create_distmodel(void) {
    cpl_polynomial* polyx = NULL;
    cpl_polynomial* polyy = NULL;
    sph_distortion_model* distmodel = NULL;
    cpl_size p[2];

    polyx = cpl_polynomial_new(2);
    p[0] = 0;
    p[1] = 0;
    cpl_polynomial_set_coeff(polyx, p, 0.0);
    p[0] = 1;
    p[1] = 0;
    cpl_polynomial_set_coeff(polyx, p, 0.000001);
    p[0] = 0;
    p[1] = 1;
    cpl_polynomial_set_coeff(polyx, p, 0.0);
    p[0] = 1;
    p[1] = 1;
    cpl_polynomial_set_coeff(polyx, p, 0.0);
    polyy = cpl_polynomial_new(2);
    p[0] = 0;
    p[1] = 0;
    cpl_polynomial_set_coeff(polyy, p, 0.0);
    p[0] = 0;
    p[1] = 1;
    cpl_polynomial_set_coeff(polyy, p, 0.0);
    p[0] = 1;
    p[1] = 0;
    cpl_polynomial_set_coeff(polyy, p, 0.0);
    p[0] = 1;
    p[1] = 1;
    cpl_polynomial_set_coeff(polyy, p, 0.000001);
    //p[0] = 0; p[1] = 2;cpl_polynomial_set_coeff(polyy,p,0.00005);

    distmodel = sph_distortion_model_new(polyx, polyy);

    cpl_polynomial_delete(polyx);
    polyx = NULL;
    cpl_polynomial_delete(polyy);
    polyy = NULL;
    return distmodel;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_common_science_calibrate_raw_test_bad_input(void) {
    sph_distortion_model* distmap = NULL;
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    cpl_image* rawim = NULL;
    sph_master_frame* result = NULL;

    distmap = cutest_sph_common_science_create_distmodel();
    cpl_test_nonnull( distmap );
    result = sph_common_science_calibrate_raw(rawim, dark, flat);
    cpl_test_null(result);
    cpl_test_noneq( cpl_error_get_code(), CPL_ERROR_NONE);
    cpl_error_reset();
    sph_master_frame_delete(result);
    result = NULL;
    sph_distortion_model_delete(distmap);
    distmap = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_common_science_calibrate_raw_test(void) {
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    cpl_image* rawim = NULL;
    sph_master_frame* result = NULL;
    int nx = 256;
    int ny = 256;

    dark = sph_master_frame_new_empty();
    dark->image = sph_test_image_tools_create_flat_image_double(nx, ny, 52.0);
    sph_test_image_tools_add_noise(dark->image, 5.5, NULL);
    dark->rmsmap = sph_test_image_tools_create_flat_image_double(nx, ny, 10.0);
    dark->ncombmap = sph_test_image_tools_create_flat_image_double(nx, ny,
            10.0);
    dark->badpixelmap = sph_test_image_tools_create_flat_image_int(nx, ny, 0);
    cpl_image_set(dark->badpixelmap, 15, 25, 1);
    flat = sph_master_frame_new_empty();
    flat->image = sph_test_image_tools_create_flat_image_double(nx, ny, 1.0);
    sph_test_image_tools_add_noise(flat->image, 0.1, NULL);
    flat->rmsmap = sph_test_image_tools_create_flat_image_double(nx, ny, 0.1);
    flat->ncombmap = sph_test_image_tools_create_flat_image_double(nx, ny,
            10.0);
    flat->badpixelmap = sph_test_image_tools_create_flat_image_int(nx, ny, 0);
    cpl_image_set(flat->badpixelmap, 10, 15, 1);

    rawim = sph_test_image_tools_create_flat_image_double(nx, ny, 152.0);
    cpl_image_set(rawim, 10, 15, 0.0);
    cpl_image_set(rawim, 15, 25, 100000.0);

    result = sph_common_science_calibrate_raw(rawim, dark, flat);

    cpl_test_assert(result);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_error_reset();
    cpl_test_eq(sph_master_frame_get_nbads(result), 2);
    cpl_test_abs(sph_master_frame_get_mean(result,NULL), 100.0, 5.0);
    cpl_test_abs(sph_master_frame_get_mean_rms(result,NULL), 15.0,
            6.0);
    cpl_test_eq(sph_master_frame_get_nbads(result), 2);
    sph_master_frame_save(result, "aresult.fits", NULL);
    sph_master_frame_delete(dark);
    dark = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    sph_master_frame_delete(result);
    result = NULL;
    cpl_image_delete(rawim);
    rawim = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_common_science_calibrate_raw_test_nodist(void) {
    sph_distortion_model* distmap = NULL;
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    cpl_image* rawim = NULL;
    sph_master_frame* result = NULL;
    int nx = 256;
    int ny = 256;
    cpl_image* wmap = NULL;

    distmap = cutest_sph_common_science_create_distmodel();
    cpl_test_nonnull( distmap );
    dark = sph_master_frame_new_empty();
    dark->image = sph_test_image_tools_create_flat_image_double(nx, ny, 52.0);
    sph_test_image_tools_add_noise(dark->image, 5.5, NULL);
    dark->rmsmap = sph_test_image_tools_create_flat_image_double(nx, ny, 10.0);
    dark->ncombmap = sph_test_image_tools_create_flat_image_double(nx, ny,
            10.0);
    dark->badpixelmap = sph_test_image_tools_create_flat_image_int(nx, ny, 0);
    cpl_image_set(dark->badpixelmap, 15, 25, 1);
    flat = sph_master_frame_new_empty();
    flat->image = sph_test_image_tools_create_flat_image_double(nx, ny, 1.0);
    sph_test_image_tools_add_noise(flat->image, 0.1, NULL);
    flat->rmsmap = sph_test_image_tools_create_flat_image_double(nx, ny, 0.1);
    flat->ncombmap = sph_test_image_tools_create_flat_image_double(nx, ny,
            10.0);
    flat->badpixelmap = sph_test_image_tools_create_flat_image_int(nx, ny, 0);
    cpl_image_set(flat->badpixelmap, 10, 15, 1);

    rawim = sph_test_image_tools_create_flat_image_double(nx, ny, 152.0);
    cpl_image_set(rawim, 10, 15, 0.0);
    cpl_image_set(rawim, 15, 25, 100000.0);

    result = sph_common_science_calibrate_raw(rawim, dark, flat);

    cpl_test_assert(result);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_error_reset();
    cpl_test_eq(sph_master_frame_get_nbads(result), 2);
    cpl_test_abs(sph_master_frame_get_mean(result,NULL), 100.0, 5.0);
    cpl_test_abs(sph_master_frame_get_mean_rms(result,NULL), 15.0,
            5.0);
    wmap = sph_master_frame_get_weights(result);
    cpl_test_assert(wmap);
    sph_master_frame_set_rms_from_weightmap(result, wmap, 1);
    cpl_test_eq(sph_master_frame_get_nbads(result), 2);
    //sph_master_frame_save(result,"aresult.fits",NULL);
    sph_master_frame_delete(dark);
    dark = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    sph_master_frame_delete(result);
    result = NULL;
    cpl_image_delete(rawim);
    rawim = NULL;
    cpl_image_delete(wmap);
    wmap = NULL;
    sph_distortion_model_delete(distmap);
    distmap = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        A test function of the testsuite
 *
 * This is a standard test case -- with ideal data.
 *
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_common_science_test_combine_nobadpix(void) {
    cpl_image* image = NULL;
    int nx = 256;
    int ny = 256;
    cpl_frameset* inframes = NULL;
    int ff = 0;
    cpl_frame* aframe = NULL;
    sph_master_frame* result = NULL;
    gsl_rng* pRNG = NULL;
    inframes = cpl_frameset_new();

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    for (ff = 0; ff < 11; ++ff) {
        if (ff < 10) {
            image = sph_test_image_tools_create_flat_image_double(nx, ny,
                    6000.0);
        } else {
            image = sph_test_image_tools_create_flat_image_double(nx, ny,
                    12000.0);
        }
        sph_test_image_tools_apply_poisson_noise(image, pRNG);
        aframe = sph_test_frame_image(image, CPL_TYPE_DOUBLE, "TEST_IDEAL",
                CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(inframes, aframe);
        cpl_image_delete(image);
        image = NULL;
    }cpl_test_error(CPL_ERROR_NONE);
    result = sph_common_science_combine(inframes, SPH_COLL_ALG_MEAN, 0, -1, -1);
    cpl_test_nonnull( result );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_zero(sph_master_frame_get_nbads(result));
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(sph_master_frame_get_mean(result,NULL), 6000.0,
            1000.000);
    sph_master_frame_delete(result);
    result = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    result = sph_common_science_combine(inframes, SPH_COLL_ALG_MEDIAN, 0, -1,
            -1);
    cpl_test_nonnull( result );
    cpl_test_zero(sph_master_frame_get_nbads(result));
    cpl_test_abs(sph_master_frame_get_mean(result,NULL), 6000.0,
            10.0);
    sph_master_frame_delete(result);
    result = NULL;
    result = sph_common_science_combine(inframes, SPH_COLL_ALG_WGT_MEAN, 0, -1,
            -1);
    cpl_test_nonnull( result );
    cpl_test_zero(sph_master_frame_get_nbads(result));
    cpl_test_abs(sph_master_frame_get_mean(result,NULL), 6000.0, 9.0);
    sph_master_frame_delete(result);
    result = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    gsl_rng_free(pRNG);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        A test function of the testsuite
 *
 * This is a standard test case -- with ideal data.
 *
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_common_science_test_combine_cube_fullset(void) {
    cpl_frameset* inframes = NULL;
    sph_master_frame* result = NULL;

    inframes = cpl_frameset_new();
    cpl_frameset_insert(inframes, cutest_sph_common_science_create_cubes__());
    cpl_frameset_insert(inframes, cutest_sph_common_science_create_cubes__());
    cpl_frameset_insert(inframes, cutest_sph_common_science_create_cubes__());
    cpl_frameset_insert(inframes, cutest_sph_common_science_create_cubes__());
    cpl_frameset_insert(inframes, cutest_sph_common_science_create_cubes__());
    cpl_frameset_insert(inframes, cutest_sph_common_science_create_cubes__());
    cpl_frameset_insert(inframes, cutest_sph_common_science_create_cubes__());
    cpl_frameset_insert(inframes, cutest_sph_common_science_create_cubes__());
    cpl_frameset_insert(inframes, cutest_sph_common_science_create_cubes__());
    cpl_frameset_insert(inframes, cutest_sph_common_science_create_cubes__());
    cpl_frameset_insert(inframes, cutest_sph_common_science_create_cubes__());
    cpl_frameset_insert(inframes, cutest_sph_common_science_create_cubes__());
    result = sph_common_science_combine(inframes, SPH_COLL_ALG_MEDIAN, 0, 1, 2);
    cpl_test_nonnull( result );
    sph_master_frame_delete(result);
    cpl_frameset_delete(inframes);
    cpl_test_error(CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        A test function of the testsuite
 *
 * This is a standard test case -- with ideal data.
 *
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_common_science_test_combine_ideal_input(void) {
    cpl_image* badpix = NULL;
    cpl_image* image = NULL;
    int nx = 256;
    int ny = 256;
    cpl_frameset* inframes = NULL;
    int ff = 0;
    cpl_frame* aframe = NULL;
    sph_master_frame* result = NULL;
    inframes = cpl_frameset_new();

    for (ff = 0; ff < 11; ++ff) {
        badpix = sph_test_image_tools_create_flat_image_int(nx, ny, 0);
        if (ff < 10) {
            image = sph_test_image_tools_create_flat_image_double(nx, ny,
                    ff + 1.0);
        } else {
            image = sph_test_image_tools_create_flat_image_double(nx, ny, 50.0);
        }
        aframe = sph_test_frame_image(image, CPL_TYPE_DOUBLE, "TEST_IDEAL",
                CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(inframes, aframe);
        cpl_image_save(badpix, cpl_frame_get_filename(aframe),
                CPL_BPP_32_SIGNED, NULL, CPL_IO_EXTEND);
        cpl_image_delete(badpix);
        badpix = NULL;
        cpl_image_delete(image);
        image = NULL;
    }

    result = sph_common_science_combine(inframes, SPH_COLL_ALG_MEAN, 0, 1, -1);
    cpl_test_nonnull( result );
    cpl_test_zero(sph_master_frame_get_nbads(result));
    cpl_test_abs(sph_master_frame_get_mean(result,NULL), 9.5454,
            0.0001);
    sph_master_frame_save(result, "result.fits", NULL);
    sph_master_frame_delete(result);
    result = NULL;

    result = sph_common_science_combine(inframes, SPH_COLL_ALG_MEDIAN, 0, 1,
            -1);
    cpl_test_nonnull( result );
    cpl_test_zero(sph_master_frame_get_nbads(result));
    cpl_test_abs(sph_master_frame_get_mean(result,NULL), 6.0, 0.0001);
    sph_master_frame_delete(result);
    result = NULL;

    cpl_frameset_delete(inframes);
    inframes = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        A test function of the testsuite
 *
 * This is a standard test case -- with ideal data.
 *
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_common_science_test_combine_noisy_ideal_input(void) {
    cpl_image* badpix = NULL;
    cpl_image* image = NULL;
    int nx = 256;
    int ny = 256;
    cpl_frameset* inframes = NULL;
    int ff = 0;
    cpl_frame* aframe = NULL;
    sph_master_frame* result = NULL;
    gsl_rng* pRNG = NULL;
    inframes = cpl_frameset_new();

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    for (ff = 0; ff < 11; ++ff) {
        badpix = sph_test_image_tools_create_flat_image_int(nx, ny, 0);
        if (ff < 10) {
            image = sph_test_image_tools_create_flat_image_double(nx, ny,
                    6000.0);
        } else {
            image = sph_test_image_tools_create_flat_image_double(nx, ny,
                    12000.0);
        }
        sph_test_image_tools_apply_poisson_noise(image, NULL);
        aframe = sph_test_frame_image(image, CPL_TYPE_DOUBLE, "TEST_IDEAL",
                CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(inframes, aframe);
        cpl_image_save(badpix, cpl_frame_get_filename(aframe),
                CPL_BPP_32_SIGNED, NULL, CPL_IO_EXTEND);
        cpl_image_delete(badpix);
        badpix = NULL;
        cpl_image_delete(image);
        image = NULL;
    }cpl_test_error(CPL_ERROR_NONE);
    result = sph_common_science_combine(inframes, SPH_COLL_ALG_MEAN, 0, 1, -1);
    cpl_test_nonnull( result );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_zero(sph_master_frame_get_nbads(result));
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(sph_master_frame_get_mean(result,NULL), 6000.0,
            1000.000);
    sph_master_frame_delete(result);
    result = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    result = sph_common_science_combine(inframes, SPH_COLL_ALG_MEDIAN, 0, 1,
            -1);
    cpl_test_nonnull( result );
    cpl_test_zero(sph_master_frame_get_nbads(result));
    cpl_test_abs(sph_master_frame_get_mean(result,NULL), 6000.0,
            10.0);
    sph_master_frame_delete(result);
    result = NULL;
    result = sph_common_science_combine(inframes, SPH_COLL_ALG_WGT_MEAN, 0, 1,
            -1);
    cpl_test_nonnull( result );
    cpl_test_zero(sph_master_frame_get_nbads(result));
    cpl_test_abs(sph_master_frame_get_mean(result,NULL), 6000.0, 9.0);
    sph_master_frame_delete(result);
    result = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    gsl_rng_free(pRNG);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        A test function of the testsuite
 *
 * This is a standard test case -- with ideal data.
 *
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_common_science_get_next_raw_test(void) {
    cpl_frameset* inframeset = NULL;
    cpl_image* im = NULL;
    int c = 0;
    cpl_frame* aframe = NULL;
    int nx = 256;
    int ny = 256;
    int ff = 0;

    inframeset = cpl_frameset_new();
    for (ff = 0; ff < 10; ++ff) {
        im = sph_test_image_tools_create_flat_image_double(nx, ny, 12000.0);
        aframe = sph_test_frame_image(im, CPL_TYPE_DOUBLE, "TEST_IDEAL",
                CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(inframeset, aframe);
        cpl_image_save(im, cpl_frame_get_filename(aframe), CPL_BPP_32_SIGNED,
                NULL, CPL_IO_EXTEND);
        cpl_image_delete(im);
        im = NULL;
    }

    im = sph_common_science_get_next_raw(inframeset, 0);
    while (im) {
        cpl_image_delete(im);
        im = NULL;
        im = sph_common_science_get_next_raw(inframeset, 0);
        c++;
    }cpl_test_eq(c, 10);
    cpl_frameset_delete(inframeset);
    inframeset = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_common_science_get_next_raw_test_swap_cube_to_normal(void) {
    cpl_frameset* inframeset = NULL;
    cpl_image* im = NULL;
    int c = 0;
    cpl_frame* aframe = NULL;
    int nx = 256;
    int ny = 256;
    int ff = 0;
    cpl_imagelist* imlist = NULL;

    inframeset = cpl_frameset_new();
    imlist = cpl_imagelist_new();
    for (ff = 0; ff < 10; ++ff) {
        im = sph_test_image_tools_create_flat_image_double(nx, ny, 12000.0);
        cpl_imagelist_set(imlist, im, ff);
    }
    aframe = sph_filemanager_create_temp_frame("TEST_IDEAL", "ATAG",
            CPL_FRAME_GROUP_RAW);
    cpl_imagelist_save(imlist, cpl_frame_get_filename(aframe),
            CPL_BPP_32_SIGNED, NULL, CPL_IO_CREATE);
    cpl_frameset_insert(inframeset, aframe);

    im = sph_common_science_get_next_raw(inframeset, 0);
    while (im) {
        cpl_image_delete(im);
        im = NULL;
        im = sph_common_science_get_next_raw(inframeset, 0);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Read plane %d", c);
        c++;
    }
    cpl_error_reset();
    cpl_test_eq(c, 10);
    cpl_frameset_delete(inframeset);
    inframeset = NULL;
    cpl_imagelist_delete(imlist);
    imlist = NULL;
    c = 0;
    inframeset = cpl_frameset_new();
    for (ff = 0; ff < 10; ++ff) {
        im = sph_test_image_tools_create_flat_image_double(nx, ny, 12000.0);
        aframe = sph_test_frame_image(im, CPL_TYPE_DOUBLE, "TEST_IDEAL",
                CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(inframeset, aframe);
        cpl_image_save(im, cpl_frame_get_filename(aframe), CPL_BPP_32_SIGNED,
                NULL, CPL_IO_EXTEND);
        cpl_image_delete(im);
        im = NULL;
    }

    im = sph_common_science_get_next_raw(inframeset, 0);
    while (im) {
        cpl_image_delete(im);
        im = NULL;
        im = sph_common_science_get_next_raw(inframeset, 0);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Read plane %d", c);
        c++;
    }
    cpl_error_reset();
    cpl_test_eq(c, 10);
    cpl_frameset_delete(inframeset);
    inframeset = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_common_science_get_next_zpl_exposure_test(void) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube2 = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_frameset* zexpcube_frames = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    const int nz = 5; //nframes
    const int nx = 10;
    const int ny = 10;
    double zero_odd = 100.0;
    double zero_even = 200.0;
    double pi_odd = 300.0;
    double pi_even = 400.0;
    double dV = 100.0;

    zexpcube_frames = cpl_frameset_new();

    zexpcube = sph_test_zpl_simulator_frame_template(
            "test_sph_common_zplexp_cube_1.fits");

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_sph_common_zplexp_single_frame.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        cpl_test_eq_error(rerr, CPL_ERROR_NONE);
        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        sph_zpl_exposure_delete(zplexp);
        cpl_frameset_insert(zexpframes, zexpframe);
    }cpl_test_eq(cpl_frameset_get_size( zexpframes ), nz);
    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_double(
            zexpframes, zexpcube);
    cpl_frameset_insert(zexpcube_frames, zexpcube);
    //cpl_frame_delete( zexpcube );
    cpl_frameset_delete(zexpframes);

    zexpcube2 = sph_test_zpl_simulator_frame_template(
            "test_sph_common_zplexp_cube_2.fits");
    zexpframes = cpl_frameset_new();
    for (int i = nz; i < 2 * nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_sph_common_zplexp_single_frame.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        cpl_test_eq_error(rerr, CPL_ERROR_NONE);
        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        sph_zpl_exposure_delete(zplexp);
        cpl_frameset_insert(zexpframes, zexpframe);
    }cpl_test_eq(cpl_frameset_get_size( zexpframes ), nz);

    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_double(
            zexpframes, zexpcube2);
    cpl_frameset_insert(zexpcube_frames, zexpcube2);
    //cpl_frame_delete( zexpcube );
    cpl_frameset_delete(zexpframes);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "All fine to here...");
    for (int i = 0; i <= 2 * nz; i++) {
        zplexp = sph_common_science_get_next_zpl_exposure(zexpcube_frames);
        if (zplexp) {
            cpl_test_abs(
                    cpl_image_get_absflux( zplexp->image_zero_odd)/(nx*ny),
                    zero_odd+i*dV, 1e-10);
            cpl_test_abs(
                    cpl_image_get_absflux( zplexp->image_zero_even)/(nx*ny),
                    zero_even+i*dV, 1e-10);
            cpl_test_abs(
                    cpl_image_get_absflux( zplexp->image_pi_odd)/(nx*ny),
                    pi_odd+i*dV, 1e-10);
            cpl_test_abs(
                    cpl_image_get_absflux( zplexp->image_pi_even)/(nx*ny),
                    pi_even+i*dV, 1e-10);
            sph_zpl_exposure_delete(zplexp);
            zplexp = NULL;
        }
    }
    cpl_frameset_delete(zexpcube_frames);
    zexpcube_frames = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_common_science_combine_weight_double_image_frameset_test(void) {
    sph_master_frame* mframe1 = NULL;
    sph_master_frame* mframe2 = NULL;
    sph_master_frame* mframe3 = NULL;
    sph_double_image* result = NULL;
    cpl_frame* frame1 = NULL;
    cpl_frame* frame2 = NULL;
    cpl_frame* frame3 = NULL;
    cpl_image* image1 = NULL;
    cpl_image* image2 = NULL;
    cpl_image* image3 = NULL;
    cpl_frameset* frames = NULL;
    cpl_frameset* frames1 = NULL;

    int nx = 100;
    int ny = 100;
    double value1 = 100.0;
    double rmsd1 = 2.0;
    double nmap1 = 20.0;

    double value2 = 300.0;
    double rmsd2 = 5.0;
    double nmap2 = 10.0;

    double value3 = 200.0;
    double rmsd3 = 3.0;
    double nmap3 = 15.0;

    double mean = 0;
    double rms = 0;

    sph_double_image* di1 = NULL;
    sph_double_image* di2 = NULL;
    sph_double_image* di3 = NULL;
    //sph_cube*					dicube = NULL;
    //cpl_frame*				frame_cube = NULL;

    //dicube = sph_cube_new("test_cube_di.fits");
    frames = cpl_frameset_new();
    //create first master frame
    image1 = sph_test_image_tools_create_flat_image_double(nx, ny, value1);
    mframe1 = sph_master_frame_new_from_cpl_image(image1);
    cpl_image_delete(image1);
    sph_utils_cpl_image_fill_double(mframe1->rmsmap, rmsd1);
    sph_utils_cpl_image_fill_double(mframe1->ncombmap, nmap1);
    frame1 = sph_filemanager_create_temp_frame("test_mean_weight_mf.fits",
            "None", CPL_FRAME_GROUP_NONE);
    //sph_master_frame_save( mframe1, cpl_frame_get_filename(frame1), NULL);
    di1 = sph_double_image_new_from_master_frames(mframe1,
            sph_master_frame_duplicate(mframe1));
    //sph_cube_append_double_image(dicube, di1, 0);

    sph_double_image_save(di1, cpl_frame_get_filename(frame1), NULL,
            CPL_IO_CREATE);
    sph_double_image_delete(di1);
    di1 = NULL;
    cpl_frameset_insert(frames, frame1);

    //create second master frame
    image2 = sph_test_image_tools_create_flat_image_double(nx, ny, value2);
    mframe2 = sph_master_frame_new_from_cpl_image(image2);
    cpl_image_delete(image2);

    sph_utils_cpl_image_fill_double(mframe2->rmsmap, rmsd2);
    sph_utils_cpl_image_fill_double(mframe2->ncombmap, nmap2);
    //cpl_image_fill_int( mframe2->badpixelmap, 1); //bad pixel for the whole image
    frame2 = sph_filemanager_create_temp_frame("test_mean_weight_mf.fits",
            "None", CPL_FRAME_GROUP_NONE);
    //sph_master_frame_save( mframe2, cpl_frame_get_filename(frame2), NULL);
    di2 = sph_double_image_new_from_master_frames(mframe2,
            sph_master_frame_duplicate(mframe2));
    //sph_cube_append_double_image(dicube, di2, 0);
    sph_double_image_save(di2, cpl_frame_get_filename(frame2), NULL,
            CPL_IO_CREATE);
    sph_double_image_delete(di2);
    di2 = NULL;
    cpl_frameset_insert(frames, frame2);
    //frame_cube = cpl_frame_new();
    //cpl_frame_set_filename( frame_cube, "test_cube_di.fits");
    //cpl_frame_set_tag( frame_cube, "NONE");
    //cpl_frameset_insert( frames, frame_cube);

    //calculate weighted mean for 2 double images

    result = sph_common_science_combine_weight_double_image_frameset(frames);

    cpl_test_nonnull( result );
    sph_double_image_save(result, "test_mean_weight_2mf_result.fits", NULL,
            CPL_IO_CREATE);
    mean = (value1 / (rmsd1 * rmsd1) + value2 / (rmsd2 * rmsd2))
            / (1. / (rmsd1 * rmsd1) + 1. / (rmsd2 * rmsd2));
    rms = sqrt(1. / (rmsd1 * rmsd1) + 1. / (rmsd2 * rmsd2));
    //printf("mean = %f\n", mean);
    cpl_test_abs( cpl_image_get_mean( result->iframe->image ), mean,
            1e-20);
    //cpl_test_abs( cpl_image_get_mean( result->ncombmap ), nmap1+nmap2, 1e-20);
    cpl_test_abs( cpl_image_get_mean( result->iframe->rmsmap ), rms,
            1e-20);

    cpl_test_abs( cpl_image_get_mean( result->pframe->image ), mean,
            1e-20);
    //cpl_test_abs( cpl_image_get_mean( result->ncombmap ), nmap1+nmap2, 1e-20);
    cpl_test_abs( cpl_image_get_mean( result->pframe->rmsmap ), rms,
            1e-20);

    sph_double_image_delete(result);
    result = NULL;
    frames1 = cpl_frameset_duplicate(frames);
    cpl_frameset_delete(frames);
    frames = NULL;

    //create third master frame
    image3 = sph_test_image_tools_create_flat_image_double(nx, ny, value3);
    mframe3 = sph_master_frame_new_from_cpl_image(image3);
    cpl_image_delete(image3);
    sph_utils_cpl_image_fill_double(mframe3->rmsmap, rmsd3);
    sph_utils_cpl_image_fill_double(mframe3->ncombmap, nmap3);
    //cpl_image_fill_int( mframe3->badpixelmap, 1); //bad pixel for the whole image
    frame3 = sph_filemanager_create_temp_frame("test_mean_weight_mf.fits",
            "None", CPL_FRAME_GROUP_NONE);
    //sph_master_frame_save( mframe3, cpl_frame_get_filename(frame3), NULL);
    di3 = sph_double_image_new_from_master_frames(mframe3,
            sph_master_frame_duplicate(mframe3));
    sph_double_image_save(di3, cpl_frame_get_filename(frame3), NULL,
            CPL_IO_CREATE);
    sph_double_image_delete(di3);
    di3 = NULL;
    cpl_frameset_insert(frames1, frame3);

    //calculate weighted mean for 3 master frames
    result = sph_common_science_combine_weight_double_image_frameset(frames1);
    cpl_test_assert( result);
    sph_double_image_save(result, "test_mean_weight_3mf_result.fits", NULL,
            CPL_IO_CREATE);
    mean = (value1 / (rmsd1 * rmsd1) + value2 / (rmsd2 * rmsd2)
            + value3 / (rmsd3 * rmsd3))
            / (1. / (rmsd1 * rmsd1) + 1. / (rmsd2 * rmsd2)
                    + 1. / (rmsd3 * rmsd3));
    rms = sqrt(
            1. / (rmsd1 * rmsd1) + 1. / (rmsd2 * rmsd2) + 1. / (rmsd3 * rmsd3));
    cpl_test_abs( cpl_image_get_mean( result->iframe->image ), mean,
            1e-20);
    //cpl_test_abs( cpl_image_get_mean( result->iframe->ncombmap ), nmap1+nmap2+nmap3, 1e-20);
    cpl_test_abs( cpl_image_get_mean( result->iframe->rmsmap ), rms,
            1e-20);
    //cpl_test_eq((int)cpl_image_get_absflux( result->badpixelmap ), nx*ny );

    sph_double_image_delete(result);

    cpl_frameset_delete(frames1);
    frames1 = NULL;
    return;

}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;

    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Science support functionality testsuite",
            cutest_init_testsuite, cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }

    if (cpl_msg_get_level() <= CPL_MSG_DEBUG && NULL ==
        sph_test_do(pSuite, "Testing cube full set combine",
                    cutest_sph_common_science_test_combine_cube_fullset ) ) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing raw calibration bad input",
                    cutest_sph_common_science_calibrate_raw_test_bad_input)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing raw calibration accuracy",
                    cutest_sph_common_science_calibrate_raw_test)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "Testing raw calibration accuracy no distortion",
                    cutest_sph_common_science_calibrate_raw_test_nodist)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_common_science_combine with ideal input",
                    cutest_sph_common_science_test_combine_ideal_input)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_common_science_combine with noisy ideal input",
                    cutest_sph_common_science_test_combine_noisy_ideal_input)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_common_science_combine with ideal input no badpix",
                    cutest_sph_common_science_test_combine_nobadpix)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_common_science_get_next_raw",
                    cutest_sph_common_science_get_next_raw_test)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_common_science_get_next_raw_cube2normal",
                    cutest_sph_common_science_get_next_raw_test_swap_cube_to_normal)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_common_science_get_next_zpl_exposure",
                    cutest_sph_common_science_get_next_zpl_exposure_test)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_common_science_combine_weight_double_image_frameset",
                    cutest_sph_common_science_combine_weight_double_image_frameset_test)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
