/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_cube.h"
#include "sph_common_keywords.h"
#include "math.h"
#include "sph_error.h"
#include "sph_test.h"

#include <math.h>
#include "sph_utils.h"
#include "sph_test.h"
#include "sph_fits.h"
static
void cutest_sph_cube_finalise_file_setup__(cpl_frameset* outframes);
/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_sph_cube Unit test for the sph_cube module
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static
int cutest_init_sph_cube_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

static
int cutest_clean_sph_cube_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Close and dump errors.
     * -------------------------------------------------------------------*/
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_cube_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_cube_new(void) {
    sph_cube* acube = NULL;
    const char* fname = "testcube.fits";
    FILE* fp = NULL;

    unlink(fname);

    acube = sph_cube_new(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(acube);

    sph_filemanager_add_tmp_file(fname);
    cpl_test_nonnull( fname );
    acube = sph_cube_new(fname);
    cpl_test_nonnull( acube );

    cpl_test_nonnull( acube->filename );
    cpl_test_eq_string(acube->filename, fname);
    cpl_test_zero(acube->file_exists);
    cpl_test_zero(acube->nx);
    cpl_test_zero(acube->ny);
    cpl_test_eq(acube->type, CPL_TYPE_UNSPECIFIED);
    cpl_test_assert(acube->proplist);

    // No file should be created yet...
    fp = fopen(fname, "r");
    cpl_test_null(fp);
    sph_cube_delete(acube);
    acube = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_cube_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_cube_new_dfs(void) {
    sph_cube* acube = NULL;
    const char* fname = "testcube.fits";
    FILE* fp = NULL;
    cpl_frameset* frameset = cpl_frameset_new();
    cpl_frame* aframe = cpl_frame_new();
    cpl_parameterlist* params = cpl_parameterlist_new();

    unlink(fname);
    acube = sph_cube_new_dfs(NULL, frameset, aframe, params, "ATAG", "ARECIPE",
                             "IRDIS", NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(acube);
    acube = sph_cube_new_dfs(fname, frameset, aframe, params, NULL, "ARECIPE",
                             "IRDIS", NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(acube);

    sph_filemanager_add_tmp_file(fname);
    acube = sph_cube_new_dfs(fname, frameset, aframe, params, "ATAG", "ARECIPE",
            "IRDIS", NULL);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( acube );

    cpl_test_nonnull( acube->filename );
    cpl_test_eq_string(acube->filename, fname);
    cpl_test_zero(acube->file_exists);
    cpl_test_zero(acube->nx);
    cpl_test_zero(acube->ny);
    cpl_test_eq(acube->type, CPL_TYPE_UNSPECIFIED);
    cpl_test_assert(acube->proplist);
    cpl_test_eq_ptr(acube->dfs_allframes, frameset);
    cpl_test_eq_ptr(acube->dfs_template_frame, aframe);
    cpl_test_eq_ptr(acube->dfs_paramlist, params);
    cpl_test_eq_string(acube->dfs_tag, "ATAG");
    cpl_test_eq_string(acube->dfs_recipe, "ARECIPE");
    cpl_test_eq_string(acube->dfs_pipename, "IRDIS");
    // No file should be created yet...
    fp = fopen(fname, "r");
    cpl_test_null(fp);
    sph_cube_delete(acube);
    acube = NULL;
    cpl_frameset_delete(frameset);
    frameset = NULL;
    cpl_frame_delete(aframe);
    aframe = NULL;
    cpl_parameterlist_delete(params);
    params = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_cube_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_cube_append_master_frame_newcube(void) {
    sph_cube* acube = NULL;
    sph_master_frame* mframe = NULL;
    const char* fname = "testcube2.fits";
    FILE* fp = NULL;

    unlink(fname);
    sph_filemanager_add_tmp_file(fname);
    cpl_test_nonnull( fname );
    acube = sph_cube_new(fname);
    cpl_test_nonnull( acube );

    mframe = sph_master_frame_new(256, 256);
    cpl_test_eq_error(sph_cube_append_master(acube,mframe,1), CPL_ERROR_NONE);
    cpl_test_eq_error(sph_cube_finalise_file(acube->filename), CPL_ERROR_NONE);

    cpl_test_eq(acube->file_exists, 1);
    cpl_test_eq(acube->nx, 256);
    cpl_test_eq(acube->ny, 256);

    // file should have been created...
    fp = fopen(fname, "r");
    cpl_test_nonnull( fp );
    fclose(fp);
    sph_cube_delete(acube);
    acube = NULL;
    sph_master_frame_delete(mframe);
    mframe = NULL;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_cube_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_cube_append_master_frame(void) {
    sph_cube* acube = NULL;
    sph_master_frame* mframe = NULL;
    const char* fname = "testcube2.fits";
    FILE* fp = NULL;
    cpl_error_code code;
    int retval;

    unlink(fname);
    sph_filemanager_add_tmp_file(fname);
    cpl_test_nonnull( fname );
    acube = sph_cube_new(fname);
    cpl_test_nonnull( acube );

    mframe = sph_master_frame_new(256, 256);
    code = sph_cube_append_master(acube, mframe, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = sph_cube_finalise_file(acube->filename);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_eq(acube->file_exists, 1);
    cpl_test_eq(acube->nx, 256);
    cpl_test_eq(acube->ny, 256);

    // file should have been created...
    fp = fopen(fname, "r");
    cpl_test_nonnull( fp );
    fclose(fp);

    code = sph_cube_append_master(acube, mframe, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = sph_cube_finalise_file(acube->filename);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    retval = sph_fits_test_iscube(fname, 0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(retval, 1);

    retval = sph_fits_get_nplanes(fname, 0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(retval, 2);

    retval = sph_fits_test_iscube(fname, -1);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_eq(retval, -1);

    retval = sph_fits_get_nplanes(fname, -1);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_eq(retval, -1);

    retval = sph_fits_test_iscube(fname, 8);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    cpl_test_eq(retval, -1);

    retval = sph_fits_get_nplanes(fname, 8);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    cpl_test_eq(retval, -1);

    retval = sph_fits_test_iscube(NULL, 0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(retval, -1);

    retval = sph_fits_get_nplanes(NULL, 0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(retval, -1);

    sph_cube_delete(acube);
    acube = NULL;
    sph_master_frame_delete(mframe);
    mframe = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_cube_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_cube_append_image(void) {
    cpl_image* image = NULL;
    const char* fname = "testcube_mext.fits";
    const char* fname0 = "testcube_mext_e00.fits";
    FILE* fp = NULL;
    cpl_propertylist* pl = NULL;

    unlink(fname);
    pl = cpl_propertylist_new();
    cpl_propertylist_update_int(pl, "MYKEY", 1000);
    sph_filemanager_add_tmp_file(fname);
    cpl_test_nonnull( fname );

    image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(sph_cube_append_image(fname,image,pl,0), CPL_ERROR_NONE);

    // file should have been created...
    fp = fopen(fname0, "r");
    cpl_test_nonnull( fp );
    fclose(fp);

    cpl_propertylist_update_int(pl, "MYKEY2", 1000);
    cpl_test_eq_error(sph_cube_append_image(fname,image,pl,0), CPL_ERROR_NONE);

    cpl_test_eq(sph_fits_get_nplanes(fname0,0), 2);

    cpl_propertylist_delete(pl);
    pl = NULL;

    pl = cpl_propertylist_load(fname0, 0);
    cpl_test_nonnull( pl );
    cpl_test_eq(cpl_propertylist_get_int(pl,"MYKEY"), 1000);
    cpl_test_eq(cpl_propertylist_get_int(pl,"MYKEY2"), 1000);

    cpl_propertylist_delete(pl);
    pl = NULL;
    cpl_image_delete(image);
    image = NULL;
    unlink(fname0);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_cube_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_cube_append_image_props_ok(void) {
    cpl_image* image = NULL;
    const char* fname = "testcube_mext.fits";
    const char* fname0 = "testcube_mext_e00.fits";
    FILE* fp = NULL;
    cpl_propertylist* pl = NULL;

    unlink(fname);
    pl = cpl_propertylist_new();
    cpl_propertylist_update_int(pl, "MYKEY", 1000);
    sph_filemanager_add_tmp_file(fname);
    cpl_test_nonnull( fname );

    image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    cpl_test_eq_error(sph_cube_append_image(fname,image,pl,0), CPL_ERROR_NONE);

    // file should have been created...
    fp = fopen(fname0, "r");
    cpl_test_nonnull( fp );
    fclose(fp);

    cpl_propertylist_update_int(pl, "MYKEY2", 1000);
    cpl_test_eq_error(sph_cube_append_image(fname,image,pl,0), CPL_ERROR_NONE);

    cpl_test_eq_error(sph_cube_append_image(fname,image,NULL,1), CPL_ERROR_NONE);
    cpl_test_eq(sph_fits_get_nplanes(fname0,0), 2);

    cpl_test_eq_error(sph_cube_append_image(fname,image,NULL,0), CPL_ERROR_NONE);

    cpl_propertylist_delete(pl);
    pl = NULL;

    pl = cpl_propertylist_load(fname0, 0);
    cpl_test_nonnull( pl );
    cpl_test_eq(cpl_propertylist_get_int(pl,"MYKEY"), 1000);
    cpl_test_eq(cpl_propertylist_get_int(pl,"MYKEY2"), 1000);

    cpl_propertylist_delete(pl);
    pl = NULL;
    sph_cube_finalise_file(fname);
    pl = cpl_propertylist_load(fname, 0);
    cpl_test_nonnull( pl );
    cpl_test_eq(cpl_propertylist_get_int(pl,"MYKEY"), 1000);
    cpl_test_eq(cpl_propertylist_get_int(pl,"MYKEY2"), 1000);
    cpl_propertylist_delete(pl);
    pl = NULL;

    cpl_image_delete(image);
    image = NULL;
    unlink(fname0);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_cube_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_cube_finalise_file_nothing_todo(void) {
    const char* fname = "testcube_mext.fits";
    const char* fname0 = "testcube_mext_e00.fits";
    const char* fname1 = "testcube_mext_e01.fits";
    FILE* fp = NULL;

    unlink(fname);
    // file should not exist...
    fp = fopen(fname0, "r");
    cpl_test_null(fp);
    fp = fopen(fname1, "r");
    cpl_test_null(fp);

    cpl_test_eq_error(sph_cube_finalise_file(fname), CPL_ERROR_NONE);

    fp = fopen(fname, "r");
    cpl_test_null(fp);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_cube_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_cube_finalise_file(void) {
    const char* fname = "testcube_mext.fits";
    const char* fname0 = "testcube_mext_e00.fits";
    const char* fname1 = "testcube_mext_e01.fits";
    FILE* fp = NULL;
    cpl_imagelist* imlist = NULL;
    cpl_propertylist* pl = NULL;

    unlink(fname);
    cutest_sph_cube_finalise_file_setup__(NULL);
    cpl_test_eq_error(sph_cube_finalise_file(fname), CPL_ERROR_NONE);

    // file should have been created...
    fp = fopen(fname, "r");
    cpl_test_nonnull( fp );
    fclose(fp);

    cpl_test_eq(sph_fits_get_nplanes(fname,0), 5);

    cpl_test_eq(sph_fits_get_nplanes(fname,1), 5);

    cpl_propertylist_delete(pl);
    pl = NULL;

    pl = cpl_propertylist_load(fname, 0);
    cpl_test_nonnull( pl );
    cpl_test_eq(cpl_propertylist_get_int(pl,"MYKEY"), 1000);
    cpl_test_zero(cpl_propertylist_has(pl,"MYKEY2"));

    cpl_propertylist_delete(pl);
    pl = NULL;
    pl = cpl_propertylist_load(fname, 1);
    cpl_test_nonnull( pl );
    cpl_test_eq(cpl_propertylist_get_int(pl,"MYKEY"), 1000);
    cpl_test_eq(cpl_propertylist_get_int(pl,"MYKEY2"), 1000);

    cpl_propertylist_delete(pl);
    pl = NULL;

    cpl_imagelist_delete(imlist);
    imlist = NULL;
    unlink(fname0);
    unlink(fname1);
    // Finalising a file twice should give no error.
    cpl_test_eq_error(sph_cube_finalise_file(fname), CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_cube_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_cube_finalise_all(void) {
    const char* fname = "testcube_mext.fits";
    const char* fname0 = "testcube_mext_e00.fits";
    const char* fname1 = "testcube_mext_e01.fits";
    FILE* fp = NULL;
    cpl_imagelist* imlist = NULL;
    cpl_propertylist* pl = NULL;
    cpl_frameset* frames = NULL;
    cpl_frame* aframe = NULL;
    frames = cpl_frameset_new();
    aframe = cpl_frame_new();
    cpl_frame_set_tag(aframe, "NONE");
    cpl_frame_set_group(aframe, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_filename(aframe, fname);
    cpl_frameset_insert(frames, aframe);

    unlink(fname);
    cutest_sph_cube_finalise_file_setup__(NULL);
    // making sure there is no cube before test
    unlink(fname);
    cpl_test_eq_error(sph_cube_finalise_all(frames), CPL_ERROR_NONE);

    // file should have been created...
    fp = fopen(fname, "r");
    cpl_test_nonnull( fp );
    fclose(fp);

    cpl_test_eq(sph_fits_get_nplanes(fname,0), 5);

    cpl_test_eq(sph_fits_get_nplanes(fname,1), 5);

    cpl_propertylist_delete(pl);
    pl = NULL;

    pl = cpl_propertylist_load(fname, 0);
    cpl_test_nonnull( pl );
    cpl_test_eq(cpl_propertylist_get_int(pl,"MYKEY"), 1000);
    cpl_test_zero(cpl_propertylist_has(pl,"MYKEY2"));

    cpl_propertylist_delete(pl);
    pl = NULL;
    pl = cpl_propertylist_load(fname, 1);
    cpl_test_nonnull( pl );
    cpl_test_eq(cpl_propertylist_get_int(pl,"MYKEY"), 1000);
    cpl_test_eq(cpl_propertylist_get_int(pl,"MYKEY2"), 1000);

    cpl_propertylist_delete(pl);
    pl = NULL;

    cpl_imagelist_delete(imlist);
    imlist = NULL;
    unlink(fname0);
    unlink(fname1);
    cpl_frameset_delete(frames);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_cube_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_cube_append_double_image_newcube(void) {
    sph_cube* acube = NULL;
    sph_double_image* dimage = NULL;
    const char* fname = "testcube2.fits";
    FILE* fp = NULL;

    unlink(fname);

    sph_filemanager_add_tmp_file(fname);
    cpl_test_nonnull( fname );
    acube = sph_cube_new(fname);
    cpl_test_nonnull( acube );

    dimage = sph_double_image_new(256, 256);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(sph_cube_append_double_image(acube,dimage, 1),
            CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq_error(sph_cube_finalise_file(acube->filename), CPL_ERROR_NONE);

    cpl_test_eq(acube->file_exists, 1);
    cpl_test_eq(acube->nx, 256);
    cpl_test_eq(acube->ny, 256);

    // file should have been created...
    fp = fopen(fname, "r");
    cpl_test_nonnull( fp );
    fclose(fp);

    cpl_test_eq(sph_fits_get_nplanes(fname,0), 1);

    cpl_test_error(CPL_ERROR_NONE);
    sph_cube_delete(acube);
    acube = NULL;
    sph_double_image_delete(dimage);
    dimage = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_cube_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_cube_append_double_image(void) {
    sph_cube* acube = NULL;
    sph_double_image* dimage = NULL;
    const char* fname = "testcube2.fits";
    FILE* fp = NULL;

    unlink(fname);
    sph_filemanager_add_tmp_file(fname);
    cpl_test_nonnull( fname );
    acube = sph_cube_new(fname);
    cpl_test_nonnull( acube );

    dimage = sph_double_image_new(256, 256);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(sph_cube_append_double_image(acube,dimage, 1),
            CPL_ERROR_NONE);
    cpl_test_eq_error(sph_cube_finalise_file(acube->filename), CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(acube->file_exists, 1);
    cpl_test_eq(acube->nx, 256);
    cpl_test_eq(acube->ny, 256);

    // file should have been created...
    fp = fopen(fname, "r");
    cpl_test_nonnull( fp );
    fclose(fp);

    cpl_test_eq(sph_cube_append_double_image(acube,dimage, 1),
            CPL_ERROR_NONE);
    cpl_test_eq_error(sph_cube_finalise_file(acube->filename), CPL_ERROR_NONE);

    cpl_test_eq(sph_fits_get_nplanes(fname,0), 2);

    sph_cube_delete(acube);
    acube = NULL;
    sph_double_image_delete(dimage);
    dimage = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Main to run the tests.
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("cutest_sph_cube", cutest_init_sph_cube_testsuite,
            cutest_clean_sph_cube_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }

    if (NULL == sph_test_do(pSuite, "Testing cube_new", cutest_sph_cube_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing cube_new_dfs",
                    cutest_sph_cube_new_dfs)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing cube_append_master_frame_newcube",
                    cutest_sph_cube_append_master_frame_newcube)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing cube_append_master_frame",
                    cutest_sph_cube_append_master_frame)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing cube_append_double_image_newcube",
                    cutest_sph_cube_append_double_image_newcube)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing cube_append_double_image_frame",
                    cutest_sph_cube_append_double_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing cube_append_image_frame",
                    cutest_sph_cube_append_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing cube_append_image_frame_props_ok",
                    cutest_sph_cube_append_image_props_ok)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing cube_finalise_file",
                    cutest_sph_cube_finalise_file)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing cube_finalise_file_nothing",
                    cutest_sph_cube_finalise_file_nothing_todo)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing cube_finalise_all",
                    cutest_sph_cube_finalise_all)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}
static
void cutest_sph_cube_finalise_file_setup__(cpl_frameset* outframes) {
    cpl_image* image = NULL;
    const char* fname = "testcube_mext.fits";
    const char* fname0 = "testcube_mext_e00.fits";
    const char* fname1 = "testcube_mext_e01.fits";
    cpl_imagelist* imlist = NULL;
    cpl_propertylist* pl = NULL;
    int np = 5;
    int pp = 0;
    cpl_frame* aframe = NULL;
    pl = cpl_propertylist_new();
    cpl_propertylist_update_int(pl, "MYKEY", 1000);
    sph_filemanager_add_tmp_file(fname);
    cpl_test_nonnull( fname );

    imlist = cpl_imagelist_new();
    for (pp = 0; pp < np; ++pp) {
        image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
        cpl_image_add_scalar(image, pp * 1.1);
        cpl_imagelist_set(imlist, image, pp);
    }
    cpl_imagelist_save(imlist, fname0, CPL_TYPE_UNSPECIFIED, pl, CPL_IO_CREATE);
    cpl_imagelist_delete(imlist);
    imlist = NULL;

    cpl_propertylist_update_int(pl, "MYKEY2", 1000);
    imlist = cpl_imagelist_new();
    for (pp = 0; pp < np; ++pp) {
        image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
        cpl_image_add_scalar(image, pp * 1.1);
        cpl_imagelist_set(imlist, image, pp);
    }
    cpl_imagelist_save(imlist, fname1, CPL_TYPE_UNSPECIFIED, pl, CPL_IO_CREATE);
    cpl_imagelist_delete(imlist);
    imlist = NULL;

    if (outframes) {
        aframe = cpl_frame_new();
        cpl_frame_set_tag(aframe, "NONE");
        cpl_frame_set_group(aframe, CPL_FRAME_GROUP_CALIB);
        cpl_frame_set_filename(aframe, fname0);
        cpl_frameset_insert(outframes, aframe);
        aframe = cpl_frame_new();
        cpl_frame_set_tag(aframe, "NONE");
        cpl_frame_set_group(aframe, CPL_FRAME_GROUP_CALIB);
        cpl_frame_set_filename(aframe, fname1);
        cpl_frameset_insert(outframes, aframe);
    }
    cpl_propertylist_delete(pl);
    pl = NULL;
}

/**@}*/
