/* $Id: $
 *
 * This file is part of the SPHERE Pipesph_filemanager
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_filemanager.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"


#include "sph_test.h"
#include "sph_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static
int cutest_init_sph_filemanager_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

static
int cutest_clean_sph_filemanager_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    sph_end_test();
    return 0;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_filemanager_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_filemanager_new(void) {
    sph_filemanager* sphfilemanager = NULL;

    sphfilemanager = sph_filemanager_new();

    cpl_test_nonnull( sphfilemanager );
    cpl_test_nonnull( sphfilemanager->workdir );
    cpl_test_nonnull( sphfilemanager->tmpfiles );
    cpl_test_zero(sphfilemanager->ntmpfiles);
    cpl_test_eq_string(sphfilemanager->workdir, ".");
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_filemanager_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_filemanager_get_numbered_filename(void) {
    sph_filemanager* sphfilemanager = NULL;
    char* filename = NULL;

    sphfilemanager = sph_filemanager_new();

    cpl_test_nonnull( sphfilemanager );
    cpl_test_nonnull( sphfilemanager->workdir );
    cpl_test_nonnull( sphfilemanager->tmpfiles );

    filename = sph_filemanager_get_numbered_filename("atest.fits", 1);
    cpl_test_eq_string(filename, "atest_00001.fits");
    cpl_free(filename);
    filename = NULL;
    filename = sph_filemanager_get_numbered_filename_new_file("atest.fits");
    cpl_test_eq_string(filename, "atest_00000.fits");
    cpl_test_zero(sphfilemanager->ntmpfiles);
    cpl_test_eq_string(sphfilemanager->workdir, ".");
    cpl_free(filename);
    filename = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_filemanager_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_filemanager_get_unique_duplicate_frame(void) {
    cpl_frame* inframe = NULL;
    cpl_frame* outframe = NULL;

    inframe = cpl_frame_new();
    cpl_test_nonnull( inframe );
    cpl_frame_set_filename(inframe, "atest.fits");
    cpl_frame_set_tag(inframe, "ATAG");
    cpl_frame_set_group(inframe, CPL_FRAME_GROUP_PRODUCT);

    outframe = sph_filemanager_get_unique_duplicate_frame(inframe);
    cpl_test_nonnull( outframe );
    cpl_test_eq_string(cpl_frame_get_filename(outframe),
            "atest_00000.fits");
    cpl_test_eq_string(cpl_frame_get_tag(outframe), "ATAG");
    cpl_test_eq(cpl_frame_get_group(outframe), CPL_FRAME_GROUP_PRODUCT);
    cpl_frame_delete(outframe);
    outframe = NULL;
    cpl_frame_delete(inframe);
    inframe = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_filemanager_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_filemanager_remove_dir(void) {
    const char* fname = "this/path/leads/to/nothing";
    const char* fname2 = "nothing";
    char* nname = NULL;
    cpl_test_nonnull( nname = sph_filemanager_remove_dir(fname) );
    cpl_test_eq_string(nname, "nothing");
    cpl_free(nname);
    nname = NULL;
    cpl_test_nonnull( nname = sph_filemanager_remove_dir(fname2) );
    cpl_test_eq_string(nname, "nothing");
    cpl_free(nname);
    nname = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_filemanager_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_filemanager_split(void) {
    char base[256];
    char ext[256];

    cpl_test_eq(sph_filemanager_split("atest.fits",base, ext),
            CPL_ERROR_NONE);
    cpl_test_eq_string(base, "atest");
    cpl_test_eq_string(ext, "fits");
    cpl_test_eq_error(sph_filemanager_split("btest",base,ext), CPL_ERROR_NONE);
    cpl_test_eq_string(base, "btest");
    cpl_test_eq_error(sph_filemanager_split(".txt",base,ext), CPL_ERROR_NONE);
    cpl_test_eq_string(ext, "txt");

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_filemanager_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_filemanager_new_filename_from_base(void) {
    char* result = NULL;

    result = sph_filemanager_new_filename_from_base("atest.fits", "hello");
    cpl_test_nonnull( result );
    cpl_test_eq_string(result, "atest_hello.fits");
    cpl_free(result);
    result = NULL;
    result = sph_filemanager_new_filename_from_base("atest", "hello");
    cpl_test_nonnull( result );
    cpl_test_eq_string(result, "atest_hello");
    cpl_free(result);
    result = NULL;
    result = sph_filemanager_new_filename_from_base("atest.fits.txt", "hello");
    cpl_test_nonnull( result );
    cpl_test_eq_string(result, "atest.fits_hello.txt");
    cpl_free(result);
    result = NULL;
    result = sph_filemanager_new_filename_from_base(".txt", "hello");
    cpl_test_nonnull( result );
    cpl_test_eq_string(result, "_hello.txt");
    cpl_free(result);
    result = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_filemanager_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_filemanager_delete(void) {
    sph_filemanager* sphfft = NULL;
    FILE* fp = NULL;
    sphfft = sph_filemanager_new();
    cpl_test_nonnull( sphfft );
    fp = fopen("testdummy.txt", "w");
    fprintf(fp, "Hello!\n");
    fclose(fp);
    fp = NULL;
    fp = fopen("testdummy.txt", "r");
    cpl_test_nonnull( fp );
    fclose(fp);
    fp = NULL;

    sph_filemanager_add_tmp_file("testdummy.txt");

    sph_filemanager_delete(0);
    fp = fopen("testdummy.txt", "r");
    cpl_test_null(fp);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test of the named function
 */
/*----------------------------------------------------------------------------*/
static void cutest_sph_filemanager_get_basename(void)
{
    const char * null;
    const char * slash = "/";
    const char * path = "/a/b";
    const char * file = "42";
    const char* base;


    null = sph_filemanager_get_basename(NULL);
    cpl_test_null(null);

    base = sph_filemanager_get_basename(slash);
    cpl_test_eq_string(base, "");

    base = sph_filemanager_get_basename(path);
    cpl_test_eq_string(base, "b");

    base = sph_filemanager_get_basename(file);
    cpl_test_eq_ptr(base, file);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {

    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("sph_filemanager_test",
            cutest_init_sph_filemanager_testsuite,
            cutest_clean_sph_filemanager_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    cutest_sph_filemanager_get_basename();

    if (NULL
            == sph_test_do(pSuite, "sph_filemanager_new",
                    cutest_sph_filemanager_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_filemanager_delete",
                    cutest_sph_filemanager_delete)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_filemanager_get_numbered_filename",
                    cutest_sph_filemanager_get_numbered_filename)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_filemanager_get_unique_duplicate_frame",
                    cutest_sph_filemanager_get_unique_duplicate_frame)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_filemanager_split",
                    cutest_sph_filemanager_split)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_filemanager_remove_dir",
                    cutest_sph_filemanager_remove_dir)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_filemanager_new_filename_from_base",
                    cutest_sph_filemanager_new_filename_from_base)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
