/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_framesorter.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"
#include "sph_test.h"


#include "sph_test.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static
int cutest_init_framesorter_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

static
int cutest_clean_framesorter_testsuite(void) {
    sph_end_test();
    return 0;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_framesorter_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_framesorter_new(void) {
    sph_framesorter* sorter = NULL;
    cpl_frameset* frameset = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    frameset = cpl_frameset_new();
    cpl_test_nonnull( frameset );
    rerr = sph_test_create_ifs_raw_dark_frameset(frameset, 3);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sorter = sph_framesorter_new(NULL);
    cpl_test_null( sorter);
    sorter = sph_framesorter_new(frameset);
    cpl_test_nonnull( sorter );
    cpl_test_null( sorter->framebag_list);
    cpl_test_eq_ptr( sorter->inframeset, frameset);
    cpl_test_null( sorter->sort_criteria);
    sph_framesorter_delete(sorter);
    sph_test_frameset_delete(frameset);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_framesorter_create function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_framesorter_sort(void) {
    sph_framesorter* sorter = NULL;
    cpl_frameset* frameset = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_parameter* param = NULL;
    cpl_parameterlist* deps = NULL;

    frameset = cpl_frameset_new();
    cpl_test_nonnull( frameset );
    rerr = sph_test_create_ifs_raw_dark_frameset(frameset, 3);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sorter = sph_framesorter_new(frameset);
    cpl_test_nonnull( sorter );
    deps = cpl_parameterlist_new();
    param = cpl_parameter_new_value("recipe.dependency.1.keyname",
            CPL_TYPE_STRING, "nothing", "none", SPH_COMMON_KEYWORD_EXPTIME);
    cpl_parameterlist_append(deps, param);
    param = cpl_parameter_new_value("recipe.dependency.2.keyname",
            CPL_TYPE_STRING, "nothing", "none", SPH_COMMON_KEYWORD_NREADS);
    cpl_parameterlist_append(deps, param);
    param = cpl_parameter_new_value("recipe.dependency.1.tolerance",
            CPL_TYPE_DOUBLE, "nothing", "none", 0.1);
    cpl_parameterlist_append(deps, param);
    param = cpl_parameter_new_value("recipe.dependency.2.tolerance",
            CPL_TYPE_INT, "nothing", "none", 0);
    cpl_parameterlist_append(deps, param);
    param = cpl_parameter_new_range("recipe.dependency.1.range",
            CPL_TYPE_DOUBLE, "nothing", "none", 1.3, 0.0, 10.0);
    cpl_parameterlist_append(deps, param);
    param = cpl_parameter_new_range("recipe.dependency.2.range", CPL_TYPE_INT,
            "nothing", "none", 1, 1, 10);
    cpl_parameterlist_append(deps, param);
    rerr = sph_framesorter_sort(sorter, deps);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( sorter->framebag_list );
    cpl_test_nonnull( sorter->framebag_list->frameset );
    cpl_test_eq(cpl_frameset_get_size(sorter->framebag_list->frameset), 3);
    sph_framesorter_delete(sorter);
    sph_test_frameset_delete(frameset);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_framesorter_check_set function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_framesorter_check_set(void) {
    sph_framesorter* sorter = NULL;
    cpl_frameset* frameset = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_parameter* param = NULL;
    cpl_frame* tframe = NULL;

    frameset = cpl_frameset_new();
    cpl_test_nonnull( frameset );
    rerr = sph_test_create_ifs_raw_dark_frameset(frameset, 3);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sorter = sph_framesorter_new(frameset);
    cpl_test_nonnull( sorter );
    sorter->sort_criteria = cpl_parameterlist_new();
    param = cpl_parameter_new_value("recipe.dependency.1.keyname",
            CPL_TYPE_STRING, "nothing", "none", SPH_COMMON_KEYWORD_EXPTIME);
    cpl_parameterlist_append(sorter->sort_criteria, param);
    param = cpl_parameter_new_value("recipe.dependency.2.keyname",
            CPL_TYPE_STRING, "nothing", "none", SPH_COMMON_KEYWORD_NREADS);
    cpl_parameterlist_append(sorter->sort_criteria, param);
    param = cpl_parameter_new_value("recipe.dependency.1.tolerance",
            CPL_TYPE_DOUBLE, "nothing", "none", 0.1);
    cpl_parameterlist_append(sorter->sort_criteria, param);
    param = cpl_parameter_new_value("recipe.dependency.2.tolerance",
            CPL_TYPE_INT, "nothing", "none", 0);
    cpl_parameterlist_append(sorter->sort_criteria, param);
    param = cpl_parameter_new_range("recipe.dependency.1.range",
            CPL_TYPE_DOUBLE, "nothing", "none", 1.3, 0.0, 10.0);
    cpl_parameterlist_append(sorter->sort_criteria, param);
    param = cpl_parameter_new_range("recipe.dependency.2.range", CPL_TYPE_INT,
            "nothing", "none", 1, 1, 10);
    cpl_parameterlist_append(sorter->sort_criteria, param);
    tframe = cpl_frameset_get_first(frameset);
    rerr = sph_framesorter_check_set(sorter, tframe, 0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( sorter->framebag_list );
    cpl_test_nonnull( sorter->framebag_list->frameset );
    cpl_test_eq(cpl_frameset_get_size(sorter->framebag_list->frameset), 1);
    tframe = cpl_frameset_get_next(frameset);
    rerr = sph_framesorter_check_set(sorter, tframe, 0);
    cpl_test_eq(cpl_frameset_get_size(sorter->framebag_list->frameset), 2);
    sph_framesorter_delete(sorter);
    sph_test_frameset_delete(frameset);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_framesorter_check_set function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_framesorter_new_bag(void) {
    sph_framesorter* sorter = NULL;
    cpl_frameset* frameset = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_vector* validvals = NULL;
    sph_framebag* bag = NULL;
    cpl_parameter* param = NULL;

    frameset = cpl_frameset_new();
    cpl_test_nonnull( frameset );
    rerr = sph_test_create_ifs_raw_dark_frameset(frameset, 3);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sorter = sph_framesorter_new(frameset);
    cpl_test_nonnull( sorter );
    sorter->sort_criteria = cpl_parameterlist_new();
    param = cpl_parameter_new_value("recipe.dependency.1.keyname",
            CPL_TYPE_STRING, "nothing", "none", SPH_COMMON_KEYWORD_EXPTIME);
    cpl_parameterlist_append(sorter->sort_criteria, param);
    param = cpl_parameter_new_value("recipe.dependency.2.keyname",
            CPL_TYPE_STRING, "nothing", "none", SPH_COMMON_KEYWORD_NREADS);
    cpl_parameterlist_append(sorter->sort_criteria, param);
    param = cpl_parameter_new_value("recipe.dependency.1.tolerance",
            CPL_TYPE_DOUBLE, "nothing", "none", 0.1);
    cpl_parameterlist_append(sorter->sort_criteria, param);
    param = cpl_parameter_new_value("recipe.dependency.2.tolerance",
            CPL_TYPE_INT, "nothing", "none", 0);
    cpl_parameterlist_append(sorter->sort_criteria, param);
    param = cpl_parameter_new_range("recipe.dependency.1.range",
            CPL_TYPE_DOUBLE, "nothing", "none", 1.3, 0.0, 10.0);
    cpl_parameterlist_append(sorter->sort_criteria, param);
    param = cpl_parameter_new_range("recipe.dependency.2.range", CPL_TYPE_INT,
            "nothing", "none", 1, 1, 10);
    cpl_parameterlist_append(sorter->sort_criteria, param);

    validvals = cpl_vector_new(2);
    cpl_test_nonnull( validvals );
    cpl_vector_set(validvals, 0, 1.3);
    cpl_vector_set(validvals, 1, 2);
    bag = sph_framesort_new_bag(sorter, validvals);
    cpl_test_nonnull( bag );
    cpl_test_abs( bag->mins[0], 1.3, 1.0e-8);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_ERROR, "Value min 0 was: %lf", bag->mins[0]);
    cpl_test_abs( bag->maxs[0], 1.4, 1.0e-8);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_ERROR, "Value max 0 was: %lf", bag->maxs[0]);
    cpl_test_abs( bag->mins[1], 2, 1.0e-8);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_ERROR, "Value min 1 was: %lf", bag->mins[1]);
    cpl_test_abs( bag->maxs[1], 2, 1.0e-8);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_ERROR, "Value max 1 was: %lf", bag->maxs[1]);
    sph_framesorter_delete(sorter);
    sph_test_frameset_delete(frameset);
    cpl_vector_delete(validvals);
    sph_framebag_delete_all(bag);
    bag = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_framesorter_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_framesorter_delete(void) {
    sph_framesorter* sorter = NULL;
    cpl_frameset* frameset = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    frameset = cpl_frameset_new();
    cpl_test_nonnull( frameset );
    rerr = sph_test_create_ifs_raw_dark_frameset(frameset, 3);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sorter = sph_framesorter_new(frameset);
    cpl_test_nonnull( sorter );

    sph_framesorter_delete(sorter);
    sph_test_frameset_delete(frameset);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {

    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("framesorter_test", cutest_init_framesorter_testsuite,
            cutest_clean_framesorter_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_framesorter_new",
                    cutest_framesorter_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_framesorter_sort",
                    cutest_framesorter_sort)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_framesorter_new_bag",
                    cutest_framesorter_new_bag)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_framesorter_check_set",
                    cutest_framesorter_check_set)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_framesorter_delete",
                    cutest_framesorter_delete)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/

