/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_keyword_manager.h"
#include "sph_common_keywords.h"
#include "sph_fits.h"
#include "sph_error.h"
#include "sph_test.h"
#include <cpl.h>

#include <math.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/
#define SPH_BASE "cutest_sph_keyword_manager"
#define SPH_CFG  SPH_BASE ".cfg"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/
static sph_keyword_manager*
cutest_util_keyword_manager_common_setup(void) {
    FILE* fp = NULL;
    sph_keyword_manager* km = NULL;
    cpl_frameset* frames = NULL;
    cpl_property* prop = NULL;

    fp = fopen(SPH_CFG, "w");
    fprintf(fp, "SPH.KEYWORDS.TEST.SOURCE.REP.NAME = \"BLUBBY\"\n");
    fprintf(fp, "SPH.KEYWORDS.TEST.SOURCE.REP.TYPE = \"INT\"\n");
    fprintf(fp, "SPH.KEYWORDS.TEST.SOURCE.REP.VALUE = \"-3.0\"\n");
    fprintf(fp, "SPH.KEYWORDS.TEST.SOURCE.REP.ALLOWED1 = \"-3.0\"\n");
    fprintf(fp, "SPH.KEYWORDS.TEST.SOURCE.REP.ALLOWED2 = \"4.0\"\n");
    fprintf(fp, "SPH.KEYWORDS.TEST.SOURCE.REP.ALLOWED3 = \"1.0\"\n");
    fprintf(fp, "SPH.KEYWORDS.REP.NAME = \"ESO DRS PC IMAGE NX\"\n");
    fprintf(fp, "SPH.KEYWORDS.REP.TYPE = \"INT\"\n");
    fprintf(fp, "SPH.KEYWORDS.REP.VALUE = \"1024\"\n");
    fprintf(fp, "SPH.KEYWORDS.REP.ALLOWED1 = \"248\"\n");
    fprintf(fp, "SPH.KEYWORDS.REP.ALLOWED2 = \"2048\"\n");
    fprintf(fp, "SPH.KEYWORDS.REP.ALLOWED3 = \"1024\"\n");
    fprintf(fp,
            "SPH.KEYWORDS.ESO_DRS_PC_PROD_TYPE.NAME= \"ESO DRS PC PROD TYPE\"\n");
    fprintf(fp, "SPH.KEYWORDS.ESO_DRS_PC_PROD_TYPE.TYPE = \"STRING\"\n");
    fprintf(fp,
            "SPH.KEYWORDS.ESO_DRS_PC_PROD_TYPE.VALUE = \"SPH PC RAW FRAME\"\n");
    fprintf(fp, "SPH.KEYWORDS.ESO_DRS_PC_BLA.NAME= \"ESO DRS PC BLA\"\n");
    fprintf(fp, "SPH.KEYWORDS.ESO_DRS_PC_BLA.TYPE = \"DOUBLE\"\n");
    fprintf(fp, "SPH.KEYWORDS.ESO_DRS_PC_BLA.VALUE = \"5.5\"\n");
    fclose(fp);

    frames = cpl_frameset_new();

    cpl_test_nonnull( frames );

    sph_test_create_ifs_master_dark_frame(frames, 1);
    prop = cpl_property_new("ESO DRS PC IMAGE NX", CPL_TYPE_INT);
    cpl_property_set_int(prop, 256);
    sph_fits_update_property("temp_master_dark.fits", prop, 0);
    cpl_property_delete(prop);
    prop = NULL;
    prop = cpl_property_new("ESO DRS PC IMAGE NY", CPL_TYPE_INT);
    cpl_property_set_int(prop, 256);
    sph_fits_update_property("temp_master_dark.fits", prop, 0);
    cpl_property_delete(prop);
    prop = NULL;
    km = sph_keyword_manager_new();
    sph_keyword_manager_read_config(SPH_CFG);
    cpl_frameset_delete(frames);
    cpl_test_zero(unlink(SPH_CFG));
    return km;
}

static
void cutest_util_keyword_manager_setup_head(void) {
    FILE* fp = NULL;

    fp = fopen("temp_master_dark.head", "w");
    //fprintf( fp, "SPH PC IMAGE NX = 248\n");
    //fprintf( fp, "SPH PC IMAGE NY = 148\n");
    fprintf(fp, "ESO DRS PC IMAGE NZ = 1212 \n");
    fprintf(fp, " \n");
    fclose(fp);
}
static
void cutest_util_keyword_manager_setup_head_double(void) {
    FILE* fp = NULL;

    fp = fopen("temp_master_dark.head", "w");
    fprintf(fp, "SPH PC IMAGE NX = 248\n");
    fprintf(fp, "SPH PC IMAGE NY = 148\n");
    fprintf(fp, "ESO DRS PC BLA = 22.2\n");
    fclose(fp);
}

static
int cutest_init_keyword_manager_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static
int cutest_clean_keyword_manager_testsuite(void) {
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_new(void) {
    sph_keyword_manager* keyman = NULL;
    const char* cfgname = sph_keyword_manager_set_cfg(SPH_CFG);

    cpl_test_eq_string(cfgname, "sphere.cfg");

    cfgname = sph_keyword_manager_get_cfg();
    cpl_test_eq_string(cfgname, SPH_CFG);

    /* Setup and run ...*/
    keyman = sph_keyword_manager_new();
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( keyman );

    /*Verify */
    sph_keyword_manager_delete();
    keyman = sph_keyword_manager_new();

    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( keyman );

    /*Verify */
    sph_keyword_manager_delete();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_read_config function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_read_config(void) {
    sph_keyword_manager* keyman = NULL;
    FILE* fp = NULL;
    sph_error_code rerr = 0;

    /* Setup and run ...*/
    fp = fopen(SPH_CFG, "w");
    fprintf(fp, "SPH.KEYWORDS.TEST.SOURCE.REP.NAME = \"BLUBBY\"\n");
    fprintf(fp, "SPH.KEYWORDS.TEST.SOURCE.REP.TYPE = \"INT\"\n");
    fprintf(fp, "SPH.KEYWORDS.TEST.SOURCE.REP.VALUE = \"-3.0\"\n");
    fprintf(fp, "SPH.KEYWORDS.TEST.SOURCE.REP.ALLOWED1 = \"-3.0\"\n");
    fprintf(fp, "SPH.KEYWORDS.TEST.SOURCE.REP.ALLOWED2 = \"4.0\"\n");
    fprintf(fp, "SPH.KEYWORDS.TEST.SOURCE.REP.ALLOWED3 = \"1.0\"\n");
    fprintf(fp, "SPH.KEYWORDS.REP.NAME = \"BLABBY\"\n");
    fprintf(fp, "SPH.KEYWORDS.REP.TYPE = \"INT\"\n");
    fprintf(fp, "SPH.KEYWORDS.REP.VALUE = \"-3.0\"\n");
    fprintf(fp, "SPH.KEYWORDS.REP.ALLOWED1 = \"-3.0\"\n");
    fprintf(fp, "SPH.KEYWORDS.REP.ALLOWED2 = \"4.0\"\n");
    fprintf(fp, "SPH.KEYWORDS.REP.ALLOWED3 = \"1.0\"\n");
    fclose(fp);

    keyman = sph_keyword_manager_new();

    rerr = sph_keyword_manager_read_config(SPH_CFG);

    /*Verify */
    cpl_test_nonnull( keyman );
    cpl_test_zero(rerr);
    cpl_test_nonnull( keyman->plistsoft );

    rerr = sph_keyword_manager_delete();

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_zero(unlink(SPH_CFG));

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_save function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_get_keyword(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    const char* czKey = NULL;
    const char* czKey2 = NULL;
    /* Setup */
    keyman = cutest_util_keyword_manager_common_setup();
    cpl_test_nonnull( keyman );

    /* Run ... */
    czKey = sph_keyword_manager_get_keyword("TEST.SOURCE.REP");
    czKey2 = sph_keyword_manager_get_keyword("REP");

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq_string( czKey, "BLUBBY");
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq_string( czKey2, "ESO DRS PC IMAGE NX");
    rerr = sph_keyword_manager_delete();

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_save function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_get_string(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    const char* czKey = NULL;

    /* Setup */
    keyman = cutest_util_keyword_manager_common_setup();
    cpl_test_nonnull( keyman );

    /* Run ... */
    czKey = sph_keyword_manager_get_string(SPH_COMMON_KEYWORD_SPH_TYPE);

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( czKey );
    cpl_test_eq_string( czKey, "SPH PC RAW FRAME");
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    rerr = sph_keyword_manager_delete();

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_get_string function.

 This test is specifically for a bug that cropped up during python
 module testing: the get_string did not return a value even though
 there was a default value defined in the config file.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_get_string_bug_A(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    const char* czKey = NULL;
    /* Setup */
    keyman = cutest_util_keyword_manager_common_setup();
    cpl_test_nonnull( keyman );

    /* Run ... */
    czKey = sph_keyword_manager_get_string(SPH_COMMON_KEYWORD_SPH_TYPE);

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( czKey );
    cpl_test_eq_string( czKey, "SPH PC RAW FRAME");
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    rerr = sph_keyword_manager_delete();

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_load_properties function.
 The keyword is defined in both the config file and the FITS file.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_load_properties(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int result = 0;
    cpl_propertylist* plist = NULL;
    /* Setup */
    keyman = cutest_util_keyword_manager_common_setup();
    cpl_test_nonnull( keyman );

    /* Run ... */
    plist = sph_keyword_manager_load_properties("temp_master_dark.fits", 0);
    cpl_test_nonnull( plist );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test( cpl_propertylist_has(plist,"REP"));
    result = cpl_propertylist_get_int(plist, "REP");

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(result, 256);

    rerr = sph_keyword_manager_delete();
    cpl_propertylist_delete(plist);
    plist = NULL;
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_trans_properties function.
 The keyword is defined in both the config file and the FITS file.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_trans_keywords(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int result = 0;
    cpl_propertylist* plist = NULL;
    cpl_propertylist* translist = NULL;
    /* Setup */
    keyman = cutest_util_keyword_manager_common_setup();
    cpl_test_nonnull( keyman );

    /* Run ... */
    plist = sph_keyword_manager_load_properties("temp_master_dark.fits", 0);
    cpl_test_nonnull( plist );
    cpl_test_error(CPL_ERROR_NONE);
    translist = sph_keyword_manager_trans_keywords(plist);
    result = cpl_propertylist_get_int(translist, "ESO DRS PC IMAGE NX");

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(result, 256);

    rerr = sph_keyword_manager_delete();
    cpl_propertylist_delete(plist);
    plist = NULL;
    cpl_propertylist_delete(translist);
    translist = NULL;
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_trans_properties function.
 The keyword is defined in both the config file and the FITS file.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_trans_keywords_no_cfg(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int result = 0;
    cpl_propertylist* plist = NULL;
    cpl_propertylist* translist = NULL;

    /* Setup */
    keyman = cutest_util_keyword_manager_common_setup();
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( keyman );

    (void)unlink(SPH_CFG);

    /* Run ... */
    plist = sph_keyword_manager_load_properties("temp_master_dark.fits", 0);
    cpl_test_nonnull( plist );
    cpl_test_error(CPL_ERROR_NONE);
    translist = sph_keyword_manager_trans_keywords(plist);
    result = cpl_propertylist_get_int(translist, "ESO DRS PC IMAGE NX");

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(result, 256);

    rerr = sph_keyword_manager_delete();
    cpl_propertylist_delete(plist);
    plist = NULL;
    cpl_propertylist_delete(translist);
    translist = NULL;

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_load_properties function.
 The keyword is defined only in the config file and not the FITS file.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_load_properties_cfg_only(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int result = 0;
    cpl_propertylist* plist = NULL;
    /* Setup */
    keyman = cutest_util_keyword_manager_common_setup();
    cpl_test_nonnull( keyman );

    /* Run ... */
    plist = sph_keyword_manager_load_properties("temp_master_dark.fits", 0);
    cpl_test_nonnull( plist );
    cpl_test_error(CPL_ERROR_NONE);
    result = cpl_propertylist_get_int(plist, "TEST.SOURCE.REP");

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(result, -3);

    rerr = sph_keyword_manager_delete();
    cpl_propertylist_delete(plist);
    plist = NULL;

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_load_properties function.
 The keyword is defined only in the config file and not the FITS file.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_load_properties_head_only(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_propertylist* plist = NULL;
    /* Setup */
    keyman = cutest_util_keyword_manager_common_setup();
    cpl_test_nonnull( keyman );

    cutest_util_keyword_manager_setup_head();

    /* Run ... */
    plist = sph_keyword_manager_load_properties("temp_master_dark.fits", 0);
    cpl_test_nonnull( plist );
//    cpl_test_error(CPL_ERROR_NONE );
//    cpl_test_eq(cpl_propertylist_has( plist, "ESO DRS PC IMAGE NZ" ), 1 );
//    result = cpl_propertylist_get_int( plist, "ESO DRS PC IMAGE NZ" );
//
//    /* Verify */
//    cpl_test_eq_error(rerr, CPL_ERROR_NONE );
//    cpl_test_error(CPL_ERROR_NONE );
//    cpl_test_eq(result, 1212 );

    rerr = sph_keyword_manager_delete();
    cpl_propertylist_delete(plist);
    plist = NULL;

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_load_properties function.
 The keyword is defined only in the config file and not the FITS file.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_load_properties_head_double(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    double result = 0;
    cpl_propertylist* plist = NULL;
    /* Setup */
    keyman = cutest_util_keyword_manager_common_setup();
    cpl_test_nonnull( keyman );

    cutest_util_keyword_manager_setup_head_double();

    /* Run ... */
    plist = sph_keyword_manager_load_properties("temp_master_dark.fits", 0);
    cpl_test_nonnull( plist );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(cpl_propertylist_has( plist, "ESO DRS PC BLA" ), 1);
    cpl_test_eq(
            cpl_property_get_type(cpl_propertylist_get_property_const( plist, "ESO DRS PC BLA" )),
            CPL_TYPE_DOUBLE);
    result = cpl_propertylist_get_double(plist, "ESO DRS PC BLA");

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs( result, 22.2, 0.01);
    rerr = sph_keyword_manager_delete();
    cpl_propertylist_delete(plist);
    plist = NULL;

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}

static
void cutest_sph_keyword_manager_load_properties_head_double_no_cfg(void) {
    sph_error_code rerr = CPL_ERROR_NONE;
    double result = 0;
    cpl_propertylist* plist = NULL;

    unlink(SPH_CFG);

    cutest_util_keyword_manager_setup_head_double();

    /* Run ... */
    plist = sph_keyword_manager_load_properties("temp_master_dark.fits", 0);
    cpl_test_nonnull( plist );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(cpl_propertylist_has( plist, "ESO DRS PC BLA" ), 1);
    cpl_test_eq(
            cpl_property_get_type(cpl_propertylist_get_property_const( plist, "ESO DRS PC BLA" )),
            CPL_TYPE_DOUBLE);
    cpl_test_eq(
            cpl_property_get_type(cpl_propertylist_get_property_const( plist, "SPH PC IMAGE NX" )),
            CPL_TYPE_INT);
    result = cpl_propertylist_get_double(plist, "ESO DRS PC BLA");

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs( result, 22.2, 0.01);
    cpl_test_eq(cpl_propertylist_get_int( plist, "SPH PC IMAGE NX" ), 248);
    rerr = sph_keyword_manager_delete();
    cpl_propertylist_delete(plist);
    plist = NULL;

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_load_properties function.
 The keyword is defined only in the config file and not the FITS file.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_load_properties_fits_only(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int result = 0;
    cpl_propertylist* plist = NULL;
    /* Setup */
    keyman = cutest_util_keyword_manager_common_setup();
    cpl_test_nonnull( keyman );

    /* Run ... */
    plist = sph_keyword_manager_load_properties("temp_master_dark.fits", 0);
    cpl_test_nonnull( plist );
    cpl_test_error(CPL_ERROR_NONE);
    result = cpl_propertylist_get_int(plist, "ESO DRS PC IMAGE NY");

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(result, 256);

    rerr = sph_keyword_manager_delete();
    cpl_propertylist_delete(plist);
    plist = NULL;

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_load_keywords function.
 The keyword is defined in both the config file and the FITS file.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_load_keywords(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int result = 0;
    /* Setup */
    keyman = cutest_util_keyword_manager_common_setup();
    cpl_test_nonnull( keyman );

    /* Run ... */
    rerr = sph_keyword_manager_load_keywords("temp_master_dark.fits", 0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    result = sph_keyword_manager_get_int("REP");

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(result, 256);

    rerr = sph_keyword_manager_delete();

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_load_keywords function.
 The keyword is defined only in the config file and not the FITS file.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_load_keywords_cfg_only(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int result = 0;
    /* Setup */
    keyman = cutest_util_keyword_manager_common_setup();
    cpl_test_nonnull( keyman );

    /* Run ... */
    rerr = sph_keyword_manager_load_keywords("temp_master_dark.fits", 0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    result = sph_keyword_manager_get_int("TEST.SOURCE.REP");

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(result, -3);

    rerr = sph_keyword_manager_delete();

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_load_keywords function.
 The keyword is defined only in the FITS file and not the config file.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_load_keywords_fits_only(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int result = 0;
    /* Setup */
    keyman = cutest_util_keyword_manager_common_setup();
    cpl_test_nonnull( keyman );

    /* Run ... */
    rerr = sph_keyword_manager_load_keywords("temp_master_dark.fits", 0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    result = sph_keyword_manager_get_int("ESO DRS PC IMAGE NY");

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(result, 256);

    rerr = sph_keyword_manager_delete();

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_load_keywords function.
 The keyword is defined only in the FITS file and not the config file.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_load_keywords_no_cfg_file(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int result = 0;
    /* Setup */
    keyman = sph_keyword_manager_new();

    cpl_test_nonnull( keyman );

    /* Run ... */
    rerr = sph_keyword_manager_load_keywords("temp_master_dark.fits", 0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    result = sph_keyword_manager_get_int("ESO DRS PC IMAGE NX");

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(result, 256);

    rerr = sph_keyword_manager_delete();

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keyword_manager_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_keyword_manager_delete(void) {
    sph_keyword_manager* keyman = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    keyman = sph_keyword_manager_new();
    cpl_test_nonnull( keyman );

    rerr = sph_keyword_manager_delete();

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    (void)sph_test_init();

    pSuite = sph_add_suite("keyword_manager_test",
            cutest_init_keyword_manager_testsuite,
            cutest_clean_keyword_manager_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_keyword_manager_new",
                    cutest_sph_keyword_manager_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_keyword_manager_read_config",
                    cutest_sph_keyword_manager_read_config)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_keyword_manager_get_keyword",
                    cutest_sph_keyword_manager_get_keyword)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_keyword_manager_get_string",
                    cutest_sph_keyword_manager_get_string)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_keyword_manager_get_string_bug_A",
                    cutest_sph_keyword_manager_get_string_bug_A)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_keyword_manager_load_keywords",
                    cutest_sph_keyword_manager_load_keywords)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_keyword_manager_load_keywords_cfg_only",
                    cutest_sph_keyword_manager_load_keywords_cfg_only)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_keyword_manager_load_keywords_fits_only",
                    cutest_sph_keyword_manager_load_keywords_fits_only)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_keyword_manager_load_keywords_no_cfg_file",
                    cutest_sph_keyword_manager_load_keywords_no_cfg_file)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_keyword_manager_load_properties",
                    cutest_sph_keyword_manager_load_properties)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_keyword_manager_load_properties_fits_only",
                    cutest_sph_keyword_manager_load_properties_fits_only)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_keyword_manager_load_properties_cfg_only",
                    cutest_sph_keyword_manager_load_properties_cfg_only)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_keyword_manager_load_properties_head_only",
                    cutest_sph_keyword_manager_load_properties_head_only)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_keyword_manager_load_properties_head_double",
                    cutest_sph_keyword_manager_load_properties_head_double)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_keyword_manager_load_properties_head_double_no_cfg",
                    cutest_sph_keyword_manager_load_properties_head_double_no_cfg)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_keyword_manager_trans_keywords",
                    cutest_sph_keyword_manager_trans_keywords)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_keyword_manager_trans_keywords_no_cfg",
                    cutest_sph_keyword_manager_trans_keywords_no_cfg)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_keyword_manager_delete",
                    cutest_sph_keyword_manager_delete)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}
