/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * cutest_sph_keyword_engine.c
 *
 *  Created on: Apr 2, 2009
 *      Author: pavlov
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_keywordnode.h"

#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include <cpl.h>

#include <math.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/
#define SPH_BASE "cutest_sph_keywordnode"
#define SPH_CFG  SPH_BASE ".cfg"

/*
 * Create the tree with the following entries:
 * @verbatim
 PIPELINE.VERSION = "0.9" \# current version of the PIPELINE
 PIPELINE.ZIMPOL.CAM1.CALNAME = "DARK"
 PIPELINE.ZIMPOL.CAM1.DIT = "30.5"
 PIPELINE.ZIMPOL.CAM1.NDIT = "20"
 PIPELINE.ZIMPOL.CAM1.CALNAME = "DARK"
 PIPELINE.ZIMPOL.CAM1.DIT = "40.5"
 PIPELINE.ZIMPOL.CAM1.NDIT = "50"
 @endverbatim
 *
 * the following tree is created:
 *
 @verbatim
 ROOT
 +-- PIPELINE
 +-- VERSION = 0.9 (double)
 +-- CAM
 +-- 1
 |   +-- CALNAME = DARK (string, i.e. char*)
 |   +-- DIT = 30.5 (double)
 |   +-- NDIT = 20 (int)
 +-- 2
 +-- CALNAME = DARK (string, i.e. char*)
 |   +-- DIT = 15.25 (double)
 |   +-- NDIT = 40 (int)

 @endverbatim
 *
 */

/* static variables */
//keywords
const double version = 0.9;
const char test = 'T'; // or 'F'
const char* cam1_calname = "DARK";
const double cam1_dit = 30.5;
const int cam1_ndit = 20;
const char* cam2_calname = "DARK";
const double cam2_dit = 15.25;
const int cam2_ndit = 40;

const double granularity = 1e-15;

const char cfgFile[] = "sph_fits_keywords_test.cfg";
const char cfgFileDel[] = "sph_fits_keywords_test_del.cfg";
const char cfgFileNum[] = "sph_fits_keywords_test_num.cfg";

static
sph_keywordnode *cutest_util_keywordnode_create_cfg(void) {
    sph_keywordnode *root = NULL;

    root = sph_keywordnode_new(NULL);
    cpl_test_nonnull( root );
    cpl_test_eq(
            sph_keywordnode_SetDouble(root, "PIPELINE.VERSION", version), 1);
    cpl_test_eq(sph_keywordnode_SetChar(root, "PIPELINE.TEST", test), 1);
    cpl_test_eq(
            sph_keywordnode_SetString(root, "PIPELINE.CAM1.CALNAME", cam1_calname),
            1);
    cpl_test_eq(
            sph_keywordnode_SetDouble(root, "PIPELINE.CAM1.DIT", cam1_dit), 1);
    cpl_test_eq(
            sph_keywordnode_SetInt(root, "PIPELINE.CAM1.NDIT", cam1_ndit), 1);
    cpl_test_eq(
            sph_keywordnode_SetString(root, "PIPELINE.CAM2.CALNAME", cam2_calname),
            1);
    cpl_test_eq(
            sph_keywordnode_SetDouble(root, "PIPELINE.CAM2.DIT", cam2_dit), 1);
    cpl_test_eq(
            sph_keywordnode_SetInt(root, "PIPELINE.CAM2.NDIT", cam2_ndit), 1);
    cpl_test_eq(sph_keywordnode_SetInt(root, "CAMERA1.DIT1", cam1_dit), 1);
    cpl_test_eq(
            sph_keywordnode_SetInt(root, "PIPELINE.CAMERA1.CAMERA2.DIT2.CAMERA2", cam2_dit),
            1);

    cpl_test_eq(sph_keywordnode_SaveConfiguration(root, cfgFile), 1);
    // save keywords configuration on the disk
    return root;
}

static
sph_keywordnode *cutest_util_keywordnode_create_cfg_start_by_number(void) {
    sph_keywordnode *root = NULL;

    root = sph_keywordnode_new(NULL);
    cpl_test_nonnull( root );
    cpl_test_eq(
            sph_keywordnode_SetDouble(root, "PIPELINE.VERSION", version), 1);
    cpl_test_eq(sph_keywordnode_SetChar(root, "PIPELINE.TEST", test), 1);
    cpl_test_eq(
            sph_keywordnode_SetString(root, "1PIPELINE.CAM1.CALNAME", cam1_calname),
            1);
    cpl_test_eq(
            sph_keywordnode_SetDouble(root, "1PIPELINE.CAM1.DIT", cam1_dit), 1);
    cpl_test_eq(
            sph_keywordnode_SetInt(root, "1PIPELINE.CAM1.NDIT", cam1_ndit), 1);
    cpl_test_eq(
            sph_keywordnode_SetString(root, "2PIPELINE.CAM2.CALNAME", cam2_calname),
            1);
    cpl_test_eq(
            sph_keywordnode_SetDouble(root, "2PIPELINE.CAM2.DIT", cam2_dit), 1);
    cpl_test_eq(
            sph_keywordnode_SetInt(root, "2PIPELINE.CAM2.NDIT", cam2_ndit), 1);

    cpl_test_eq(sph_keywordnode_SaveConfiguration(root, cfgFileNum), 1);
    // save keywords configuration on the disk
    return root;
}

static
int cutest_init_keywordnode_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    //sph_init();
    sph_test_nop_code();
    return 0;
}

static
int cutest_clean_keywordnode_testsuite(void) {
    sph_end_test();
    return 0;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_keywordnode_ function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_keywordnode_LoadConfiguration(void) {
    sph_keywordnode *root = NULL;
    root = cutest_util_keywordnode_create_cfg();
    cpl_test_nonnull( root );
    sph_keywordnode_destroy(root);
    root = NULL;
    root = sph_keywordnode_load_configuration(cfgFile);
    cpl_test_nonnull( root );
    sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
            SPH_ERROR_INFO, " - cutest_keywordnode_LoadConfiguration -");
    //printf(" - cutest_keywordnode_LoadConfiguration:");
    sph_keywordnode_destroy(root);
}

/*
 * this function is just to test whether keyword with is saved properly
 */
static
void cutest_keywordnode_keywordnode_save_number(void) {
    sph_keywordnode *root = NULL;
    root = cutest_util_keywordnode_create_cfg_start_by_number();
    cpl_test_nonnull( root );
    sph_keywordnode_destroy(root);
}

static
void cutest_keywordnode_keywordnode_destroy(void) {
    sph_keywordnode *root = cutest_util_keywordnode_create_cfg();

    sph_keywordnode *node;
    double value;

    sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
            SPH_ERROR_INFO, " - cutest_keywordnode_keywordnode_destroy -");
    //printf(" - cutest_keywordnode_keywordnode_destroy: \n");

    cpl_test_nonnull( root );

    if ((node = sph_keywordnode_get_keywordnode_by_name(root,
            "PIPELINE.VERSION")) != NULL) {
        //printf(" -- Before node-destroy of PIPELINE.VERSION keywordnode:\n");

        cpl_test_eq(
                sph_keywordnode_GetDouble(root,"PIPELINE.VERSION", &value), 1);
        cpl_test_abs(value, version, granularity);
        //printf(" -- PIPELINE.VERSION = %f \n", value);
        sph_keywordnode_destroy(node);
    } else {
        //printf("-- PIPELINE.VERSION keyword has not been created, please verify \n.");
        sph_error_raise(
                SPH_ERROR_GENERAL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                " -- PIPELINE.VERSION keyword has not been created, please verify --");
    }cpl_test_eq(sph_keywordnode_GetDouble(root,"PIPELINE.VERSION", &value),
            0);
    cpl_test_eq(sph_keywordnode_SaveConfiguration(root, cfgFileDel), 1);
    // look at this file (PIPELINE.VERSION must be erased!)
    sph_keywordnode_destroy(root);
}

static
void cutest_keywordnode_GetInt(void) {
    int value;
    sph_keywordnode *root = NULL;
    root = cutest_util_keywordnode_create_cfg();
    cpl_test_nonnull( root );
    cpl_test_eq(sph_keywordnode_GetInt(root,"PIPELINE.CAM1.NDIT", &value),
            1);
    cpl_test_eq(value, cam1_ndit);
    //printf(" - cutest_keywordnode_GetInt:");
    //printf(" -- PIPELINE.CAM1.NDIT = %i \n", value);
    sph_keywordnode_destroy(root);
}

static
void cutest_keywordnode_GetDouble(void) {
    double value;
    sph_keywordnode *root = NULL;
    root = cutest_util_keywordnode_create_cfg();
    cpl_test_nonnull( root );
    cpl_test_eq(sph_keywordnode_GetDouble(root,"PIPELINE.VERSION", &value),
            1);
    cpl_test_abs(value, version, granularity);
    //printf(" - cutest_keywordnode_GetDouble:");
    //printf(" -- PIPELINE.VERSION = %f \n", value);
    sph_keywordnode_destroy(root);
}

static
void cutest_keywordnode_GetString(void) {
    char string[100];
    char qualname[100], qualname_list[100];
    sph_keywordnode *root = NULL;
    sph_keywordnode *node;
    cpl_propertylist* proplist;
    int k;

    root = cutest_util_keywordnode_create_cfg();
    cpl_test_nonnull( root );
    cpl_test_eq(
            sph_keywordnode_GetString(root,"PIPELINE.CAM1.CALNAME", string, sizeof(string)),
            1);
    cpl_test_eq_string(string, cam1_calname);
    //printf(" -- PIPELINE.CAM1.CALNAME = %s \n", string);
    //printf(" -- Set Double instead of String \n");
    sph_keywordnode_SetInt(root, "PIPELINE.CAM1.CALNAME", 10);
    sph_keywordnode_GetInt(root, "PIPELINE.CAM1.CALNAME", &k);
    //printf(" -- Get New Integer : PIPELINE.CAM1.CALNAME = %d", k);

    //node = GetKeywordNodeByName(root, "PIPELINE.CAMERA1.CAMERA2.DIT2.CAMERA2");
    node = sph_keywordnode_get_keywordnode_by_name(root,
            "PIPELINE.CAM1.CALNAME");
    sph_keywordnode_BuildQualifiedName(node, qualname, 100);
    //printf("-- ### BuildQualifiedName: %s ###\n\n", qualname);
    proplist = cpl_propertylist_new();

    //node = GetKeywordNodeByName(root, "PIPELINE.CAM1.CALNAME");
    //BuildQualifiedName(node, qualname_list, 100);
    sph_keywordnode_BuildQualifiedNameList(node, qualname_list, 100, proplist);
    //printf("-- ### BuildQualifiedNameList: %s ###\n", qualname_list);
    //printf("-- ### starting loop for the list...\n");
    //for (i=0; i < cpl_propertylist_get_size(proplist);i++){
    //cpl_property* p = cpl_propertylist_get(proplist,i);
    //printf("Prop %d: %s = %d \n", i, cpl_property_get_name(p), cpl_property_get_int(p));
    //}
    sph_keywordnode_destroy(root);
    cpl_propertylist_delete(proplist);
    proplist = NULL;

}

static
void cutest_keywordnode_GetChar(void) {
    char s_char;
    sph_keywordnode *root = NULL;
    root = cutest_util_keywordnode_create_cfg();
    cpl_test_nonnull( root );
    cpl_test_eq(sph_keywordnode_GetChar(root,"PIPELINE.TEST", &s_char), 1);
    cpl_test_eq(s_char, test);
    //printf(" - cutest_keywordnode_GetChar:");
    //printf(" -- PIPELINE.TEST = %c \n", s_char);
    sph_keywordnode_destroy(root);
}

static
void cutest_keywordnode_GetType(void) {
    sph_keywordnode *root = NULL;
    keyword_TYPE type, type1;
    root = cutest_util_keywordnode_create_cfg();
    cpl_test_nonnull( root );
    type = sph_keywordnode_GetType(
            sph_keywordnode_LocateByName(root, "PIPELINE.VERSION", 0));
    cpl_test_eq(type, keyword_DOUBLE);
    type1 = sph_keywordnode_GetType(
            sph_keywordnode_get_keywordnode_by_name(root,
                    "PIPELINE.CAM1.CALNAME"));
    cpl_test_eq(type1, keyword_STRING);
    //printf(" - cutest_keywordnode_GetType:");
    sph_keywordnode_destroy(root);
}

static
void cutest_keywordnode_QueryVerify(void) {
    sph_keywordnode *root = cutest_util_keywordnode_create_cfg();
    sph_keywordnode *node;

    // Query template with verifications starts here
    //printf(" - cutest_keywordnode_QueryVerify: \n");
    sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
            SPH_ERROR_INFO, " -  cutest_keywordnode_QueryVerify -");

    cpl_test_nonnull( root );

    if ((node = sph_keywordnode_get_keywordnode_by_name(root,
            "PIPELINE.VERSION")) != NULL) {
        keyword_TYPE type;
        double value;

        if ((type = sph_keywordnode_GetType(node)) == keyword_DOUBLE) {
            if (sph_keywordnode_GetDouble(root, "PIPELINE.VERSION", &value)) {
                // printf(" -- PIPELINE.VERSION = %f \n", value);
                cpl_test_abs(value, version, granularity);
            }
        } else {
            //printf(" -- PIPELINE.VERSION is not a type of keyword_DOUBLE \n");
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    " -- PIPELINE.VERSION is not a type of keyword_DOUBLE --");
        }
        cpl_test_eq(type, keyword_DOUBLE);
    } else {
        //printf(" -- PIPELINE.VERSION keyword doesn't exist! \n");
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                " -- PIPELINE.VERSION  keyword doesn't exist! --");

    }
    //Query template with verifications ends here

    // CU unit verifications
    cpl_test_nonnull( node );
    sph_keywordnode_destroy(root);
}

static
void cutest_keywordnode_ChildQueries(void) {
    sph_keywordnode *root = cutest_util_keywordnode_create_cfg();

    //Query template for the multiple childs starts here
    sph_keywordnode *node;
    //printf(" - cutest_keywordnode_ChildQueries: \n");
    sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
            SPH_ERROR_INFO, " - cutest_keywordnode_ChildQueries -");

    cpl_test_nonnull( root );

    if ((node = sph_keywordnode_get_keywordnode_by_name(root, "PIPELINE.CAM"))
            != NULL) {
        int i = 0; // this line is only for cunit purpose, don't use in the template
        for (node = sph_keywordnode_GetFirstChild(node); node != NULL; node =
                sph_keywordnode_GetNext(node)) {
            double dit;
            i++; // this line is only for the cunit purpose,  don't use in the template
            if (sph_keywordnode_GetDouble(node, "DIT", &dit)) {
                //printf("DIT for camera %d = %f\n", GetNr(node), dit);
                cpl_test_eq(i, sph_keywordnode_GetNr(node));
                // this line is only for the cunit purpose, don't use in the template
                if (sph_keywordnode_GetNr(node) == 1) {
                    cpl_test_eq(dit, cam1_dit);
                }
                if (sph_keywordnode_GetNr(node) == 2) {
                    cpl_test_eq(dit, cam2_dit);
                }
            } else {
                //printf("DIT for camera %d unknown\n", sph_keywordnode_GetNr(node));
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, " -- DIT for camera %d unknown: ",
                        sph_keywordnode_GetNr(node));
            }

        }
    }
    //Query template for the multiple childs ends here
    sph_keywordnode_destroy(root);
}

static void sph_keywordnode_test_one(void)
{
    double vers;
    sph_keywordnode *node=NULL;
    sph_keywordnode *root = sph_keywordnode_new(node);
    sph_keywordnode *root1;
    sph_keywordnode *root1node;
    const double test2 = 2.0;
    double vers1;
    const int str_size=100;
    char sVal[str_size];
    int code;

    cpl_test_nonnull(root);

    code = sph_keywordnode_SetDouble(root, "TEST", test2);
    cpl_test_eq(code, 1);
    //root->dump();

    code = sph_keywordnode_GetDouble (root, "TEST", &vers);
    cpl_test_eq(code, 1);
    cpl_test_abs(vers, test2, 0.0);

    code = sph_keywordnode_GetDouble (root, "TEST", &vers);
    cpl_test_eq(code, 1);
    cpl_test_abs(vers, test2, 0.0);

    sph_keywordnode_destroy(root);
    root1 = sph_keywordnode_load_configuration(cfgFile);
    if (root1 != NULL) {

        code = sph_keywordnode_GetString(root1, "LSYS.ERROR1", sVal, str_size);
        cpl_test_zero(code);

        sph_keywordnode_destroy(root1);
    }
    root1 = sph_keywordnode_load_configuration(cfgFile);
    if (root1 != NULL) {

        code = sph_keywordnode_GetDouble (root1, "SPH.VERSION", &vers1);
        cpl_test_zero(code);
        cpl_test_abs(vers, test2, 0.0);

        code = sph_keywordnode_GetString(root1, "SPH.KEYWORD.RECIPE.BIAS.DIT.NAME",
                                         sVal, str_size);
        cpl_test_zero(code);

        root1node = sph_keywordnode_LocateByName(root1, "SPH.KEYWORD.RECIPE.BIAS.DIT", 0);
        cpl_test_null(root1node);

        if (root1node != NULL) {
            keyword_TYPE type
                = sph_keywordnode_GetType(sph_keywordnode_LocateByName(root1node,
                                                                       "DEFAULTVALUE",
                                                                       0));
            if (type == keyword_DOUBLE) {
                double ditvalue;
                if (sph_keywordnode_GetDouble(root1node, "DEFAULTVALUE", &ditvalue)) {
                    printf("DEVAULT value for BIAS = %f\n", ditvalue);
                } else {
                    printf("DEFAULTVALUE for BIAS unknown\n");
                }
            } else {
                printf("DEFAULTVALUE is not a DOUBLE!\n");
            }
        }

        code = sph_keywordnode_SaveConfiguration(root1, SPH_CFG);
        cpl_test_eq(code, 1);
        cpl_test_zero(unlink(SPH_CFG));

        sph_keywordnode_destroy(root1);
    }
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of sph_keywordnode and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();

    pSuite = sph_add_suite("keywordnode_test", cutest_init_keywordnode_testsuite,
            cutest_clean_keywordnode_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL
            == sph_test_do(pSuite, "cutest_keywordnode_LoadConfiguration",
                    cutest_keywordnode_LoadConfiguration)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_keywordnode_GetDouble",
                    cutest_keywordnode_GetDouble)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_keywordnode_GetInt",
                    cutest_keywordnode_GetInt)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_keywordnode_GetString",
                    cutest_keywordnode_GetString)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_keywordnode_GetChar",
                    cutest_keywordnode_GetChar)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_keywordnode_GetType",
                    cutest_keywordnode_GetType)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_keywordnode_QueryVerify",
                    cutest_keywordnode_QueryVerify)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_keywordnode_ChildQueries",
                    cutest_keywordnode_ChildQueries)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_keywordnode_keywordnode_destroy",
                    cutest_keywordnode_keywordnode_destroy)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_keywordnode_keywordnode_save_number",
                    cutest_keywordnode_keywordnode_save_number)) {
        return sph_test_get_error();
    }

    sph_keywordnode_test_one();

    cpl_test_zero(unlink(cfgFile));
    cpl_test_zero(unlink(cfgFileDel));
    cpl_test_zero(unlink(cfgFileNum));

    return sph_test_end();
}

