/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_ldt.h"
#include "sph_dataset.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"
#include "sph_test.h"


#include "sph_test.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/
static sph_ldt* __ldt = NULL;
/**@{*/

static sph_ldt*
create_ldt_standard(void) {
    sph_ifs_lenslet_model* model = NULL;
    cpl_image* inimage = NULL;
    sph_pixel_description_table* pdt = NULL;
    sph_ldt* ldt = NULL;
    int nx = 256;
    int ny = 256;
    int ll = 0;
    sph_lenslet_descriptor* ldescr = NULL;
    cpl_vector* vec = NULL;

    model = sph_test_create_small_lenslet_model();

    inimage = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
    pdt = sph_pixel_description_table_new_from_model(model, 0.0, 0.0);
    ldt = sph_ldt_new_from_pdt_image(pdt, model, inimage);
    sph_pixel_description_table_delete(pdt);
    cpl_image_delete(inimage);
    vec = cpl_vector_duplicate(ldt->reflambdas);
    for (ll = 0; ll < ldt->nlens; ++ll) {
        ldescr = ldt->arr[ll];
        if (ldescr) {
            cpl_vector_fill(vec, 2.0);
            sph_lenslet_descriptor_fill_values(ldescr, vec);
            cpl_vector_fill(vec, 0.0);
            sph_lenslet_descriptor_fill_bpix(ldescr, vec);
            cpl_vector_fill(vec, 3.0);
            sph_lenslet_descriptor_fill_rms(ldescr, vec);
            cpl_vector_fill(vec, 4.0);
            sph_lenslet_descriptor_fill_ncomb(ldescr, vec);
        }
    }

    cpl_vector_delete(vec);
    vec = NULL;
    return ldt;
}

static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    __ldt = create_ldt_standard();
    if (!__ldt)
        return 1;
    if (cpl_error_get_code())
        return 1;
    return 0;
}
static
int cutest_clean_testsuite(void) {
    if (__ldt)
        sph_ldt_delete(__ldt);
    __ldt = NULL;
    return sph_end_test();

}

static
void cutest_standard_happy_case(void) {
    sph_master_frame* mframe = NULL;
    int nx = 256;
    int ny = 256;
    cpl_propertylist* plist = NULL;

    cpl_test_nonnull( __ldt );
    mframe = sph_ldt_create_mframe(__ldt, nx, ny, 1, 0.0, NULL);
    cpl_test_eq(
            cpl_image_save( mframe->image, "test_ldt_ip_image.fits", CPL_TYPE_DOUBLE, plist = sph_ifs_lenslet_model_get_as_propertylist( __ldt->model ), CPL_IO_CREATE ),
            CPL_ERROR_NONE);
    //cpl_test_abs(cpl_image_get_flux(mframe->image),__ldt->nlens * 2.0, 1.0);
    sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
            SPH_ERROR_INFO, "flux:%f vs. %f\n",
            cpl_image_get_flux(mframe->image), __ldt->nlens * 2.0);
    sph_master_frame_delete(mframe);
    cpl_propertylist_delete(plist);
    plist = NULL;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("ldt_create_image", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "standard happy case",
                    cutest_standard_happy_case)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
