/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_ldt.h"
#include "sph_dataset.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"
#include "cutest_sph_ldt_fixtures.h"

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>

/*----------------------------------------------------------------------------*/
/*-                            Declarations of private functions              */
/*----------------------------------------------------------------------------*/

sph_ldt*
cutest_sph_ldt_fixtures_setup_core_ldt(void) {
    sph_ifs_lenslet_model* model = sph_test_create_small_lenslet_model();
    sph_ldt* tab = sph_ldt_new(model);

    cpl_test_nonnull( model );
    cpl_test_nonnull( tab );

    for (int lens = 0; lens < tab->nlens; ++lens) {
        const int nlambdas = model->speclength_pixels;
        sph_polygon* poly =
            sph_ifs_lenslet_model_get_poly(model, tab->arr[lens]->u,
                                           tab->arr[lens]->v);
        cpl_test_nonnull( poly );
        sph_polygon_delete(poly);
        poly = NULL;
        for (int zz = 0; zz < nlambdas; ++zz) {
            if ((abs(tab->arr[lens]->u) <= 1 && tab->arr[lens]->v == 0)
                    || (tab->arr[lens]->u == 1 && tab->arr[lens]->v == -1)
                    || (tab->arr[lens]->u == -1 && tab->arr[lens]->v == 1)
                    || (tab->arr[lens]->u == 0 && abs(tab->arr[lens]->v) <= 1)) {
                tab->arr[lens]->values[zz] = 10.0;
                tab->arr[lens]->bpix[zz] = 0;
            } else {
                tab->arr[lens]->values[zz] = 0.0;
                tab->arr[lens]->bpix[zz] = 1;
            }
            if (tab->arr[lens]->u == 1 && tab->arr[lens]->v == -1) {
                tab->arr[lens]->values[zz] = 20.0;
                tab->arr[lens]->bpix[zz] = 0.0;
            }
        }
    }
    sph_ldt_save(tab, "test_ldt_core.fits", CPL_IO_CREATE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_zero(unlink("test_ldt_core.fits"));
    return tab;
}
sph_ldt*
cutest_sph_ldt_fixtures_setup_big_ldt(void) {
    sph_ldt* tab = NULL;
    sph_ifs_lenslet_model* model = NULL;
    model = sph_ifs_lenslet_model_new();
    cpl_test_nonnull( model );

    tab = sph_ldt_new(model);
    cpl_test_nonnull( tab );
    return tab;
}
sph_ldt*
cutest_sph_ldt_fixtures_setup_big_ldt_ids(void) {
    sph_ldt* tab = NULL;
    sph_ifs_lenslet_model* model = NULL;
    sph_polygon* poly = NULL;
    int zz = 0;
    int lens = 0;
    int nlambdas = 10;
    double radius0 = 0.0;

    model = sph_ifs_lenslet_model_new();
    cpl_test_nonnull( model );

    tab = sph_ldt_new(model);

    for (lens = 0; lens < tab->nlens; ++lens) {
        double x = 0.0;
        double y = 0.0;
        double xx = 0.0;
        double yy = 0.0;
        poly = sph_ifs_lenslet_model_get_poly(model, tab->arr[lens]->u,
                tab->arr[lens]->v);
        cpl_test_nonnull( poly );
        x = sph_polygon_get_midx(poly);
        y = sph_polygon_get_midy(poly);
        sph_polygon_get_midxy_(poly, &xx, &yy);
        cpl_test_rel(x, xx, 2.0 * DBL_EPSILON);
        cpl_test_rel(y, yy, 2.0 * DBL_EPSILON);
        sph_polygon_delete(poly);
        poly = NULL;
        for (zz = 0; zz < nlambdas; ++zz) {
            const double z =
                (zz - 0.5 * nlambdas) * model->pixsize_microns * 10.0;
            const double radius = sqrt(x * x + y * y + z * z);
            if (radius < radius0) {
                tab->arr[lens]->values[zz] = lens + 1.0;
                tab->arr[lens]->bpix[zz] = 0.0;
            } else {
                tab->arr[lens]->values[zz] = lens + 1.0;
                tab->arr[lens]->bpix[zz] = 1.0;
            }
        }
    }

    cpl_test_nonnull( tab );
    return tab;
}

sph_ldt*
cutest_sph_ldt_fixtures_setup_ball_ldt(void) {
    sph_ldt* tab = NULL;
    sph_ifs_lenslet_model* model = NULL;
    sph_polygon* poly = NULL;
    int zz = 0;
    int lens = 0;
    int nlambdas = 10;
    double radius0 = 0.0;

    model = sph_test_create_small_lenslet_model();
    cpl_test_nonnull( model );
    radius0 = model->detsize_microns / 4.0;

    tab = sph_ldt_new(model);
    cpl_test_nonnull( tab );

    nlambdas = model->speclength_pixels - 1;

    for (lens = 0; lens < tab->nlens; ++lens) {
        double x = 0.0;
        double y = 0.0;
        double xx = 0.0;
        double yy = 0.0;
        poly = sph_ifs_lenslet_model_get_poly(model, tab->arr[lens]->u,
                tab->arr[lens]->v);
        cpl_test_nonnull( poly );
        x = sph_polygon_get_midx(poly);
        y = sph_polygon_get_midy(poly);
        sph_polygon_get_midxy_(poly, &xx, &yy);
        cpl_test_rel(x, xx, 2.0 * DBL_EPSILON);
        cpl_test_rel(y, yy, 2.0 * DBL_EPSILON);
        sph_polygon_delete(poly);
        poly = NULL;
        for (zz = 0; zz < nlambdas; ++zz) {
            const double z =
                (zz - 0.5 * nlambdas) * model->pixsize_microns * 10.0;
            const double radius = sqrt(x * x + y * y + z * z);
            if (radius < radius0) {
                tab->arr[lens]->values[zz] = 10.0;
                tab->arr[lens]->bpix[zz] = 0.0;
            } else {
                tab->arr[lens]->values[zz] = 0.0;
                tab->arr[lens]->bpix[zz] = 1.0;
            }
        }
    }
    sph_ldt_save(tab, "test_ldt.fits", CPL_IO_CREATE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_zero(unlink("test_ldt.fits"));
    return tab;
}
