/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_loci.h"
#include "sph_time.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_filemanager_clean();
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_loci_new(void) {
    sph_loci* loci = NULL;

    loci = sph_loci_new_empty();
    cpl_test_nonnull( loci );
    sph_loci_delete(loci);
    loci = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_loci_init_subsections(void) {
    sph_loci* loci = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    loci = sph_loci_new(300, 4.0, 0.5, 512.0, 512.0, 1.0, 100.0, 200.0);
    cpl_test_nonnull( loci );
    rerr = sph_loci_init_subsections(loci, 5.0, SPH_LOCI_NORMAL);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(loci->nsubsections, 1021);
    cpl_test_nonnull( loci->subsections );
    cpl_test_abs(loci->subsections[loci->nsubsections-1]->sub_maxr,
            loci->maxR, 0.001);
    cpl_test_abs(loci->subsections[0]->sub_minr, loci->minR, 0.001);
    cpl_test_abs(loci->subsections[0]->sub_maxr,
            loci->minR + loci->dr, 0.001);
    cpl_test_abs(
            loci->subsections[(int)cpl_vector_get(loci->rings_min_indices,1)]->sub_minr,
            loci->minR + loci->dr, 0.001);
    sph_loci_delete(loci);
    loci = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_loci_draw_subsections(void) {
    sph_loci* loci = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* image = NULL;
    loci = sph_loci_new(300, 4.0, 0.5, 512, 512, 1.0, 100, 200.0);
    cpl_test_nonnull( loci );
    rerr = sph_loci_init_subsections(loci, 5.0, SPH_LOCI_NORMAL);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    image = cpl_image_new(1024, 1024, CPL_TYPE_DOUBLE);
    sph_loci_draw_subsections(loci, image);
    cpl_image_save(image, "loci_subsections.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    sph_loci_delete(loci);
    loci = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(image);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_loci_calculate_weights(void) {
    sph_loci* loci = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* image = NULL;
    cpl_image* im = NULL;
    cpl_image* maskim = NULL;
    cpl_frameset* frameset = NULL;
    cpl_vector* weights = NULL;
    cpl_vector* angles_start = NULL;
    cpl_vector* angles_stop = NULL;
    int ii = 0;
    int nims = 20;

    loci = sph_loci_new(50, 4.0, 0.5, 128, 128, 1.0, 50, 100);
    cpl_test_nonnull( loci );
    rerr = sph_loci_init_subsections(loci, 5.0, SPH_LOCI_NORMAL);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    sph_loci_draw_subsections(loci, image);
    cpl_image_save(image, "loci_subsections.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    frameset = cpl_frameset_new();
    angles_start = cpl_vector_new(nims);
    angles_stop = cpl_vector_new(nims);
    for (ii = 0; ii < nims; ++ii) {
        im = cpl_image_duplicate(image);
        cpl_vector_set(angles_start, ii, ii * 10.0);
        cpl_vector_set(angles_stop, ii, ii * 10.0 + 10.0);
        if (ii == 2 || ii == 9) {
            sph_test_image_tools_add_gauss(im, 180.0, 130.0, 3.0, 10000.0);
            sph_test_image_tools_add_gauss(im, 180.0, 140.0, 3.0, 10000.0);
        }
        sph_test_image_tools_add_noise(im, 5.0, NULL);
        cpl_frameset_insert(
                frameset,
                sph_test_frame_image(im, CPL_TYPE_DOUBLE, "LOCI_TEST",
                        CPL_FRAME_GROUP_RAW));
        cpl_image_delete(im);
        im = NULL;
    }cpl_test_nonnull( frameset );
    sph_loci_set_frameset(loci, frameset);
    cpl_test_nonnull( loci->frameset );
    cpl_test_error(CPL_ERROR_NONE);
    sph_loci_load_new_subsection_set(loci);
    sph_time_start_timer(0);
    weights = sph_loci_calculate_weights(loci, 1, 2, angles_start, angles_stop);
    cpl_test_nonnull( weights );
    //cpl_vector_dump(weights,stdout);
    cpl_test_eq(cpl_vector_get_size(weights), 20);
    cpl_test_abs(cpl_vector_get(weights,9), 1, 0.1);
    cpl_test_abs(cpl_vector_get(weights,2), 0, 0.1);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_vector_delete(weights);
    sph_loci_delete(loci);
    loci = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(image);
    cpl_frameset_delete(frameset);
    cpl_vector_delete(angles_start);
    cpl_vector_delete(angles_stop);
    cpl_image_delete(maskim);
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_loci_calculate_ref_image(void) {
    sph_loci* loci = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* image = NULL;
    cpl_image* im = NULL;
    cpl_image* maskim = NULL;
    cpl_frameset* frameset = NULL;
    cpl_vector* weights = NULL;
    cpl_vector* angles_start = NULL;
    cpl_vector* angles_stop = NULL;
    int ii = 0;
    int nims = 20;
    cpl_image* refimage = NULL;

    loci = sph_loci_new(50, 4.0, 0.5, 128, 128, 1.0, 50, 100);
    cpl_test_nonnull( loci );
    rerr = sph_loci_init_subsections(loci, 20.0, SPH_LOCI_NORMAL);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    frameset = cpl_frameset_new();
    angles_start = cpl_vector_new(nims);
    angles_stop = cpl_vector_new(nims);
    for (ii = 0; ii < nims; ++ii) {
        im = cpl_image_duplicate(image);
        cpl_vector_set(angles_start, ii, ii * 10.0);
        cpl_vector_set(angles_stop, ii, ii * 10.0 + 10.0);
        if (ii == 2 || ii == 9) {
            sph_test_image_tools_add_gauss(im, 180.0, 130.0, 3.0, 10000.0);
            sph_test_image_tools_add_gauss(im, 180.0, 140.0, 3.0, 10000.0);
        }
        sph_test_image_tools_add_noise(im, 5.0, NULL);
        cpl_frameset_insert(
                frameset,
                sph_test_frame_image(im, CPL_TYPE_DOUBLE, "LOCI_TEST",
                        CPL_FRAME_GROUP_RAW));
        cpl_image_delete(im);
        im = NULL;
    }cpl_test_nonnull( frameset );
    sph_loci_set_frameset(loci, frameset);
    cpl_test_nonnull( loci->frameset );
    cpl_test_error(CPL_ERROR_NONE);
    sph_loci_load_new_subsection_set(loci);
    sph_time_start_timer(0);
    weights = sph_loci_calculate_weights(loci, 1, 2, angles_start, angles_stop);
    cpl_test_nonnull( weights );
    //cpl_vector_dump(weights,stdout);
    cpl_test_eq(cpl_vector_get_size(weights), 20);
    cpl_test_abs(cpl_vector_get(weights,9), 1, 0.1);
    cpl_test_abs(cpl_vector_get(weights,2), 0, 0.1);
    cpl_test_error(CPL_ERROR_NONE);

    im = sph_loci_subsection_get_subtract_image(loci->subsections[1], 2);
    cpl_image_save(im, "refimage.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_test_lt(20.0, fabs(cpl_image_get_mean(im)));
    refimage = sph_loci_calculate_ref_image(loci, 1, weights);
    cpl_test_nonnull( refimage );
    cpl_image_subtract(im, refimage);
    cpl_test_abs(cpl_image_get_mean(im), 0.0, 5.0);

    cpl_image_delete(refimage);
    refimage = NULL;
    cpl_image_delete(im);
    im = NULL;
    cpl_test_error(CPL_ERROR_NONE);

    cpl_vector_delete(weights);
    sph_loci_delete(loci);
    loci = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(image);
    cpl_frameset_delete(frameset);
    cpl_vector_delete(angles_start);
    cpl_vector_delete(angles_stop);
    cpl_image_delete(maskim);
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_loci_calculate_imagelist(void) {
    sph_loci* loci = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* image = NULL;
    cpl_image* im = NULL;
    cpl_image* maskim = NULL;
    cpl_frameset* frameset = NULL;
    cpl_vector* weights = NULL;
    cpl_vector* angles_start = NULL;
    cpl_vector* angles_stop = NULL;
    int ii = 0;
    int nims = 20;
    cpl_imagelist* refimagelist = NULL;

    loci = sph_loci_new(50, 4.0, 0.5, 128, 128, 1.0, 50, 100);
    cpl_test_nonnull( loci );
    rerr = sph_loci_init_subsections(loci, 20.0, SPH_LOCI_NORMAL);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    frameset = cpl_frameset_new();
    angles_start = cpl_vector_new(nims);
    angles_stop = cpl_vector_new(nims);
    for (ii = 0; ii < nims; ++ii) {
        im = cpl_image_duplicate(image);
        cpl_vector_set(angles_start, ii, ii * 5.0);
        cpl_vector_set(angles_stop, ii, ii * 5.0 + 5.0);
        if (ii == 2 || ii == 9) {
            //sph_test_image_tools_add_gauss(im,180.0,130.0,3.0,100000.0);
            sph_test_image_tools_add_gauss(im, 180.0, 140.0, 3.0, 100000.0);
        }
        if (ii == 2) {
            //sph_test_image_tools_add_gauss(im,180.0,130.0,2.0,1000.0);
            sph_test_image_tools_add_gauss(im, 187.0, 142.0, 2.0, 1000.0);
        }
        sph_test_image_tools_add_noise(im, 5.0, NULL);
        cpl_frameset_insert(
                frameset,
                sph_test_frame_image(im, CPL_TYPE_DOUBLE, "LOCI_TEST",
                        CPL_FRAME_GROUP_RAW));
        cpl_image_delete(im);
        im = NULL;
    }cpl_test_nonnull( frameset );
    sph_loci_set_frameset(loci, frameset);
    cpl_test_nonnull( loci->frameset );
    cpl_test_error(CPL_ERROR_NONE);
    sph_loci_load_new_subsection_set(loci);
    sph_time_start_timer(0);
    refimagelist = sph_loci_calculate_imagelist(loci, 1, angles_start,
            angles_stop);
    cpl_test_nonnull( refimagelist );
    cpl_test_eq(cpl_imagelist_get_size(refimagelist), nims);
    for (ii = 0; ii < nims; ++ii) {
        if (ii != 2 && ii != 9) {
            cpl_test_abs(
                    cpl_image_get_mean( cpl_imagelist_get(refimagelist,ii)),
                    0.0, 5.0);
        }
    }
    cpl_imagelist_save(refimagelist, "refimagelist.fits", CPL_TYPE_FLOAT, NULL,
            CPL_IO_CREATE);
    cpl_imagelist_delete(refimagelist);
    refimagelist = NULL;
    cpl_test_error(CPL_ERROR_NONE);

    cpl_vector_delete(weights);
    sph_loci_delete(loci);
    loci = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(image);
    cpl_frameset_delete(frameset);
    cpl_vector_delete(angles_start);
    cpl_vector_delete(angles_stop);
    cpl_image_delete(maskim);
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_loci_process_single(void) {
    sph_loci* loci = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* image = NULL;
    cpl_image* im = NULL;
    cpl_image* maskim = NULL;
    cpl_frameset* frameset = NULL;
    cpl_vector* weights = NULL;
    cpl_vector* angles_start = NULL;
    cpl_vector* angles_stop = NULL;
    int ii = 0;
    int nims = 20;
    cpl_image* refimage = NULL;

    loci = sph_loci_new(50, 4.0, 0.5, 128, 128, 1.0, 50, 100);
    cpl_test_nonnull( loci );
    rerr = sph_loci_init_subsections(loci, 20.0, SPH_LOCI_NORMAL);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    frameset = cpl_frameset_new();
    angles_start = cpl_vector_new(nims);
    angles_stop = cpl_vector_new(nims);
    for (ii = 0; ii < nims; ++ii) {
        im = cpl_image_duplicate(image);
        cpl_vector_set(angles_start, ii, ii * 5.0);
        cpl_vector_set(angles_stop, ii, ii * 5.0 + 5.0);
        if (ii == 2 || ii == 9) {
            //sph_test_image_tools_add_gauss(im,180.0,130.0,3.0,100000.0);
            sph_test_image_tools_add_gauss(im, 180.0, 140.0, 3.0, 100000.0);
        }
        if (ii == 2) {
            //sph_test_image_tools_add_gauss(im,180.0,130.0,2.0,1000.0);
            sph_test_image_tools_add_gauss(im, 187.0, 142.0, 2.0, 1000.0);
        }
        sph_test_image_tools_add_noise(im, 5.0, NULL);
        cpl_frameset_insert(
                frameset,
                sph_test_frame_image(im, CPL_TYPE_DOUBLE, "LOCI_TEST",
                        CPL_FRAME_GROUP_RAW));
        cpl_image_delete(im);
        im = NULL;
    }cpl_test_nonnull( frameset );
    sph_loci_set_frameset(loci, frameset);
    cpl_test_nonnull( loci->frameset );
    cpl_test_error(CPL_ERROR_NONE);
    sph_loci_load_new_subsection_set(loci);
    sph_time_start_timer(0);
    refimage = sph_loci_process_single(loci, 2, 256, 256, angles_start,
            angles_stop);
    cpl_test_nonnull( refimage );
    cpl_test_abs(cpl_image_get_mean( refimage), 0.0, 5.0);
    cpl_image_save(refimage, "loci_result.fits", CPL_TYPE_FLOAT, NULL,
            CPL_IO_CREATE);
    cpl_image_delete(refimage);
    refimage = NULL;
    cpl_test_error(CPL_ERROR_NONE);

    cpl_vector_delete(weights);
    sph_loci_delete(loci);
    loci = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(image);
    cpl_frameset_delete(frameset);
    cpl_vector_delete(angles_start);
    cpl_vector_delete(angles_stop);
    cpl_image_delete(maskim);
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_loci_get_valid_set(void) {
    sph_loci* loci = NULL;
    cpl_vector* angles0 = NULL;
    cpl_vector* angles1 = NULL;
    cpl_vector* set = NULL;
    int ii = 0;
    double da = 9.0;

    loci = sph_loci_new(100, 3.0, 5.0, 128.0, 128.0, 1.0, 50, 100);
    cpl_test_nonnull( loci );
    cpl_test_eq(sph_loci_init_subsections(loci,10.0, SPH_LOCI_NORMAL),
            CPL_ERROR_NONE);
    loci->loaded_subsect_minindex = 0;
    loci->loaded_subsect_maxindex = 50;

    angles0 = cpl_vector_new(40);
    angles1 = cpl_vector_new(40);
    loci->subsections[1]->imlist = cpl_imagelist_new();
    for (ii = 0; ii < 40; ++ii) {
        cpl_vector_set(angles0, ii, da * ii);
        cpl_vector_set(angles1, ii, da * ii + da);
        cpl_imagelist_set(loci->subsections[1]->imlist,
                sph_test_image_tools_create_flat_image_double(256, 256, 1.0),
                ii);
    }
    set = sph_loci_get_valid_set(loci, 1, 10, angles0, angles1);
    cpl_test_nonnull( set );
    cpl_test_eq(cpl_vector_get_size(set), 35);
    cpl_vector_delete(set);
    set = NULL;
    set = sph_loci_get_valid_set(loci, 1, 0, angles0, angles1);
    cpl_test_nonnull( set );
    cpl_test_eq(cpl_vector_get_size(set), 35);
    cpl_vector_delete(set);
    set = NULL;
    cpl_vector_delete(angles0);
    angles0 = NULL;
    cpl_vector_delete(angles1);
    angles1 = NULL;
    sph_loci_delete(loci);
    loci = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_loci_load_new_subsection_set(void) {
    sph_loci* loci = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* image = NULL;
    cpl_image* maskim = NULL;
    cpl_frameset* frameset = NULL;
    int ii = 0;
    loci = sph_loci_new(50, 4.0, 0.5, 128, 128, 1.0, 50, 80);
    cpl_test_nonnull( loci );
    rerr = sph_loci_init_subsections(loci, 1.0, SPH_LOCI_NORMAL);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    frameset = cpl_frameset_new();
    image = sph_test_image_tools_create_flat_image_double(256, 256, 10.0);
    for (ii = 0; ii < 10; ++ii) {
        cpl_frameset_insert(
                frameset,
                sph_test_frame_image(image, CPL_TYPE_DOUBLE, "LOCI_TEST",
                        CPL_FRAME_GROUP_RAW));
    }
    sph_time_start_timer(0);
    cpl_test_nonnull( frameset );
    sph_loci_set_frameset(loci, frameset);
    cpl_test_nonnull( loci->frameset );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(sph_loci_load_new_subsection_set(loci),
            CPL_ERROR_NONE);
    cpl_test_lt(15, loci->nsubsections);
    cpl_test_nonnull( loci->subsections[15] );
    maskim = sph_loci_subsection_get_opt_mask_image(loci->subsections[15]);
    cpl_image_save(maskim, "loci_mask.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_test_nonnull( maskim );
    cpl_imagelist_multiply_image(loci->subsections[15]->imlist, maskim);
    cpl_imagelist_save(loci->subsections[15]->imlist, "loci_imlist.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    sph_loci_delete(loci);
    loci = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(image);
    cpl_frameset_delete(frameset);
    cpl_image_delete(maskim);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing LOCI", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing constructor", cutest_sph_loci_new)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing init subsections",
                    cutest_sph_loci_init_subsections)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing draw subsections",
                    cutest_sph_loci_draw_subsections)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing load subsections",
                    cutest_sph_loci_load_new_subsection_set)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing weight calculation",
                    cutest_sph_loci_calculate_weights)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing getting of set",
                    cutest_sph_loci_get_valid_set)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing ref image calculation",
                    cutest_sph_loci_calculate_ref_image)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing calc imagelist",
                    cutest_sph_loci_calculate_imagelist)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing process single",
                    cutest_sph_loci_process_single)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
