/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>
#include <math.h>



#include "sph_master_frame.h"
#include "sph_paint_polygon.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"
#include "sph_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_sph_master_frame Unit test for the sph_master_frame module            
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

sph_master_frame* __mframe = NULL;

static sph_master_frame*
cutest_sph_paint_polygon_create_mframe__(void) {
    int nx = 256;
    int ny = 256;
    if (__mframe)
        return __mframe;
    __mframe = sph_master_frame_new(nx, ny);
    return __mframe;
}

static
int cutest_sph_paint_polygon_init_testsuite(void) {

    sph_test_nop_code();
    __mframe = cutest_sph_paint_polygon_create_mframe__();
    return __mframe == NULL ? 1 : 0;
}

static
int cutest_sph_paint_polygon_clean_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Close and dump errors.
     * -------------------------------------------------------------------*/
    sph_master_frame_delete(__mframe);
    sph_error_dump(SPH_ERROR_ERROR);

    return sph_end_test();
}
static
void cutest_sph_paint_polygon_null_params(void) {
    sph_error_code rerr = CPL_ERROR_NONE;

    cpl_test_nonnull( __mframe );
    SPH_RAISE_CPL_RESET;
    rerr = sph_paint_polygon_on_master_image(__mframe, NULL, 0);
    cpl_test_noneq(rerr, CPL_ERROR_NONE);
    SPH_RAISE_CPL_RESET;
    rerr = sph_paint_polygon_on_master_bpix(__mframe, NULL, 0);
    cpl_test_noneq(rerr, CPL_ERROR_NONE);
    SPH_RAISE_CPL_RESET;
    rerr = sph_paint_polygon_on_master_rmsmap(__mframe, NULL, 0);
    cpl_test_noneq(rerr, CPL_ERROR_NONE);
    SPH_RAISE_CPL_RESET;
    rerr = sph_paint_polygon_on_master_weightmap(__mframe, NULL, 0);
    cpl_test_noneq(rerr, CPL_ERROR_NONE);
    SPH_RAISE_CPL_RESET;
    sph_error_reset();
}
static
void cutest_sph_paint_polygon_happy_case(void) {
    sph_polygon* poly = NULL;

    poly = sph_polygon_new();
    sph_polygon_add_point(poly, 10.0, 10.0);
    sph_polygon_add_point(poly, 110.0, 10.0);
    sph_polygon_add_point(poly, 60.0, 110.0);
    cpl_test_eq(sph_paint_polygon_on_master_image( __mframe, poly, 1.5 ),
            CPL_ERROR_NONE);
    cpl_test_eq(sph_master_frame_save( __mframe, "sph_paint_polygon_mframe.fits", NULL ),
            CPL_ERROR_NONE);

    cpl_test_abs( cpl_image_get_flux(__mframe->image), 1.5, 0.0001);

    sph_polygon_delete(poly);

    poly = sph_polygon_new();
    sph_polygon_add_point(poly, 10.0, 10.0);
    sph_polygon_add_point(poly, 10.0, 110.0);
    sph_polygon_add_point(poly, 60.0, 110.0);
    cpl_test_eq(sph_paint_polygon_on_master_image( __mframe, poly, 1.5 ),
            CPL_ERROR_NONE);
    cpl_test_eq(sph_master_frame_save( __mframe, "sph_paint_polygon_mframe.fits", NULL ),
            CPL_ERROR_NONE);

    cpl_test_abs( cpl_image_get_flux( __mframe->image ), 3.0, 0.01);

    sph_polygon_delete(poly);
    sph_master_frame_delete(__mframe);
    __mframe = NULL;
    __mframe = cutest_sph_paint_polygon_create_mframe__();
}

static
void cutest_sph_paint_polygon_happy_case_all_ext(void) {
    sph_polygon* poly = NULL;

    poly = sph_polygon_new();
    sph_polygon_add_point(poly, 10.0, 10.0);
    sph_polygon_add_point(poly, 110.0, 10.0);
    sph_polygon_add_point(poly, 60.0, 110.0);
    cpl_test_eq(
            sph_paint_polygon_on_master( __mframe, poly, 1.5, 2, 3.5, 4.5, NULL ),
            CPL_ERROR_NONE);
    cpl_test_eq(sph_master_frame_save(__mframe,"sph_paint_polygon_mframe_allext.fits", NULL),
            CPL_ERROR_NONE);

    cpl_test_abs( cpl_image_get_flux(__mframe->image), 1.5, 0.0001);
    cpl_test_abs( cpl_image_get_flux(__mframe->badpixelmap),
            2 * sph_polygon_area(poly), 400.0);
    //tolerance large
    // since "border" of 1
    // pixel depth.
    cpl_test_abs( cpl_image_get_flux(__mframe->rmsmap), 3.5, 0.0001);
    cpl_test_abs( cpl_image_get_flux(__mframe->ncombmap), 4.5,
            0.0001);

    sph_polygon_delete(poly);

    poly = sph_polygon_new();
    sph_polygon_add_point(poly, 10.0, 10.0);
    sph_polygon_add_point(poly, 60.0, 110.0);
    sph_polygon_add_point(poly, 10.0, 110.0);
    cpl_test_eq(sph_paint_polygon_on_master_image(__mframe,poly, 1.5),
            CPL_ERROR_NONE);
    cpl_test_eq(sph_master_frame_save(__mframe,"sph_paint_polygon_mframe.fits", NULL),
            CPL_ERROR_NONE);

    cpl_test_abs( cpl_image_get_flux(__mframe->image), 3.0, 0.0001);

    sph_polygon_delete(poly);
    sph_master_frame_delete(__mframe);
    __mframe = NULL;
    __mframe = cutest_sph_paint_polygon_create_mframe__();
}
static
void cutest_sph_paint_polygon_overlap_neg(void) {
    sph_polygon* poly = NULL;

    poly = sph_polygon_new();
    sph_polygon_add_point(poly, 10.0, 10.0);
    sph_polygon_add_point(poly, 110.0, 10.0);
    sph_polygon_add_point(poly, 60.0, 110.0);
    cpl_test_eq(sph_paint_polygon_on_master_image(__mframe,poly, -1.5),
            CPL_ERROR_NONE);
    cpl_test_eq(sph_master_frame_save(__mframe,"sph_paint_polygon_mframe.fits", NULL),
            CPL_ERROR_NONE);

    cpl_test_abs( cpl_image_get_flux(__mframe->image), -1.5, 0.0001);

    sph_polygon_delete(poly);

    poly = sph_polygon_new();
    sph_polygon_add_point(poly, 10.0, 10.0);
    sph_polygon_add_point(poly, 60.0, 80.0);
    sph_polygon_add_point(poly, 10.0, 110.0);
    cpl_test_eq(sph_paint_polygon_on_master_image(__mframe,poly, -1.5),
            CPL_ERROR_NONE);
    cpl_test_eq(sph_master_frame_save(__mframe,"sph_paint_polygon_mframe.fits", NULL),
            CPL_ERROR_NONE);

    cpl_test_abs( cpl_image_get_flux(__mframe->image), -3.0, 0.0001);

    sph_polygon_delete(poly);
    sph_master_frame_delete(__mframe);
    __mframe = NULL;
    __mframe = cutest_sph_paint_polygon_create_mframe__();
}
static
void cutest_sph_paint_polygon_overlap(void) {
    sph_polygon* poly = NULL;

    poly = sph_polygon_new();
    sph_polygon_add_point(poly, 10.0, 10.0);
    sph_polygon_add_point(poly, 110.0, 10.0);
    sph_polygon_add_point(poly, 60.0, 110.0);
    cpl_test_eq(sph_paint_polygon_on_master_image(__mframe,poly, 1.5),
            CPL_ERROR_NONE);
    cpl_test_eq(sph_master_frame_save(__mframe,"sph_paint_polygon_mframe.fits", NULL),
            CPL_ERROR_NONE);

    cpl_test_abs( cpl_image_get_flux(__mframe->image), 1.5, 0.0001);

    sph_polygon_delete(poly);

    poly = sph_polygon_new();
    sph_polygon_add_point(poly, 10.0, 10.0);
    sph_polygon_add_point(poly, 60.0, 80.0);
    sph_polygon_add_point(poly, 10.0, 110.0);
    cpl_test_eq(sph_paint_polygon_on_master_image(__mframe,poly, 1.5),
            CPL_ERROR_NONE);
    cpl_test_eq(sph_master_frame_save(__mframe,"sph_paint_polygon_mframe.fits", NULL),
            CPL_ERROR_NONE);

    cpl_test_abs( cpl_image_get_flux(__mframe->image), 3.0, 0.0001);

    sph_polygon_delete(poly);
    sph_master_frame_delete(__mframe);
    __mframe = NULL;
    __mframe = cutest_sph_paint_polygon_create_mframe__();
}

static
void cutest_sph_paint_polygon_many_random(void) {
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_polygon* poly = NULL;
    gsl_rng* pRNG = NULL;
    int npolys = 100;
    int nx = 0;
    int ny = 0;
    double val = 0.0;
    double tval = 0.0;
    int ii = 0;
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    nx = cpl_image_get_size_x(__mframe->image);
    ny = cpl_image_get_size_y(__mframe->image);

    for (ii = 0; ii < npolys; ++ii) {
        poly = sph_polygon_new();
        sph_polygon_add_point(poly, 0.0, 0.0);
        sph_polygon_add_point(poly, 5.0, 10.0);
        sph_polygon_add_point(poly, 10.0, 0.0);
        sph_polygon_shift(poly, gsl_ran_flat(pRNG, 0.0, nx - 10.0),
                gsl_ran_flat(pRNG, 0.0, ny - 10.0));
        val = gsl_ran_flat(pRNG, 1.0, 10.0);
        tval += val;
        rerr |= sph_paint_polygon_on_master_image(__mframe, poly, val);
        sph_polygon_delete(poly);
    }cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(sph_master_frame_save(__mframe,"sph_paint_polygon_mframe.fits", NULL),
            CPL_ERROR_NONE);

    cpl_test_abs( cpl_image_get_flux(__mframe->image), tval, 0.1);
    sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
            SPH_ERROR_INFO, "Flux: %f, %f", cpl_image_get_flux(__mframe->image),
            tval);
    sph_master_frame_delete(__mframe);
    __mframe = NULL;
    __mframe = cutest_sph_paint_polygon_create_mframe__();
    gsl_rng_free(pRNG);
    pRNG = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Main to run the tests.
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;

    (void)sph_test_init();


    pSuite = sph_add_suite("cutest_sph_paint_polygon",
            cutest_sph_paint_polygon_init_testsuite,
            cutest_sph_paint_polygon_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "NULL params",
                    cutest_sph_paint_polygon_null_params)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "happy case",
                    cutest_sph_paint_polygon_happy_case)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "happy case all ext",
                    cutest_sph_paint_polygon_happy_case_all_ext)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "overlap", cutest_sph_paint_polygon_overlap)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "many random",
                    cutest_sph_paint_polygon_many_random)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "overlap_neg",
                    cutest_sph_paint_polygon_overlap_neg)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
