/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include "sph_pixel_polyfit_table.h"
#include "sph_pixel_polyfit.h"

#include "sph_test.h"
#include "sph_test_image_tools.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_fitting.h"
#include "sph_utils.h"

#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    sph_end_test();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_pixel_polyfit_table_new(void) {
    sph_pixel_polyfit_table* ptab = NULL;

    ptab = sph_pixel_polyfit_table_new(256, 128, 2);
    cpl_test_nonnull( ptab );
    cpl_test_eq(ptab->nx, 256);
    cpl_test_eq(ptab->ny, 128);
    cpl_test_eq(ptab->order, 2);
    cpl_test_assert(ptab->table);
    sph_pixel_polyfit_table_delete(ptab);
    ptab = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_pixel_polyfit_wrap(void) {
    sph_pixel_polyfit* ptab = NULL;
    cpl_imagelist* imlist = cpl_imagelist_new();
    cpl_image*     img;
    cpl_error_code code;

    img = sph_test_image_tools_create_flat_image_double(256, 128, 1.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(img);

    code = cpl_imagelist_set(imlist, img, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    img = sph_test_image_tools_create_flat_image_double(256, 128, 2.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(img);

    code = cpl_imagelist_set(imlist, img, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    img = sph_test_image_tools_create_flat_image_double(256, 128, 3.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(img);

    code = cpl_imagelist_set(imlist, img, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    ptab = sph_pixel_polyfit_wrap(imlist, NULL);
    imlist = NULL;

    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( ptab );
    cpl_test_nonnull(ptab->fit);
    cpl_test_nonnull(ptab->pfit);
    cpl_test_nonnull(ptab->pbpm);
    cpl_test_null(ptab->error);
    cpl_test_eq(ptab->nx, 256);
    cpl_test_eq(ptab->ny, 128);
    cpl_test_eq(ptab->order, 2);

    sph_pixel_polyfit_delete(ptab);
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_pixel_polyfit_table_set_invalid(void) {
    sph_pixel_polyfit_table* ptab = NULL;
    cpl_mask* bpixmask = NULL;
    sph_master_frame* mframe = NULL;
    int bpix = 0;
    cpl_size       n;
    cpl_imagelist* imlist = cpl_imagelist_new();
    cpl_image*     img;
    cpl_error_code code;

    img = sph_test_image_tools_create_flat_image_double(256, 128, 1.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(img);

    code = cpl_imagelist_set(imlist, img, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    img = sph_test_image_tools_create_flat_image_double(256, 128, 2.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(img);

    code = cpl_imagelist_set(imlist, img, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    img = sph_test_image_tools_create_flat_image_double(256, 128, 3.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(img);

    code = cpl_imagelist_set(imlist, img, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    ptab = sph_pixel_polyfit_table_new_from_imlist(imlist, NULL);
    cpl_test_nonnull( ptab );
    cpl_test_eq(ptab->nx, 256);
    cpl_test_eq(ptab->ny, 128);
    cpl_test_eq(ptab->order, 2);
    cpl_test_nonnull(ptab->table);

    sph_pixel_polyfit_table_set_invalid(ptab, 12, 12);

    bpixmask = sph_pixel_polyfit_table_create_badpixel_mask(ptab, -100000.0,
            10000000.0, 10000.0);
    cpl_test_nonnull( bpixmask );
    cpl_test_eq(cpl_mask_get(bpixmask,13,13), 1);
    cpl_test_zero(cpl_mask_get(bpixmask,12,12));

    cpl_mask_delete(bpixmask);

    mframe = sph_pixel_polyfit_table_get_coeff_as_image(ptab, 1);
    cpl_test_nonnull( mframe );
    cpl_test_eq(cpl_image_get(mframe->badpixelmap,13,13,&bpix), 1);
    cpl_test_zero(cpl_image_get(mframe->badpixelmap,12,12,&bpix));

    sph_master_frame_delete(mframe);

    sph_pixel_polyfit_table_save_cube(ptab, "ptab.fits");
    cpl_test_fits("ptab.fits");

    n = cpl_fits_count_extensions("ptab.fits");
    cpl_test_eq(n, 3);
    cpl_test_zero(unlink("ptab.fits"));

    sph_pixel_polyfit_table_delete(ptab);
    cpl_imagelist_delete(imlist);

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing sph_pixel_polyfit_table",
            cutest_init_testsuite, cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "new", cutest_test_pixel_polyfit_table_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "new from imlist",
                    cutest_test_pixel_polyfit_wrap)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "set invalid",
                    cutest_test_pixel_polyfit_table_set_invalid)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
