/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <cpl.h>

#include "sph_skycalc.h"
#include "sph_time.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_utils.h"



/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_zpl_exposure Unit test of sph_zpl_exposure
 *
 */
/*----------------------------------------------------------------------------*/

static int cutest_init_skycalc_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static int cutest_clean_skycalc_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Close and dump errors.
     * -------------------------------------------------------------------*/
    return sph_end_test();
}

static
void cutest_test_sph_skycalc(void) {
    sph_skycalc_input skyinp[7];
    sph_skycalc_output skyout;
    char iso8601[80];
    int year, month, day, hh, mm;
    double ss;
    double mjd;

    sph_skycalc_setup(skyinp, "19 19 19", "2 2 2", 2000, "1990 10 20",
            "21 30 00");
    sph_skycalc_set_timezone(skyinp, LT); //set to a local time
    sph_skycalc(skyinp, &skyout);
    //printf("SKYOUT1: objra = %f\n", skyout.objra);
    //printf("SKYOUT1: objdec = %f\n", skyout.objdec);
    //printf("SKYOUT1: objepoch = %f\n", skyout.objepoch);
    //printf("SKYOUT1: alt = %f\n", skyout.alt);
    //printf("SKYOUT1: jd = %f\n", skyout.jd);
    //printf("SKYOUT1: barjd = %f\n", skyout.barjd);
    //printf("SKYOUT1: pa = %f\n", skyout.pa);
    cpl_test_abs(skyout.alt, -22.509, 0.001);
    cpl_test_abs(skyout.pa, 129.2, 0.1);

    sph_skycalc_set_time(skyinp, "21 35 00");
    sph_skycalc(skyinp, &skyout);
    cpl_test_abs( skyout.pa, 128.4, 0.1);

    //test seconds in a double value, LT
    sph_skycalc_setup(skyinp, "29 29 19.5", "2 2 2.4", 2000, "1990 10 20",
            "21 30 00");
    sph_skycalc_set_timezone(skyinp, LT); //set to a local time
    sph_skycalc(skyinp, &skyout);
    //printf("SKYOUT1: objra = %f\n", skyout.objra);
    //printf("SKYOUT1: objdec = %f\n", skyout.objdec);
    //printf("SKYOUT1: objepoch = %f\n", skyout.objepoch);
    //printf("SKYOUT1: jd = %f\n", skyout.jd);
    //printf("SKYOUT1: barjd = %f\n", skyout.barjd);
    //printf("SKYOUT1: pa = %f\n", skyout.pa);
    cpl_test_abs(skyout.pa, -116.2, 0.1);

    //iso8601_string, LT
    sph_skycalc_setup_iso8601_string(skyinp, "19 19 19", "2 2 2", 2000,
            "1990-10-20T21:30:00.0");
    sph_skycalc_set_timezone(skyinp, LT); //set to a local time
    sph_skycalc(skyinp, &skyout);
    //printf("SKYOUT2: objra = %f\n", skyout.objra);
    //printf("SKYOUT2: objdec = %f\n", skyout.objdec);
    //printf("SKYOUT2: objepoch = %f\n", skyout.objepoch);
    //printf("SKYOUT2: jd = %f\n", skyout.jd);
    //printf("SKYOUT2: barjd = %f\n", skyout.barjd);
    //printf("SKYOUT2: pa = %f\n", skyout.pa);
    cpl_test_abs(skyout.pa, 129.2, 0.1);

    //iso8601, LT
    sph_skycalc_setup_iso8601(skyinp, "19 19 19", "2 2 2", 2000, 1990, 10, 20,
            21, 30, 0.0);
    sph_skycalc_set_timezone(skyinp, LT); //set to a local time
    sph_skycalc(skyinp, &skyout);
    //printf("SKYOUT3: objra = %f\n", skyout.objra);
    //printf("SKYOUT3: objdec = %f\n", skyout.objdec);
    //printf("SKYOUT3: objepoch = %f\n", skyout.objepoch);
    //printf("SKYOUT3: jd = %f\n", skyout.jd);
    //printf("SKYOUT3: barjd = %f\n", skyout.barjd);
    //printf("SKYOUT3: pa = %f\n", skyout.pa);
    cpl_test_abs(skyout.pa, 129.2, 0.1);

    //mjd, LT
    sph_time_iso8601_from_string(&year, &month, &day, &hh, &mm, &ss,
            "1990-10-20T21:30:00.0");
    //printf("year=%d  month=%d  day=%d Time=%d:%d:%f\n", year, month, day, hh, mm, ss );
    sph_time_mjd_from_string(&mjd, "1990-10-20T21:30:0.0");
    //printf("mjd = %f\n", mjd);
    sph_time_mjd_to_iso8601string(mjd, iso8601);
    //sph_time_mjd_to_iso8601string(skyout.jd, iso8601);
    //printf("iso8601: %s\n", iso8601);

    sph_skycalc_setup_mjd(skyinp, "19 19 19", "2 2 2", 2000, mjd);
    sph_skycalc_set_timezone(skyinp, LT); //set to a local time
    sph_skycalc(skyinp, &skyout);
    //printf("SKYOUT4: objra = %f\n", skyout.objra);
    //printf("SKYOUT4: objdec = %f\n", skyout.objdec);
    //printf("SKYOUT4: objepoch = %f\n", skyout.objepoch);
    //printf("SKYOUT4: jd = %f\n", skyout.jd);
    //printf("SKYOUT4: barjd = %f\n", skyout.barjd);
    //printf("SKYOUT4: pa = %f\n", skyout.pa);
    cpl_test_abs(skyout.pa, 129.2, 0.1);

    // set, mjd
    sph_time_mjd_from_string(&mjd, "1990-10-20T23:30:0.0");
    sph_skycalc_set_mjd(skyinp, mjd);
    //sph_skycalc_set_timezone( skyinp, LT ); //set to a local time
    sph_skycalc(skyinp, &skyout);
    //printf("SKYOUT45: objra = %f\n", skyout.objra);
    //printf("SKYOUT45: objdec = %f\n", skyout.objdec);
    //printf("SKYOUT45: objepoch = %f\n", skyout.objepoch);
    //printf("SKYOUT45: jd = %f\n", skyout.jd);
    //printf("SKYOUT45: barjd = %f\n", skyout.barjd);
    //printf("SKYOUT45: pa = %f\n", skyout.pa);
    cpl_test_abs(skyout.pa, 117.2, 0.1);

    //Time, UT
    sph_skycalc_setup(skyinp, "9 19 19", "2 2 2", 2000, "1990 10 20",
            "21 30 00");
    sph_skycalc_set_timezone(skyinp, UT); //set to universal time
    sph_skycalc(skyinp, &skyout);
    //printf("SKYOUT5: objra = %f\n", skyout.objra);
    //printf("SKYOUT5: objdec = %f\n", skyout.objdec);
    //printf("SKYOUT5: objepoch = %f\n", skyout.objepoch);
    //printf("SKYOUT5: alt = %f\n", skyout.alt);
    //printf("SKYOUT5: jd = %f\n", skyout.jd);
    //printf("SKYOUT5: barjd = %f\n", skyout.barjd);
    //printf("SKYOUT5: pa = %f\n", skyout.pa);
    cpl_test_abs(skyout.pa, 124.6, 0.1);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("cutest_skycalc", cutest_init_skycalc_testsuite,
            cutest_clean_skycalc_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL == sph_test_do(pSuite, "sph_skycalc", cutest_test_sph_skycalc)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}
