/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_utils.h"
#include "sph_master_frame.h"
#include "sph_double_image.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_error.h"
#include "sph_test.h"

#include <cpl.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static
int cutest_init_utils_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static
int cutest_clean_utils_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_utils_create_master_frameset function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_utils_create_master_frameset(void) {
    cpl_frameset* inframes = NULL;
    cpl_frameset* newframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    FILE* fp = fopen("sph_utils_create_master_frameset_web.txt", "w");

    cpl_test_nonnull(fp);

    cpl_test(fprintf(fp, "<table><tr><td>Test passed</td></tr></table>\n"));
    cpl_test_zero(fclose(fp));

    /* Setup... */
    inframes = cpl_frameset_new();
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( inframes );

    rerr = sph_test_create_ifs_raw_dark_frameset(inframes, 2);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /* Run ... */
    newframes = sph_utils_create_master_frameset(inframes);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( newframes);

    sph_test_frameset_delete(inframes);
    cpl_frameset_delete(newframes);
    cpl_test_error(CPL_ERROR_NONE);

}

/*
 **/
/*----------------------------------------------------------------------------*/
static
void cutest_utils_attach_plist(void) {
    cpl_propertylist* plist = NULL;
    cpl_propertylist* compplist = NULL;
    sph_master_frame* mframe = NULL;
    cpl_frame* aframe = NULL;

    plist = cpl_propertylist_new();
    cpl_test_nonnull( plist );
    mframe = sph_master_frame_new(100, 100);
    cpl_test_nonnull( mframe );
    aframe = cpl_frame_new();
    cpl_frame_set_tag(aframe, "TESTFRAME");
    cpl_frame_set_filename(aframe, "sph_utils_test_attach_plist.fits");

    sph_master_frame_save(mframe, cpl_frame_get_filename(aframe), plist);

    compplist = sph_keyword_manager_load_properties(
            cpl_frame_get_filename(aframe), 0);
    cpl_test_nonnull( compplist );

    cpl_test_zero(cpl_propertylist_has( compplist, "ESO TESTIT" ));

    cpl_propertylist_append_double(plist, "ESO TESTIT", 10.0);
    sph_utils_attach_plist(plist, aframe, 0);
    cpl_propertylist_delete(compplist);
    compplist = NULL;
    compplist = sph_keyword_manager_load_properties(
            cpl_frame_get_filename(aframe), 0);
    cpl_test_nonnull( compplist );
    cpl_test_eq(cpl_propertylist_has( compplist, "ESO TESTIT" ), 1);
    cpl_test_abs(
            cpl_propertylist_get_double( compplist, "ESO TESTIT" ), 10.0, 0.01);
    cpl_propertylist_delete(compplist);
    compplist = NULL;
    sph_master_frame_delete(mframe);
    cpl_propertylist_delete(plist);
    plist = NULL;
    cpl_frame_delete(aframe);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Unit test for the sph_utils_cpl_image_fill_double
 */
/*----------------------------------------------------------------------------*/
static
void cutest_utils_cpl_image_fill_double(void) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* image = NULL;
    double dvalue = 10.0;

    image = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);
    rerr = sph_utils_cpl_image_fill_double(image, dvalue);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs( cpl_image_get_mean( image ), 10.0, 1e-20);
    //cpl_image_save( image, "test_fill_double.fits.tmp", CPL_TYPE_FLOAT, NULL, CPL_IO_DEFAULT );
    cpl_image_delete(image);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Unit test for the sph_utils_cpl_image_fill_double
 */
/*----------------------------------------------------------------------------*/
static
void cutest_utils_append_to_propertynames(void) {
    cpl_propertylist* plist = NULL;

    plist = cpl_propertylist_new();
    cpl_test_assert(plist);

    cpl_propertylist_append_double(plist,"ANAME",10.0);

    cpl_test_eq(
    		sph_utils_append_to_propertynames(plist,"L"),
    		CPL_ERROR_NONE);
    cpl_test_zero(
    		cpl_propertylist_has(plist, "ANAME") );
    cpl_test(
    		cpl_propertylist_has(plist, "ANAME L") );
    cpl_test_abs(
    		cpl_propertylist_get_double(plist, "ANAME L"),
    		10.0, 0.0001);

    cpl_propertylist_delete(plist);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;

    (void)sph_test_init();

    pSuite = sph_add_suite("sph_utils_test", cutest_init_utils_testsuite,
            cutest_clean_utils_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_utils_cpl_image_fill_double",
                    cutest_utils_cpl_image_fill_double)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_utils_create_master_frameset",
                    cutest_utils_create_master_frameset)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_utils_attach_plist",
                    cutest_utils_attach_plist)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_utils_append_to_propertynames",
                    cutest_utils_append_to_propertynames)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
