/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_zpl_exposure.h"
#include "sph_utils.h"
#include "sph_common_keywords.h"
#include "sph_zpl_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_master_frame.h"
#include "sph_framecombination.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"
#include "sph_test_zpl_simulator.h"

#include <cpl.h>

#include <stdlib.h>
#include <string.h>
#include <math.h>

/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/
#define SPH_BASE "cutest_zpl_exposure"
#define SPH_CFG  SPH_BASE ".cfg"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_zpl_exposure Unit test of sph_zpl_exposure
 *
 */
/*----------------------------------------------------------------------------*/
static
void cutest_zpl_exposure_create_head_file(const char* infile) {
    FILE* fp;
    char outfilename[256];

    sprintf(outfilename, "%s.head", infile);
    fp = fopen(outfilename, "w");
    if (fp != NULL) {
        fprintf(fp, "ESO DRS ZPL HWPZ SETTING = 48.5\n");
        fclose(fp);
    }

}

static
sph_keyword_manager* cutest_zpl_exposure_create_cfg_file(void) {
    FILE* fp;
    sph_keyword_manager* km = NULL;

    fp = fopen(SPH_CFG, "w");
    fprintf(fp,
            "SPH.KEYWORDS.ESO_DRS_ZPL_HWPZ_SETTING.NAME = \"ESO DRS ZPL HWPZ SETTING\"\n");
    fprintf(fp, "SPH.KEYWORDS.ESO_DRS_ZPL_HWPZ_SETTING.TYPE = \"DOUBLE\"\n");
    fprintf(fp, "SPH.KEYWORDS.ESO_DRS_ZPL_HWPZ_SETTING.VALUE = \"10.0\"\n");
    fclose(fp);

    km = sph_keyword_manager_new();
    cpl_test_nonnull(km);

    return km;
}

static int cutest_init_zpl_exposure_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static int cutest_clean_zpl_exposure_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Close and dump errors.
     * -------------------------------------------------------------------*/
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_master_frame_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_zpl_exposure_new_empty(void) {
    sph_zpl_exposure* zplexp = NULL;

    zplexp = sph_zpl_exposure_new_empty();
    cpl_test_nonnull( zplexp );

    cpl_test(
            cpl_propertylist_has( zplexp->properties, SPH_COMMON_KEYWORD_SPH_TYPE ));
    cpl_test_eq_string(
            cpl_propertylist_get_string( zplexp->properties, SPH_COMMON_KEYWORD_SPH_TYPE ),
            SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP);
    sph_zpl_exposure_delete(zplexp);
    cpl_test_error(CPL_ERROR_NONE);

    return;
}

static
void cutest_test_zpl_exposure_new(void) {
    sph_zpl_exposure* zplexp = NULL;
    const int nx = 100;
    const int ny = 200;

    zplexp = sph_zpl_exposure_new(nx, ny);
    cpl_test_nonnull( zplexp );

    cpl_test(
            cpl_propertylist_has( zplexp->properties, SPH_COMMON_KEYWORD_SPH_TYPE ));
    cpl_test_eq_string(
            cpl_propertylist_get_string( zplexp->properties, SPH_COMMON_KEYWORD_SPH_TYPE ),
            SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP);

    cpl_test_eq(cpl_image_get_size_x( zplexp->image_zero_odd ), nx);
    cpl_test_eq(cpl_image_get_size_y( zplexp->image_zero_odd ), ny);

    cpl_test_eq(cpl_image_get_size_x( zplexp->image_zero_even ), nx);
    cpl_test_eq(cpl_image_get_size_y( zplexp->image_zero_even ), ny);

    cpl_test_eq(cpl_image_get_size_x( zplexp->image_pi_odd ), nx);
    cpl_test_eq(cpl_image_get_size_y( zplexp->image_pi_odd ), ny);

    cpl_test_eq(cpl_image_get_size_x( zplexp->image_pi_even ), nx);
    cpl_test_eq(cpl_image_get_size_y( zplexp->image_pi_even ), ny);

    sph_zpl_exposure_delete(zplexp);
    cpl_test_error(CPL_ERROR_NONE);

    return;
}

static
void cutest_test_zpl_exposure_save(void) {
    sph_zpl_exposure* zplexp = NULL;
    sph_zpl_exposure* zexpload = NULL;
    cpl_frame* zexpframe = NULL;

    zplexp = sph_test_zpl_simulator_create_exposure(100, 100);

    /* internal test of ph_test_zpl_simulator_create_exposure & _4flat */
    cpl_test_abs(cpl_image_get_mean( zplexp->image_zero_odd ),
                           ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_zero_even ),
                           ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_odd ),
                           PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_even ),
                           PI_EVEN_DOUBLE, 1e-20);

    // save zpl exp frame
    zexpframe = sph_filemanager_create_temp_frame("test_zplexp.fits", "PREPROC",
            CPL_FRAME_GROUP_NONE);
    sph_zpl_exposure_save(zplexp, zexpframe, NULL);
    //load a zpl exposure from pure zpl exp frame
    zexpload = sph_zpl_exposure_load(zexpframe, 0);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_zero_even ),
                           ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_odd ),
                           PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_even ),
                           PI_EVEN_DOUBLE,
                           1e-20);

    cpl_frame_delete(zexpframe);
    sph_zpl_exposure_delete(zexpload);
    sph_zpl_exposure_delete(zplexp);
    sph_filemanager_clean();

    return;
}

static
void cutest_test_zpl_exposure_save_append(void) {
    sph_zpl_exposure* zplexp = NULL;
    sph_zpl_exposure* zplexp_append = NULL;
    sph_zpl_exposure* zexpload = NULL;
    cpl_frame* zexpframe = NULL;

    zplexp = sph_test_zpl_simulator_create_exposure(100, 100);
    cpl_test_nonnull( zplexp );

    /* internal test of ph_test_zpl_simulator_create_exposure & _4flat */

    cpl_test_abs( cpl_image_get_mean( zplexp->image_zero_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_zero_even ),
                           ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_odd ),
                           PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_even ),
                           PI_EVEN_DOUBLE, 1e-20);

    // save_open zpl exp frame
    zexpframe = sph_filemanager_create_temp_frame("test_zplexp.fits", "PREPROC",
            CPL_FRAME_GROUP_NONE);
    sph_zpl_exposure_save_open(zplexp, zexpframe, NULL, NULL);

    zplexp_append = sph_test_zpl_simulator_create_exposure_4flat_double(100,
            100, 10.0, 20.0, 30.0, 40.0);
    sph_zpl_exposure_save_append(zplexp_append, zexpframe, NULL);
    sph_zpl_exposure_finalize_file(zexpframe, NULL);

    //load a zpl exposure from the zpl exp frame cube: first plane = 0
    zexpload = sph_zpl_exposure_load(zexpframe, 0);
    cpl_test_nonnull( zexpload );
    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_zero_even ),
                           ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_odd ),
                           PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_even ),
                           PI_EVEN_DOUBLE,
                           1e-20);
        sph_zpl_exposure_delete(zexpload);

    //load a zpl exposure from the zpl exp frame cube: second plane = 1
    zexpload = sph_zpl_exposure_load(zexpframe, 1);
    cpl_test_nonnull( zexpload );
    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_odd ),
                            10.0, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_zero_even ),
                           20.0, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_odd ),
                           30.0, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_even ),
                           40.0, 1e-20);

    sph_zpl_exposure_delete(zexpload);
    sph_zpl_exposure_delete(zplexp);
    sph_zpl_exposure_delete(zplexp_append);
    cpl_frame_delete(zexpframe);

    sph_filemanager_clean();

    return;
}

static
void cutest_test_zpl_exposure_finalize_with_ovsc_table(void) {
    sph_zpl_exposure* zplexp = NULL;
    sph_zpl_exposure* zplexp_append = NULL;
    sph_zpl_exposure* zexpload = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_propertylist* ovsc_pl = NULL;
    cpl_table* ovsc_table = NULL;
    double a[8] = { 100.0, 1.0, 200.0, 2.0, 300.0, 3.0, 400.0, 4.0 };
    double b[8] = { 110.0, 11.0, 210.0, 12.0, 310.0, 31.0, 410.0, 14.0 };

    zplexp = sph_test_zpl_simulator_create_exposure(100, 100);
    cpl_test_nonnull( zplexp );
    /* internal test of ph_test_zpl_simulator_create_exposure & _4flat */
    cpl_test_abs( cpl_image_get_mean( zplexp->image_zero_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_zero_even ),
                           ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_odd ),
                           PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_even ),
                           PI_EVEN_DOUBLE, 1e-20);

    // save_open zpl exp frame
    zexpframe = sph_filemanager_create_temp_frame("test_zplexp_finalize.fits",
            "PREPROC", CPL_FRAME_GROUP_NONE);
    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist(a);
    if (zplexp->ovsc)
        cpl_propertylist_delete(zplexp->ovsc);
    zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);
    cpl_propertylist_delete(ovsc_pl);
    ovsc_pl = NULL;

    ovsc_table = sph_zpl_exposure_ovsc_table_create_empty();
    sph_zpl_exposure_save_open(zplexp, zexpframe, NULL, ovsc_table);

    //printf("ADU1_ZERO_OVSC_MEAN = %f\n", cpl_table_get_double( ovsc_table,  ADU1_ZERO_OVSC_MEAN, 0, NULL ));

    zplexp_append = sph_test_zpl_simulator_create_exposure_4flat_double(100,
            100, 10.0, 20.0, 30.0, 40.0);
    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist(b);
    if (zplexp_append->ovsc)
        cpl_propertylist_delete(zplexp_append->ovsc);
    zplexp_append->ovsc = cpl_propertylist_duplicate(ovsc_pl);
    cpl_propertylist_delete(ovsc_pl);

    sph_zpl_exposure_save_append(zplexp_append, zexpframe, ovsc_table);
    //sph_cube_finalise_file( cpl_frame_get_filename(zexpframe));

    //printf("ADU1_ZERO_OVSC_MEAN = %f\n", cpl_table_get_double( ovsc_table,  ADU1_ZERO_OVSC_MEAN, 1, NULL ));

    sph_zpl_exposure_finalize_file(zexpframe, ovsc_table);
    //load a zpl exposure from the zpl exp frame cube: first plane = 0
    zexpload = sph_zpl_exposure_load(zexpframe, 0);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_zero_even ),
                           ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_odd ),
                           PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_even ),
                           PI_EVEN_DOUBLE,
                           1e-20);

    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_ZERO_OVSC_MEAN ),
            a[0], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_ZERO_OVSC_RMS ),
            a[1], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_ZERO_OVSC_MEAN ),
            a[2], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_ZERO_OVSC_RMS ),
            a[3], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_PI_OVSC_MEAN ),
            a[4], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_PI_OVSC_RMS ),
            a[5], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_PI_OVSC_MEAN ),
            a[6], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_PI_OVSC_RMS ),
            a[7], 1e-20);

    sph_zpl_exposure_delete(zexpload);
    //load a zpl exposure from the zpl exp frame cube: second plane = 1
    zexpload = sph_zpl_exposure_load(zexpframe, 1);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_odd ),
                            10.0, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_zero_even ),
                           20.0, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_odd ),
                           30.0, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_even ),
                           40.0, 1e-20);

    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_ZERO_OVSC_MEAN ),
            b[0], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_ZERO_OVSC_RMS ),
            b[1], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_ZERO_OVSC_MEAN ),
            b[2], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_ZERO_OVSC_RMS ),
            b[3], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_PI_OVSC_MEAN ),
            b[4], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_PI_OVSC_RMS ),
            b[5], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_PI_OVSC_MEAN ),
            b[6], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_PI_OVSC_RMS ),
            b[7], 1e-20);

    sph_zpl_exposure_delete(zexpload);
    sph_zpl_exposure_delete(zplexp);
    sph_zpl_exposure_delete(zplexp_append);
    cpl_table_delete(ovsc_table);
    cpl_frame_delete(zexpframe);

    sph_filemanager_clean();

    return;
}

static
void cutest_test_zpl_exposure_load(void) {
    sph_zpl_exposure* zexpload = NULL;
    cpl_frame* zexpcube = NULL;

    //create flat image cube
    zexpcube = sph_test_zpl_simulator_create_exposure_cube_frame(100, 100, 5,
            "test_zplexp_cube.fits");

    //load a zpl exposure from zpl exp cube frame
    cpl_test_nonnull( zexpcube );

    //load zpl exposure at the plane number 2 from cube of zpl exposures
    zexpload = sph_zpl_exposure_load(zexpcube, 2);
    cpl_test_nonnull( zexpload );

    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_zero_even ),
                           ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_odd ),
                           PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_even ),
                           PI_EVEN_DOUBLE,
                           1e-20);

    cpl_frame_delete(zexpcube);
    sph_zpl_exposure_delete(zexpload);
    sph_filemanager_clean();

    return;
}

static
void cutest_test_zpl_exposure_frames_double(void) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frameset* zexpframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_propertylist* pl = NULL;
    sph_master_frame* mf_zero_odd = NULL;
    //sph_master_frame*    mf_zero_even= NULL;
    //sph_master_frame*    mf_pi_odd    = NULL;
    //sph_master_frame*    mf_pi_even    = NULL;
    cpl_parameterlist* parlist = NULL;
    cpl_parameter* p = NULL;

    const int nz = 5; //nframes
    const int nx = 100;
    const int ny = 100;
    double zero_odd = 100.0;
    double zero_even = 200.0;
    double pi_odd = 300.0;
    double pi_even = 400.0;
    double dV = 100.0;

    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    zexpcube = sph_test_zpl_simulator_frame_template(
            "test_zplexp_cube_cam1_double.fits");

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_double(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame.fits", "NONE", CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        cpl_test_eq_error(rerr, CPL_ERROR_NONE);
        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(zexpframes, zexpframe);
        sph_zpl_exposure_delete(zplexp);
    }cpl_test_eq(cpl_frameset_get_size( zexpframes ), nz);

    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_double(
            zexpframes, zexpcube);
    //cpl_test_noneq(rerr, CPL_ERROR_NONE);

    parlist = cpl_parameterlist_new();
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_enum("zpl.test.coll_alg", CPL_TYPE_INT,
            "No description", "zpl.test", 1, 3, 0, 1, 2);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_range("zpl.test.coll_alg.clean_mean.reject_high",
            CPL_TYPE_INT, "No description", "zpl.test", 0, 0, 20);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_range("zpl.test.coll_alg.clean_mean.reject_low",
            CPL_TYPE_INT, "No description", "zpl.test", 0, 0, 20);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_range("zpl.test.clean_mean.sigma", CPL_TYPE_DOUBLE,
            "No description", "zpl.test", 5.0, 0.0, 20.);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parlist, p);

    mf_zero_odd =
            sph_framecombination_new_master_frame_from_cpl_frame_multi_ext(
                    zexpcube, 0, SPH_COLL_ALG_MEAN, parlist);
    cpl_test_nonnull( mf_zero_odd );
    sph_master_frame_save(mf_zero_odd, "test_mf_zero_odd.fits", NULL);
    //printf("absflux: %.3f\n", cpl_image_get_absflux( mf_zero_odd->image ));
    cpl_test_abs( cpl_image_get_absflux( mf_zero_odd->image ),
            300.0*nx*ny, 1e-20);

    cpl_frame_delete(zexpcube);
    cpl_frameset_delete(zexpframes);
    cpl_parameterlist_delete(parlist);
    cpl_propertylist_delete(pl);
    sph_master_frame_delete(mf_zero_odd);
    sph_filemanager_clean();

    return;

}

static
void cutest_test_zpl_exposure_frames_int_plus(void) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frameset* zexpframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_propertylist* pl = NULL;
    sph_master_frame* mf_zero_odd = NULL;
    sph_master_frame* mf_zero_even = NULL;
    sph_master_frame* mf_pi_odd = NULL;
    sph_master_frame* mf_pi_even = NULL;
    cpl_parameterlist* parlist = NULL;
    cpl_parameter* p = NULL;

    const int nz = 3; //nframes
    const int nx = 100;
    const int ny = 100;
    int zero_odd = 169;
    int zero_even = 29;
    int pi_odd = 29;
    int pi_even = 169;
    int dV = 1;

    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    zexpcube = sph_test_zpl_simulator_frame_template(
            "test_zplexp_cube_cam1_int_plus.fits");

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_int(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame_cam1_int_plus.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        cpl_test_eq_error(rerr, CPL_ERROR_NONE);
        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(zexpframes, zexpframe);
        sph_zpl_exposure_delete(zplexp);
    }cpl_test_eq(cpl_frameset_get_size( zexpframes ), nz);

    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_int(
            zexpframes, zexpcube);
    //cpl_test_noneq(rerr, CPL_ERROR_NONE);

    parlist = cpl_parameterlist_new();
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_enum("zpl.test.coll_alg", CPL_TYPE_INT,
            "No description", "zpl.test", 1, 3, 0, 1, 2);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_range("zpl.test.coll_alg.clean_mean.reject_high",
            CPL_TYPE_INT, "No description", "zpl.test", 0, 0, 20);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_range("zpl.test.coll_alg.clean_mean.reject_low",
            CPL_TYPE_INT, "No description", "zpl.test", 0, 0, 20);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_range("zpl.test.clean_mean.sigma", CPL_TYPE_DOUBLE,
            "No description", "zpl.test", 5.0, 0.0, 20.);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parlist, p);

    mf_zero_odd =
            sph_framecombination_new_master_frame_from_cpl_frame_multi_ext(
                    zexpcube, 0, SPH_COLL_ALG_MEAN, parlist);
    sph_master_frame_save(mf_zero_odd, "test_mf_zero_odd.fits", NULL);
    //printf("absflux: %.3f\n", cpl_image_get_absflux( mf_zero_odd->image ));
    cpl_test_abs( cpl_image_get_absflux( mf_zero_odd->image ),
            (zero_odd+1)*nx*ny, 1e-20);

    mf_zero_even =
            sph_framecombination_new_master_frame_from_cpl_frame_multi_ext(
                    zexpcube, 1, SPH_COLL_ALG_MEAN, parlist);
    sph_master_frame_save(mf_zero_even, "test_mf_zero_even.fits", NULL);
    //printf("absflux: %.3f\n", cpl_image_get_absflux( mf_zero_odd->image ));
    cpl_test_abs( cpl_image_get_absflux( mf_zero_even->image ),
            (zero_even+1)*nx*ny, 1e-20);

    mf_pi_odd = sph_framecombination_new_master_frame_from_cpl_frame_multi_ext(
            zexpcube, 2, SPH_COLL_ALG_MEAN, parlist);
    sph_master_frame_save(mf_pi_odd, "test_mf_pi_odd.fits", NULL);
    //printf("absflux: %.3f\n", cpl_image_get_absflux( mf_zero_odd->image ));
    cpl_test_abs( cpl_image_get_absflux( mf_pi_odd->image ),
            (pi_odd+1)*nx*ny, 1e-20);

    mf_pi_even = sph_framecombination_new_master_frame_from_cpl_frame_multi_ext(
            zexpcube, 3, SPH_COLL_ALG_MEAN, parlist);
    sph_master_frame_save(mf_pi_even, "test_mf_pi_even.fits", NULL);
    //printf("absflux: %.3f\n", cpl_image_get_absflux( mf_zero_odd->image ));
    cpl_test_abs( cpl_image_get_absflux( mf_pi_even->image ),
            (pi_even+1)*nx*ny, 1e-20);

    cpl_frame_delete(zexpcube);
    cpl_frameset_delete(zexpframes);
    cpl_parameterlist_delete(parlist);
    cpl_propertylist_delete(pl);
    sph_master_frame_delete(mf_zero_odd);
    sph_master_frame_delete(mf_zero_even);
    sph_master_frame_delete(mf_pi_odd);
    sph_master_frame_delete(mf_pi_even);

    sph_filemanager_clean();

    return;

}

static
void cutest_test_zpl_exposure_frames_int_minus(void) {
    sph_zpl_exposure* zplexp = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frameset* zexpframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_propertylist* pl = NULL;
    sph_master_frame* mf_zero_odd = NULL;
    sph_master_frame* mf_zero_even = NULL;
    sph_master_frame* mf_pi_odd = NULL;
    sph_master_frame* mf_pi_even = NULL;
    cpl_parameterlist* parlist = NULL;
    cpl_parameter* p = NULL;

    const int nz = 3; //nframes
    const int nx = 100;
    const int ny = 100;
    int zero_odd = 51;
    int zero_even = 151;
    int pi_odd = 151;
    int pi_even = 51;
    int dV = -1;

    pl = cpl_propertylist_new();
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_NAME,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    zexpcube = sph_test_zpl_simulator_frame_template(
            "test_zplexp_cube_cam1_int_minus.fits");

    zexpframes = cpl_frameset_new();
    for (int i = 0; i < nz; i++) {
        zplexp = sph_test_zpl_simulator_create_exposure_4flat_int(nx, ny,
                zero_odd + i * dV, zero_even + i * dV, pi_odd + i * dV,
                pi_even + i * dV);

        zexpframe = sph_filemanager_create_temp_frame(
                "test_zplexp_single_frame_cam1_int_minus.fits", "NONE",
                CPL_FRAME_GROUP_NONE);
        rerr = sph_zpl_exposure_save(zplexp, zexpframe, NULL);

        cpl_test_eq_error(rerr, CPL_ERROR_NONE);
        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR( "could not save zpl exposure");
            return;
        }
        cpl_frameset_insert(zexpframes, zexpframe);
        sph_zpl_exposure_delete(zplexp);
    }cpl_test_eq(cpl_frameset_get_size( zexpframes ), nz);

    rerr = sph_test_zpl_simulator_create_cube_from_exposure_frames_int(
            zexpframes, zexpcube);
    //cpl_test_noneq(rerr, CPL_ERROR_NONE);

    parlist = cpl_parameterlist_new();
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_enum("zpl.test.coll_alg", CPL_TYPE_INT,
            "No description", "zpl.test", 1, 3, 0, 1, 2);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_range("zpl.test.coll_alg.clean_mean.reject_high",
            CPL_TYPE_INT, "No description", "zpl.test", 0, 0, 20);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_range("zpl.test.coll_alg.clean_mean.reject_low",
            CPL_TYPE_INT, "No description", "zpl.test", 0, 0, 20);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_range("zpl.test.clean_mean.sigma", CPL_TYPE_DOUBLE,
            "No description", "zpl.test", 5.0, 0.0, 20.);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parlist, p);

    mf_zero_odd =
            sph_framecombination_new_master_frame_from_cpl_frame_multi_ext(
                    zexpcube, 0, SPH_COLL_ALG_MEAN, parlist);
    sph_master_frame_save(mf_zero_odd, "test_mf_zero_odd.fits", NULL);
    //printf("absflux: %.3f\n", cpl_image_get_absflux( mf_zero_odd->image ));
    cpl_test_abs( cpl_image_get_absflux( mf_zero_odd->image ),
            (zero_odd-1)*nx*ny, 1e-20);

    mf_zero_even =
            sph_framecombination_new_master_frame_from_cpl_frame_multi_ext(
                    zexpcube, 1, SPH_COLL_ALG_MEAN, parlist);
    sph_master_frame_save(mf_zero_even, "test_mf_zero_even.fits", NULL);
    //printf("absflux: %.3f\n", cpl_image_get_absflux( mf_zero_odd->image ));
    cpl_test_abs( cpl_image_get_absflux( mf_zero_even->image ),
            (zero_even-1)*nx*ny, 1e-20);

    mf_pi_odd = sph_framecombination_new_master_frame_from_cpl_frame_multi_ext(
            zexpcube, 2, SPH_COLL_ALG_MEAN, parlist);
    sph_master_frame_save(mf_pi_odd, "test_mf_pi_odd.fits", NULL);
    //printf("absflux: %.3f\n", cpl_image_get_absflux( mf_zero_odd->image ));
    cpl_test_abs( cpl_image_get_absflux( mf_pi_odd->image ),
            (pi_odd-1)*nx*ny, 1e-20);

    mf_pi_even = sph_framecombination_new_master_frame_from_cpl_frame_multi_ext(
            zexpcube, 3, SPH_COLL_ALG_MEAN, parlist);
    sph_master_frame_save(mf_pi_even, "test_mf_pi_even.fits", NULL);
    //printf("absflux: %.3f\n", cpl_image_get_absflux( mf_zero_odd->image ));
    cpl_test_abs( cpl_image_get_absflux( mf_pi_even->image ),
            (pi_even-1)*nx*ny, 1e-20);

    cpl_frame_delete(zexpcube);
    cpl_frameset_delete(zexpframes);
    cpl_parameterlist_delete(parlist);
    cpl_propertylist_delete(pl);
    sph_master_frame_delete(mf_zero_odd);
    sph_master_frame_delete(mf_zero_even);
    sph_master_frame_delete(mf_pi_odd);
    sph_master_frame_delete(mf_pi_even);

    sph_filemanager_clean();
    return;

}

static
void cutest_test_zpl_exposure_new_from_images(void) {
    sph_zpl_exposure* zplexp = NULL;
    sph_zpl_exposure* zexpload = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_image* image_zero_odd = NULL;
    cpl_image* image_zero_even = NULL;
    cpl_image* image_pi_odd = NULL;
    cpl_image* image_pi_even = NULL;

    image_zero_odd = sph_test_image_tools_create_flat_image_double(100, 100,
            10.0);
    image_zero_even = sph_test_image_tools_create_flat_image_double(100, 100,
            11.0);
    image_pi_odd = sph_test_image_tools_create_flat_image_double(100, 100,
            20.0);
    image_pi_even = sph_test_image_tools_create_flat_image_double(100, 100,
            21.0);

    zplexp = sph_zpl_exposure_new_from_cplimages(image_zero_odd,
            image_zero_even, image_pi_odd, image_pi_even);
    cpl_image_delete(image_zero_odd);
    cpl_image_delete(image_zero_even);
    cpl_image_delete(image_pi_odd);
    cpl_image_delete(image_pi_even);

    cpl_test_nonnull( zplexp );
    cpl_test_abs( cpl_image_get_mean( zplexp->image_zero_odd ), 10.0,
                            1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_zero_even ),
                           11.0, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_odd ),
                           20.0, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_even ),
                           21.0, 1e-20);

    // save zpl exp frame
    zexpframe = sph_filemanager_create_temp_frame("test_zplexp.fits", "PREPROC",
            CPL_FRAME_GROUP_NONE);
    sph_zpl_exposure_save(zplexp, zexpframe, NULL);
    //load a zpl exposure from pure zpl exp frame
    zexpload = sph_zpl_exposure_load(zexpframe, 0);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_odd ),
                            10.0, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_zero_even ),
                           11.0, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_odd ),
                           20.0, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_even ),
                           21.0, 1e-20);

    cpl_frame_delete(zexpframe);
    sph_zpl_exposure_delete(zplexp);
    sph_zpl_exposure_delete(zexpload);
    sph_filemanager_clean();

    return;
}

static
void cutest_test_zpl_exposure_load_with_head(void) {
    sph_zpl_exposure* zexpload = NULL;
    sph_zpl_exposure* zexptest = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* test_frame = NULL;
    sph_keyword_manager* keyman = NULL;

    //create config
    keyman = cutest_zpl_exposure_create_cfg_file();

    cpl_test_nonnull( keyman );

    //create flat image cube
    zexpcube = sph_test_zpl_simulator_create_exposure_cube_frame(100, 100, 5,
            "test_zplexp_cube.fits");

    //create .head file
    cutest_zpl_exposure_create_head_file("test_zplexp_cube");

    //load a zpl exposure from zpl exp cube frame
    cpl_test_nonnull( zexpcube );

    //load zpl exposure at the plane number 2 from cube of zpl exposures
    zexpload = sph_zpl_exposure_load(zexpcube, 2);
    cpl_test_nonnull( zexpload );

    //zexpload->properties = sph_keyword_manager_load_properties("test_zplexp_cube.fits",0);
    cpl_test(
            cpl_propertylist_has( zexpload->properties, "ESO DRS ZPL HWPZ SETTING"));
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->properties, "ESO DRS ZPL HWPZ SETTING"),
            48.5, 0.001);

    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_odd ),
            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(
                           cpl_image_get_mean( zexpload->image_zero_even ), ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(
                           cpl_image_get_mean( zexpload->image_pi_odd ), PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(
                           cpl_image_get_mean( zexpload->image_pi_even ), PI_EVEN_DOUBLE,
                           1e-20);

    test_frame = sph_filemanager_create_temp_frame("test_zplexp.fits.tmp",
            "NONE", CPL_FRAME_GROUP_NONE);
    sph_zpl_exposure_save(zexpload, test_frame, NULL);
    unlink("test_zplexp_cube.fits");
    unlink("test_zplexp_cube.head");
    zexptest = sph_zpl_exposure_load(test_frame, 0);
    cpl_test(
            cpl_propertylist_has( zexptest->properties, "ESO DRS ZPL HWPZ SETTING"));
    cpl_test_abs(
            cpl_propertylist_get_double( zexptest->properties, "ESO DRS ZPL HWPZ SETTING"),
            48.5, 0.001);

    cpl_frame_delete(zexpcube);
    sph_zpl_exposure_delete(zexpload);
    cpl_frame_delete(test_frame);
    sph_zpl_exposure_delete(zexptest);

    sph_keyword_manager_delete();
    return;
}

static
void cutest_test_zpl_exposure_save_load_with_ovsc_table(void) {
    sph_zpl_exposure* zplexp = NULL;
    sph_zpl_exposure* zexpload = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_propertylist* ovsc_pl = NULL;
    double a[8] = { 100.0, 1.0, 200.0, 2.0, 300.0, 3.0, 400.0, 4.0 };

    zplexp = sph_test_zpl_simulator_create_exposure(100, 100);

    /* internal test of ph_test_zpl_simulator_create_exposure & _4flat */
    cpl_test_abs( cpl_image_get_mean( zplexp->image_zero_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_zero_even ),
                           ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_odd ),
                           PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_even ),
                           PI_EVEN_DOUBLE, 1e-20);

    // save zpl exp frame
    zexpframe = sph_filemanager_create_temp_frame("test_zplexp_ovsc_table.fits",
            "PREPROC", CPL_FRAME_GROUP_NONE);
    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist(a);
    if (zplexp->ovsc)
        cpl_propertylist_delete(zplexp->ovsc);
    zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);
    cpl_propertylist_delete(ovsc_pl);
    sph_zpl_exposure_save(zplexp, zexpframe, NULL);

    //load a zpl exposure from pure zpl exp frame
    zexpload = sph_zpl_exposure_load(zexpframe, 0);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_zero_even ),
                           ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_odd ),
                           PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_even ),
                           PI_EVEN_DOUBLE,
                           1e-20);

    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_ZERO_OVSC_MEAN ),
            a[0], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_ZERO_OVSC_RMS ),
            a[1], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_ZERO_OVSC_MEAN ),
            a[2], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_ZERO_OVSC_RMS ),
            a[3], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_PI_OVSC_MEAN ),
            a[4], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_PI_OVSC_RMS ),
            a[5], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_PI_OVSC_MEAN ),
            a[6], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_PI_OVSC_RMS ),
            a[7], 1e-20);
    sph_zpl_exposure_delete(zplexp);
    sph_zpl_exposure_delete(zexpload);
    cpl_frame_delete(zexpframe);
    sph_filemanager_clean();

    return;

}

static
void cutest_test_zpl_exposure_duplicate_with_ovsc_table(void) {
    sph_zpl_exposure* zplexp = NULL;
    sph_zpl_exposure* zexpload = NULL;
    sph_zpl_exposure* zexpdupl = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_propertylist* ovsc_pl = NULL;
    double a[8] = { 100.0, 1.0, 200.0, 2.0, 300.0, 3.0, 400.0, 4.0 };

    zplexp = sph_test_zpl_simulator_create_exposure(100, 100);

    /* internal test of ph_test_zpl_simulator_create_exposure & _4flat */
    cpl_test_abs( cpl_image_get_mean( zplexp->image_zero_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_zero_even ),
                           ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_odd ),
                           PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_even ),
                           PI_EVEN_DOUBLE, 1e-20);

    // save zpl exp frame
    zexpframe = sph_filemanager_create_temp_frame("test_zplexp_ovsc_table.fits",
            "PREPROC", CPL_FRAME_GROUP_NONE);
    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist(a);
    if (zplexp->ovsc)
        cpl_propertylist_delete(zplexp->ovsc);
    zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);
    cpl_propertylist_delete(ovsc_pl);
    sph_zpl_exposure_save(zplexp, zexpframe, NULL);

    //load a zpl exposure from pure zpl exp frame
    zexpload = sph_zpl_exposure_load(zexpframe, 0);
    zexpdupl = sph_zpl_exposure_duplicate(zexpload);
    cpl_test_abs( cpl_image_get_mean( zexpdupl->image_zero_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpdupl->image_zero_even ),
                           ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpdupl->image_pi_odd ),
                           PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zexpdupl->image_pi_even ),
                           PI_EVEN_DOUBLE,
                           1e-20);

    cpl_test_abs(
            cpl_propertylist_get_double( zexpdupl->ovsc, ADU1_ZERO_OVSC_MEAN ),
            a[0], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpdupl->ovsc, ADU1_ZERO_OVSC_RMS ),
            a[1], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpdupl->ovsc, ADU2_ZERO_OVSC_MEAN ),
            a[2], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpdupl->ovsc, ADU2_ZERO_OVSC_RMS ),
            a[3], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpdupl->ovsc, ADU1_PI_OVSC_MEAN ),
            a[4], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpdupl->ovsc, ADU1_PI_OVSC_RMS ),
            a[5], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpdupl->ovsc, ADU2_PI_OVSC_MEAN ),
            a[6], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpdupl->ovsc, ADU2_PI_OVSC_RMS ),
            a[7], 1e-20);

    sph_zpl_exposure_delete(zplexp);
    sph_zpl_exposure_delete(zexpload);
    sph_zpl_exposure_delete(zexpdupl);
    cpl_frame_delete(zexpframe);

    sph_filemanager_clean();

    return;

}

static
void cutest_test_zpl_exposure_subtract_overscans(void) {
    sph_zpl_exposure* zplexp = NULL;
    sph_zpl_exposure* zexpload = NULL;
    cpl_frame* zexpframe = NULL;
    cpl_propertylist* ovsc_pl = NULL;
    double a[8] = { 10.0, 1.0, 20.0, 2.0, 30.0, 3.0, 40.0, 4.0 };

    zplexp = sph_test_zpl_simulator_create_exposure(100, 100);

    /* internal test of ph_test_zpl_simulator_create_exposure & _4flat */
    cpl_test_abs( cpl_image_get_mean( zplexp->image_zero_odd ),
                            ZERO_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_zero_even ),
                           ZERO_EVEN_DOUBLE,
                           1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_odd ),
                           PI_ODD_DOUBLE, 1e-20);
    cpl_test_abs(cpl_image_get_mean( zplexp->image_pi_even ),
                           PI_EVEN_DOUBLE, 1e-20);

    // save zpl exp frame
    zexpframe = sph_filemanager_create_temp_frame(
            "test_zplexp_subtract_ovescans.fits", "PREPROC",
            CPL_FRAME_GROUP_NONE);
    ovsc_pl = sph_test_zpl_simulator_create_ovsc_propertylist(a);
    if (zplexp->ovsc)
        cpl_propertylist_delete(zplexp->ovsc);
    zplexp->ovsc = cpl_propertylist_duplicate(ovsc_pl);
    cpl_propertylist_delete(ovsc_pl);

    //subtract overscans & save
    sph_zpl_exposure_subtract_overscans(zplexp);
    sph_zpl_exposure_save(zplexp, zexpframe, NULL);

    //load a zpl exposure from pure zpl exp frame
    zexpload = sph_zpl_exposure_load(zexpframe, 0);

    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_odd ),
            (ZERO_ODD_DOUBLE - a[0] + ZERO_ODD_DOUBLE - a[2])/2.0, 1e-10);
    cpl_test_abs( cpl_image_get_mean( zexpload->image_zero_even ),
                            (ZERO_EVEN_DOUBLE - a[0] + ZERO_EVEN_DOUBLE - a[2] )/2.0,
                            1e-10);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_odd ),
                           (PI_ODD_DOUBLE - a[4] + PI_ODD_DOUBLE - a[6])/2.0, 1e-10);
    cpl_test_abs(cpl_image_get_mean( zexpload->image_pi_even ),
                           (PI_EVEN_DOUBLE - a[4] + PI_EVEN_DOUBLE - a[6])/2.0, 1e-10);

    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_ZERO_OVSC_MEAN ),
            a[0], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_ZERO_OVSC_RMS ),
            a[1], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_ZERO_OVSC_MEAN ),
            a[2], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_ZERO_OVSC_RMS ),
            a[3], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_PI_OVSC_MEAN ),
            a[4], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU1_PI_OVSC_RMS ),
            a[5], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_PI_OVSC_MEAN ),
            a[6], 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( zexpload->ovsc, ADU2_PI_OVSC_RMS ),
            a[7], 1e-20);
    sph_zpl_exposure_delete(zplexp);
    sph_zpl_exposure_delete(zexpload);
    cpl_frame_delete(zexpframe);

    sph_filemanager_clean();

    return;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    (void)sph_test_init();

    cpl_test_nonnull(sph_keyword_manager_set_cfg(SPH_CFG));

    pSuite = sph_add_suite("cutest_zpl_exposure",
            cutest_init_zpl_exposure_testsuite,
            cutest_clean_zpl_exposure_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL
            == sph_test_do(pSuite, "sph_zpl_exposure_new_empty",
                    cutest_test_zpl_exposure_new_empty)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_zpl_exposure_new",
                    cutest_test_zpl_exposure_new)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_zpl_exposure_save",
                    cutest_test_zpl_exposure_save)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_zpl_exposure_load",
                    cutest_test_zpl_exposure_load)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_zpl_exposure_frames_double",
                    cutest_test_zpl_exposure_frames_double)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_zpl_exposure_frames_int_plus",
                    cutest_test_zpl_exposure_frames_int_plus)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_zpl_exposure_frames_int_minus",
                    cutest_test_zpl_exposure_frames_int_minus)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_test_zpl_exposure_new_from_images",
                    cutest_test_zpl_exposure_new_from_images)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_zpl_exposure_save_append",
                    cutest_test_zpl_exposure_save_append)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_zpl_exposure_finalize with ovsc_table",
                    cutest_test_zpl_exposure_finalize_with_ovsc_table)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_zpl_exposure_load_with_head",
                    cutest_test_zpl_exposure_load_with_head)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_zpl_exposure_save_load_with_ovsc_table",
                    cutest_test_zpl_exposure_save_load_with_ovsc_table)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_zpl_exposure_duplicate_with_ovsc_table",
                    cutest_test_zpl_exposure_duplicate_with_ovsc_table)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_zpl_exposure_subtract_overscans",
                    cutest_test_zpl_exposure_subtract_overscans)) {
        return sph_test_get_error();
    }

    cpl_test_zero(unlink(SPH_CFG));

    return sph_test_end();
}
