/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ifs_instrument_model.h"


#include "sph_test.h"
#include "sph_utils.h"
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include "sph_framecombination.h"
#include "sph_utils.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_keyword_manager.h"
#include "sph_test_zpl_simulator.h"
#include "sph_triple_image.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup Requirement Test: sph_framecombination
 *
 * The requirement tests for the framecombination module
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static
int reqtest_init_framecombination_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

static
int reqtest_clean_framecobinatin_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    sph_end_test();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief
 */
/*----------------------------------------------------------------------------*/
static
void reqtest_sph_framecombination(void) {
    int rerr = CPL_ERROR_NONE;
    sph_test_pupilimage_creator* pupil_creator = NULL;
    gsl_rng* pRNG = NULL;
    sph_framecombination* combiner = NULL;
    cpl_image* im = NULL;
    cpl_image* im0 = NULL;
    cpl_image* imrot = NULL;
    int ii = 0;
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    cpl_test_nonnull( pRNG );
    pupil_creator = sph_test_pupilimage_creator_new();
    cpl_test_nonnull( pupil_creator );
    pupil_creator->nx = 64;
    pupil_creator->ny = 64;
    sph_test_pupilimage_creator_create_binary_star_frames(pupil_creator, 10.0,
            20.0, 5.0, 100.0, 2.0, gsl_ran_gaussian(pRNG, 0.01) + 3.0, 1);
    //sph_test_pupilimage_creator_add_speckles( pupil_creator, 10, 10.0, pRNG );
    rerr = cpl_error_get_code();
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    combiner = sph_framecombination_new();
    cpl_test_nonnull( combiner );
    sph_framecombination_set_method_gimros(combiner);
    im = cpl_image_load(
            cpl_frame_get_filename(
                    cpl_frameset_get_first(
                            sph_test_pupilimage_creator_get_frameset(
                                    pupil_creator))), CPL_TYPE_DOUBLE, 0, 0);
    cpl_test_nonnull( im );
    im0 = cpl_image_duplicate(im);
    cpl_test_nonnull( im0 );
    for (ii = 0; ii < 3; ++ii) {
        imrot = sph_framecombination_rotate_image(combiner, im, 120.0);
        cpl_test_nonnull( imrot );
        cpl_image_delete(im);
        im = imrot;
    }
    cpl_image_save(imrot, "rotated_image.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(im0, "rotated_image.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(im0, imrot);
    cpl_image_save(im0, "rotated_image.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    /* Setup ...*/
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(im0);
    im0 = NULL;
    sph_framecombination_delete(combiner);
    combiner = NULL;
    gsl_rng_free(pRNG);
    sph_test_pupilimage_creator_delete(pupil_creator);

    /*Verify */
    return;
}
static
void reqtest_sph_framecombination_cplwarp(void) {
    int rerr = CPL_ERROR_NONE;
    sph_test_pupilimage_creator* pupil_creator = NULL;
    gsl_rng* pRNG = NULL;
    sph_framecombination* combiner = NULL;
    cpl_image* im = NULL;
    cpl_image* im0 = NULL;
    cpl_image* imrot = NULL;
    int ii = 0;
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    cpl_test_nonnull( pRNG );
    pupil_creator = sph_test_pupilimage_creator_new();
    cpl_test_nonnull( pupil_creator );
    pupil_creator->nx = 64;
    pupil_creator->ny = 64;
    sph_test_pupilimage_creator_create_binary_star_frames(pupil_creator, 10.0,
            20.0, 5.0, 100.0, 2.0, gsl_ran_gaussian(pRNG, 0.01) + 3.0, 1);
    //sph_test_pupilimage_creator_add_speckles( pupil_creator, 10, 10.0, pRNG );
    rerr = cpl_error_get_code();
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    combiner = sph_framecombination_new();
    cpl_test_nonnull( combiner );
    sph_framecombination_set_method_cplwarp(combiner, CPL_KERNEL_TANH);
    im = cpl_image_load(
            cpl_frame_get_filename(
                    cpl_frameset_get_first(
                            sph_test_pupilimage_creator_get_frameset(
                                    pupil_creator))), CPL_TYPE_DOUBLE, 0, 0);
    cpl_test_nonnull( im );
    im0 = cpl_image_duplicate(im);
    cpl_test_nonnull( im0 );
    for (ii = 0; ii < 360; ++ii) {
        imrot = sph_framecombination_rotate_image(combiner, im, 1.0);
        cpl_test_nonnull( imrot );
        cpl_image_delete(im);
        im = imrot;
    }
    cpl_image_save(imrot, "rotated_image_cplwarp.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(im0, "rotated_image_cplwarp.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(im0, imrot);
    cpl_image_save(im0, "rotated_image_cplwarp.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    /* Setup ...*/
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(im0);
    im0 = NULL;
    sph_framecombination_delete(combiner);
    combiner = NULL;
    gsl_rng_free(pRNG);
    sph_test_pupilimage_creator_delete(pupil_creator);

    /*Verify */
    return;
}
static
void reqtest_sph_framecombination_gimros(void) {
    int rerr = CPL_ERROR_NONE;
    sph_test_pupilimage_creator* pupil_creator = NULL;
    gsl_rng* pRNG = NULL;
    sph_framecombination* combiner = NULL;
    cpl_image* im = NULL;
    cpl_image* im0 = NULL;
    cpl_image* imrot = NULL;
    int ii = 0;
    cpl_error_reset();
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    cpl_test_nonnull( pRNG );
    pupil_creator = sph_test_pupilimage_creator_new();
    cpl_test_nonnull( pupil_creator );
    pupil_creator->nx = 64;
    pupil_creator->ny = 64;
    sph_test_pupilimage_creator_create_binary_star_frames(pupil_creator, 10.0,
            20.0, 5.0, 100.0, 2.0, gsl_ran_gaussian(pRNG, 0.01) + 3.0, 1);
    rerr = cpl_error_get_code();
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    combiner = sph_framecombination_new();
    cpl_test_nonnull( combiner );
    sph_framecombination_set_method_gimros(combiner);
    im = cpl_image_load(
            cpl_frame_get_filename(
                    cpl_frameset_get_first(
                            sph_test_pupilimage_creator_get_frameset(
                                    pupil_creator))), CPL_TYPE_DOUBLE, 0, 0);
    cpl_test_nonnull( im );
    im0 = cpl_image_duplicate(im);
    cpl_test_nonnull( im0 );
    for (ii = 0; ii < 4; ++ii) {
        imrot = sph_framecombination_rotate_image(combiner, im, 90.0);
        cpl_test_nonnull( imrot );
        cpl_image_delete(im);
        im = imrot;
    }
    cpl_image_save(imrot, "rotated_image_gimros.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(im0, "rotated_image_gimros.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(im0, imrot);
    cpl_image_save(im0, "rotated_image_gimros.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    /* Setup ...*/
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(im0);
    im0 = NULL;
    sph_framecombination_delete(combiner);
    combiner = NULL;

    /*Verify */
    sph_test_pupilimage_creator_delete(pupil_creator);
    gsl_rng_free(pRNG);
    return;
}
static
void reqtest_sph_framecombination_gslfft_null_rot(void) {
    sph_framecombination* combiner = NULL;
    cpl_image* im0 = NULL;
    cpl_image* imrot = NULL;
    cpl_error_reset();

    im0 = sph_test_image_tools_create_flat_image_double(256, 256, 0.0);
    cpl_image_fill_gaussian(im0, 220.0, 127.5, 1000.0, 5.0, 5.0);
    combiner = sph_framecombination_new();
    cpl_test_nonnull( combiner );
    sph_framecombination_set_method_gslfft(combiner, 0.0);
    imrot = sph_framecombination_rotate_image(combiner, im0, 0.0);
    cpl_image_save(im0, "rotated_image_gslfft_null_rot.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_DEFAULT);
    cpl_image_save(imrot, "rotated_image_gslfft_null_rot.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_EXTEND);
    cpl_image_subtract(im0, imrot);
    cpl_image_save(im0, "rotated_image_gslfft_null_rot.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_EXTEND);
    /* Setup ...*/
    cpl_test_lt(cpl_image_get_max(im0), 0.01);
    cpl_test_lt(fabs(cpl_image_get_min(im0)), 0.01);
    cpl_test_lt(fabs(cpl_image_get_mean(im0)), 0.01);

    cpl_image_delete(imrot);
    imrot = NULL;
    cpl_image_delete(im0);
    im0 = NULL;
    sph_framecombination_delete(combiner);
    combiner = NULL;

    /*Verify */
    return;
}
static
void reqtest_sph_framecombination_gslfft_sense_of_rot(void) {
    sph_framecombination* combiner = NULL;
    cpl_image* im = NULL;
    cpl_image* im0 = NULL;
    cpl_image* imrot = NULL;
    cpl_error_reset();

    im0 = sph_test_image_tools_create_flat_image_double(256, 256, 0.0);
    cpl_image_fill_gaussian(im0, 220.0, 128.5, 1000.0, 5.0, 5.0);
    im = sph_test_image_tools_create_flat_image_double(256, 256, 0.0);
    cpl_image_fill_gaussian(im, 128.5, 220.0, 1000.0, 5.0, 5.0);
    combiner = sph_framecombination_new();
    cpl_test_nonnull( combiner );
    sph_framecombination_set_method_gslfft(combiner, 0.0);
    imrot = sph_framecombination_rotate_image(combiner, im, -89.9999);
    cpl_image_save(im, "rotated_image_gslfft_rot.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(im0, "rotated_image_gslfft_rot.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_save(imrot, "rotated_image_gslfft_rot.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_EXTEND);
    cpl_image_subtract(im0, imrot);
    cpl_image_save(im0, "rotated_image_gslfft_rot.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    /* Setup ...*/
    cpl_test_lt(cpl_image_get_max(im0), 0.01);
    cpl_test_lt(fabs(cpl_image_get_min(im0)), 0.01);
    cpl_test_lt(fabs(cpl_image_get_mean(im0)), 0.01);
    cpl_image_delete(imrot);
    imrot = NULL;
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(im0);
    im0 = NULL;
    sph_framecombination_delete(combiner);
    combiner = NULL;
    /*Verify */
    return;
}
static
void reqtest_sph_framecombination_gslfft(void) {
    int rerr = CPL_ERROR_NONE;
    sph_test_pupilimage_creator* pupil_creator = NULL;
    gsl_rng* pRNG = NULL;
    sph_framecombination* combiner = NULL;
    cpl_image* im = NULL;
    cpl_image* im0 = NULL;
    cpl_image* imrot = NULL;
    int ii = 0;
    cpl_error_reset();
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    cpl_test_nonnull( pRNG );
    pupil_creator = sph_test_pupilimage_creator_new();
    cpl_test_nonnull( pupil_creator );
    pupil_creator->nx = 64;
    pupil_creator->ny = 64;
    sph_test_pupilimage_creator_create_binary_star_frames(pupil_creator, 10.0,
            20.0, 5.0, 100.0, 2.0, gsl_ran_gaussian(pRNG, 0.01) + 3.0, 1);
    rerr = cpl_error_get_code();
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    combiner = sph_framecombination_new();
    cpl_test_nonnull( combiner );
    sph_framecombination_set_method_gslfft(combiner, 0.0);
    im = cpl_image_load(
            cpl_frame_get_filename(
                    cpl_frameset_get_first(
                            sph_test_pupilimage_creator_get_frameset(
                                    pupil_creator))), CPL_TYPE_DOUBLE, 0, 0);
    cpl_test_nonnull( im );
    im0 = cpl_image_duplicate(im);
    cpl_test_nonnull( im0 );
    for (ii = 0; ii < 360; ++ii) {
        imrot = sph_framecombination_rotate_image(combiner, im, 1.0);
        cpl_test_nonnull( imrot );
        cpl_image_delete(im);
        im = imrot;
    }
    cpl_image_save(imrot, "rotated_image_gslfft.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(im0, "rotated_image_gslfft.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(im0, imrot);
    cpl_image_save(im0, "rotated_image_gslfft.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    /* Setup ...*/
    cpl_image_delete(im0);
    im0 = NULL;
    cpl_image_delete(im);
    im = NULL;
    sph_framecombination_delete(combiner);
    combiner = NULL;
    gsl_rng_free(pRNG);
    sph_test_pupilimage_creator_delete(pupil_creator);
    /*Verify */
    return;
}

static
void reqtest_sph_framecombination_gslfft_combine_with_filter(void) {
    gsl_rng* pRNG = NULL;
    sph_test_pupilimage_creator* pupil_creator = NULL;
    sph_framecombination* combiner = NULL;
    cpl_image* im = NULL;
    cpl_frameset* frames = NULL;
    cpl_parameterlist* colparams = NULL;
    sph_master_frame* result = NULL;
    cpl_frame* resultframe = NULL;
    cpl_frame* aframe = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int ii = 0;

    cpl_error_reset();
    pupil_creator = sph_test_pupilimage_creator_new();
    cpl_test_nonnull( pupil_creator );
    pupil_creator->nx = 64;
    pupil_creator->ny = 64;
    sph_test_pupilimage_creator_create_binary_star_frames(pupil_creator, 10.0,
            45.0, 10.0, 100.0, 2.0, 3.0, 18);

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    cpl_test_nonnull( pRNG );

    rerr = cpl_error_get_code();
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    combiner = sph_framecombination_new();
    cpl_test_nonnull( combiner );

    sph_framecombination_set_method_gslfft(combiner, 0.99);

    frames = sph_test_pupilimage_creator_get_frameset(pupil_creator);
    cpl_test_nonnull( frames );
    for (ii = 0; ii < cpl_frameset_get_size(frames); ++ii) {
        aframe = cpl_frameset_get_position(frames, ii);
        im = cpl_image_new(pupil_creator->nx, pupil_creator->ny,
                CPL_TYPE_DOUBLE);
        sph_test_image_tools_add_noise(im, 100.0, pRNG);
        pl = sph_keyword_manager_load_properties(cpl_frame_get_filename(aframe),
                0);
        cpl_image_save(im, cpl_frame_get_filename(aframe), CPL_TYPE_DOUBLE, pl,
                CPL_IO_DEFAULT);
        cpl_propertylist_delete(pl);
        cpl_image_delete(im);
    }

    colparams = cpl_parameterlist_new();
    cpl_test_nonnull( colparams );

    rerr = cpl_parameterlist_append(
            colparams,
            cpl_parameter_new_range("clean_mean.reject_low", CPL_TYPE_INT,
                    "none", "none", 0, 0, 100));
    rerr += cpl_parameterlist_append(
            colparams,
            cpl_parameter_new_range("clean_mean.reject_high", CPL_TYPE_INT,
                    "none", "none", 0, 0, 100));

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    result = sph_framecombination_combine_master_frames_overwrite(combiner,
            frames, SPH_COLL_ALG_CLEAN_MEAN, colparams);
    /*Verify */
    cpl_test_nonnull( result );
    resultframe = sph_filemanager_create_temp_frame(
            "framecomb_combine_masterframe_filter.fits", "TEST",
            CPL_FRAME_GROUP_CALIB);
    cpl_test_nonnull( resultframe );
    sph_master_frame_save(result, cpl_frame_get_filename(resultframe), NULL);

    gsl_rng_free(pRNG);
    sph_framecombination_delete(combiner);
    sph_master_frame_delete(result);
    cpl_frame_delete(resultframe);
    cpl_parameterlist_delete(colparams);
    sph_test_pupilimage_creator_delete(pupil_creator);
    return;
}

static
void reqtest_sph_framecombination_gslfft_centre_odd_size_frame(void) {
    int rerr = CPL_ERROR_NONE;
    sph_test_pupilimage_creator* pupil_creator = NULL;
    gsl_rng* pRNG = NULL;
    sph_framecombination* combiner = NULL;
    cpl_image* im = NULL;
    cpl_image* im0 = NULL;
    cpl_image* imrot = NULL;
    cpl_image* imrotback = NULL;
    double xcent = 0.0;
    double ycent = 0.0;
    cpl_error_reset();
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    cpl_test_nonnull( pRNG );
    pupil_creator = sph_test_pupilimage_creator_new();
    cpl_test_nonnull( pupil_creator );
    pupil_creator->nx = 65;
    pupil_creator->ny = 65;
    sph_test_pupilimage_creator_create_binary_star_frames(pupil_creator, 10.0,
            20.0, 5.0, 100.0, 1.0e99, 3.0, 1);
    rerr = cpl_error_get_code();
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    combiner = sph_framecombination_new();
    cpl_test_nonnull( combiner );
    sph_framecombination_set_method_gslfft(combiner, 0.0);
    im = cpl_image_load(
            cpl_frame_get_filename(
                    cpl_frameset_get_first(
                            sph_test_pupilimage_creator_get_frameset(
                                    pupil_creator))), CPL_TYPE_DOUBLE, 0, 0);
    cpl_test_nonnull( im );
    im0 = cpl_image_duplicate(im);
    cpl_test_nonnull( im0 );
    imrot = sph_framecombination_rotate_image(combiner, im, 45.0);
    cpl_test_nonnull( imrot );
    cpl_image_delete(im);
    im = imrot;
    cpl_image_save(imrot, "rotated_image_gslfft_odd_size.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_DEFAULT);
    xcent = cpl_image_get_centroid_x(imrot);
    ycent = cpl_image_get_centroid_y(imrot);
    //printf( "Centre of rotated image: %f, %f\n", xcent, ycent);
    cpl_image_save(im0, "rotated_image_gslfft_odd_size.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_EXTEND);
    xcent = cpl_image_get_centroid_x(im0);
    ycent = cpl_image_get_centroid_y(im0);
    //printf( "Centre of unrotated image: %f, %f\n", xcent, ycent);
    cpl_image_subtract(im0, imrot);
    cpl_image_save(im0, "rotated_image_gslfft_odd_size.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_EXTEND);
    imrotback = sph_framecombination_rotate_image(combiner, imrot, -45.0);
    cpl_image_save(imrotback, "rotated_image_gslfft_odd_size.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    xcent = cpl_image_get_centroid_x(imrotback);
    ycent = cpl_image_get_centroid_y(imrotback);
    //printf( "Centre of back rotated image: %f, %f\n", xcent, ycent);

    /*Verify */
    gsl_rng_free(pRNG);
    cpl_image_delete(im0);
    im0 = NULL;
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(imrotback);
    imrotback = NULL;
    sph_framecombination_delete(combiner);
    sph_test_pupilimage_creator_delete(pupil_creator);
    return;
}
static
void reqtest_sph_framecombination_gslfft_centre_even_size_frame(void) {
    int rerr = CPL_ERROR_NONE;
    sph_test_pupilimage_creator* pupil_creator = NULL;
    gsl_rng* pRNG = NULL;
    sph_framecombination* combiner = NULL;
    cpl_image* im = NULL;
    cpl_image* im0 = NULL;
    cpl_image* imrot = NULL;
    cpl_image* imrotback = NULL;
    double xcent = 0.0;
    double ycent = 0.0;
    cpl_error_reset();
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    cpl_test_nonnull( pRNG );
    pupil_creator = sph_test_pupilimage_creator_new();
    cpl_test_nonnull( pupil_creator );
    pupil_creator->nx = 64;
    pupil_creator->ny = 64;
    sph_test_pupilimage_creator_create_binary_star_frames(pupil_creator, 10.0,
            20.0, 5.0, 100.0, 1.0e99, 3.0, 1);
    rerr = cpl_error_get_code();
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    combiner = sph_framecombination_new();
    cpl_test_nonnull( combiner );
    sph_framecombination_set_method_gslfft(combiner, 0.0);
    im = cpl_image_load(
            cpl_frame_get_filename(
                    cpl_frameset_get_first(
                            sph_test_pupilimage_creator_get_frameset(
                                    pupil_creator))), CPL_TYPE_DOUBLE, 0, 0);
    cpl_test_nonnull( im );
    im0 = cpl_image_duplicate(im);
    cpl_test_nonnull( im0 );
    imrot = sph_framecombination_rotate_image(combiner, im, 90.0);
    cpl_test_nonnull( imrot );
    cpl_image_delete(im);
    im = imrot;
    cpl_image_save(imrot, "rotated_image_gslfft_even_size.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
    xcent = cpl_image_get_centroid_x(imrot);
    ycent = cpl_image_get_centroid_y(imrot);
    //printf( "Centre of rotated image: %f, %f\n", xcent, ycent);
    cpl_image_save(im0, "rotated_image_gslfft_even_size.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_EXTEND);
    xcent = cpl_image_get_centroid_x(im0);
    ycent = cpl_image_get_centroid_y(im0);
    //printf( "Centre of unrotated image: %f, %f\n", xcent, ycent);
    cpl_image_subtract(im0, imrot);
    cpl_image_save(im0, "rotated_image_gslfft_even_size.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_EXTEND);
    imrotback = sph_framecombination_rotate_image(combiner, imrot, -90.0);
    cpl_image_save(imrotback, "rotated_image_gslfft_even_size.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    xcent = cpl_image_get_centroid_x(imrotback);
    ycent = cpl_image_get_centroid_y(imrotback);
    //printf( "Centre of back rotated image: %f, %f\n", xcent, ycent);

    /* Setup ...*/

    gsl_rng_free(pRNG);
    cpl_image_delete(imrot);
    imrot = NULL;
    cpl_image_delete(im0);
    im0 = NULL;
    cpl_image_delete(imrotback);
    imrotback = NULL;
    sph_framecombination_delete(combiner);
    sph_test_pupilimage_creator_delete(pupil_creator);

    /*Verify */
    return;
}
static
void reqtest_sph_framecombination_gslfft_centre_shift_even_size_frame(void) {
    int rerr = CPL_ERROR_NONE;
    sph_test_pupilimage_creator* pupil_creator = NULL;
    gsl_rng* pRNG = NULL;
    sph_framecombination* combiner = NULL;
    cpl_image* im = NULL;
    cpl_image* im0 = NULL;
    cpl_image* imrot = NULL;
    cpl_image* imrotback = NULL;
    double xcent = 0.0;
    double ycent = 0.0;
    cpl_error_reset();
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    cpl_test_nonnull( pRNG );
    pupil_creator = sph_test_pupilimage_creator_new();
    cpl_test_nonnull( pupil_creator );
    pupil_creator->nx = 64;
    pupil_creator->ny = 64;
    sph_test_pupilimage_creator_create_binary_star_frames(pupil_creator, 10.0,
            20.0, 5.0, 100.0, 1.0e99, 3.0, 1);
    //sph_test_pupilimage_creator_add_speckles( pupil_creator, 10, 10.0, pRNG );
    rerr = cpl_error_get_code();
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    combiner = sph_framecombination_new();
    cpl_test_nonnull( combiner );
    sph_framecombination_set_method_gslfft(combiner, 0.0);
    im = cpl_image_load(
            cpl_frame_get_filename(
                    cpl_frameset_get_first(
                            sph_test_pupilimage_creator_get_frameset(
                                    pupil_creator))), CPL_TYPE_DOUBLE, 0, 0);
    cpl_test_nonnull( im );
    im0 = cpl_image_duplicate(im);
    cpl_test_nonnull( im0 );
    imrot = sph_framecombination_shift_image(combiner, im, 5.0, 7.0);
    cpl_test_nonnull( imrot );
    cpl_image_delete(im);
    im = imrot;
    cpl_image_save(imrot, "rotated_image_gslfft_shift_even_size.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
    xcent = cpl_image_get_centroid_x(imrot);
    ycent = cpl_image_get_centroid_y(imrot);
    //printf( "Centre of rotated image: %f, %f\n", xcent, ycent);
    cpl_image_save(im0, "rotated_image_gslfft_shift_even_size.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    xcent = cpl_image_get_centroid_x(im0);
    ycent = cpl_image_get_centroid_y(im0);
    //printf( "Centre of unrotated image: %f, %f\n", xcent, ycent);
    cpl_image_subtract(im0, imrot);
    cpl_image_save(im0, "rotated_image_gslfft_even_size.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_EXTEND);
    imrotback = sph_framecombination_shift_image(combiner, im, -5.0, -7.0);
    cpl_image_save(imrotback, "rotated_image_gslfft_even_size.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    xcent = cpl_image_get_centroid_x(imrotback);
    ycent = cpl_image_get_centroid_y(imrotback);
    //printf( "Centre of back rotated image: %f, %f\n", xcent, ycent);

    /* Setup ...*/

    gsl_rng_free(pRNG);
    cpl_image_delete(imrot);
    imrot = NULL;
    cpl_image_delete(im0);
    im0 = NULL;
    cpl_image_delete(imrotback);
    imrotback = NULL;
    sph_framecombination_delete(combiner);
    sph_test_pupilimage_creator_delete(pupil_creator);

    /*Verify */
    return;
}
static
void reqtest_sph_framecombination_gslfft_centre_shift_odd_size_frame(void) {
    int rerr = CPL_ERROR_NONE;
    sph_test_pupilimage_creator* pupil_creator = NULL;
    gsl_rng* pRNG = NULL;
    sph_framecombination* combiner = NULL;
    cpl_image* im = NULL;
    cpl_image* im0 = NULL;
    cpl_image* imrot = NULL;
    cpl_image* imrotback = NULL;
    double xcent = 0.0;
    double ycent = 0.0;
    cpl_error_reset();
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    cpl_test_nonnull( pRNG );
    pupil_creator = sph_test_pupilimage_creator_new();
    cpl_test_nonnull( pupil_creator );
    pupil_creator->nx = 65;
    pupil_creator->ny = 65;
    sph_test_pupilimage_creator_create_binary_star_frames(pupil_creator, 10.0,
            20.0, 5.0, 100.0, 1.0e99, 3.0, 1);
    //sph_test_pupilimage_creator_add_speckles( pupil_creator, 10, 10.0, pRNG );
    rerr = cpl_error_get_code();
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    combiner = sph_framecombination_new();
    cpl_test_nonnull( combiner );
    sph_framecombination_set_method_gslfft(combiner, 0.0);
    im = cpl_image_load(
            cpl_frame_get_filename(
                    cpl_frameset_get_first(
                            sph_test_pupilimage_creator_get_frameset(
                                    pupil_creator))), CPL_TYPE_DOUBLE, 0, 0);
    cpl_test_nonnull( im );
    im0 = cpl_image_duplicate(im);
    cpl_test_nonnull( im0 );
    imrot = sph_framecombination_shift_image(combiner, im, 5.2, 7.6);
    cpl_test_nonnull( imrot );
    cpl_image_delete(im);
    im = imrot;
    cpl_image_save(imrot, "shifted_image_gslfft_odd_size.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_DEFAULT);
    xcent = cpl_image_get_centroid_x(imrot);
    ycent = cpl_image_get_centroid_y(imrot);
    //printf( "Centre of rotated image: %f, %f\n", xcent, ycent);
    cpl_image_save(im0, "shifted_image_gslfft_odd_size.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_EXTEND);
    xcent = cpl_image_get_centroid_x(im0);
    ycent = cpl_image_get_centroid_y(im0);
    //printf( "Centre of unrotated image: %f, %f\n", xcent, ycent);
    cpl_image_subtract(im0, imrot);
    cpl_image_save(im0, "shifted_image_gslfft_odd_size.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_EXTEND);
    imrotback = sph_framecombination_shift_image(combiner, imrot, -5.2, -7.6);
    cpl_image_save(imrotback, "shifted_image_gslfft_odd_size.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    xcent = cpl_image_get_centroid_x(imrotback);
    ycent = cpl_image_get_centroid_y(imrotback);
    //printf( "Centre of back rotated image: %f, %f\n", xcent, ycent);

    gsl_rng_free(pRNG);
    cpl_image_delete(imrot);
    imrot = NULL;
    cpl_image_delete(im0);
    im0 = NULL;
    cpl_image_delete(imrotback);
    imrotback = NULL;
    sph_framecombination_delete(combiner);
    sph_test_pupilimage_creator_delete(pupil_creator);

    /*Verify */
    return;
}
static
void reqtest_sph_framecombination_combine_master_frames_overwrite(void) {
    sph_test_pupilimage_creator* pupil_creator = NULL;
    gsl_rng* pRNG = NULL;
    sph_framecombination* combiner = NULL;
    cpl_frameset* frames = NULL;
    cpl_parameterlist* colparams = NULL;
    sph_master_frame* result = NULL;
    cpl_frame* resultframe = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_error_reset();
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    cpl_test_nonnull( pRNG );
    pupil_creator = sph_test_pupilimage_creator_new();
    cpl_test_nonnull( pupil_creator );
    pupil_creator->nx = 64;
    pupil_creator->ny = 64;
    sph_test_pupilimage_creator_create_binary_star_frames(pupil_creator, 10.0,
            45.0, 10.0, 100.0, 2.0, 3.0, 18);
    //sph_test_pupilimage_creator_add_speckles( pupil_creator, 10, 10.0, pRNG );
    rerr = cpl_error_get_code();
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    combiner = sph_framecombination_new();
    cpl_test_nonnull( combiner );
    sph_framecombination_set_method_gslfft(combiner, 0.0);
    frames = sph_test_pupilimage_creator_get_frameset(pupil_creator);
    cpl_test_nonnull( frames );
    colparams = cpl_parameterlist_new();
    cpl_test_nonnull( colparams );
    rerr = cpl_parameterlist_append(
            colparams,
            cpl_parameter_new_range("clean_mean.reject_low", CPL_TYPE_INT,
                    "none", "none", 0, 0, 100));
    rerr += cpl_parameterlist_append(
            colparams,
            cpl_parameter_new_range("clean_mean.reject_high", CPL_TYPE_INT,
                    "none", "none", 0, 0, 100));
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    result = sph_framecombination_combine_master_frames_overwrite(combiner,
            frames, SPH_COLL_ALG_CLEAN_MEAN, colparams);
    /*Verify */
    cpl_test_nonnull( result );
    resultframe = sph_filemanager_create_temp_frame(
            "framecomb_combine_masterframe.fits", "TEST",
            CPL_FRAME_GROUP_CALIB);
    cpl_test_nonnull( resultframe );
    sph_master_frame_save(result, cpl_frame_get_filename(resultframe), NULL);
    gsl_rng_free(pRNG);
    sph_framecombination_delete(combiner);
    sph_master_frame_delete(result);
    cpl_frame_delete(resultframe);
    cpl_parameterlist_delete(colparams);
    sph_test_pupilimage_creator_delete(pupil_creator);
    return;
}

static
void reqtest_sph_framecombination_combine_master_frames_overwrite_gimros(void) {
    sph_test_pupilimage_creator* pupil_creator = NULL;
    gsl_rng* pRNG = NULL;
    sph_framecombination* combiner = NULL;
    cpl_frameset* frames = NULL;
    cpl_parameterlist* colparams = NULL;
    sph_master_frame* result = NULL;
    cpl_frame* resultframe = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_error_reset();
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    cpl_test_nonnull( pRNG );
    pupil_creator = sph_test_pupilimage_creator_new();
    cpl_test_nonnull( pupil_creator );
    pupil_creator->nx = 64;
    pupil_creator->ny = 64;
    sph_test_pupilimage_creator_create_binary_star_frames(pupil_creator, 10.0,
            45.0, 10.0, 100.0, 2.0, 3.0, 18);
    //sph_test_pupilimage_creator_add_speckles( pupil_creator, 10, 10.0, pRNG );
    rerr = cpl_error_get_code();
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    combiner = sph_framecombination_new();
    cpl_test_nonnull( combiner );
    sph_framecombination_set_method_gimros(combiner);
    frames = sph_test_pupilimage_creator_get_frameset(pupil_creator);
    cpl_test_nonnull( frames );
    colparams = cpl_parameterlist_new();
    cpl_test_nonnull( colparams );
    rerr = cpl_parameterlist_append(
            colparams,
            cpl_parameter_new_range("clean_mean.reject_low", CPL_TYPE_INT,
                    "none", "none", 0, 0, 100));
    rerr += cpl_parameterlist_append(
            colparams,
            cpl_parameter_new_range("clean_mean.reject_high", CPL_TYPE_INT,
                    "none", "none", 0, 0, 100));
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    result = sph_framecombination_combine_master_frames_overwrite(combiner,
            frames, SPH_COLL_ALG_CLEAN_MEAN, colparams);
    /*Verify */
    cpl_test_nonnull( result );
    resultframe = sph_filemanager_create_temp_frame(
            "framecomb_combine_masterframe_gimros.fits", "TEST",
            CPL_FRAME_GROUP_CALIB);
    cpl_test_nonnull( resultframe );
    sph_master_frame_save(result, cpl_frame_get_filename(resultframe), NULL);
    gsl_rng_free(pRNG);
    sph_framecombination_delete(combiner);
    sph_master_frame_delete(result);
    cpl_frame_delete(resultframe);
    cpl_parameterlist_delete(colparams);
    sph_test_pupilimage_creator_delete(pupil_creator);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_framecombination_double_image_new_from_cpl_frameset function.
 */
/*----------------------------------------------------------------------------*/
static
void reqtest_sph_framecombination_double_image_new_from_cpl_frameset(void) {
    cpl_frameset* inframes = NULL;
    cpl_frame* aframe = NULL;
    sph_double_image* dimage = NULL;
    cpl_parameterlist* plist = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int ii = 0;
    int nframes = 5;
    char czFilename[256];

    inframes = cpl_frameset_new();
    cpl_test_nonnull( inframes );
    for (ii = 0; ii < nframes; ++ii) {
        dimage = sph_double_image_new_empty();
        cpl_test_nonnull( dimage );
        dimage->iframe = sph_master_frame_new(248, 248);
        cpl_test_nonnull( dimage->iframe );
        dimage->pframe = sph_master_frame_new(248, 248);
        cpl_test_nonnull( dimage->pframe );
        rerr += cpl_image_fill_noise_uniform(dimage->iframe->image, ii - 0.1,
                ii + 0.1);
        rerr += cpl_image_fill_noise_uniform(dimage->pframe->image,
                -1.0 * ii - 0.1, -1.0 * ii + 0.1);
        sprintf(czFilename, "temp_double_image_%d.fits", ii);
        rerr += sph_double_image_save(dimage, czFilename, NULL, CPL_IO_CREATE);
        aframe = cpl_frame_new();
        rerr += cpl_frame_set_filename(aframe, czFilename);
        rerr += cpl_frame_set_tag(aframe, "BLUBB");
        rerr += cpl_frameset_insert(inframes, aframe);
        sph_double_image_delete(dimage);
    }

    cpl_test_eq(cpl_frameset_get_size( inframes ), nframes);
    plist = cpl_parameterlist_new();
    cpl_test_nonnull( plist );
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.reject_low", CPL_TYPE_INT, NULL,
                    NULL, 1, 0, 20));
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.reject_high", CPL_TYPE_INT,
                    NULL, NULL, 1, 0, 20));
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.sigma", CPL_TYPE_DOUBLE, NULL,
                    NULL, 5.0, 0, 200));
    cpl_test_eq(cpl_parameterlist_get_size( plist ), 3);
    dimage = sph_framecombination_double_image_new_from_cpl_frameset(inframes,
            SPH_COLL_ALG_CLEAN_MEAN, plist);
    cpl_test_nonnull( dimage );

    double mean = sph_master_frame_get_mean(dimage->iframe, NULL);

    cpl_test_abs( mean, 2.0, 0.1);

    mean = sph_master_frame_get_mean(dimage->pframe, NULL);
    cpl_test_abs( mean, -2.0, 0.1);
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(plist);
    sph_double_image_delete(dimage);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_framecombination_double_image_new_from_cpl_frameset function.
 */
/*----------------------------------------------------------------------------*/
static
void reqtest_sph_framecombination_double_image_new_from_cpl_frameset_median(void) {
    cpl_frameset* inframes = NULL;
    cpl_frame* aframe = NULL;
    sph_double_image* dimage = NULL;
    cpl_parameterlist* plist = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int ii = 0;
    int nframes = 5;
    char czFilename[256];

    inframes = cpl_frameset_new();
    cpl_test_nonnull( inframes );
    for (ii = 0; ii < nframes; ++ii) {
        dimage = sph_double_image_new_empty();
        cpl_test_nonnull( dimage );
        dimage->iframe = sph_master_frame_new(248, 248);
        cpl_test_nonnull( dimage->iframe );
        dimage->pframe = sph_master_frame_new(248, 248);
        cpl_test_nonnull( dimage->pframe );
        if (ii != 0) {
            rerr += cpl_image_fill_noise_uniform(dimage->iframe->image, 1.99,
                    2.01);
            rerr += cpl_image_fill_noise_uniform(dimage->pframe->image, -2.01,
                    -1.99);
        } else {
            rerr += cpl_image_fill_noise_uniform(dimage->iframe->image, 10.99,
                    11.01);
            rerr += cpl_image_fill_noise_uniform(dimage->pframe->image, -20.01,
                    -19.99);
        }
        sprintf(czFilename, "temp_double_image_%d.fits", ii);
        rerr += sph_double_image_save(dimage, czFilename, NULL, CPL_IO_CREATE);
        aframe = cpl_frame_new();
        rerr += cpl_frame_set_filename(aframe, czFilename);
        rerr += cpl_frame_set_tag(aframe, "BLUBB");
        rerr += cpl_frameset_insert(inframes, aframe);
        sph_double_image_delete(dimage);
    }

    cpl_test_eq(cpl_frameset_get_size( inframes ), nframes);
    plist = cpl_parameterlist_new();
    cpl_test_nonnull( plist );
    cpl_test_zero(cpl_parameterlist_get_size( plist ));
    dimage = sph_framecombination_double_image_new_from_cpl_frameset(inframes,
            SPH_COLL_ALG_MEDIAN, plist);
    cpl_test_nonnull( dimage );

    double mean = sph_master_frame_get_mean(dimage->iframe, NULL);

    cpl_test_abs( mean, 2.0, 0.1);

    mean = sph_master_frame_get_mean(dimage->pframe, NULL);
    cpl_test_abs( mean, -2.0, 0.1);

    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(plist);
    sph_double_image_delete(dimage);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_framecombination_quad_image_new_from_cpl_frameset function.
 */
/*----------------------------------------------------------------------------*/
static
void reqtest_sph_framecombination_quad_image_new_from_cpl_frameset(void) {
    cpl_frameset* quadframes = NULL;
    cpl_parameterlist* plist = NULL;
    sph_quad_image* quadimage = NULL;

    quadframes =
            sph_test_simulator_create_quad_image_frames_from_zplexp_frames();
    plist = cpl_parameterlist_new();
    cpl_test_nonnull( plist );
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.reject_low", CPL_TYPE_INT, NULL,
                    NULL, 0, 0, 20));
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.reject_high", CPL_TYPE_INT,
                    NULL, NULL, 0, 0, 20));
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.sigma", CPL_TYPE_DOUBLE, NULL,
                    NULL, 5.0, 0, 200));
    cpl_test_eq(cpl_parameterlist_get_size( plist ), 3);
    quadimage = sph_framecombination_quad_image_new_from_cpl_frameset(
            quadframes, SPH_COLL_ALG_CLEAN_MEAN, plist);
    cpl_test_nonnull( quadimage );
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->iframe->image ), 170.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->pframe->image ), 30.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->iframe->image ), 30.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->pframe->image ), 170.0,
            1e-20);
    sph_quad_image_save(quadimage,
            "test_sph_framecombination_quad_image.fits.tmp", NULL);
    sph_quad_image_delete(quadimage);
    cpl_frameset_delete(quadframes);
    quadframes = NULL;
    cpl_parameterlist_delete(plist);
    plist = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_framecombination_quad_image_new_from_cpl_frameset_multi_cubes function.
 */
/*----------------------------------------------------------------------------*/
static
void reqtest_sph_framecombination_quad_image_new_from_cpl_frameset_multi_cubes(void) {
    cpl_frame* zexpcube_plus = NULL;
    cpl_frame* zexpcube_minus = NULL;
    cpl_frameset* zexpcubes = NULL;
    cpl_parameterlist* plist = NULL;
    sph_quad_image* quadimage = NULL;
    const int nframes = 3;

    zexpcubes = cpl_frameset_new();

    sph_test_simulator_create_zplpreproc_cube_plus(&zexpcube_plus, nframes);
    cpl_test_nonnull( zexpcube_plus );
    cpl_frameset_insert(zexpcubes, zexpcube_plus);

    plist = cpl_parameterlist_new();
    cpl_test_nonnull( plist );
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.reject_low", CPL_TYPE_INT, NULL,
                    NULL, 0, 0, 20));
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.reject_high", CPL_TYPE_INT,
                    NULL, NULL, 0, 0, 20));
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.sigma", CPL_TYPE_DOUBLE, NULL,
                    NULL, 5.0, 0, 200));
    cpl_test_eq(cpl_parameterlist_get_size( plist ), 3);

    quadimage =
            sph_framecombination_quad_image_new_from_cpl_frameset_multi_cubes(
                    zexpcubes, SPH_COLL_ALG_CLEAN_MEAN, plist);
    cpl_test_nonnull( quadimage );
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->iframe->image ), 170.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->pframe->image ), 30.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->iframe->image ), 30.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->pframe->image ), 170.0,
            1e-20);
    sph_quad_image_delete(quadimage);

    sph_test_simulator_create_zplpreproc_cube_minus(&zexpcube_minus, nframes);
    cpl_frameset_insert(zexpcubes, zexpcube_minus);
    quadimage =
            sph_framecombination_quad_image_new_from_cpl_frameset_multi_cubes(
                    zexpcubes, SPH_COLL_ALG_CLEAN_MEAN, plist);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->iframe->image ), 110.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->pframe->image ), 90.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->iframe->image ), 90.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->pframe->image ), 110.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->iframe->ncombmap ), 6.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->pframe->ncombmap ), 6.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->iframe->ncombmap ), 6.0,
            1e-20);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->pframe->ncombmap ), 6.0,
            1e-20);
    sph_quad_image_save(quadimage,
            "test_sph_framecombination_quad_image.fits.tmp", NULL);
    sph_quad_image_delete(quadimage);
    cpl_frameset_delete(zexpcubes);
    cpl_parameterlist_delete(plist);

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_framecombination_master_frame_new_mean_weight_from_cpl_framest function.
 */
/*----------------------------------------------------------------------------*/
static
void reqtest_sph_framecombination_master_frame_new_mean_weight_from_cpl_frameset(void) {
    sph_master_frame* mframe1 = NULL;
    sph_master_frame* mframe2 = NULL;
    sph_master_frame* mframe3 = NULL;
    sph_master_frame* result = NULL;
    cpl_frame* frame1 = NULL;
    cpl_frame* frame2 = NULL;
    cpl_frame* frame3 = NULL;
    cpl_image* image1 = NULL;
    cpl_image* image2 = NULL;
    cpl_image* image3 = NULL;
    cpl_frameset* frames = NULL;

    int nx = 100;
    int ny = 100;
    double value1 = 100.0;
    double rmsd1 = 2.0;
    double nmap1 = 20.0;

    double value2 = 300.0;
    double rmsd2 = 5.0;
    double nmap2 = 10.0;

    double value3 = 200.0;
    double rmsd3 = 3.0;
    double nmap3 = 15.0;

    double mean = 0;
    double rms = 0;

    frames = cpl_frameset_new();
    //create first master frame
    image1 = sph_test_image_tools_create_flat_image_double(nx, ny, value1);
    mframe1 = sph_master_frame_new_from_cpl_image(image1);
    sph_utils_cpl_image_fill_double(mframe1->rmsmap, rmsd1);
    sph_utils_cpl_image_fill_double(mframe1->ncombmap, nmap1);
    frame1 = sph_filemanager_create_temp_frame("test_mean_weight_mf.fits",
            "None", CPL_FRAME_GROUP_NONE);
    sph_master_frame_save(mframe1, cpl_frame_get_filename(frame1), NULL);
    cpl_frameset_insert(frames, frame1);

    //create second master frame
    image2 = sph_test_image_tools_create_flat_image_double(nx, ny, value2);
    mframe2 = sph_master_frame_new_from_cpl_image(image2);
    sph_utils_cpl_image_fill_double(mframe2->rmsmap, rmsd2);
    sph_utils_cpl_image_fill_double(mframe2->ncombmap, nmap2);
    cpl_image_multiply_scalar(mframe2->badpixelmap, 0);
    cpl_image_add_scalar(mframe2->badpixelmap, 1); //bad pixel for the whole image
    frame2 = sph_filemanager_create_temp_frame("test_mean_weight_mf.fits",
            "None", CPL_FRAME_GROUP_NONE);
    sph_master_frame_save(mframe2, cpl_frame_get_filename(frame2), NULL);
    cpl_frameset_insert(frames, frame2);

    //calculate weighted mean for 2 master frames
    result = sph_framecombination_master_frame_new_mean_weight_from_cpl_framest(
            frames);
    cpl_test_nonnull( result );
    sph_master_frame_save(result, "test_mean_weight_2mf_result.fits", NULL);
    mean = (value1 / (rmsd1 * rmsd1) + value2 / (rmsd2 * rmsd2))
            / (1. / (rmsd1 * rmsd1) + 1. / (rmsd2 * rmsd2));
    rms = sqrt(1. / (rmsd1 * rmsd1) + 1. / (rmsd2 * rmsd2));
    cpl_test_abs( cpl_image_get_mean( result->image ), mean, 1e-20);
    cpl_test_abs( cpl_image_get_mean( result->ncombmap ), nmap1+nmap2,
            1e-20);
    cpl_test_abs( cpl_image_get_mean( result->rmsmap ), rms, 1e-20);
    sph_master_frame_delete(result);

    //create third master frame
    image3 = sph_test_image_tools_create_flat_image_double(nx, ny, value3);
    mframe3 = sph_master_frame_new_from_cpl_image(image3);
    sph_utils_cpl_image_fill_double(mframe3->rmsmap, rmsd3);
    sph_utils_cpl_image_fill_double(mframe3->ncombmap, nmap3);
    cpl_image_multiply_scalar(mframe3->badpixelmap, 0);
    cpl_image_add_scalar(mframe3->badpixelmap, 1); //bad pixel for the whole image
    frame3 = sph_filemanager_create_temp_frame("test_mean_weight_mf.fits",
            "None", CPL_FRAME_GROUP_NONE);
    sph_master_frame_save(mframe3, cpl_frame_get_filename(frame3), NULL);
    cpl_frameset_insert(frames, frame3);

    //calculate weighted mean for 3 master frames
    result = sph_framecombination_master_frame_new_mean_weight_from_cpl_framest(
            frames);
    cpl_test_assert( result);
    sph_master_frame_save(result, "test_mean_weight_3mf_result.fits", NULL);
    mean = (value1 / (rmsd1 * rmsd1) + value2 / (rmsd2 * rmsd2)
            + value3 / (rmsd3 * rmsd3))
            / (1. / (rmsd1 * rmsd1) + 1. / (rmsd2 * rmsd2)
                    + 1. / (rmsd3 * rmsd3));
    rms = sqrt(
            1. / (rmsd1 * rmsd1) + 1. / (rmsd2 * rmsd2) + 1. / (rmsd3 * rmsd3));
    cpl_test_abs( cpl_image_get_mean( result->image ), mean, 1e-20);
    cpl_test_abs( cpl_image_get_mean( result->ncombmap ),
            nmap1+nmap2+nmap3, 1e-20);
    cpl_test_abs( cpl_image_get_mean( result->rmsmap ), rms, 1e-20);
    cpl_test_eq((int)cpl_image_get_absflux( result->badpixelmap ), nx*ny);
    sph_master_frame_delete(result);
    sph_master_frame_delete(mframe3);
    cpl_image_delete(image3);
    sph_master_frame_delete(mframe2);
    cpl_image_delete(image2);
    sph_master_frame_delete(mframe1);
    cpl_image_delete(image1);
    cpl_frameset_delete(frames);

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_framecombination_triple_image_new_from_cpl_frameset function.
 */
/*----------------------------------------------------------------------------*/
static
void reqtest_sph_framecombination_triple_image_new_from_cpl_frameset(void) {
    cpl_parameterlist* plist = NULL;
    cpl_frame* triframe = NULL;
    cpl_frameset* triframes = NULL;
    sph_triple_image* trimage = NULL;
    sph_triple_image* result = NULL;
    int nx = 256;
    int ny = 256;
    int nframes = 3;
    cpl_error_code rerr = CPL_ERROR_NONE;

    plist = cpl_parameterlist_new();
    cpl_test_nonnull( plist );
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.reject_low", CPL_TYPE_INT, NULL,
                    NULL, 0, 0, 20));
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.reject_high", CPL_TYPE_INT,
                    NULL, NULL, 0, 0, 20));
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.sigma", CPL_TYPE_DOUBLE, NULL,
                    NULL, 5.0, 0, 200));
    cpl_test_eq(cpl_parameterlist_get_size( plist ), 3);

    triframes = cpl_frameset_new();
    for (int ii = 0; ii < nframes; ii++) {
        trimage = sph_test_zpl_simulator_create_triple_image(nx, ny,
                (double) ii + 10.0, (double) ii + 20.0, (double) ii + 30.0);
        triframe = sph_filemanager_create_temp_frame(
                "test_framecombination_trimage.fits.tmp", "NONE",
                CPL_FRAME_GROUP_NONE);
        sph_triple_image_save(trimage, cpl_frame_get_filename(triframe), NULL);
        cpl_frameset_insert(triframes, triframe);
        sph_triple_image_delete(trimage);
        trimage = NULL;
    }
    //printf("cpl_frameset_size: %i\n", cpl_frameset_get_size( triframes));
    //printf("filename: %s\n", cpl_frame_get_filename( cpl_frameset_get_first(triframes) ) );
    //printf("filename: %s\n", cpl_frame_get_filename( cpl_frameset_get_next(triframes) ) );
    //printf("filename: %s\n", cpl_frame_get_filename( cpl_frameset_get_next(triframes) ) );

    result = sph_framecombination_triple_image_new_from_cpl_frameset(triframes,
            SPH_COLL_ALG_CLEAN_MEAN, plist);
    cpl_test_nonnull( result );

    rerr = sph_triple_image_quality_check(result);
    sph_triple_image_save(result,
            "test_framecombination_trimage_result.fits.tmp", NULL);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs(
            cpl_propertylist_get_double(result->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_IFRAME),
            11.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(result->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_QFRAME),
            21.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(result->qclist, SPH_COMMON_KEYWORD_QC_MEAN_TRIPLEIMAGE_UFRAME),
            31.0, 1e-20);

    sph_triple_image_delete(trimage);
    cpl_frameset_delete(triframes);
    sph_triple_image_delete(result);
    cpl_parameterlist_delete(plist);

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_framecombination_double_image_from_frameset function.
 */
/*----------------------------------------------------------------------------*/
static
void reqtest_sph_framecombination_double_image_from_frameset(void) {
    cpl_frameset* inframes = NULL;
    cpl_parameterlist* plist = NULL;
    cpl_frame* zexpimgcube = NULL;
    cpl_frame* zexpimgcube1 = NULL;
    sph_double_image* result = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;

    plist = cpl_parameterlist_new();
    cpl_test_nonnull( plist );
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.reject_low", CPL_TYPE_INT, NULL,
                    NULL, 0, 0, 20));
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.reject_high", CPL_TYPE_INT,
                    NULL, NULL, 0, 0, 20));
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.sigma", CPL_TYPE_DOUBLE, NULL,
                    NULL, 5.0, 0, 200));
    cpl_test_eq(cpl_parameterlist_get_size( plist ), 3);

    sph_test_simulator_create_zplpreproc_imaging_cube(&zexpimgcube, nframes);
    sph_test_simulator_create_zplpreproc_imaging_cube_set(&zexpimgcube1, 25.0,
            30.0, 1.0, nframes);

    inframes = cpl_frameset_new();
    cpl_frameset_insert(inframes, zexpimgcube);
    cpl_frameset_insert(inframes, zexpimgcube1);

    //mean
    result = sph_framecombination_double_image_from_frameset(inframes,
            SPH_COLL_ALG_CLEAN_MEAN, plist);
    cpl_test_nonnull( result );

    rerr = sph_double_image_quality_check(result);
    sph_double_image_save(result,
            "test_framecombination_doubleimage_mean_result.fits.tmp", NULL,
            CPL_IO_CREATE);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs(
            cpl_propertylist_get_double(result->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            23.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(result->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME),
            26.0, 1e-20);
    sph_double_image_delete(result);

    //median
    result = sph_framecombination_double_image_from_frameset(inframes,
            SPH_COLL_ALG_MEDIAN, plist);
    rerr = sph_double_image_quality_check(result);
    sph_double_image_save(result,
            "test_framecombination_doubleimage_median_result.fits.tmp", NULL,
            CPL_IO_CREATE);
    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Values are: %f %f", cpl_propertylist_get_double(result->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), cpl_propertylist_get_double(result->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME));
    // Something strange happened here when going to CPL v 6.0a -- Alex: check the test is correct now!
    cpl_test_abs(
            cpl_propertylist_get_double(result->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            23.0, 1e-10);
    // had to change these from 21 ?
    cpl_test_abs(
            cpl_propertylist_get_double(result->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME),
            26.0, 1e-10);
    // and 22 after going to cpl 6

    sph_double_image_delete(result);
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(plist);

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_framecombination_quad_image_from_frameset function.
 */
/*----------------------------------------------------------------------------*/
static
void reqtest_sph_framecombination_quad_image_from_frameset(void) {
    cpl_frame* zexpcube_plus = NULL;
    cpl_frame* zexpcube_minus = NULL;
    cpl_frameset* zexpcubes = NULL;
    cpl_parameterlist* plist = NULL;
    sph_quad_image* quadimage = NULL;
    const int nframes = 3;

    zexpcubes = cpl_frameset_new();

    sph_test_simulator_create_zplpreproc_cube_plus(&zexpcube_plus, nframes);
    cpl_frameset_insert(zexpcubes, zexpcube_plus);

    plist = cpl_parameterlist_new();
    cpl_test_nonnull( plist );
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.reject_low", CPL_TYPE_INT, NULL,
                    NULL, 0, 0, 20));
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.reject_high", CPL_TYPE_INT,
                    NULL, NULL, 0, 0, 20));
    cpl_parameterlist_append(
            plist,
            cpl_parameter_new_range("clean_mean.sigma", CPL_TYPE_DOUBLE, NULL,
                    NULL, 5.0, 0, 200));
    cpl_test_eq(cpl_parameterlist_get_size( plist ), 3);

    // one cube (frame)
    //mean
    quadimage = sph_framecombination_quad_image_from_frameset(zexpcubes,
            SPH_COLL_ALG_CLEAN_MEAN, plist);
    cpl_test_nonnull( quadimage );
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->iframe->image ), 170.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->pframe->image ), 30.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->iframe->image ), 30.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->pframe->image ), 170.0,
            1e-10);
    sph_quad_image_delete(quadimage);

    //add second cube-frame to the list of the inputframes
    sph_test_simulator_create_zplpreproc_cube_minus(&zexpcube_minus, nframes);
    cpl_frameset_insert(zexpcubes, zexpcube_minus);

    // two cubes (frames)
    //mean
    quadimage = sph_framecombination_quad_image_from_frameset(zexpcubes,
            SPH_COLL_ALG_CLEAN_MEAN, plist);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->iframe->image ), 110.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->pframe->image ), 90.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->iframe->image ), 90.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->pframe->image ), 110.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->iframe->ncombmap ), 6.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->pframe->ncombmap ), 6.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->iframe->ncombmap ), 6.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->pframe->ncombmap ), 6.0,
            1e-10);
    //sph_quad_image_save( quadimage, "test_sph_framecombination_quad_image.fits.tmp", NULL);
    sph_quad_image_delete(quadimage);

    //median
    quadimage = sph_framecombination_quad_image_from_frameset(zexpcubes,
            SPH_COLL_ALG_CLEAN_MEAN, plist);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->iframe->image ), 110.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->pframe->image ), 90.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->iframe->image ), 90.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->pframe->image ), 110.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->iframe->ncombmap ), 6.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->zero_image->pframe->ncombmap ), 6.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->iframe->ncombmap ), 6.0,
            1e-10);
    cpl_test_abs(
            cpl_image_get_mean( quadimage->pi_image->pframe->ncombmap ), 6.0,
            1e-10);
    sph_quad_image_delete(quadimage);
    cpl_frameset_delete(zexpcubes);
    cpl_parameterlist_delete(plist);

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of the sph_framecombination class of SPH API
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("reqtest_sph_framecombination",
            reqtest_init_framecombination_testsuite,
            reqtest_clean_framecobinatin_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    " reqtest_sph_framecombination_quad_image_from_frameset",
                    reqtest_sph_framecombination_quad_image_from_frameset)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    " reqtest_sph_framecombination_double_image_from_frameset",
                    reqtest_sph_framecombination_double_image_from_frameset)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "reqtest_sph_framecombination_triple_image_new_from_cpl_frameset",
                    reqtest_sph_framecombination_triple_image_new_from_cpl_frameset)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "reqtest_sph_framecombination_master_frame_new_mean_weight_from_cpl_frameset",
                    reqtest_sph_framecombination_master_frame_new_mean_weight_from_cpl_frameset)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "reqtest_sph_framecombination",
                    reqtest_sph_framecombination)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "reqtest_sph_framecombination_quad_image_new_from_cpl_frameset_multi_cubes",
                    reqtest_sph_framecombination_quad_image_new_from_cpl_frameset_multi_cubes)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "reqtest_sph_framecombination_quad_image_new_from_cpl_frameset",
                    reqtest_sph_framecombination_quad_image_new_from_cpl_frameset)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "reqtest_sph_framecombination_double_image_new_from_cpl_frameset",
                    reqtest_sph_framecombination_double_image_new_from_cpl_frameset)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "reqtest_sph_framecombination_double_image_new_from_cpl_frameset_median",
                    reqtest_sph_framecombination_double_image_new_from_cpl_frameset_median)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "reqtest_sph_framecombination_gslfft_combine_with_filter",
                    reqtest_sph_framecombination_gslfft_combine_with_filter)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "reqtest_sph_framecombination_cplwarp",
                    reqtest_sph_framecombination_cplwarp)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "reqtest_sph_framecombination_gslfft",
                    reqtest_sph_framecombination_gslfft)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "reqtest_sph_framecombination_gslfft_null_rot",
                    reqtest_sph_framecombination_gslfft_null_rot)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "reqtest_sph_framecombination_gslfft_sense_of_rot",
                    reqtest_sph_framecombination_gslfft_sense_of_rot)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "reqtest_sph_framecombination_gslfft_even_size",
                    reqtest_sph_framecombination_gslfft_centre_even_size_frame)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "reqtest_sph_framecombination_gslfft_odd_size",
                    reqtest_sph_framecombination_gslfft_centre_odd_size_frame)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "reqtest_sph_framecombination_gslfft_shift_even_size",
                    reqtest_sph_framecombination_gslfft_centre_shift_even_size_frame)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "reqtest_sph_framecombination_gslfft_shift_odd_size",
                    reqtest_sph_framecombination_gslfft_centre_shift_odd_size_frame)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "reqtest_sph_framecombination_gimros",
                    reqtest_sph_framecombination_gimros)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "reqtest_sph_framecombination_combine_master_frames_overwrite",
                    reqtest_sph_framecombination_combine_master_frames_overwrite)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "reqtest_sph_framecombination_combine_master_frames_overwrite_gimros",
                    reqtest_sph_framecombination_combine_master_frames_overwrite_gimros)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
