import os
from adari_core.data_libs.master_dark_bias import MasterRawCutsReport
from adari_core.plots.text import TextPlot
from .sphere_utils import SphereSetupInfo, SphereReportMixin, data_reader

class SphereMasterDarkReport(SphereReportMixin, MasterRawCutsReport):
    def __init__(self):
        super().__init__("sphere_master_dark")
        self.data_readers["raw_im"] = data_reader
        self.img = False
        self.multi = False

        self.files_needed_raw = [
            "IRD_DARK_RAW",
            "IRD_INS_BG_RAW",
            "IFS_CAL_BACKGROUND_RAW",
            "IFS_DARK_RAW",
        ]
        self.files_needed = [
            "IFS_MASTER_DARK",
            "IFS_CAL_BACKGROUND",
            "IRD_MASTER_DARK",
            "IRD_INS_BG",
            "ZPL_MASTER_DARK_CAM1",
            "ZPL_MASTER_DARK_CAM2",
            "ZPL_MASTER_DARK_IMAGING_CAM1",
            "ZPL_MASTER_DARK_IMAGING_CAM2",
        ]

    def parse_sof(self):

        master_dark = []
        raw_dark = []

        for filename, catg in self.inputs:
            if catg in self.files_needed_raw:
                raw_dark.append(filename)
            if catg in self.files_needed:
                master_dark.append(filename)
            if catg == "ZPL_MASTER_DARK_CAM1" or catg == "ZPL_MASTER_DARK_CAM2":
                self.multi = True
            if (
                catg == "ZPL_MASTER_DARK_IMAGING_CAM1"
                or catg == "ZPL_MASTER_DARK_IMAGING_CAM2"
            ):
                self.multi = True
                self.img = True

        # Build and return the file name list
        file_lists = []
        if self.multi:
            for item in master_dark:
                file_lists.append(
                    {
                        "master_im": item,
                    }
                )
        else:
            for item1 in master_dark:
                for item2 in raw_dark:
                    file_lists.append(
                        {
                            "raw_im": item2,
                            "master_im": item1,
                        }
                    )
        return file_lists

    def generate_panels(self, **kwargs):
        panels = {}
        if self.multi:
            if self.img:
                exts = ["IFRAME.IMAGE", "LEAKAGE.IMAGE"]
            else:
                exts = [
                    "PHASE-0.ODD.IMAGE",
                    "PHASE-0.EVEN.IMAGE",
                    "PHASE-PI.ODD.IMAGE",
                    "PHASE-PI.EVEN.IMAGE",
                ]
        else:
            exts = ["IMAGE"]
        vspace = 0.3
        rext = "PRIMARY"

        for ext in exts:
            if "raw_im" in self.hdus[0].keys():
                new_panels = super().generate_panels(
                    master_im_ext=ext,
                    raw_im_ext=rext,
                    im_clipping="percentile",
                    im_n_clipping=50,
                    interpolation="nearest",
                )
            else:
                new_panels = super().generate_panels(
                    master_im_ext=ext,
                )

            for i, (panel, panel_descr) in enumerate(new_panels.items()):
                panel_descr["report_description"] = (
                    f"SPHERE dark panel - "
                    f"{os.path.basename(panel_descr['master_im'])}, "
                    f"{panel_descr['master_im_ext']}"
                )

                master_im = self.hdus[i]["master_im"]
                procatg = str(master_im["PRIMARY"].header.get("HIERARCH ESO PRO CATG"))

                # Text Plot
                fname = os.path.basename(str(master_im.filename()))
                t1 = TextPlot(columns=1, v_space=vspace)

                col1 = (
                    str(master_im["PRIMARY"].header.get("INSTRUME")),
                    "EXTNAME: " + ext,
                    "PRO CATG: " + procatg,
                    "FILE NAME: " + fname,
                    "RAW1 NAME: "
                    + str(
                        master_im["PRIMARY"].header.get(
                            "HIERARCH ESO PRO REC1 RAW1 NAME"
                        )
                    ),
                )
                t1.add_data(col1)
                panel.assign_plot(t1, 0, 0, xext=2)

                t2 = TextPlot(columns=1, v_space=vspace, xext=1)
                if self.multi:
                    self.metadata = SphereSetupInfo.zpl_dark(master_im)
                elif procatg == "IRD_INS_BG":
                    self.metadata = SphereSetupInfo.ird_ins_bg(master_im)
                elif procatg == "IRD_MASTER_DARK" or procatg == "IFS_MASTER_DARK":
                    self.metadata = SphereSetupInfo.basic(master_im)
                elif procatg == "IFS_CAL_BACKGROUND":
                    self.metadata = SphereSetupInfo.ifs_cal_background(master_im)
                col2 = self.metadata
                t2.add_data(col2)
                panel.assign_plot(t2, 2, 0, xext=1)

                master_path = os.path.basename(panel_descr["master_im"])
                if (
                    master_path == "zpl_master_dark_imaging_cam1.fits"
                    or master_path == "zpl_master_dark_imaging_cam2.fits"
                ):
                    histogram = panel.retrieve(3, 1)

                    bin_per_val = self.hist_bins_max + 1  # Dummy val to start while
                    vals_per_bin = 1
                    rmin, rmax = histogram.get_vlim()
                    while bin_per_val > self.hist_bins_max:
                        bin_per_val = (rmax - rmin + 1) // vals_per_bin
                        vals_per_bin += 1
                    histogram.set_int_bins(bin_size=((rmax - rmin + 1) // bin_per_val))
                    histogram.bins = bin_per_val

            panels = {**panels, **new_panels}

        return panels


rep = SphereMasterDarkReport()
