import astropy.io.fits as fits


def data_reader(filename):
    hdu = fits.open(filename, mode="readonly")
    rext = "PRIMARY"
    if len(hdu[rext].data.shape) > 2:
        hdu[rext].data = hdu[rext].data[0]
    return hdu

class SphereReportMixin(object):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._version = "TBD"

class SphereSetupInfo(object):
    @staticmethod
    def basic(hdul):
        metadata = [
            "DET.READ.CURNAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET READ CURNAME", "N/A")),
            "DET.SEQ1.DIT: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET SEQ1 DIT", "N/A")),
            "DET.NDIT: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET NDIT", "N/A")),
        ]
        return metadata

    @staticmethod
    def opti2(hdul):
        metadata = [
            "INS2.OPTI2.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS2 OPTI2 NAME", "N/A")),
        ]
        return metadata

    @staticmethod
    def opti3(hdul):
        metadata = []
        for i in range(1, 7):
            metadata.append(
                "INS3.OPTI"
                + str(i)
                + ".NAME: "
                + str(
                    hdul["PRIMARY"].header.get(
                        "HIERARCH ESO INS3 OPTI" + str(i) + " NAME", "N/A"
                    )
                ),
            )
        return metadata

    @staticmethod
    def dark(hdul):
        metadata = SphereSetupInfo.basic(hdul)
        return metadata

    @staticmethod
    def ifs_cal_background(hdul):
        metadata = SphereSetupInfo.basic(hdul)
        meta2 = [
            "INS4.FILT2.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS4 FILT2 NAME", "N/A")),
            "INS4.OPTI13.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS4 OPTI13 NAME", "N/A")),
        ]
        metadata = metadata + meta2
        return metadata

    @staticmethod
    def ifs_specpos(hdul):
        return SphereSetupInfo.opti2(hdul)

    @staticmethod
    def ifs_distortion_map(hdul):
        return SphereSetupInfo.opti2(hdul)

    @staticmethod
    def ifs_standard(hdul):
        metadata = SphereSetupInfo.opti2(hdul)
        metadata = metadata + [
            "INS4.DROT2.MODE: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS4 DROT2 MODE", "N/A")),
        ]
        return metadata

    @staticmethod
    def ifs_flat(hdul):
        return SphereSetupInfo.opti2(hdul)

    @staticmethod
    def ifs_wave_cal(hdul):
        return SphereSetupInfo.opti2(hdul)

    @staticmethod
    def ird_ins_bg(hdul):
        metadata = SphereSetupInfo.basic(hdul)
        meta2 = [
            "INS4.FILT2.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS4 FILT2 NAME", "N/A")),
            "INS1.FILT.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS1 FILT NAME", "N/A")),
            "INS1.OPTI2.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS1 OPTI2 NAME", "N/A")),
        ]
        metadata = metadata + meta2
        return metadata

    @staticmethod
    def ird_distortion_map(hdul):
        metadata = [
            "INS4.FILT2.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS4 FILT2 NAME", "N/A")),
            "INS4.OPTI10.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS4 OPTI10 NAME", "N/A")),
            "INS1.FILT.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS1 FILT NAME", "N/A")),
            "INS1.OPTI1.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS1 OPTI1 NAME", "N/A")),
            "INS1.OPTI2.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS1 OPTI2 NAME", "N/A")),
        ]
        return metadata

    @staticmethod
    def ird_science_imaging(hdul):
        metadata = [
            "INS1.FILT.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS1 FILT NAME", "N/A")),
            "INS1.OPTI1.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS1 OPTI1 NAME", "N/A")),
            "INS1.OPTI2.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS1 OPTI2 NAME", "N/A")),
            "INS4.DROT2.MODE: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS4 DROT2 MODE", "N/A")),
            "AOS.IRLOOP.STATE: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO AOS IRLOOP STATE", "N/A")),
        ]
        return metadata

    @staticmethod
    def ird_wave_cal(hdul):
        metadata = SphereSetupInfo.basic(hdul)
        meta2 = [
            "INS1.FILT.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS1 FILT NAME", "N/A")),
            "INS1.OPTI1.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS1 OPTI1 NAME", "N/A")),
            "INS1.OPTI2.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS1 OPTI2 NAME", "N/A")),
        ]
        metadata = metadata + meta2
        return metadata

    @staticmethod
    def ird_flat(hdul):
        metadata = SphereSetupInfo.basic(hdul)
        meta2 = [
            "INS4.FILT2.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS4 FILT2 NAME", "N/A")),
            "INS4.OPTI10.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS4 OPTI10 NAME", "N/A")),
            "INS1.FILT.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS1 FILT NAME", "N/A")),
            "INS1.OPTI1.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS1 OPTI1 NAME", "N/A")),
            "INS1.OPTI2.NAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS1 OPTI2 NAME", "N/A")),
        ]
        metadata = metadata + meta2
        return metadata

    @staticmethod
    def zpl_int_flat(hdul):
        metadata = [
            "DET.READ.CURNAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET READ CURNAME", "N/A")),
            "DET.DIT1: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET DIT1", "N/A")),
            "DET.NDIT: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET NDIT", "N/A")),
        ]
        metadata = metadata + SphereSetupInfo.opti3(hdul)
        return metadata

    @staticmethod
    def zpl_bias(hdul):
        metadata = [
            "DET.READ.CURNAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET READ CURNAME", "N/A")),
            "DET.NDIT: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET NDIT", "N/A")),
        ]
        return metadata

    @staticmethod
    def zpl_dark(hdul):
        metadata = [
            "DET.READ.CURNAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET READ CURNAME", "N/A")),
            "DET.SEQ1.EXPTIME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET SEQ1 EXPTIME", "N/A")),
            "DET.NDIT: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET NDIT", "N/A")),
        ]
        return metadata

    @staticmethod
    def zpl_modem_efficiency(hdul):
        metadata = SphereSetupInfo.opti3(hdul)
        metadata = metadata + [
            "DET.READ.CURNAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET READ CURNAME", "N/A")),
            "EXPTIME: " + str(hdul["PRIMARY"].header.get("EXPTIME", "N/A")),
            "DET.NDIT: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET NDIT", "N/A")),
        ]
        return metadata

    @staticmethod
    def zpl_standard(hdul):
        metadata = SphereSetupInfo.opti3(hdul)
        metadata = metadata + [
            "INS.COMB.VCOR: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO INS COMB VCOR", "N/A")),
            "DET.READ.CURNAME: "
            + str(hdul["PRIMARY"].header.get("HIERARCH ESO DET READ CURNAME", "N/A")),
        ]
        return metadata
