/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <math.h>
#include "sph_common_keywords.h"
#include "sph_ird_keywords.h"

#include "sph_ird_tags.h"
#include "sph_ird_flux_calib.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_ird_instrument_model.h"
#include "sph_framecombination.h"
#include "sph_ird_common_science.h"
#include "sph_keyword_manager.h"
#include "sph_common_science.h"
/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IRD_FLUX_CALIB_GENERAL;
extern sph_error_code SPH_IRD_FLUX_CALIB_FRAMES_MISSING;

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ird_flux_calib_run Create Master Dark Recipe
 *
 * This module provides the algorithm inplementation for the creation of the
 * master dakr
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ird_flux_calib.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_ird_flux_calib recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ird_flux_calib_run(sph_ird_flux_calib* self) {
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    sph_master_frame* left_coro = NULL;
    sph_master_frame* right_coro = NULL;
    sph_master_frame* left_nocoro = NULL;
    sph_master_frame* right_nocoro = NULL;
    sph_master_frame* result_coro = NULL;
    sph_master_frame* result_nocoro = NULL;
    cpl_propertylist* plist = NULL;
    cpl_frameset* rawframes = NULL;
    double flux_left_coro = 0.0;
    double flux_right_coro = 0.0;
    double flux_left_nocoro = 0.0;
    double flux_right_nocoro = 0.0;
    sph_ird_common_science* sci = NULL;
    cpl_frameset* left_frameset = NULL;
    cpl_frameset* right_frameset = NULL;
    cpl_frameset* total_frameset = NULL;

    sci = sph_ird_common_science_new(self->inframes, self->inparams,
            self->dark_frame, self->flat_frame, NULL, NULL, NULL,
            self->flat_frame, NULL, SPH_RECIPE_NAME_IRD_ASTROMETRY,
            SPH_IRD_TAG_ASTROMETRY_CALIB);
    sph_ird_common_science_process_cubes(sci, self->rawframes_no_coro, NULL,
                               &left_frameset, &right_frameset, &total_frameset,
                                         NULL, NULL, NULL);
    left_nocoro = sph_common_science_combine(left_frameset, self->coll_alg, 0,
            1, -1);
    right_nocoro = sph_common_science_combine(right_frameset, self->coll_alg, 0,
            1, -1);
    result_nocoro = sph_common_science_combine(total_frameset, self->coll_alg,
            0, 1, -1);
    if (left_nocoro && right_nocoro) {
        flux_left_nocoro = cpl_image_get_absflux(
                sph_master_frame_extract_image(left_nocoro, 1));
        flux_right_nocoro = cpl_image_get_absflux(
                sph_master_frame_extract_image(right_nocoro, 1));
        sph_master_frame_delete(left_nocoro);
        left_nocoro = NULL;
        sph_master_frame_delete(right_nocoro);
        right_nocoro = NULL;
    }
    cpl_frameset_delete(left_frameset);
    left_frameset = NULL;
    cpl_frameset_delete(right_frameset);
    right_frameset = NULL;
    cpl_frameset_delete(total_frameset);
    total_frameset = NULL;
    sph_ird_common_science_process_cubes(sci, self->rawframes_no_coro, NULL,
            &left_frameset, &right_frameset, &total_frameset,
                                         NULL, NULL, NULL);
    left_coro = sph_common_science_combine(left_frameset, self->coll_alg, 0, 1,
            -1);
    right_coro = sph_common_science_combine(right_frameset, self->coll_alg, 0,
            1, -1);
    result_coro = sph_common_science_combine(total_frameset, self->coll_alg, 0,
            1, -1);
    if (left_coro && right_coro) {
        flux_left_coro = cpl_image_get_absflux(
                sph_master_frame_extract_image(left_coro, 1));
        flux_right_coro = cpl_image_get_absflux(
                sph_master_frame_extract_image(right_coro, 1));
        sph_master_frame_delete(left_coro);
        left_coro = NULL;
        sph_master_frame_delete(right_coro);
        right_coro = NULL;
    }
    cpl_frameset_delete(left_frameset);
    left_frameset = NULL;
    cpl_frameset_delete(right_frameset);
    right_frameset = NULL;
    cpl_frameset_delete(total_frameset);
    total_frameset = NULL;
    if (result_nocoro && result_coro) {
        cpl_propertylist_append_double(result_nocoro->properties,
                SPH_IRD_KEYWORD_FLUX_RIGHT_CORO, flux_right_coro);
        cpl_propertylist_append_double(result_nocoro->properties,
                SPH_IRD_KEYWORD_FLUX_LEFT_CORO, flux_left_coro);
        cpl_propertylist_append_double(result_nocoro->properties,
                SPH_IRD_KEYWORD_FLUX_LEFT_NO_CORO, flux_left_nocoro);
        cpl_propertylist_append_double(result_nocoro->properties,
                SPH_IRD_KEYWORD_FLUX_RIGHT_NO_CORO, flux_right_nocoro);
        if (flux_left_coro) {
            cpl_propertylist_append_double(result_nocoro->properties,
                    SPH_IRD_KEYWORD_CORO_ATTN_LEFT,
                    flux_left_coro / flux_left_nocoro);
        } else {
            cpl_propertylist_append_double(result_nocoro->properties,
                    SPH_IRD_KEYWORD_CORO_ATTN_LEFT, SPH_IRD_INVALID_DOUBLE);
        }
        if (flux_right_coro) {
            cpl_propertylist_append_double(result_nocoro->properties,
                    SPH_IRD_KEYWORD_CORO_ATTN_RIGHT,
                    flux_right_coro / flux_right_nocoro);
        } else {
            cpl_propertylist_append_double(result_nocoro->properties,
                    SPH_IRD_KEYWORD_CORO_ATTN_RIGHT, SPH_IRD_INVALID_DOUBLE);
        }
        sph_master_frame_divide_master_frame(result_nocoro, result_coro);
        rawframes = cpl_frameset_new();
        sph_utils_copy_frames(self->rawframes_coro, rawframes);
        sph_utils_copy_frames(self->rawframes_no_coro, rawframes);
        sph_master_frame_save_dfs(result_nocoro, self->flux_calib_outfilename,
                rawframes, NULL, self->inparams, SPH_IRD_TAG_FLUX_CALIB_CALIB,
                SPH_RECIPE_NAME_IRD_FLUX_CALIB, SPH_PIPELINE_NAME_IRDIS, plist);
        sph_master_frame_delete(result_nocoro);
        result_nocoro = NULL;
        sph_master_frame_delete(result_coro);
        result_coro = NULL;
    }

    if (dark) {
        sph_master_frame_delete(dark);
        dark = NULL;
    }
    if (flat) {
        sph_master_frame_delete(flat);
        flat = NULL;
    }
    sph_ird_common_science_delete(sci);
    sci = NULL;
    return (int) cpl_error_get_code();

}

/**@}*/
