/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <math.h>
#include "sph_common_keywords.h"
#include "sph_ird_keywords.h"

#include "sph_ird_tags.h"
#include "sph_ird_science_dpi.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_ird_instrument_model.h"
#include "sph_framecombination.h"
#include "sph_common_science.h"
#include "sph_ird_common_science.h"
#include "sph_ird_common_dpi.h"
#include "sph_keyword_manager.h"
#include "sph_differential_imaging.h"
#include "sph_fits.h"
#include "sph_filemanager.h"
#include "sph_simple_adi.h"
/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IRD_SCIENCE_DPI_GENERAL;
extern sph_error_code SPH_IRD_SCIENCE_DPI_FRAMES_MISSING;

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ird_science_dpi_run Create Master Dark Recipe
 *
 * This module provides the algorithm inplementation for the creation of the
 * master dakr
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ird_science_dpi.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_ird_science_dpi recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ird_science_dpi_run(sph_ird_science_dpi* self) {
    sph_double_image* ipframe = NULL;
    cpl_propertylist* plist = NULL;
    sph_master_frame* leftFOV = NULL;
    sph_master_frame* rightFOV = NULL;
    cpl_frameset* left_frames_out = NULL;
    cpl_frameset* right_frames_out = NULL;
    sph_ird_common_science* sci = NULL;
    sph_transform* transform = NULL;
    cpl_frameset* total_frames_out = NULL;
    cpl_frameset* adi_frames = NULL;
    int master_dark_present = 0;
    int ff = 0;
    int pp = 0;
    int np = 0;
    cpl_frame* oframe = NULL;
    cpl_frame* lframe = NULL;
    cpl_frame* rframe = NULL;
    cpl_frame* tframe = NULL;
    char* tfname = NULL;
    sph_cube* dpi_cube = NULL;
    char* basename = NULL;
    cpl_vector* angles = NULL;
    cpl_frameset* fctable_frames = NULL;



    /*------------------------------------------------------------------
     -  selecting the right dark
     --------------------------------------------------------------------*/
    if(self->dark_frame)
    	master_dark_present = 1;

    if ((self->skybg_fit_frame || self->skybg_frame || self->insbg_fit_frame || self->insbg_frame) && self->dark_frame)
	      cpl_frame_delete(self->dark_frame); // delete dark because it's going to be re-assigned below

    if(self->skybg_fit_frame){
       	self->dark_frame = cpl_frame_duplicate(self->skybg_fit_frame);
       	cpl_msg_info(cpl_func,"Using SKY_BG_FIT frame as background!");
    }
    else {
    	if(self->skybg_frame){
    		self->dark_frame = cpl_frame_duplicate(self->skybg_frame);
    		cpl_msg_info(cpl_func,"Using SKY_BG frame as background!");
           	}
    	else {
    		if(self->insbg_fit_frame){
    			self->dark_frame = cpl_frame_duplicate(self->insbg_fit_frame);
    			cpl_msg_info(cpl_func,"Using INS_BG_FIT frame as background!");
    		}
           	else{
           		if(self->insbg_frame){
           			self->dark_frame = cpl_frame_duplicate(self->insbg_frame);
           		    cpl_msg_info(cpl_func,"Using INS_BG frame as background!");
           		}
           		else {
           			if(self->dark_frame){
           				if(master_dark_present)
           					cpl_msg_info(cpl_func,"Using master dark as background!");
           			}
           			else {
           				sph_error_raise( SPH_IRD_SCIENCE_DPI_FRAMES_MISSING,
                                __FILE__, __func__, __LINE__ ,
                                SPH_ERROR_ERROR,
           				                         "No dark or background whatsoever supplied! "
           				                         "Must have one of %s, %s, %s, %s, or %s!",
           				                         SPH_IRD_TAG_INS_BG_FIT_CALIB,
           				                         SPH_IRD_TAG_INS_BG_CALIB,
           				                         SPH_IRD_TAG_SKY_BG_FIT_CALIB,
           				                         SPH_IRD_TAG_SKY_BG_CALIB,
           				                         SPH_IRD_TAG_DARK_CALIB);
           				cpl_ensure_code(0, cpl_error_get_code());

           			}

           		}
           	}
    	}
    }





    if (self->transform_method == 0) {
        if (self->filter_method == 0) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_NONE, 0.0,
                    0.0);
        } else if (self->filter_method == 1) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_TH,
                    self->filter_radius, 0.0);
        } else if (self->filter_method == 2) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_FERMI,
                    self->filter_radius, self->fermi_temp);
        } else if (self->filter_method == 3) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_BUTTER,
                    self->butter_pass, self->butter_stop);
        }
    } else if (self->transform_method == 1) {
        transform = sph_transform_new_cpl_warp();
    }

    if (self->fcasciis != NULL) {
        fctable_frames = cpl_frameset_duplicate(self->fcasciis);
        if (fctable_frames == NULL) {
            goto EXIT;
        }
    } else {
        if (self->fcframe) {
            fctable_frames = sph_ird_common_science_create_fctables(
                    self->rawframes, self->fcframe, self->save_interprod, 1);
            if (fctable_frames == NULL) {
                goto EXIT;
            }
        }
    }

    plist = cpl_propertylist_new();

    sci = sph_ird_common_science_new(self->inframes, self->inparams,
            self->dark_frame, self->flat_frame, self->static_badpixel_frame,
            self->distmap_frame, NULL, self->flat_frame,
            fctable_frames,
            SPH_RECIPE_NAME_IRD_SCIENCE_DPI, SPH_IRD_TAG_SCIENCE_DPI_CALIB);
    sci->flag_sdi = 1;
    sci->rotate_flag = 0;
    sci->maxr = self->maxr;
    sci->minr = self->minr;
    if (transform)
        sph_ird_common_science_set_transform(sci, transform);
    sph_ird_common_science_process_cubes(sci, self->rawframes, fctable_frames,
                                         &left_frames_out, &right_frames_out,
                                         self->save_interprod ?
                                         &total_frames_out :
                                         NULL,
                                      SPH_IRD_TAG_SCIENCE_DPI_CALIB_LEFT_CUBE,
                                      SPH_IRD_TAG_SCIENCE_DPI_CALIB_RIGHT_CUBE,
                                      SPH_IRD_TAG_SCIENCE_DPI_CALIB_TOTAL_CUBE);

    if (self->flag_adi == 0) {
        leftFOV = sph_common_science_combine(left_frames_out, self->coll_alg, 0,
                1, -1);
        rightFOV = sph_common_science_combine(right_frames_out, self->coll_alg,
                0, 1, -1);
        if (leftFOV && rightFOV) {
            ipframe = sph_ird_common_dpi_create_ipframe(leftFOV, rightFOV);
        } else {
            SPH_ERR("Could not create product frame.");
        }
        if (ipframe) {
            cpl_propertylist_append_double(plist, SPH_IRD_KEYWORD_DPI_TOTAL_POL,
                    sph_master_frame_get_absflux(ipframe->pframe, NULL));
            cpl_propertylist_append_double(plist, SPH_IRD_KEYWORD_DPI_TOTAL_INT,
                    sph_master_frame_get_absflux(ipframe->iframe, NULL));
            /* FIXME: Set WCS to dummy (pixel) value for now */
            sph_utils_reset_wcs_12d(plist);
            sph_double_image_save_dfs(ipframe, self->science_dpi_outfilename,
                    self->inframes, cpl_frameset_get_first(self->rawframes),
                    self->inparams, SPH_IRD_TAG_SCIENCE_DPI_CALIB,
                    SPH_RECIPE_NAME_IRD_SCIENCE_DPI, SPH_PIPELINE_NAME_IRDIS,
                    plist);
        }
    } else {
        if (total_frames_out == NULL)
            total_frames_out = cpl_frameset_new();
        cpl_ensure_code(
                cpl_frameset_get_size(self->rawframes) == cpl_frameset_get_size(left_frames_out),
                CPL_ERROR_INCOMPATIBLE_INPUT);
        cpl_ensure_code(
                cpl_frameset_get_size(self->rawframes) == cpl_frameset_get_size(right_frames_out),
                CPL_ERROR_INCOMPATIBLE_INPUT);
        for (ff = 0; ff < cpl_frameset_get_size(self->rawframes); ++ff) {
            oframe = cpl_frameset_get_position(self->rawframes, ff);
            lframe = cpl_frameset_get_position(left_frames_out, ff);
            rframe = cpl_frameset_get_position(right_frames_out, ff);
            np = sph_fits_get_nplanes(cpl_frame_get_filename(lframe), 0);
            tframe = cpl_frame_duplicate(lframe);
            basename = sph_filemanager_remove_dir(
                    cpl_frame_get_filename(oframe));
            tfname = sph_filemanager_new_filename_from_base(basename, "dpi");
            cpl_free(basename);
            basename = NULL;
            cpl_frame_set_filename(tframe, tfname);
            dpi_cube = sph_cube_new_dfs(tfname, sci->inframes, oframe,
                    sci->inparams, sci->tag, sci->recipe,
                    SPH_PIPELINE_NAME_IRDIS, NULL);

            for (pp = 0; pp < np; ++pp) {
                leftFOV = sph_master_frame_load_(lframe,
                        pp);
                rightFOV = sph_master_frame_load_(rframe,
                        pp);
                ipframe = sph_ird_common_dpi_create_ipframe(leftFOV, rightFOV);
                sph_cube_append_double_image(dpi_cube, ipframe, (float) pp);
                sph_master_frame_delete(leftFOV);
                leftFOV = NULL;
                sph_master_frame_delete(rightFOV);
                rightFOV = NULL;
                sph_double_image_delete(ipframe);
                ipframe = NULL;
            }
            cpl_free(tfname);
            tfname = NULL;
            sph_cube_finalise_file(dpi_cube->filename);
            sph_cube_delete(dpi_cube);
            dpi_cube = NULL;
            cpl_frameset_insert(total_frames_out, tframe);
        }
        angles = sph_common_science_extract_angles_from_fctables(
                fctable_frames);

        adi_frames = sph_simple_adi_process_cubes(total_frames_out,
                self->rawframes, angles, sci->transform, sci->irdmodel, 0, 1, 2,
                self->fullset);

        // Since ADI only works on master frame for the moment,
        // only fill the P frame part of double image
        ipframe = sph_double_image_new_empty();
        ipframe->pframe = sph_common_science_combine(adi_frames, self->coll_alg,
                0, 1, -1);

        sph_master_frame_save_dfs(ipframe->pframe,
                self->science_dpi_outfilename, self->inframes,
                cpl_frameset_get_first(self->rawframes), self->inparams,
                SPH_IRD_TAG_SCIENCE_DPI_CALIB, SPH_RECIPE_NAME_IRD_SCIENCE_DPI,
                SPH_PIPELINE_NAME_IRDIS, NULL);

        sph_double_image_delete(ipframe);
        ipframe = NULL;
        cpl_vector_delete(angles);
        angles = NULL;
        cpl_frameset_delete(adi_frames);
        adi_frames = NULL;
    }

    if (self->save_interprod) {
        if (total_frames_out != NULL) {
            cpl_frameset_join(self->inframes, total_frames_out);
        }
        if (left_frames_out != NULL) {
            cpl_frameset_join(self->inframes, left_frames_out);
        }
        if (right_frames_out != NULL) {
            cpl_frameset_join(self->inframes, right_frames_out);
        }
    }
    cpl_frameset_delete(left_frames_out);
    left_frames_out = NULL;
    cpl_frameset_delete(right_frames_out);
    right_frames_out = NULL;
    cpl_frameset_delete(total_frames_out);
    total_frames_out = NULL;
    if(self->save_addprod){
        if (leftFOV) {
            sph_ird_common_science_put_wcs(plist,512.0,512.0);
            sph_master_frame_save_dfs(leftFOV, self->science_dpi_outfilename_left,
                self->inframes, cpl_frameset_get_first(self->rawframes),
                self->inparams, SPH_IRD_TAG_SCIENCE_DPI_CALIB_LEFT,
                SPH_RECIPE_NAME_IRD_SCIENCE_DPI, SPH_PIPELINE_NAME_IRDIS,
                plist);
        }
        if (rightFOV) {
            sph_ird_common_science_put_wcs(plist,512.0,512.0);
            sph_master_frame_save_dfs(rightFOV, self->science_dpi_outfilename_right,
                self->inframes, cpl_frameset_get_first(self->rawframes),
                self->inparams, SPH_IRD_TAG_SCIENCE_DPI_CALIB_RIGHT,
                SPH_RECIPE_NAME_IRD_SCIENCE_DPI, SPH_PIPELINE_NAME_IRDIS,
                plist);
        }
    }

    EXIT: sph_double_image_delete(ipframe);
    ipframe = NULL;
    cpl_propertylist_delete(plist);
    plist = NULL;
    sph_master_frame_delete(leftFOV);
    leftFOV = NULL;
    sph_master_frame_delete(rightFOV);
    rightFOV = NULL;
    sph_ird_common_science_delete(sci);
    sci = NULL;
    sph_filemanager_delete(self->save_interprod);
    return (int) cpl_error_get_code();
}

/**@}*/

