/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <cpl.h>
#include <math.h>
#include "sph_zpl_master_dark_imaging.h"
#include "sph_zpl_common_preproc.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_zpl_utils.h"
#include "sph_zpl_exposure_imaging.h"
#include "sph_double_image.h"
#include "sph_error.h"
#include "sph_utils.h"
#include "sph_common_keywords.h"
#include "sph_framecombination.h"

/*-------------------------------------------------------------------------------
 * The Structure Definition
 *
 * This structure contains the members of the sph_zpl_master_dark_imaging_cunit that are
 * to reduce data for the one zimpol channel (camera). This structure is used as
 * an interface between a sphere "standard" recipe structure and re-designed
 * zimpol "_run" function which treats both zimpol channel.
 *
 * ------------------------------------------------------------------------------
 */

typedef struct _sph_zpl_master_dark_imaging_camera_unit {
    cpl_frameset* inframes; /* The recipe input frames */

    cpl_frameset* current_raw_frameset; /* The frameset for one product */

    cpl_parameterlist* inparams; /* The recipe input parameters */

    cpl_parameterlist* framecomb_parameterlist; /* The recipe input parameters */

    char* master_dark_imaging_outfilename; /* The parameter of zpl.master_dark_imaging.outfilename_cam1 _cam2 */

    short subtract_overscan; /* The parameter of zpl.master_dark_imaging.subtract_overscan */

    int coll_alg; /* The parameter of zpl.master_dark_imaging.coll_alg */

    int clean_mean_reject_high; /* The parameter of zpl.master_dark_imaging.coll_alg.clean_mean.reject_high */

    int clean_mean_reject_low; /* The parameter of zpl.master_dark_imaging.coll_alg.clean_mean.reject_low */

    double clean_mean_sigma; /* The parameter of zpl.master_dark_imaging.clean_mean.sigma */

    double sigma_clip; /* The parameter of zpl.master_dark_imaging.sigma_clip */

    short  keep_intermediate                     ; /* The parameter of zpl.master_dark_imaging.keep_intermediate */

    cpl_frameset* rawframes; /* The ZPL_BIAS_IMAGING_RAW frames */

    cpl_frameset* preproc_frames; /* The ZPL_DARK_IMAGING_PREPROC_CAM1 _CAM2 frames */

    cpl_frame* master_bias_imaging_frame; /* The ZPL_MASTER_BIAS_IMAGING_CAM1 _CAM2 frames */

    sph_double_image* master_dark_imaging_doubleimage; /* The ZPL_MASTER_DARK_IMAGING_CAM1 _CAM2 product */

    char* eso_pro_catg; /*Here the ZPL_MASTER_DARK_IMAGING_CAM1 or _CAM2 will be written */

} sph_zpl_master_dark_imaging_cunit;

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 Private function prototypes
 -----------------------------------------------------------------------------*/
static sph_zpl_master_dark_imaging_cunit* _sph_zpl_master_dark_imaging_cunit_create(
        sph_zpl_master_dark_imaging* self, int camera);
static sph_error_code _sph_zpl_master_dark_imaging_cunit_delete(
        sph_zpl_master_dark_imaging_cunit* cunit);
static cpl_error_code _sph_zpl_master_dark_imaging_cunit_run(
        sph_zpl_master_dark_imaging_cunit* cunit);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_zpl_master_dark_imaging_run Create Master Dark Imaging Recipe
 *
 * This module provides the algorithm implementation for the creation of the
 * master dark
 *
 * @par Synopsis:
 * @code
 *   #include "sph_zpl_master_dark_imaging.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_zpl_master_dark_imaging recipe.
 The error code returned is always a cpl error code (to allow maximal compatibility
 with esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_zpl_master_dark_imaging_run(
        sph_zpl_master_dark_imaging* self) {
    sph_zpl_master_dark_imaging_cunit* cunit = NULL;
    cpl_error_code recipe_error = CPL_ERROR_NONE;

    SPH_INFO_MSG("Starting sph_zpl_master_dark_imaging_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return cpl_error_get_code();
    }

    if (self->preproc_frames_cam1 == NULL && self->preproc_frames_cam2 == NULL && self->preproc_frames == NULL) {

        if (self->rawframes) {
            self->preproc_frames_cam1 = cpl_frameset_new();
            self->preproc_frames_cam2 = cpl_frameset_new();
            SPH_INFO_MSG("Start pre-processing...")
            //recipe_error = sph_zpl_master_dark_preproc( self );
            recipe_error = sph_zpl_common_preproc_imaging_recipe(self->rawframes,
                    self->outfilename_cam1, self->outfilename_cam2,
                    self->preproc_frames_cam1, self->preproc_frames_cam2);
        } else {
            SPH_ERR(
                    "There are neither raw frames nor pre-processed ones! Please verify that the tags are set correctly!");
            return sph_error_get_last_code();
        }
        if (recipe_error) {
            SPH_ERR("Pre-processing step is failed.");
            cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }


        if ( sph_zpl_utils_check_format( self->rawframes ) == 1) {
            SPH_INFO_MSG("New style format of the raw data...")
            if ( cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 ) ) {
                self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam2 );
                cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
            } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam1 );
                cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
            } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                SPH_WARNING("Rawdata set contains fits files from camera-1 and camera-2. Please, make sure that you provide calibrations for both cameras."
                        "The products will be created individually for each camera (off-line pipeline extended support!")
            } else {
                SPH_ERR("Pre-processing step is failed.");
                cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
                return (int) recipe_error;
            }
        } else if ( sph_zpl_utils_check_format( self->rawframes ) == 0) {
            SPH_INFO_MSG("Old style format of the raw data...")
        } else {
            SPH_ERR("Not supported raw data format or new format and old format are mixed in the input dataset. "
                    "Please, make sure that you use either old or new style of the rawdata format! Stop recipe...");
            cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }
    }

    //reduce data for the camera-1
    //create cunit structure
    if (self->preproc_frames_cam1) {
        SPH_INFO_MSG("Create cunit structure for the camera-1...");
        cunit = _sph_zpl_master_dark_imaging_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);

        if (cunit) {

            recipe_error = _sph_zpl_master_dark_imaging_cunit_run(cunit);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("reduction for the camera-1 is failed!");
            } else {
                //put the output product in the main self structure
                self->master_dark_imaging_doubleimage_cam1 =
                        sph_double_image_duplicate(
                                cunit->master_dark_imaging_doubleimage);
            }
        } else {
            SPH_ERR(
                    "Creation of the cunit structure for the camera-1 is failed...trying for the camera-2")
        }
    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-1 -> nothing to reduce..trying for the camera-2");
    }
    if (cunit) {
        _sph_zpl_master_dark_imaging_cunit_delete(cunit);
        cunit = NULL;
    }

    //reset an error log system to reduce data for camera-2
    SPH_RAISE_CPL_RESET;

    //reduce data for the camera-2
    //create cunit structure
    if (self->preproc_frames_cam2) {
        SPH_INFO_MSG("Create cunit structure for the camera-2...");
        cunit = _sph_zpl_master_dark_imaging_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID);

        if (cunit) {
            recipe_error = _sph_zpl_master_dark_imaging_cunit_run(cunit);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("reduction for the camera-2 is failed!");
            } else {
                //put the output product in the main self structure
                self->master_dark_imaging_doubleimage_cam2 =
                        sph_double_image_duplicate(
                                cunit->master_dark_imaging_doubleimage);
            }
        } else {
            SPH_ERR(
                    "Creation of the cunit structure for the camera-2 is failed!")
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-2 -> nothing to reduce.");
    }

    if (cunit) {
        _sph_zpl_master_dark_imaging_cunit_delete(cunit);
        cunit = NULL;
    }

    //reset an error log system to reduce data
    SPH_RAISE_CPL_RESET;

    //reduce data for the new style data (on-line pipeline)
    //create cunit structure
    if (self->preproc_frames) {
        SPH_INFO_MSG("Create cunit structure for the new style format...");
        cunit = _sph_zpl_master_dark_imaging_cunit_create(self, SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID);

        if (cunit) {
            recipe_error = _sph_zpl_master_dark_imaging_cunit_run(cunit);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("reduction is failed for the newstyle pre-processed data!");
            } else {
                //put the output product in the main self structure
                self->master_dark_imaging_doubleimage =
                        sph_double_image_duplicate( cunit->master_dark_imaging_doubleimage );
            }
        } else {
            SPH_ERR(
                    "Creation of the cunit structure newstyle pre-processed data is failed!")
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the new style format -> nothing to reduce.");
    }

    if (cunit) {
         _sph_zpl_master_dark_imaging_cunit_delete(cunit);
         cunit = NULL;
     }


    if (self->master_dark_imaging_doubleimage_cam1 == NULL
            && self->master_dark_imaging_doubleimage_cam2 == NULL && self->master_dark_imaging_doubleimage == NULL) {
        SPH_ERR(" No outputs have been created !!!");
    }

    if (!self->keep_intermediate && self->rawframes != NULL) {
        SPH_INFO_MSG("Unliking intermediate data!");
        if (self->preproc_frames_cam1) sph_utils_frames_unlink(self->preproc_frames_cam1);
        if (self->preproc_frames_cam2) sph_utils_frames_unlink(self->preproc_frames_cam2);
        if (self->preproc_frames) sph_utils_frames_unlink(self->preproc_frames);
    }

    SPH_INFO_MSG("sph_zpl_master_dark_imaging_run...End");
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}

sph_zpl_master_dark_imaging_cunit* _sph_zpl_master_dark_imaging_cunit_create(
        sph_zpl_master_dark_imaging* self, int camera_id) {
    sph_zpl_master_dark_imaging_cunit* result = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return NULL;
    }

    result = cpl_calloc(1, sizeof(sph_zpl_master_dark_imaging_cunit));

    if (result == NULL) {
        SPH_ERR( "Could not allocate the structure for run unit.");
        return result;
    }

    //copy and duplicate some elements of the initial "sph_zpl_masters_dark_imaging" structure into the cunit structure
    result->inframes = self->inframes;
    result->inparams = self->inparams;
    result->framecomb_parameterlist = self->framecomb_parameterlist;
    result->subtract_overscan = self->subtract_overscan;
    result->coll_alg = self->coll_alg;
    result->clean_mean_reject_high = self->clean_mean_reject_high;
    result->clean_mean_reject_low = self->clean_mean_reject_low;
    result->clean_mean_sigma = self->clean_mean_sigma;
    result->sigma_clip = self->sigma_clip;
    result->keep_intermediate = self->keep_intermediate;

    result->rawframes = self->rawframes;
    if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {
        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam1);
        result->master_dark_imaging_outfilename = cpl_strdup(
                self->master_dark_imaging_outfilename_cam1);
        result->eso_pro_catg = cpl_strdup(
                SPH_ZPL_TAG_DARK_IMAGING_CALIB_CAM1);
        if (self->master_bias_imaging_frame_cam1) {
            result->master_bias_imaging_frame = cpl_frame_duplicate(
                    self->master_bias_imaging_frame_cam1);
        }
    } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {
        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam2);
        result->master_dark_imaging_outfilename = cpl_strdup(
                self->master_dark_imaging_outfilename_cam2);
        result->eso_pro_catg = cpl_strdup(
                SPH_ZPL_TAG_DARK_IMAGING_CALIB_CAM2);
        if (self->master_bias_imaging_frame_cam2) {
            result->master_bias_imaging_frame = cpl_frame_duplicate(
                    self->master_bias_imaging_frame_cam2);
        }
    } else if ( camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID ) {
        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames);
        result->master_dark_imaging_outfilename = cpl_strdup(
                self->master_dark_imaging_outfilename);
        result->eso_pro_catg = cpl_strdup(
                SPH_ZPL_TAG_DARK_IMAGING_CALIB);
        if (self->master_bias_imaging_frame) {
            result->master_bias_imaging_frame = cpl_frame_duplicate(
                    self->master_bias_imaging_frame);
        }
    } else {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Wrong camera id = %d (it must be 1 or 2 for oldstyle format, 0 for newstyle format)",
                camera_id);
        _sph_zpl_master_dark_imaging_cunit_delete(result);
        result = NULL;
    }

    if (sph_error_get_last_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Post condition error check shows the following error %d",
                sph_error_get_last_code());
        _sph_zpl_master_dark_imaging_cunit_delete(result);
        result = NULL;
    }

    return result;

}

sph_error_code _sph_zpl_master_dark_imaging_cunit_delete(
        sph_zpl_master_dark_imaging_cunit* cunit) {
    sph_error_code rerr = CPL_ERROR_NONE;

    if (cunit == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return rerr;
    }

    /* Code to delete recipe pointers GENERATED DO NOT EDIT */

    if (cunit->preproc_frames != NULL) {
        cpl_frameset_delete(cunit->preproc_frames);
        cunit->preproc_frames = NULL;
    }

    if (cunit->master_dark_imaging_doubleimage) {
        sph_double_image_delete(cunit->master_dark_imaging_doubleimage);
        cunit->master_dark_imaging_doubleimage = NULL;
    }
    if (cunit->master_dark_imaging_outfilename)
        cpl_free(cunit->master_dark_imaging_outfilename);
    if (cunit->eso_pro_catg)
        cpl_free(cunit->eso_pro_catg);
    if (cunit->preproc_frames)
        cpl_frameset_delete(cunit->preproc_frames);

    //calibrations
    if (cunit->master_bias_imaging_frame)
        cpl_frame_delete(cunit->master_bias_imaging_frame);

    if (cunit->current_raw_frameset) {
        cpl_frameset_delete(cunit->current_raw_frameset);
        cunit->current_raw_frameset = NULL;
    }
    sph_polygon_free_all();
    cpl_free(cunit);
    return rerr;

}

cpl_error_code _sph_zpl_master_dark_imaging_cunit_run(
        sph_zpl_master_dark_imaging_cunit* self) {
    sph_double_image*       master_bias_imaging_doubleimage = NULL;
    cpl_frameset*           ovsc_subtract_frames = NULL;
    int                     recipe_error = CPL_ERROR_NONE;
    cpl_propertylist*       pl = NULL;

    SPH_INFO_MSG("Starting static _sph_zpl_master_dark_imaging_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (self->preproc_frames) {

        if (self->subtract_overscan) {
            SPH_INFO_MSG("Subtract overscans from preproc_frames...")
            ovsc_subtract_frames = sph_zpl_utils_subtract_overscans(
                    self->preproc_frames);
            if (ovsc_subtract_frames) {
                self->master_dark_imaging_doubleimage =
                        sph_framecombination_double_image_from_frameset(
                                ovsc_subtract_frames, self->coll_alg,
                                self->framecomb_parameterlist);
                if (!self->keep_intermediate) {
                    sph_utils_frames_unlink(ovsc_subtract_frames);
                }
                cpl_frameset_delete(ovsc_subtract_frames); ovsc_subtract_frames = NULL;
            } else {
                SPH_ERROR_RAISE_WARNING(
                        SPH_ERROR_WARNING,
                        "Overscans could not be subtracted, trying to proceed further.")
                self->master_dark_imaging_doubleimage =
                        sph_framecombination_double_image_from_frameset(
                                self->preproc_frames, self->coll_alg,
                                self->framecomb_parameterlist);
            }
        } else {
            self->master_dark_imaging_doubleimage =
                    sph_framecombination_double_image_from_frameset(
                            self->preproc_frames, self->coll_alg,
                            self->framecomb_parameterlist);
        }

    } else {
        SPH_ERR("no preproc_frames are set up; please verify TAGS")
        return sph_error_get_last_code();
    }

    if (!self->master_dark_imaging_doubleimage) {
        sph_error_raise(
                SPH_ERROR_GENERAL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "Could not create zimpol master dark imaging:\n"
                        "sph_framecombination_double_from_frameset returns null pointer.\n"
                        "cpl error code is: %d", cpl_error_get_code());
        return cpl_error_get_code();
    }

    //if master bias imaging exists load and subtract it from the master dark
    if (self->master_bias_imaging_frame) {
        master_bias_imaging_doubleimage = sph_double_image_load(
                cpl_frame_get_filename(self->master_bias_imaging_frame), 0);
        if (master_bias_imaging_doubleimage) {
            recipe_error = sph_double_image_subtract_double_image(
                    self->master_dark_imaging_doubleimage,
                    master_bias_imaging_doubleimage);
            if (recipe_error != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by subtracting master bias imaging:\n"
                                "cpl error code is: %d", cpl_error_get_code());
                sph_double_image_delete(master_bias_imaging_doubleimage);
                master_bias_imaging_doubleimage = NULL;
                return cpl_error_get_code();
            }
            sph_double_image_delete(master_bias_imaging_doubleimage);
        }
        SPH_INFO_MSG("Subtracted a given master bias imaging.")
    }

    //quality check (calculate statistical moments for the master dark)
    recipe_error = sph_double_image_quality_check_names(
            self->master_dark_imaging_doubleimage,
			NULL, &pframe_names_leakage);
    if (recipe_error) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Could not create zimpol master dark imaging:\n"
                        "sph_double_image_quality_check returns cpl error.\n"
                        "cpl error code is: %d", recipe_error);
        return recipe_error;
    }SPH_INFO_MSG("Quality check calculated without badpixels.");

    //mask hot/bad pixels based on the given sigma_clip parameter
    recipe_error = sph_double_image_mask_sigma(
            self->master_dark_imaging_doubleimage, self->sigma_clip);
    if (recipe_error) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not create zimpol master dark:\n"
                        "sph_double_image_mask_sigma returns cpl error.\n"
                        "cpl error code is: %d", recipe_error);
        return recipe_error;
    }SPH_INFO_MSG("Calculated mask based on the sigma clipping");

    //re-calculate qc keywords to exclude bad pixels from the result
    recipe_error = sph_double_image_quality_check_names(
            self->master_dark_imaging_doubleimage,
			NULL, &pframe_names_leakage);
    if (recipe_error) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Could not create zimpol master dark imaging:\n"
                        "sph_double_image_quality_check returns cpl error.\n"
                        "cpl error code is: %d", recipe_error);
        return recipe_error;
    }SPH_INFO_MSG(
            "Quality check re-done to exclude badpixels from the calculation.");

    /* Attention:
     * The following function includes a step to update SPH_COMMON_KEYWORD_SPH_TYPE by
     * SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_DOUBLE_IMAGE. It is necessary because cpl_dsf_save_image
     * updates the quad image property list by the property list from the first input frames.
     * In the case of some misbehavior exclude this step for checking.
     */

    if (self->rawframes) {
        SPH_INFO_MSG(
                "Save master dark imaging as a dfs product: header info taken from the first raw frame.");
        pl = sph_zpl_utils_get_camera_header(self->preproc_frames, self->rawframes);
        cpl_propertylist_append(pl, self->master_dark_imaging_doubleimage->properties);

        cpl_propertylist_update_string(pl, SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_RIGHT,
        		SPH_IMAGING_PFRAME_NAME_LEAKAGE_NAME);

        recipe_error = sph_double_image_save_dfs(
                self->master_dark_imaging_doubleimage,
                self->master_dark_imaging_outfilename,
                self->inframes,
                cpl_frameset_get_first(self->rawframes),
                self->inparams,
                self->eso_pro_catg, //SPH_ZPL_TAG_DARK_IMAGING_CALIB,
                SPH_RECIPE_NAME_ZPL_MASTER_DARK_IMAGING,
                SPH_PIPELINE_NAME_ZIMPOL,
                pl);
        if ( pl ) {
            cpl_propertylist_delete( pl ); pl = NULL;
        }
    } else if (self->preproc_frames) {
        SPH_INFO_MSG(
                "Save master dark imaging as a dfs product: header info taken from the pre-processed frame.")
        recipe_error = sph_double_image_save_dfs(
                self->master_dark_imaging_doubleimage,
                self->master_dark_imaging_outfilename,
                self->inframes,
                cpl_frameset_get_first(self->preproc_frames),
                self->inparams,
                self->eso_pro_catg, //SPH_ZPL_TAG_DARK_IMAGING_CALIB,
                SPH_RECIPE_NAME_ZPL_MASTER_DARK_IMAGING,
                SPH_PIPELINE_NAME_ZIMPOL,
                self->master_dark_imaging_doubleimage->properties);

    } else {
        SPH_ERR("Neither raw frames nor pre-processed frames provided!");
        recipe_error = sph_error_get_last_code();
    }

    if (recipe_error) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Could not create zimpol master dark imaging:\n"
                        "sph_double_image_save returns error code.\n"
                        "cpl error code is: %d", recipe_error);
    }

    SPH_INFO_MSG("static _sph_zpl_master_dark_imaging_cunit_run...End");
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}
/**@}*/
