/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <libgen.h>
#include <cpl.h>
#include <math.h>
#include "sph_zpl_science_imaging.h"
#include "sph_zpl_common_preproc.h"
#include "sph_zpl_exposure_imaging.h"
#include "sph_double_image.h"
#include "sph_error.h"
#include "sph_utils.h"
#include "sph_zpl_utils.h"
#include "sph_common_keywords.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_filemanager.h"
#include "sph_keywordnode.h"
#include "sph_common_science.h"
#include "sph_fctable.h"
#include "sph_transform.h"
#include "sph_strehl.h"
#include "sph_phot.h"
#include "sph_zpl_subtract_dark_scaled.h"

/*-------------------------------------------------------------------------------
 * The Structure Definition
 *
 * This structure contains the members of the sph_zpl_science_imaging_camera_unit that are
 * to reduce data for the one zimpol channel (camera). This structure is used as
 * an interface between a sphere "standard" recipe structure and re-designed
 * zimpol "_run" function which treats both zimpol channel.
 *
 * ------------------------------------------------------------------------------
 */
typedef struct _sph_zpl_science_imaging_camera_unit {
    cpl_frameset* inframes; /* The recipe input frames */

    cpl_frameset* current_raw_frameset; /* The frameset for one product */

    cpl_parameterlist* inparams; /* The recipe input parameters */

    cpl_parameterlist* framecomb_parameterlist; /* The recipe input parameters */

    char* science_imaging_outfilename; /* The parameter of zpl.science_imaging.outfilename_cam1 or _cam2 */

    short subtract_overscan; /* The parameter of zpl.science_p1.subtract_overscan */

    int coll_alg; /* The parameter of zpl.science_imaging.coll_alg */

    double filter_radius; /* The parameter of zpl.science_imaging.filter_radius */

    short  keep_intermediate ; /* The parameter of zpl.science_imaging.keep_intermediate */

    short save_interprod; /* The parameter of zpl.science_imaging.save_interprod */

    short star_center_iframe; /* The parameter of zpl.science_imaging.star_center_iframe */

    double center_xoffset; /* The parameter of zpl.science_imaging.center_xoffset_cam1 or _cam2 */

    double center_yoffset; /* The parameter of zpl.science_imaging.center_yoffset_cam1  or _cam2*/

    cpl_frameset* rawframes; /* The ZPL_SCIENCE_IMAGING_RAW frames */

    cpl_frameset* preproc_frames; /* The ZPL_SCIENCE_IMAGING_PREPROC_CAM1 or _CAM2 frames */

    cpl_frame* master_bias_frame; /* The ZPL_MASTER_BIAS_IMAGING_CAM1 or _CAM2frames */

    cpl_frame* master_dark_frame; /* The ZPL_MASTER_DARK_IMAGING_CAM1 or _CAM2 frames */

    cpl_frame* intensity_flat_frame; /* The ZPL_INT_FLAT_FIELD_IMAGING_CAM1 or CAM2 frames */

    cpl_frameset* fctable_frames; /* The ZPL_FIELD_CENTER_TABLE frames */

    cpl_frame*    filter_frame;   /* The ZPL_FILTER_TABLE calibration frame */

    cpl_frame*    star_center_frame ; /* The ZPL_STAR_CENTER_IMG_CALIB_CAM1 or CAM2 frames */

    sph_double_image* science_imaging_di; /* The ZPL_SCIENCE_IMAGING_REDUCED or _CAM1 or _CAM2 product */

    char* eso_pro_catg; /* Here the ZPL_SCIENCE_IMAGING_REDUCED_CAM1 or _CAM2 will be written */

} sph_zpl_science_imaging_camera_unit;

typedef enum {
	SPH_ZPL_MODE_CAMERA1,
	SPH_ZPL_MODE_CAMERA2,
	SPH_ZPL_MODE_ONLINE
}sph_zpl_mode;

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
 Private function prototypes
 -----------------------------------------------------------------------------*/
static sph_zpl_science_imaging_camera_unit* _sph_zpl_science_imaging_camera_unit_create(
        sph_zpl_science_imaging* self, int camera);
static sph_error_code _sph_zpl_science_imaging_camera_unit_delete(
		sph_zpl_science_imaging_camera_unit* camera_unit);

static cpl_error_code
    _sph_zpl_science_imaging_camera_unit_run(sph_zpl_science_imaging_camera_unit* self,
                                       const sph_zpl_science_imaging* cboth,
									   const sph_zpl_mode mode);

static void _calc_strehl_zpoint_and_append(const cpl_image * image_cam,
		const sph_zpl_science_imaging* cboth, cpl_propertylist* pl,
		const  sph_zpl_mode mode);

static cpl_boolean
_sph_zpl_science_imaging_is_ao_enabled(const cpl_frame * self);


static cpl_boolean
_sph_zpl_science_imaging_is_ao_enabled_from_list(const cpl_propertylist * list);

static cpl_boolean _sph_zpl_science_imaging_plist_contains_value(const cpl_propertylist * list,
		const char * key, const char * expected_value);
/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_zpl_science_imaging_run Science Imaging Recipe
 *
 * This module provides the algorithm implementation for the science recipe
 *
 * @par Synopsis:
 * @code
 *   #include "sph_zpl_science_imaging.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

cpl_error_code sph_zpl_science_imaging_run(sph_zpl_science_imaging* self) {
    sph_zpl_science_imaging_camera_unit* camera_unit = NULL;
    cpl_error_code recipe_error = CPL_ERROR_NONE;

    SPH_INFO_MSG("Starting sph_zpl_science_imaging_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return cpl_error_get_code();
    }

    if (self->preproc_frames_cam1 == NULL && self->preproc_frames_cam2 == NULL && self->preproc_frames == NULL) {

        if (self->rawframes) {
            self->preproc_frames_cam1 = cpl_frameset_new();
            self->preproc_frames_cam2 = cpl_frameset_new();
            SPH_INFO_MSG("Start pre-processing...")
            //recipe_error = sph_zpl_master_dark_preproc( self );
            recipe_error = sph_zpl_common_preproc_imaging_recipe(self->rawframes,
                    self->outfilename_cam1, self->outfilename_cam2,
                    self->preproc_frames_cam1, self->preproc_frames_cam2);
        } else {
            SPH_ERR(
                    "There are neither raw frames nor pre-processed ones! Please verify that the tags are set correctly!");
            return sph_error_get_last_code();
        }
        if (recipe_error) {
            SPH_ERR("Pre-processing step is failed.");
            cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }


        if ( sph_zpl_utils_check_format( self->rawframes ) == 1) {
            SPH_INFO_MSG("New style format of the raw data...")
            if ( cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 ) ) {
                self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam2 );
                cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
            } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam1 );
                cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
            } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                SPH_WARNING("Rawdata set contains fits files from camera-1 and camera-2. Please, make sure that you provide calibrations for both cameras."
                        "The products will be created individually for each camera (off-line pipeline extended support!")
            } else {
                SPH_ERR("Pre-processing step is failed.");
                cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
                return (int) recipe_error;
            }
        } else if ( sph_zpl_utils_check_format( self->rawframes ) == 0) {
            SPH_INFO_MSG("Old style format of the raw data...")
        } else {
            SPH_ERR("Not supported raw data format or new format and old format are mixed in the input dataset. "
                    "Please, make sure that you use either old or new style of the rawdata format! Stop recipe...");
            cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }
    }

    //reduce data for the camera-1
    //create camera_unit structure
    if (self->preproc_frames_cam1) {
        SPH_INFO_MSG("Create camera_unit structure for the camera-1...");
        camera_unit = _sph_zpl_science_imaging_camera_unit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);

        if (camera_unit) {

            recipe_error = _sph_zpl_science_imaging_camera_unit_run(camera_unit, self,
            		SPH_ZPL_MODE_CAMERA1);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("reduction for the camera-1 is failed!");
            } else {
                //put the output product in the main self structure
                self->science_imaging_di_cam1 = sph_double_image_duplicate(
                        camera_unit->science_imaging_di);
            }
        } else {
            SPH_ERR(
                    "Creation of the camera_unit structure for the camera-1 is failed...trying for the camera-2")
        }
    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-1 -> nothing to reduce..trying for the camera-2");
    }
    if (camera_unit) {
        _sph_zpl_science_imaging_camera_unit_delete(camera_unit);
        camera_unit = NULL;
    }

    //reset an error log system to reduce data for camera-2
    SPH_RAISE_CPL_RESET;

    //reduce data for the camera-2
    //create camera_unit structure
    if (self->preproc_frames_cam2) {
        SPH_INFO_MSG("Create camera_unit structure for the camera-2...");
        camera_unit = _sph_zpl_science_imaging_camera_unit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID);

        if (camera_unit) {
            recipe_error = _sph_zpl_science_imaging_camera_unit_run(camera_unit, self,
            		SPH_ZPL_MODE_CAMERA2);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("reduction for the camera-2 is failed!");
            } else {
                //put the output product in the main self structure
                self->science_imaging_di_cam2 = sph_double_image_duplicate(
                        camera_unit->science_imaging_di);
            }
        } else {
            SPH_ERR(
                    "Creation of the camera_unit structure for the camera-2 is failed!")
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-2 -> nothing to reduce.");
    }

    if (camera_unit) {
        _sph_zpl_science_imaging_camera_unit_delete(camera_unit);
        camera_unit = NULL;
    }

    //reset an error log system to reduce data
    SPH_RAISE_CPL_RESET;

    //reduce data for the new style data (on-line pipeline)
    //create camera_unit structure
    if ( self->preproc_frames ) {
        SPH_INFO_MSG("Create camera_unit structure for the new style format...");
        camera_unit = _sph_zpl_science_imaging_camera_unit_create(self, SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID);

        if (camera_unit) {
            recipe_error = _sph_zpl_science_imaging_camera_unit_run(camera_unit, self,
            		SPH_ZPL_MODE_ONLINE);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR("reduction is failed for the newstyle pre-processed data!");
            } else {
                //put the output product in the main self structure
                self->science_imaging_di = sph_double_image_duplicate(
                        camera_unit->science_imaging_di);
            }
        } else {
            SPH_ERR(
                    "Creation of the camera_unit structure newstyle pre-processed data is failed!")
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the new style format -> nothing to reduce.");
    }

    if (camera_unit) {
        _sph_zpl_science_imaging_camera_unit_delete(camera_unit);
        camera_unit = NULL;
    }



    if (self->science_imaging_di_cam1 == NULL
            && self->science_imaging_di_cam2 == NULL && self->science_imaging_di == NULL) {
        SPH_ERR(" No outputs have been created !!!");
    }

    if (!self->keep_intermediate && self->rawframes != NULL) {
        SPH_INFO_MSG("Unliking intermediate data!");
        if (self->preproc_frames_cam1) sph_utils_frames_unlink(self->preproc_frames_cam1);
        if (self->preproc_frames_cam2) sph_utils_frames_unlink(self->preproc_frames_cam2);
        if (self->preproc_frames) sph_utils_frames_unlink(self->preproc_frames);
    }

    SPH_INFO_MSG("sph_zpl_intensity_flat_imaging_run...End");
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}

sph_zpl_science_imaging_camera_unit* _sph_zpl_science_imaging_camera_unit_create(
        sph_zpl_science_imaging* self, int camera_id) {
    sph_zpl_science_imaging_camera_unit* result = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return NULL;
    }

    result = cpl_calloc(1, sizeof(sph_zpl_science_imaging_camera_unit));

    if (result == NULL) {
        SPH_ERR( "Could not allocate the structure for run unit.");
        return result;
    }

    //copy and duplicate some elements of the initial "sph_zpl_masters_dark_imaging" structure into the camera_unit structure
    result->inframes = self->inframes;
    result->inparams = self->inparams;
    result->framecomb_parameterlist = self->framecomb_parameterlist;
    result->subtract_overscan = self->subtract_overscan;
    result->coll_alg = self->coll_alg;
    result->filter_radius = self->filter_radius;
    result->save_interprod = self->save_interprod;
    result->keep_intermediate = self->keep_intermediate;
    result->star_center_iframe = self->star_center_iframe;

    result->rawframes = self->rawframes;
    result->fctable_frames = self->fctable_frames;
    result->filter_frame = self->filter_frame;

    if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {
        result->center_xoffset = self->center_xoffset_cam1;
        result->center_yoffset = self->center_yoffset_cam1;

        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam1);
        result->science_imaging_outfilename = cpl_strdup(
                self->science_imaging_outfilename_cam1);
        result->eso_pro_catg = cpl_strdup(
                SPH_ZPL_TAG_SCIENCE_IMAGING_CALIB_CAM1);

        if (self->master_bias_frame_cam1) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame_cam1);
        }
        if (self->master_dark_frame_cam1) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame_cam1);
        }
        if (self->intensity_flat_frame_cam1) {
            result->intensity_flat_frame = cpl_frame_duplicate(
                    self->intensity_flat_frame_cam1);
        }
        if (self->star_center_frame_cam1) {
            result->star_center_frame = cpl_frame_duplicate(
                    self->star_center_frame_cam1);
        }

    } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {
        result->center_xoffset = self->center_xoffset_cam2;
        result->center_yoffset = self->center_yoffset_cam2;

        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam2);
        result->science_imaging_outfilename = cpl_strdup(
                self->science_imaging_outfilename_cam2);
        result->eso_pro_catg = cpl_strdup(
                SPH_ZPL_TAG_SCIENCE_IMAGING_CALIB_CAM2);

        if (self->master_bias_frame_cam2) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame_cam2);
        }
        if (self->master_dark_frame_cam2) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame_cam2);
        }
        if (self->intensity_flat_frame_cam2) {
            result->intensity_flat_frame = cpl_frame_duplicate(
                    self->intensity_flat_frame_cam2);
        }
        if (self->star_center_frame_cam2) {
            result->star_center_frame = cpl_frame_duplicate(
                    self->star_center_frame_cam2);
        }

    } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID ) {
        result->center_xoffset = self->center_xoffset_cam1;
        result->center_yoffset = self->center_yoffset_cam1;

        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames);
        result->science_imaging_outfilename = cpl_strdup(
                self->science_imaging_outfilename);
        result->eso_pro_catg = cpl_strdup(
                SPH_ZPL_TAG_SCIENCE_IMAGING_CALIB);

        if (self->master_bias_frame) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame);
        }
        if (self->master_dark_frame) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame);
        }
        if (self->intensity_flat_frame) {
            result->intensity_flat_frame = cpl_frame_duplicate(
                    self->intensity_flat_frame);
        }
        if (self->star_center_frame) {
            result->star_center_frame = cpl_frame_duplicate(
                    self->star_center_frame);
        }
    } else {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Wrong camera id = %d (it must be 1 or 2 for oldstyle format, 0 for newstyle format)",
                camera_id);
        _sph_zpl_science_imaging_camera_unit_delete(result);
        result = NULL;
    }

    if (sph_error_get_last_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Post condition error check shows the following error %d",
                sph_error_get_last_code());
        _sph_zpl_science_imaging_camera_unit_delete(result);
        result = NULL;
    }

    return result;

}

sph_error_code _sph_zpl_science_imaging_camera_unit_delete(
        sph_zpl_science_imaging_camera_unit* camera_unit) {
    sph_error_code rerr = CPL_ERROR_NONE;

    if (camera_unit == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return rerr;
    }

    /* Code to delete recipe pointers GENERATED DO NOT EDIT */

    if (camera_unit->preproc_frames != NULL) {
        cpl_frameset_delete(camera_unit->preproc_frames);
        camera_unit->preproc_frames = NULL;
    }

    if (camera_unit->science_imaging_di) {
        sph_double_image_delete(camera_unit->science_imaging_di);
        camera_unit->science_imaging_di = NULL;
    }
    if (camera_unit->science_imaging_outfilename)
        cpl_free(camera_unit->science_imaging_outfilename);
    if (camera_unit->eso_pro_catg)
        cpl_free(camera_unit->eso_pro_catg);
    if (camera_unit->preproc_frames)
        cpl_frameset_delete(camera_unit->preproc_frames);

    //calibrations
    if (camera_unit->master_bias_frame)
        cpl_frame_delete(camera_unit->master_bias_frame);
    if (camera_unit->master_dark_frame)
        cpl_frame_delete(camera_unit->master_dark_frame);
    if (camera_unit->intensity_flat_frame)
        cpl_frame_delete(camera_unit->intensity_flat_frame);
    if (camera_unit->star_center_frame)
        cpl_frame_delete(camera_unit->star_center_frame);

    if (camera_unit->current_raw_frameset) {
        cpl_frameset_delete(camera_unit->current_raw_frameset);
        camera_unit->current_raw_frameset = NULL;
    }
    sph_polygon_free_all();
    cpl_free(camera_unit);
    return rerr;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_zpl_science_imaging recipe.
 The error  code returned is always a cpl error code
 (to allow maximal compatibility with  esorex, gasgano, etc.) even if during
 recipe execution an error in the SPHERE API is the cause. In this case
 (and if the underlying error is not a cpl error)  the cpl error code is set
 to the cpl_error_code that matches the failure reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/

static cpl_frame* _sph_zpl_science_imaging_calibrate_transform_frame(
        const char* filename, cpl_frameset* inframes, sph_fctable* fctable,
        sph_double_image* master_bias_doubleimage,
        sph_double_image* master_dark_doubleimage,
        sph_double_image* intensity_flat_doubleimage ) {

    sph_zpl_exposure_imaging* zplexp = NULL;
    sph_cube* dicube = NULL;
    cpl_frame* result = NULL;
    sph_transform* transform = NULL;
    int iplane = 0;
    int rerr = CPL_ERROR_NONE;

    cpl_ensure(inframes, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(filename, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(fctable, CPL_ERROR_NULL_INPUT, NULL);

    SPH_INFO_MSG("Calibrate and transform frames from the cube...")

    transform = sph_transform_new_default();
    if (!transform) {
        SPH_ERR("sph_transform_new_dafault() constructor returns NULL...");
        return NULL;
    }
    dicube = sph_cube_new(filename);
    if (!dicube) {
        SPH_ERR("sph_cube_new constructor returns NULL...");
        sph_transform_delete(transform);
        return NULL;
    }

    zplexp = sph_common_science_get_next_zpl_exposure_imaging(inframes);
    if (!zplexp) {
        SPH_ERR(
                "no zplexp can be loaded, sph_common_science_get_next_zpl_exposure_imaging returns NULL...");
        sph_transform_delete(transform);
        sph_cube_delete(dicube);
        return NULL;
    }

    while (zplexp) {
        sph_double_image* doubleimage = 
        sph_double_image_new_from_zpl_exposure_imaging(zplexp);

        if (!doubleimage) {
            SPH_ERR(
                    "can't create double image from the zplexp, sph_double_image_new_from_zpl_exposure_imaging"
                    " constructor returns NULL...");
            sph_transform_delete(transform);
            sph_cube_delete(dicube);
            sph_zpl_exposure_imaging_delete(zplexp);
            sph_double_image_delete(doubleimage);
            return NULL;
        }

        if (master_bias_doubleimage) {
            rerr = sph_double_image_subtract_double_image(doubleimage,
                    master_bias_doubleimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by subtracting master bias:\n"
                                "cpl error code is: %d", cpl_error_get_code());
                sph_transform_delete(transform);
                sph_cube_delete(dicube);
                sph_zpl_exposure_imaging_delete(zplexp);
                sph_double_image_delete(doubleimage);
                return NULL;
            }
            SPH_INFO_MSG("Subtracted master bias.")
        }

        if (master_dark_doubleimage) {
            rerr = sph_zpl_subtract_dark_double_image_scaled(doubleimage,
                    master_dark_doubleimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by subtracting dark:\n"
                                "cpl error code is: %d", cpl_error_get_code());
                sph_transform_delete(transform);
                sph_cube_delete(dicube);
                sph_zpl_exposure_imaging_delete(zplexp);
                sph_double_image_delete(doubleimage);
                return NULL;
            }
            SPH_INFO_MSG("Subtracted master dark.")
        }

        if (intensity_flat_doubleimage) {
            rerr = sph_double_image_divide_double_image(doubleimage,
                    intensity_flat_doubleimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(
                        SPH_ERROR_GENERAL,
                        __FILE__,
                        __func__,
                        __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by applying (dividing) intensity flat:\n"
                                "cpl error code is: %d", cpl_error_get_code());
                sph_transform_delete(transform);
                sph_cube_delete(dicube);
                sph_zpl_exposure_imaging_delete(zplexp);
                sph_double_image_delete(doubleimage);
                return NULL;
            }
            SPH_INFO_MSG("Applied intensity flat field.")
        }

        //if double image is not squared interpolate it and adjusting the borderlines
        //with the regard to the zpl imaging format
        if (!sph_double_image_is_squared(doubleimage)) {
            SPH_INFO_MSG(
                    "double image is not squared, it will be linear interpolated in y-direction, "
                    "the total flux will be preserved (a small flux lost is due to the border lines)!")
            rerr = sph_double_image_interpolate_y_zpl_imaging( doubleimage, CPL_TRUE );

            if (rerr) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "sph_doble_image_interpolate_y returns cpl error.\n"
                                "cpl error code is: %d", rerr);
                sph_transform_delete(transform);
                sph_cube_delete(dicube);
                sph_zpl_exposure_imaging_delete(zplexp);
                sph_double_image_delete(doubleimage);
                return NULL;
            }
        }

        if ( !sph_double_image_is_squared(doubleimage) ) {
            SPH_ERR("doubleimage is still not squared, "
                    "check the size of the input data...stop processing.\n");
            sph_transform_delete(transform);
            sph_cube_delete(dicube);
            sph_zpl_exposure_imaging_delete(zplexp);
            sph_double_image_delete(doubleimage);
            return NULL;
        }

/*
        SPH_INFO_MSG("sph_transform_apply_to_doubleimage...")
        rerr = sph_transform_apply_to_doubleimage(transform, doubleimage, NULL,
                    sph_fctable_get_centre_x_element(fctable, iplane),
                    sph_fctable_get_centre_y_element(fctable, iplane),
                    sph_fctable_get_angle(fctable, iplane), 1);
*/

        SPH_INFO_MSG("sph_transform_apply_to_doubleimage...")
        rerr = sph_transform_apply_to_doubleimage_doublemode(transform, doubleimage, NULL,
                    sph_fctable_get_centre_x_left_element( fctable, iplane ),
                    sph_fctable_get_centre_y_left_element( fctable, iplane),
                    sph_fctable_get_centre_x_right_element( fctable, iplane ),
                    sph_fctable_get_centre_y_right_element( fctable, iplane),
                    sph_fctable_get_angle(fctable, iplane), 1);

        if (rerr) {
            sph_error_raise(
                SPH_ERROR_GENERAL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "sph_transform_apply_to_doubleimage functions returns cpl error.\n"
                        "cpl error code is: %d", rerr);
            sph_transform_delete(transform);
            sph_cube_delete(dicube);
            sph_zpl_exposure_imaging_delete(zplexp);
            sph_double_image_delete(doubleimage);
            return NULL;
        }


        SPH_INFO_MSG("append the transformed double image to the cube...")
        rerr = sph_cube_append_double_image(dicube, doubleimage, 1);
        if (rerr) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "can't append the transformed double image to the cube: "
                            "sph_cube_append_double_image function returns cpl error.\n"
                            "cpl error code is: %d", rerr);

            sph_transform_delete(transform);
            sph_cube_delete(dicube);
            sph_zpl_exposure_imaging_delete(zplexp);
            sph_double_image_delete(doubleimage);
            return NULL;
        }

        sph_double_image_delete(doubleimage);
        sph_zpl_exposure_imaging_delete(zplexp);
        zplexp = sph_common_science_get_next_zpl_exposure_imaging(inframes);
        iplane++;
    }

    rerr = sph_cube_finalise_file(dicube->filename);
    if (rerr) {
        sph_error_raise(
                SPH_ERROR_GENERAL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "can't finalize output cube for the filename %s: sph_cube_finalise_file function returns cpl error.\n"
                        "cpl error code is: %d", dicube->filename, rerr);
        sph_cube_delete(dicube);
        if (transform) {
            sph_transform_delete(transform);
            transform = NULL;
        }
        return NULL;
    }

    sph_cube_delete(dicube);

    result = cpl_frame_new();
    cpl_frame_set_filename(result, filename);
    cpl_frame_set_tag(result, SPH_ZPL_TAG_SCIENCE_IMAGING_CALIB_TMP);
    sph_transform_delete(transform);

    return result;

}

static cpl_frameset* _sph_zpl_science_imaging_calibrate_frames(
        sph_zpl_science_imaging_camera_unit* self, cpl_frameset* inframes,
        sph_double_image* master_bias_doubleimage,
        sph_double_image* master_dark_doubleimage,
        sph_double_image* intensity_flat_doubleimage) {
    cpl_frameset* results = NULL;
    const cpl_frame* curframe = NULL;
    cpl_propertylist*   pl_star_center = NULL;
    cpl_image* image = NULL;
    double clx = 512.0;
    double cly = 512.0;
    double crx = 512.0;
    double cry = 512.0;

    cpl_ensure( self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(inframes, CPL_ERROR_NULL_INPUT, NULL);

    SPH_INFO_MSG("Calibrate cubes...")

    if ( self->star_center_frame ){
        pl_star_center = cpl_propertylist_load( cpl_frame_get_filename( self->star_center_frame), 0);
    }


    results = cpl_frameset_new();
    curframe = cpl_frameset_get_first_const(inframes);
    image = cpl_image_load(cpl_frame_get_filename( curframe ),
                           CPL_TYPE_DOUBLE, 0, 0  );

    if ( image != NULL ){
        clx = (double) (cpl_image_get_size_x( image ))/ 2.0;
        cly = crx = cry = clx;
        cpl_image_delete( image ); image = NULL;
    }

    while (curframe) {
        char* fctable_name = NULL;
        sph_fctable* fctable = NULL;
        cpl_frameset* ovsc_subtract_frames = NULL;
        char outfilename[612];
        char outfilename_tmp[612];
        cpl_frame* calframe = NULL;
        cpl_frameset* curframes = cpl_frameset_new();
        cpl_frameset_insert(curframes, cpl_frame_duplicate(curframe));

        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_WARNING,
                "The filename of the current cube: %s", cpl_frame_get_filename( curframe ));
        if (self->fctable_frames) {
            SPH_INFO_MSG("Searching a corresponding fctable frame...")
            fctable = sph_fctable_find_fctable(curframe, self->fctable_frames);
            if (fctable == NULL)
                SPH_INFO_MSG(
                        "The fctable not found in the given fctable input frameset!")
        }

        if (fctable == NULL) {
            cpl_propertylist*   pl = NULL;
            SPH_INFO_MSG("Creating fctable for the current frame.");
            fctable = sph_fctable_create_fctable_new(curframe, SPH_FCTAB_DOUBLE);

            pl = cpl_propertylist_load(cpl_frame_get_filename(curframe), 0);
            if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DROT2_MODE) ){
                const char*   drot2_mode = 
                    cpl_propertylist_get_string(pl, SPH_COMMON_KEYWORD_DROT2_MODE);

                  if (!strcmp(drot2_mode, SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_PUPIL_STAB)){
                         sph_fctable_add_angle_scalar( fctable, -SPH_ZPL_KEYWORD_PUPIL_STAB_IMAGING_ANGLE_OFFSET);
                  }
            }
            cpl_propertylist_delete(pl);

            if ( self->center_xoffset != 0.0 || self->center_yoffset != 0.0 ) {
                          SPH_INFO_MSG("Applying offset from the center of image.")
                          sph_fctable_add_center_offset_double_mode ( fctable,
                                  self->center_xoffset, self->center_yoffset,
                                  self->center_xoffset, self->center_yoffset);
            }

            if ( pl_star_center ) {
                if ( cpl_propertylist_has( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_XCOORD) &&
                     cpl_propertylist_has( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_YCOORD) &&
                     cpl_propertylist_has( pl_star_center, SPH_ZPL_STAR_CENTER_PFRAME_XCOORD) &&
                     cpl_propertylist_has( pl_star_center, SPH_ZPL_STAR_CENTER_PFRAME_YCOORD) ){

                    SPH_INFO_MSG("Applying star center calibration.")
                    if ( !self->star_center_iframe ){
                       SPH_INFO_MSG("Center coordinates of IFRAME and PFRAME[Dark Current] are used to de-rotate double image.");
                       sph_fctable_add_center_offset_double_mode ( fctable,
                         - clx + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_XCOORD),
                         - cly + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_YCOORD),
                         - crx + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_PFRAME_XCOORD),
                         - cry + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_PFRAME_YCOORD) );
                    } else {
                        SPH_INFO_MSG("Center coordinates of IFRAME  are used to de-rotate double image.");
                        sph_fctable_add_center_offset_double_mode ( fctable,
                          - clx + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_XCOORD),
                          - cly + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_YCOORD),
                          - crx + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_XCOORD),
                          - cry + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_YCOORD) );
                    }

                }
            }



            if (self->save_interprod) {
                fctable_name = sph_fctable_construct_filename(curframe, 1);
                sph_fctable_save_ascii(fctable, fctable_name);
                cpl_free(fctable_name);
            }
        }


        if (fctable == NULL || cpl_error_get_code()) {
            SPH_ERR("fctable has null pointer or error occured by creating fctable");
            if (results) {
                cpl_frameset_delete(results);
                results = NULL;
            }
            if (curframes) {
                cpl_frameset_delete(curframes);
                curframes = NULL;
            }
            if ( pl_star_center ){
                cpl_propertylist_delete( pl_star_center );
                pl_star_center = NULL;
            }
            return NULL;
        }

        SPH_INFO_MSG("Creating filename for intermediate products...");
        (void)strcpy(outfilename, self->science_imaging_outfilename);
        sprintf(outfilename_tmp, "_%s",
                sph_filemanager_get_basename( cpl_frame_get_filename(curframe)));
        strcat(outfilename, outfilename_tmp);
        SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
                "Intermediate outfilename = %s", outfilename)

        if (self->subtract_overscan) {
            SPH_INFO_MSG("Subtract overscans from rawframes...")
            ovsc_subtract_frames = sph_zpl_utils_subtract_overscans(curframes);
            if (!ovsc_subtract_frames) {
                SPH_ERROR_RAISE_WARNING(
                        SPH_ERROR_WARNING,
                        "Overscans could not be subtracted, trying to proceed further "
                        "without subtraction...")
                ovsc_subtract_frames = cpl_frameset_duplicate(curframes);
                sph_error_reset(); //reset is needed because sph_zpl_utils_subtract_overscans raises error when it returns NULL
            }
        } else {
            SPH_ERROR_RAISE_WARNING(
                    SPH_ERROR_WARNING,
                    "Overscans could not be subtracted, trying to proceed further "
                    "without subtraction...")
            ovsc_subtract_frames = cpl_frameset_duplicate(curframes);
        }

        calframe = _sph_zpl_science_imaging_calibrate_transform_frame(
                (const char*) outfilename, ovsc_subtract_frames, fctable,
                master_bias_doubleimage, master_dark_doubleimage,
                intensity_flat_doubleimage );
        if (!calframe) {
            SPH_ERR(
                    "can't calibrate the given frame(plane), the sph_zpl_science_p1_calibrate_transform_frame returns NULL");
            if (results) {
                cpl_frameset_delete(results);
                results = NULL;
            }
            if (curframes) {
                cpl_frameset_delete(curframes);
                curframes = NULL;
            }
            if (ovsc_subtract_frames) {
                cpl_frameset_delete(ovsc_subtract_frames);
                ovsc_subtract_frames = NULL;
            }
            if (fctable) {
                sph_fctable_delete(fctable);
                fctable = NULL;
            }
            if ( pl_star_center ){
                 cpl_propertylist_delete( pl_star_center );
                 pl_star_center = NULL;
             }
            return NULL;
        }

        cpl_frameset_insert(results, calframe);

        if (!self->keep_intermediate) {
            sph_utils_frames_unlink(ovsc_subtract_frames);
        }
        cpl_frameset_delete(ovsc_subtract_frames);  ovsc_subtract_frames = NULL;
        cpl_frameset_delete(curframes); curframes = NULL;
        sph_fctable_delete(fctable); fctable = NULL;

        curframe = cpl_frameset_get_next_const(inframes);
    }

    cpl_propertylist_delete( pl_star_center );
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return results;

}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Process one camera unit
 @param self        Camera unit object to process
 @param cboth       Read access to both-camera object, for strehl estimate
 @param mode        Mode we are using
 @return CPL_ERROR_NONE or the relevant CPL error code on error

 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
    _sph_zpl_science_imaging_camera_unit_run(sph_zpl_science_imaging_camera_unit* self,
                                       const sph_zpl_science_imaging* cboth,
									   const sph_zpl_mode mode) {

    int recipe_error = CPL_ERROR_NONE;
    sph_double_image* master_bias_doubleimage = NULL;
    sph_double_image* master_dark_doubleimage = NULL;
    sph_double_image* intensity_flat_doubleimage = NULL;
    cpl_frameset* calframes = NULL;
    sph_master_frame* imframe = NULL;
    sph_master_frame* pmframe = NULL;
    double xpix = -9999.0;
    double ypix = -9999.0;

    SPH_INFO_MSG("Starting static sph_zpl_science_imaging_camera_unit_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (!self->preproc_frames) {
        SPH_ERR("no preproc frames are set up!")
        return sph_error_get_last_code();
    }

    //self->science_imaging_di = NULL;

    //if master bias exists load it
    if (self->master_bias_frame) {
        SPH_INFO_MSG("Loading master bias...\n");
        master_bias_doubleimage = sph_double_image_load(
                cpl_frame_get_filename(self->master_bias_frame), 0);
    }

    //if master dark exists load it
    if (self->master_dark_frame) {
        SPH_INFO_MSG("Loading master dark...\n");
        master_dark_doubleimage = sph_double_image_load(
                cpl_frame_get_filename(self->master_dark_frame), 0);
    }

    //if intensity flat field exists load
    if (self->intensity_flat_frame) {
        SPH_INFO_MSG("Loading intensity flat field...\n");
        intensity_flat_doubleimage = sph_double_image_load(
                cpl_frame_get_filename(self->intensity_flat_frame), 0);
    }


    //calibrate and transform frames (de-dither)
    calframes = _sph_zpl_science_imaging_calibrate_frames(self,
            self->preproc_frames, master_bias_doubleimage,
            master_dark_doubleimage, intensity_flat_doubleimage);
//    cpl_frameset_dump(calframes, stdout);
//    cpl_msg_info(__func__, "Press ENTER to continue... (L1053)");
//    getchar();

    if (!calframes) {
        SPH_ERR("Can't calibrate, de-dither and de-rotate given frames...");
        if (master_bias_doubleimage) {
            sph_double_image_delete(master_bias_doubleimage);
            master_bias_doubleimage = NULL;
        }
        if (master_dark_doubleimage) {
            sph_double_image_delete(master_dark_doubleimage);
            master_dark_doubleimage = NULL;
        }
        if (intensity_flat_doubleimage) {
            sph_double_image_delete(intensity_flat_doubleimage);
            intensity_flat_doubleimage = NULL;
        }
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
    }

    //combine
    imframe = sph_common_science_combine(calframes, self->coll_alg,
            SPH_MASTER_FRAME_IMAGE_EXT, SPH_MASTER_FRAME_BPIX_EXT, -1);
    if (!imframe) {
        SPH_ERR(
                "Can't combine calframes, imframe from sph_common_science_combine is NULL...");
        if (master_bias_doubleimage) {
            sph_double_image_delete(master_bias_doubleimage);
            master_bias_doubleimage = NULL;
        }
        if (master_dark_doubleimage) {
            sph_double_image_delete(master_dark_doubleimage);
            master_dark_doubleimage = NULL;
        }
        if (intensity_flat_doubleimage) {
            sph_double_image_delete(intensity_flat_doubleimage);
            intensity_flat_doubleimage = NULL;
        }
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
    }

    pmframe = sph_common_science_combine(calframes, self->coll_alg,
            SPH_MASTER_FRAME_IMAGE_EXT + 4, SPH_MASTER_FRAME_BPIX_EXT, -1);
    if (!pmframe) {
        SPH_ERR(
                "Can't combine calframes, pframe from sph_common_science_combine is NULL...");
        if (master_bias_doubleimage) {
            sph_double_image_delete(master_bias_doubleimage);
            master_bias_doubleimage = NULL;
        }
        if (master_dark_doubleimage) {
            sph_double_image_delete(master_dark_doubleimage);
            master_dark_doubleimage = NULL;
        }
        if (intensity_flat_doubleimage) {
            sph_double_image_delete(intensity_flat_doubleimage);
            intensity_flat_doubleimage = NULL;
        }
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
    }

    if ( !self->save_interprod ) {
        // PIPE-7143: calframes is what we need to not unlink in science_p*
        sph_utils_frames_unlink( calframes );
    }

    if ( calframes ) {
        cpl_frameset_delete(calframes); calframes = NULL;
    }

    self->science_imaging_di = sph_double_image_new_from_master_frames(imframe,
            pmframe);
    if (!self->science_imaging_di) {
        SPH_ERR(
                "Can't create double image from iframe and pframe master frames, self->science_di is NULL...");
        if (master_bias_doubleimage) {
            sph_double_image_delete(master_bias_doubleimage);
            master_bias_doubleimage = NULL;
        }
        if (master_dark_doubleimage) {
            sph_double_image_delete(master_dark_doubleimage);
            master_dark_doubleimage = NULL;
        }
        if (intensity_flat_doubleimage) {
            sph_double_image_delete(intensity_flat_doubleimage);
            intensity_flat_doubleimage = NULL;
        }
        if (imframe) {
            sph_master_frame_delete(imframe);
            imframe = NULL;
        }
        if (pmframe) {
            sph_master_frame_delete(pmframe);
            pmframe = NULL;
        }
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
    }

    //quality check (calculate statistical moments for the basic imaging reduced double image)
    recipe_error = sph_double_image_quality_check_names(self->science_imaging_di, NULL,
    		&pframe_names_leakage);
    if (recipe_error) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "sph_double_image_quality_check returns cpl error.\n"
                        "cpl error code is: %d", cpl_error_get_code());
        if (master_bias_doubleimage) {
            sph_double_image_delete(master_bias_doubleimage);
            master_bias_doubleimage = NULL;
        }
        if (master_dark_doubleimage) {
            sph_double_image_delete(master_dark_doubleimage);
            master_dark_doubleimage = NULL;
        }
        if (intensity_flat_doubleimage) {
            sph_double_image_delete(intensity_flat_doubleimage);
            intensity_flat_doubleimage = NULL;
        }
        return recipe_error;
    }

    //save
    /* Attention:
     * The following function includes a step to update SPH_COMMON_KEYWORD_SPH_TYPE by
     * SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_DOUBLE_IMAGE. It is necessary because cpl_dsf_save_image
     * updates the double image property list by the property list from the first input frames.
     * In the case of some misbehavior exclude this step for checking.
     */

    SPH_INFO_MSG("Saving...");
    // reference pixels for WCS
    xpix = cpl_image_get_size_x(self->science_imaging_di->iframe->image)/2.;
    ypix = cpl_image_get_size_y(self->science_imaging_di->iframe->image)/2.;

    if (self->rawframes) {
        cpl_propertylist* pl = NULL;
        cpl_propertylist* pl_wcs = NULL;

        SPH_INFO_MSG(
                "Save science imaging as a dfs product: header info taken from the first raw frame.");
        pl = sph_zpl_utils_get_camera_header(self->preproc_frames, self->rawframes);
        //create WCS keywords and add to pl
        pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl( (const cpl_frame*) cpl_frameset_get_first(self->rawframes),
                                                         (const double) xpix, (const double) ypix);
        if ( pl_wcs && pl){
            cpl_propertylist_append( pl, pl_wcs );
            cpl_propertylist_delete( pl_wcs );
        }

        _calc_strehl_zpoint_and_append(self->science_imaging_di->iframe->image, cboth, pl, mode);

        cpl_propertylist_update_string(pl, SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_RIGHT,
        		SPH_IMAGING_PFRAME_NAME_LEAKAGE_NAME);

        recipe_error = sph_double_image_save_dfs(self->science_imaging_di,
                self->science_imaging_outfilename, self->inframes,
                cpl_frameset_get_first(self->rawframes),
                self->inparams,
                self->eso_pro_catg, //SPH_ZPL_TAG_SCIENCE_IMAGING_CALIB_CAM1 or _CAM2,
                SPH_RECIPE_NAME_ZPL_SCIENCE_IMAGING, SPH_PIPELINE_NAME_ZIMPOL,
                pl);
        cpl_propertylist_delete(pl);

    } else if (self->preproc_frames) {

        cpl_propertylist * pl =
        		sph_zpl_utils_get_camera_header(self->preproc_frames, self->rawframes);

        cpl_propertylist* pl_wcs = NULL;
        SPH_INFO_MSG(
                "Save science imaging as a dfs product: header info taken from the pre-processed frame.");
        //create WCS keywords into pl
        pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl( (const cpl_frame*) cpl_frameset_get_first(self->preproc_frames),
                                                         (const double) xpix, (const double) ypix);

        if ( pl_wcs && pl){
            cpl_propertylist_append( pl, pl_wcs );
            cpl_propertylist_delete( pl_wcs );
        }

        recipe_error = sph_double_image_save_dfs(self->science_imaging_di,
                self->science_imaging_outfilename, self->inframes,
                cpl_frameset_get_first(self->preproc_frames),
                self->inparams,
                self->eso_pro_catg, //SPH_ZPL_TAG_SCIENCE_IMAGING_CALIB_CAM1 or _CAM2,
                SPH_RECIPE_NAME_ZPL_SCIENCE_IMAGING, SPH_PIPELINE_NAME_ZIMPOL,
                pl_wcs);
        cpl_propertylist_delete(pl);

    } else {
        SPH_ERR("Neither raw frames nor pre-processed frames provided!");
        recipe_error = sph_error_get_last_code();
    }

    if (recipe_error) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Could not save science imaging reduced double image:\n"
                        "sph_double_image_save_dfs returns error code.\n"
                        "cpl error code is: %d", cpl_error_get_code());
        if (master_bias_doubleimage) {
            sph_double_image_delete(master_bias_doubleimage);
            master_bias_doubleimage = NULL;
        }
        if (master_dark_doubleimage) {
            sph_double_image_delete(master_dark_doubleimage);
            master_dark_doubleimage = NULL;
        }
        if (intensity_flat_doubleimage) {
            sph_double_image_delete(intensity_flat_doubleimage);
            intensity_flat_doubleimage = NULL;
        }
        return recipe_error;
    }

    if (master_bias_doubleimage) {
        sph_double_image_delete(master_bias_doubleimage);
        master_bias_doubleimage = NULL;
    }
    if (master_dark_doubleimage) {
        sph_double_image_delete(master_dark_doubleimage);
        master_dark_doubleimage = NULL;
    }
    if (intensity_flat_doubleimage) {
        sph_double_image_delete(intensity_flat_doubleimage);
        intensity_flat_doubleimage = NULL;
    }

    SPH_INFO_MSG("static _sph_zpl_science_imaging_run...End");
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}

static cpl_boolean _sph_zpl_science_imaging_plist_contains_value(const cpl_propertylist * list,
		const char * key, const char * expected_value){

	if(!cpl_propertylist_has(list, key)) return CPL_FALSE;
	const char * content = cpl_propertylist_get_string(list, key);

	return strcmp(content, expected_value) == 0;
}

static cpl_boolean
_sph_zpl_science_imaging_is_ao_enabled_from_list(const cpl_propertylist * list){

	const char * ao_keywords[] = {SPH_PHOT_ZPL_AOS_HOLOOP,
			SPH_PHOT_ZPL_AOS_PUPLOOP, SPH_PHOT_ZPL_AOS_TTLOOP};

	const char * expected_ao_value = "CLOSED_LOOP";
	const char * expected_value_for_IRLOOP = "OPEN_LOOP";

	if(!_sph_zpl_science_imaging_plist_contains_value(list,
			SPH_PHOT_ZPL_AOS_IRLOOP, expected_value_for_IRLOOP)){

		cpl_msg_warning(cpl_func, "%s was not set to %s", SPH_PHOT_ZPL_AOS_IRLOOP, expected_value_for_IRLOOP);
	}

	cpl_size sz = sizeof(ao_keywords) / sizeof(ao_keywords[0]);

	for(cpl_size i = 0; i < sz; ++i){
		if(!_sph_zpl_science_imaging_plist_contains_value(list,
				ao_keywords[i], expected_ao_value)) return CPL_FALSE;
	}

	return CPL_TRUE;
}


static void _calc_strehl_zpoint_and_append(const cpl_image * image_cam,
		const sph_zpl_science_imaging* cboth, cpl_propertylist* pl,
		const  sph_zpl_mode mode){

	if(mode == SPH_ZPL_MODE_ONLINE){
		cpl_msg_warning(cpl_func,
				"Strehl and zeropoint calculation disabled for legacy ONLINE mode");
		return;
	}

    cpl_errorstate okstate      = cpl_errorstate_get();
    const cpl_boolean is_arm1 = mode == SPH_ZPL_MODE_CAMERA1;
    const char * rec_name = "science_imaging";

	double flx = 0.0;

    if(!_sph_zpl_science_imaging_is_ao_enabled(cpl_frameset_get_first(cboth->rawframes))){

    	cpl_msg_info(cpl_func, "AO is disabled, Strehl ratio computation will be skipped");

    	double bkg = 0.0;

    	const cpl_error_code fail =
    	        sph_strehl_disabled_ao_flux_zimpol(image_cam,
    			cpl_frameset_get_first(cboth->rawframes), cboth->inparams, rec_name, is_arm1, &flx, &bkg);

    	if(fail){
    		cpl_msg_warning(cpl_func, "Strehl flux estimation for disabled AO failed: %s, "
    				"aborting zeropint calculation", cpl_error_get_message());
			cpl_errorstate_dump(okstate, CPL_FALSE,
								cpl_errorstate_dump_one_warning);
			cpl_errorstate_set(okstate);
    		return;
    	} else {
    		cpl_msg_info(cpl_func, "Strehl flux estimation for disabled AO successful!");
    	}

    	sph_strehl_fill_qc_pars_zimpol_ao_disabled(pl, flx, bkg);
    }
    else{

    	sph_strehl_qc_pars qc_out = {0, 0, 0, 0, 0, 0, 0, 0, 0};

    	cpl_msg_info(cpl_func, "AO is enabled, Strehl ratio computation will be performed");

    	cpl_error_code fail = sph_strehl_zimpol(
    			image_cam, cpl_frameset_get_first(cboth->rawframes),
				cboth->filter_frame, rec_name, cboth->inparams,
				is_arm1, &qc_out);

    	if(fail){
    		cpl_msg_warning(cpl_func, "Strehl estimation failed: %s, "
    				"aborting zeropint calculation", cpl_error_get_message());
			cpl_errorstate_dump(okstate, CPL_FALSE,
								cpl_errorstate_dump_one_warning);
			cpl_errorstate_set(okstate);
    		return;
    	} else {
    		cpl_msg_info(cpl_func, "Strehl estimation successful!");
    	}

    	sph_strehl_fill_qc_pars_zimpol(pl, &qc_out);
    	flx = qc_out.star_flux;
    }

	cpl_propertylist * ph_list = sph_phot_zimpol(flx, is_arm1,
			cpl_frameset_get_first(cboth->rawframes), cboth->star_frame,
			cboth->filter_frame);

	if(ph_list){
		cpl_propertylist_append(pl, ph_list);
		cpl_propertylist_delete(ph_list);
	} else {
		const char * target =
			cpl_propertylist_has(pl, SPH_COMMON_TARGET_NAME_KEY) &&
			cpl_propertylist_get_type(pl, SPH_COMMON_TARGET_NAME_KEY) ==
			CPL_TYPE_STRING
			? cpl_propertylist_get_string(pl, SPH_COMMON_TARGET_NAME_KEY)
			: "<missing>";
		cpl_boolean no_error = cpl_errorstate_is_equal(okstate);

		cpl_msg_warning(cpl_func, "Photometry computation failed (%s)! "
						"It is ok if this target (%s) is not a "
						"standard star...",
						no_error ? "without error" : "with error(s)",
						target);
		if (!no_error) {
			cpl_errorstate_dump(okstate, CPL_FALSE,
								cpl_errorstate_dump_one_warning);
			cpl_errorstate_set(okstate);
		}
	}

}

static cpl_boolean
_sph_zpl_science_imaging_is_ao_enabled(const cpl_frame * raw_frame){

	cpl_propertylist * list = cpl_propertylist_load(cpl_frame_get_filename(raw_frame), 0);
	cpl_boolean to_ret = _sph_zpl_science_imaging_is_ao_enabled_from_list(list);
	cpl_propertylist_delete(list);
	return to_ret;
}
/**@}*/
