/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "sph_error.h"
#include "sph_andromeda_core.h"
#include <math.h>
#include "sph_psfcube.h"
#include "sph_andromeda_support.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/
cpl_image*
sph_andromeda_core_get_weight_2D( cpl_imagelist* diffims, int hom ) {
    cpl_image*      result  = NULL;
    int             kmax    = 0;
    cpl_image*      tempim   = NULL;
    cpl_image*      var2d   = NULL;
    cpl_imagelist*  diffims_copy    = NULL;

    int             ii  = 0;
    double*         pdata = NULL;
    int             nx  = 0;
    int             ny  = 0;
    double          mean = 0.0;

    cpl_ensure(diffims,CPL_ERROR_NULL_INPUT,NULL);

    diffims_copy = cpl_imagelist_duplicate(diffims);
    cpl_imagelist_multiply(diffims_copy,diffims_copy);

    kmax = cpl_imagelist_get_size(diffims);
    tempim = cpl_imagelist_collapse_create(diffims);
    cpl_image_divide_scalar(tempim,kmax);
    cpl_image_multiply(tempim,tempim);
    var2d = cpl_imagelist_collapse_create(diffims_copy);
    cpl_image_divide_scalar(var2d,kmax);
    cpl_image_subtract(var2d,tempim);
    cpl_image_delete(tempim);
    nx = cpl_image_get_size_x(var2d);
    ny = cpl_image_get_size_y(var2d);

    if ( hom ) {
        mean = cpl_image_get_mean(var2d);
        result = cpl_image_new(nx,ny,CPL_TYPE_DOUBLE);
        cpl_image_add_scalar(result,1.0/mean);
    }
    else {
        pdata = cpl_image_unwrap(var2d);
        for (ii = 0; ii < nx*ny; ++ii) {
            if ( pdata[ii] ) pdata[ii] = 1.0/pdata[ii];
        }
        result = cpl_image_wrap_double(nx,ny,pdata);
    }
    cpl_imagelist_delete(diffims_copy);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief The core of the Andromeda algorithm
 * @param diffimages    the input list of difference images
 * @param posind        the indices in diffimages of positive images
 * @param negind        the indices in diffimages of negative images
 * @param angles        the original angles of the images
 * @param psfcube       A cube of shifted PSF images
 * @param hom_var       flag to set homogenous variance (no time dep.)
 * @param weights       optional input of weight maps for difference images
 * @param rhomin        minimum radius
 * @param rhomax        maximum radius
 * @param gamma         the gamma values
 * @param positivity    if set require flux to be positive
 * @param mask          optional input mask
 * @param snr           optional output of SNR
 * @param likelyhood    optional output of the likelyhood
 * @param std           optional output of standard dev. on flux
 *
 * @return flux image
 *
 * This function provides the core Andromeda functionality. Its main purpose
 * is to calculate a flux image from the input list of difference images.
 * Optionally, the map of the signal to noise (snr), the likelyhood and
 * or the variance on the flux is also produced.
 *
 * Required inputs are the list of difference images, as obtained with the
 * sph_andromeda_support_angular_difference function, the list of angles for
 * the original images and the list of indices for the positive and negative
 * original images. A special cube (sph_psfcube) is also needed as input. This
 * cube stores a series of PSF images, each shifted by a different amount.
 * The function sph_andromeda_support_calc_psf_shift_subpix should be used
 * to create this cube from a (64x64 pixels or smaller!) PSF reference image.
 *
 * Andromeda also needs to estimate the variance maps for the difference images.
 * This can be either done using a supplied list of difference image weightmaps
 * or by assuming a variance that is constant in time (the weights input list
 * is NULL). A single weight image is then calculated using the time variance of
 * the difference images. If weights is NULL and hom_var is also set to 1 the
 * variance is also assumed to be homogeneous over the whole region.
 *
 *
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_andromeda_core_get_flux(
        cpl_imagelist* diffimages,
        cpl_vector* posind,
        cpl_vector* negind,
        cpl_vector* angles,
        sph_psfcube* psfcube,
        int  hom_var,
        cpl_imagelist* weights,
        double rhomin,
        double rhomax,
        cpl_vector* gamma,
        int positivity,
        cpl_image* mask,
        cpl_image** snr,
        cpl_image** likelyhood,
        cpl_image** std
        )
{
    cpl_image*      result  = NULL;
    cpl_image*      distim  = NULL;
    cpl_image*      maskim  = NULL;
    int             npix    = 0;
    int             xx      = 0;
    int             yy      = 0;
    int             ii      = 0;
    int             kmax    = 0;
    cpl_bivector*   indices = NULL;
    sph_andromeda_core* core    = NULL;
    double          num_part    = 0.0;
    double          den_part    = 0.0;
    cpl_image*      cut_psf      = NULL;
    cpl_image*      numerator      = NULL;
    cpl_image*      denominator      = NULL;
    cpl_image*      cut_psf_den      = NULL;
    cpl_image*      cut_weight      = NULL;
    cpl_image*      cut_weight_diff      = NULL;
    cpl_image*      weights_2D  = NULL;
    cpl_imagelist*  weighted_diff_ims   = NULL;
    int             variance_is_3D  = 0;
    int             bpix    = 0;
    double          gamval = 0.0;

    cpl_ensure(diffimages,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(posind,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(negind,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(psfcube,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(gamma,CPL_ERROR_NULL_INPUT,NULL);

    npix = cpl_image_get_size_x(cpl_imagelist_get(diffimages,0));
    cpl_ensure(npix>2,CPL_ERROR_ILLEGAL_INPUT,NULL);

    distim = sph_andromeda_support_cal_freq_image(npix,npix,0.5*(npix-1.0),0.5*(npix-1.0));

    maskim = sph_andromeda_support_calc_mask_image(distim,rhomin,rhomax);
    if ( mask ) {
        cpl_image_multiply(maskim,mask);
    }
    weighted_diff_ims = cpl_imagelist_duplicate(diffimages);
    if ( weights ) {
        cpl_imagelist_multiply(weighted_diff_ims,weights);
        variance_is_3D = 1;
    }
    else {
        weights_2D = sph_andromeda_core_get_weight_2D(diffimages,hom_var);
        cpl_imagelist_multiply_image(weighted_diff_ims,weights_2D);
    }

    indices = cpl_bivector_wrap_vectors(posind,negind);
    kmax = cpl_bivector_get_size(indices);
    core = sph_andromeda_support_init_core(diffimages,angles,indices,psfcube);
    numerator = cpl_image_new(npix,npix,CPL_TYPE_DOUBLE);
    denominator = cpl_image_new(npix,npix,CPL_TYPE_DOUBLE);
    for (yy = npix/2 - ceil(rhomax); yy < npix/2 + ceil(rhomax) - 1; ++yy) {
        for (xx = npix/2-ceil(rhomax); xx < npix/2.0 + ceil(rhomax) - 1; ++xx) {
            if ( cpl_image_get(maskim,xx,yy,&bpix) ) {
                num_part = 0.0;
                den_part = 0.0;
                for (ii = 0; ii < kmax; ++ii) {
                    sph_andromeda_support_get_region(xx,yy,ii,core);
                    gamval = cpl_vector_get(gamma,ii);
                    cut_psf = sph_andromeda_support_place_psfs(core,gamval);
                    cut_psf_den = cpl_image_multiply_create(cut_psf,cut_psf);
                    if ( variance_is_3D ) {
                        cut_weight = cpl_image_extract(
                                cpl_imagelist_get(weights,ii),
                                core->minx,core->miny,core->maxx,core->maxy);
                    }
                    else {
                        cut_weight = cpl_image_extract(
                                weights_2D,
                                core->minx,core->miny,core->maxx,core->maxy);
                    }
                    cut_weight_diff = cpl_image_extract(
                            cpl_imagelist_get(weighted_diff_ims,ii),
                            core->minx,core->miny,core->maxx,core->maxy);
                    if ( cpl_error_get_code() == CPL_ERROR_ILLEGAL_INPUT ) {
                        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                                "Trying to extract window with x = %d -> %d and y = %d -> %d",
                                core->minx,core->maxx,core->miny,core->maxy);
                        cpl_image_delete(cut_psf_den);
                        cpl_image_delete(cut_weight);
                        cpl_image_delete(cut_weight_diff);
                        return NULL;
                    }
                    cpl_image_multiply(cut_psf,cut_weight_diff);
                    SPH_RAISE_CPL_RESET
                    cpl_image_multiply(cut_psf_den,cut_weight);
                    SPH_RAISE_CPL_RESET
                    num_part += cpl_image_get_flux(cut_psf);
                    den_part += cpl_image_get_flux(cut_psf_den);

                    cpl_image_delete(cut_psf);
                    cpl_image_delete(cut_psf_den);
                    cpl_image_delete(cut_weight_diff);
                    cpl_image_delete(cut_weight);
                }
                cpl_image_set(numerator,xx,yy,num_part);
                cpl_image_set(denominator,xx,yy,den_part);
            }
        }
    }

    if ( positivity ) {
        cpl_image_threshold(numerator,-DBL_MAX,0.0,0.0,0.0);
    }
    result = cpl_image_divide_create(numerator,denominator);
    if ( snr ) {
        *snr = cpl_image_power_create(denominator,-0.5);
        cpl_image_multiply(*snr,numerator);
        if ( likelyhood ) {
            *likelyhood = cpl_image_multiply_create(*snr,*snr);
            SPH_RAISE_CPL
            cpl_image_multiply_scalar(*likelyhood,0.5);
            SPH_RAISE_CPL
        }
        if ( std ) {
            *std = cpl_image_duplicate(result);
            cpl_image_divide(*std,*snr);
        }
    }

    cpl_imagelist_delete(weighted_diff_ims);
    cpl_bivector_unwrap_vectors(indices);
    cpl_image_delete(maskim);
    cpl_image_delete(distim);
    cpl_image_delete(weights_2D);
    cpl_free(core);
    cpl_image_delete(numerator);
    cpl_image_delete(denominator);
    return result;
}


/**@}*/
