/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "sph_error.h"
#include "sph_psfcube.h"
#include "sph_fft.h"
#include "sph_andromeda_support.h"
#include <math.h>


/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/


/*----------------------------------------------------------------------------*/
/**
 * @brief Initialise the core
 * @param imlist    the list of difference images
 * @param angles    the list of angles
 * @param indices   a bivector containing pos and neg angle indices
 * @param psfcube   the PSF cube
 *
 * @return a new core or NULL on error
 *
 *  This function allocates memory for a structure to hold the
 *  andromeda_core information.
 *
 *
 */
/*----------------------------------------------------------------------------*/

sph_andromeda_core*
sph_andromeda_support_init_core(
        cpl_imagelist* imlist,
        cpl_vector* angles,
        cpl_bivector* indices,
        sph_psfcube* psfcube) {
    sph_andromeda_core*     core  = NULL;

    core = cpl_calloc(1,sizeof(sph_andromeda_core));
    core->angles=angles;
    core->indices=indices;
    core->imlist = imlist;
    core->minx = 0;
    core->miny = 0;
    core->maxx = cpl_image_get_size_x(cpl_imagelist_get(imlist,0));
    core->maxy = cpl_image_get_size_y(cpl_imagelist_get(imlist,0));
    core->nx = cpl_image_get_size_x(cpl_imagelist_get(imlist,0));
    core->ny = cpl_image_get_size_y(cpl_imagelist_get(imlist,0));
    core->psfcube = psfcube;
    return core;

}



/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the object
 * @param self   The object to dump
 */
/*----------------------------------------------------------------------------*/
void sph_andromeda_core_delete(sph_andromeda_core* self)
{
    if (self != NULL) {
        cpl_vector_delete(self->angles);
        cpl_bivector_delete(self->indices);
        cpl_imagelist_delete(self->imlist);
        sph_psfcube_delete(self->psfcube);
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Dump the object to the provided stream
 * @param stream The stream to dump to
 * @param self   The object to dump
 */
/*----------------------------------------------------------------------------*/
void sph_andromeda_core_dump(FILE* stream, const sph_andromeda_core* self) {
   fprintf(stream, "maxx       : %d\n", self->maxx);
   fprintf(stream, "maxy       : %d\n", self->maxy);
   fprintf(stream, "minx       : %d\n", self->minx);
   fprintf(stream, "miny       : %d\n", self->miny);
   fprintf(stream, "nx         : %d\n", self->nx);
   fprintf(stream, "ny         : %d\n", self->ny);
   fprintf(stream, "negbot     : %d\n", self->negbot);
   fprintf(stream, "negleft    : %d\n", self->negleft);
   fprintf(stream, "posbot     : %d\n", self->posbot);
   fprintf(stream, "posleft    : %d\n", self->posleft);
   fprintf(stream, "possubx    : %f\n", self->possubx);
   fprintf(stream, "possuby    : %f\n", self->possuby);
   fprintf(stream, "negsubx    : %f\n", self->negsubx);
   fprintf(stream, "negsuby    : %f\n", self->negsuby);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Calculate a cube of shifted PSF images
 * @param image     the input PSF
 * @param ndivs     the number of subdivision (1-100)
 *
 * @return a new psfcube or NULL on error
 *
 * This function creates a cube of shifted PSF's. The cube contains all
 * shifted versions of the input PSF from dx = 0, ..., 1.0 to dy = 0, ..., 1.0
 * with the range divied by ndivs divisions.
 *
 */
/*----------------------------------------------------------------------------*/
sph_psfcube*
sph_andromeda_support_calc_psf_shift_subpix( cpl_image* image, int ndivs ) {
    sph_psfcube*        result = NULL;
    cpl_vector*         xin = NULL;
    cpl_vector*         yin = NULL;
    cpl_imagelist*      imlist = NULL;
    int                 nx      = ndivs;
    int                 ny      = ndivs;
    int                 xx      = 0;
    int                 yy      = 0;
    cpl_image*          imshift = NULL;
    sph_fft*            fft = NULL;
    double              prec = 0.0;

    cpl_ensure(image,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure( ndivs > 0 && ndivs < 100, CPL_ERROR_ILLEGAL_INPUT, NULL);


    prec = 1.0/ndivs;

    imlist = cpl_imagelist_new();
    cpl_imagelist_set(imlist,cpl_image_duplicate(image),0);
    xin = cpl_vector_new(1);
    cpl_vector_set(xin,0,0.0);
    yin = cpl_vector_new(1);
    cpl_vector_set(yin,0,0.0);
    result = sph_psfcube_new(imlist,xin,yin);
    if ( !result ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_OUTPUT, "Could not create the PSF cube.");
        cpl_vector_delete(xin);
        cpl_vector_delete(yin);
        cpl_imagelist_delete(imlist);
        return NULL;
    }

    fft = sph_fft_new(SPH_FFT_GSL_MIXEDRADIX);
    if ( !fft ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_OUTPUT, "Could not set up FFT.");
        sph_psfcube_delete(result);
        return NULL;
    }

    for (yy = 0; yy < ny; ++yy) {
        for (xx = 0; xx < nx; ++xx) {
            if ( xx || yy ) {
                imshift = sph_fft_shift_image(fft,image,prec *xx, prec * yy);
                sph_psfcube_insert(result,imshift,prec * xx, prec * yy );
            }
        }
    }
    sph_fft_delete(fft);
    return result;
}

#ifdef SPH_ANDROMEDA_USE
// Currently unused function:
static
cpl_imagelist*
sph_andromeda_support_adi( cpl_imagelist* inlist,
        const cpl_vector* angles,
        const cpl_vector* posind,
        const cpl_vector* negind) {
    cpl_imagelist*      result = NULL;
    int                 ii      = 0;
    cpl_image*          impos = NULL;
    cpl_image*          im = NULL;
    cpl_image*          imneg = NULL;
    cpl_image*          imneg_rot = NULL;
    sph_fft*            fft     = NULL;
    int                 negi        = 0;
    int                 posi        = 0;

    cpl_ensure( inlist, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( angles, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( posind, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( negind, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( cpl_vector_get_size(posind) ==
            cpl_vector_get_size(negind),
            CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure( cpl_imagelist_get_size(inlist) ==
            cpl_vector_get_size(angles),
            CPL_ERROR_ILLEGAL_INPUT, NULL);

    fft = sph_fft_new(SPH_FFT_FFTW3_DP);
    result = cpl_imagelist_new();
    for (ii = 0; ii < cpl_vector_get_size(posind); ++ii) {
        posi = (int)cpl_vector_get(posind,ii);
        negi = (int)cpl_vector_get(negind,ii);
        impos = cpl_imagelist_get(inlist,posi);
        imneg = cpl_imagelist_get(inlist,negi);
        imneg_rot = sph_fft_rotate_image(fft,imneg,cpl_vector_get(angles,posi)-cpl_vector_get(angles,negi));
        im = cpl_image_subtract_create(impos,imneg_rot);
        cpl_imagelist_set(result,im,ii);
        cpl_image_delete(imneg_rot); imneg_rot = NULL;
    }
    sph_fft_delete(fft); fft = NULL;
    return result;
}
#endif

cpl_vector*
sph_andromeda_support_calc_normalisations( const cpl_imagelist* inlist,
        const cpl_vector* posind,
        const cpl_vector* negind,
        double rmin,
        double rmax,
        int least_sq)
{
    cpl_vector*      result = NULL;
    int                 ii      = 0;
    cpl_image*          impos = NULL;
    cpl_image*          imneg = NULL;
    int                 negi        = 0;
    int                 posi        = 0;
    cpl_image*          distim      = NULL;
    cpl_image*          annulus     = NULL;
    cpl_image*          divisor     = NULL;
    int                 nx          = 0;
    int                 ny          = 0;
    double              gam         = 0.0;

    cpl_ensure( inlist, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( posind, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( negind, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( cpl_vector_get_size(posind) ==
            cpl_vector_get_size(negind),
            CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure( cpl_imagelist_get_size(inlist) > 0,
            CPL_ERROR_ILLEGAL_INPUT, NULL);

    nx = cpl_image_get_size_x( cpl_imagelist_get_const(inlist,0) );
    ny = cpl_image_get_size_y( cpl_imagelist_get_const(inlist,0) );
    result = cpl_vector_new(cpl_vector_get_size(posind));
    distim = sph_andromeda_support_cal_freq_image(nx,ny,(nx-1.0)/2.0,(ny-1.0)/2.0);
    if ( !distim ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_OUTPUT,"Could not create distim");
        return NULL;
    }
    annulus = sph_andromeda_support_calc_mask_image(distim,rmin,rmax);
    cpl_image_delete(distim);distim = NULL;
    if ( !annulus ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_OUTPUT,"Could not create annulus");
        return NULL;
    }
    for (ii = 0; ii < cpl_vector_get_size(posind); ++ii) {
        posi = (int)cpl_vector_get(posind,ii);
        negi = (int)cpl_vector_get(negind,ii);
        gam = 0.0;
        if ( least_sq ) {
            const cpl_image* impos2;
            imneg = cpl_image_multiply_create(cpl_imagelist_get_const(inlist,negi),
                                              annulus);
            divisor = cpl_image_multiply_create(imneg,
                                                cpl_imagelist_get_const(inlist,negi));
            impos2 = cpl_imagelist_get_const(inlist, posi);
            cpl_image_multiply(imneg,impos2);
            gam = cpl_image_get_flux(imneg)/cpl_image_get_flux(divisor);
            cpl_image_delete(divisor);divisor = NULL;
            cpl_image_delete(imneg);imneg = NULL;
        }
        else {
            imneg = cpl_image_multiply_create(cpl_imagelist_get_const(inlist,negi),
                                              annulus);
            impos = cpl_image_multiply_create(cpl_imagelist_get_const(inlist,posi),
                                              annulus);
            gam = cpl_image_get_flux(impos)/cpl_image_get_flux(imneg);
            cpl_image_delete(imneg);imneg = NULL;
            cpl_image_delete(impos); impos = NULL;
        }
        cpl_vector_set(result,ii,gam);
    }
    cpl_image_delete(annulus); annulus = NULL;
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create the difference images for a list of images
 * @param imlist    the input list of images
 * @param varilist  input list of variances (may be null)
 * @param posind    the list of positive image indices
 * @param negind    the list of negative image indices
 * @param rmin      minimum radius
 * @param rmax      maximum radius
 * @param least_sq  flag to control least squares gamma
 * @param gamma     (optional output) vector of gamma
 * @param varout    (optional output) output variance images
 * @param weigthsout (optional output) output weight images
 *
 * @return new imagelist with all difference images or NULL on error
 *
 * The input imagelist I_i must contain both positive (P) and negative (N_)
 * images. The indices for the positive images and negative images must
 * be supplied as two cpl_vectors.
 * The resulting imagelist contains all images with R_j = P_i - N_i.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist*
sph_andromeda_support_angular_difference( const cpl_imagelist* inlist,
                                          const cpl_imagelist* varilist,
                                          const cpl_vector* posind,
                                          const cpl_vector* negind,
                                          double rmin,
                                          double rmax,
                                          int least_sq,
                                          cpl_vector** ingamma,
                                          cpl_imagelist** varout,
                                          cpl_imagelist** weightsout)
{
    cpl_imagelist*      result = NULL;
    int                 ii      = 0;
    int                 negi        = 0;
    int                 posi        = 0;
    cpl_image*          diffim      = NULL;
    cpl_image*          wmask     = NULL;
    double              gam         = 0.0;
    cpl_vector*         vgamma  = NULL;

    cpl_ensure( inlist, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( posind, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( negind, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( cpl_vector_get_size(posind) ==
            cpl_vector_get_size(negind),
            CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure( cpl_imagelist_get_size(inlist) > 0,
            CPL_ERROR_ILLEGAL_INPUT, NULL);
    if ( varilist ) {
        cpl_ensure( cpl_imagelist_get_size(inlist) ==
                cpl_imagelist_get_size(varilist),
                CPL_ERROR_ILLEGAL_INPUT, NULL);
    }

    result = cpl_imagelist_new();
    if ( ingamma ) {
        *ingamma = sph_andromeda_support_calc_normalisations(inlist, posind,
                                                             negind, rmin, rmax,
                                                             least_sq);
        vgamma = *ingamma;
    }
    else {
        vgamma = cpl_vector_new(cpl_vector_get_size(posind));
        cpl_vector_fill(vgamma,1.0);
    }
    if (varout ) {
        *varout = cpl_imagelist_new();
    }
    if ( weightsout ) {
        *weightsout = cpl_imagelist_new();
    }
    for (ii = 0; ii < cpl_vector_get_size(posind); ++ii) {
        const cpl_image*     impos = NULL;
        const cpl_image*     imneg = NULL;
        cpl_image*          im = NULL;
        posi = (int)cpl_vector_get(posind,ii);
        negi = (int)cpl_vector_get(negind,ii);
        impos = cpl_imagelist_get_const(inlist,posi);
        imneg = cpl_imagelist_get_const(inlist,negi);
        gam = cpl_vector_get(vgamma,ii);
        im = cpl_image_multiply_scalar_create(imneg,gam);
        diffim = cpl_image_subtract_create(impos,im);
        cpl_image_delete(im);
        cpl_imagelist_set(result,diffim,ii);
        if ( varilist ) {
            impos = cpl_imagelist_get_const(varilist,posi);
            imneg = cpl_imagelist_get_const(varilist,negi);
            im = cpl_image_multiply_scalar_create(imneg,gam*gam);
            cpl_image_add(im,impos);
            if ( varout ) {
                cpl_imagelist_set(*varout,cpl_image_duplicate(im),ii);
            }
            if ( weightsout ) {
                wmask = cpl_image_duplicate(im);
                cpl_image_threshold(wmask,FLT_MIN,FLT_MIN,0.0,1.0);
                cpl_image_threshold(im,FLT_MIN,DBL_MAX,1.0,DBL_MAX);

                cpl_image_power(im,-1.0);
                cpl_image_multiply(wmask,im);
                cpl_imagelist_set(*weightsout,wmask,ii);
            }
            cpl_image_delete(im);
        }
    }
    if ( !ingamma) {
        cpl_vector_delete(vgamma);
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Calculate a map of radial distances
 * @param n the size of the resulting image in x
 * @param m the size of the resulting image in y
 * @param cx the centre x coordinate
 * @param cy the centry y coordinate
 *
 *
 * @return the map as a cpl_image
 *
 * This functions creates a rectangular array in which each element is
 * proportional to its frequency.
 * Its returns a rectangular array in which each pixel = euclidian
 * distance from the corner or from a given origin (Center),
 * e.g., ((n-1)/2,(m-1)/2).
 * This array may be used for a variety
 * of purposes, including frequency-domain filtering and
 * making pretty pictures.
 * The returned result R = sqrt(F^2+G^2)
 * depends on if cx and/or cy are >= 0.
 * If cx < 0and cy < 0:
 *     F = xx for 0 <= xx <= n/2
 *     F = n-xx otherwise
 *     G = yy for 0 <= yy <= m/2
 *     G = m-yy otherwise
 * otherwise
 *    F = xx -cx
 *    G = yy -cy
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_andromeda_support_cal_freq_image( int n, int m, double cx, double cy ) {
    cpl_image*      result = NULL;
    int             xx  = 0;
    int             yy  = 0;
    double          F = 0.0;
    double          G = 0.0;
    double          R = 0.0;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    result = cpl_image_new(n,m,CPL_TYPE_DOUBLE);
    for (xx = 0; xx < n; ++xx) {
        if ( cx < 0 && cy < 0 ) {
            if ( xx >= 0 && xx <= n/2 ) F = xx;
            else F = n-xx;
        }
        else {
            F = xx - cx;
        }
        for (yy = 0; yy < m; ++yy) {
            if ( cx < 0 && cy < 0 ) {
                if ( yy >= 0 && yy <= m/2 ) G = yy;
                else G = m-yy;
            }
            else {
                G = yy - cy;
            }
            R = sqrt( F * F + G * G);
            cpl_image_set( result, xx + 1, yy + 1, R );
        }
    }

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Calculate a mask image for all pixels in annulus
 * @param distimage     image containing distances (see below)
 * @param rmin          minimum radial distance
 * @param rmax          maximum radial distance
 *
 * @return  new mask as cpl_image
 *
 * Returns a new image with all pixels inside the given radius set to 1,
 * and to 0 otherwise.
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_andromeda_support_calc_mask_image( const cpl_image* distim,
        double rmin, double rmax )
{
    cpl_image*      result = NULL;

    cpl_ensure(distim, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(rmin < rmax, CPL_ERROR_ILLEGAL_INPUT, NULL);

    result = cpl_image_duplicate(distim);
    cpl_image_threshold(result,rmin,rmax,0.0,0.0);
    cpl_image_threshold(result,-1.0,0.0,0.0,1.0);
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create pairs of angles
 * @param angles angles to pair up
 * @param angmin the minimum separation angle
 * @param waste optional output vector of unused indices
 *
 * @return the pairs of indices paired up.
 *
 * Given a monotonic increasing array of angles, this function computes and returns the
 * couples of indices of the array for which the separation is the closest to
 * the value ANGMIN, by using the highest possible number of angles, all if
 *  possible. If this is not the case, the list of the unused indices is
 * returned in the optional output parameter waste.
 *
 *
 */
/*----------------------------------------------------------------------------*/
cpl_bivector*
sph_andromeda_support_create_angpairs( cpl_vector* angles,
                                            double angmin,
                                            cpl_vector** waste )
{
    cpl_bivector*   result = NULL;
    int             ii      = 0;
    int             jj      = 0;
    int             nangs   = 0;
    double          ang0    = 0.0;
    int             found   = 0;

    cpl_vector*     indexi = NULL;
    cpl_vector*     indexj = NULL;
    cpl_vector*     unmatch = NULL;
    int             nfound = 0;
    int             nunmatched = 0;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    nangs = cpl_vector_get_size(angles);

    if ( nangs < 2 ) {
        SPH_ERROR_RAISE_ERR(
                CPL_ERROR_ILLEGAL_INPUT,
                "Need at least 2 angles!");
        return NULL;
    }
    cpl_vector_sort(angles,1);
    indexi = cpl_vector_new(nangs);
    indexj = cpl_vector_new(nangs);
    unmatch = cpl_vector_new(nangs);
    cpl_vector_fill(unmatch,1.0);
    for (ii = 0; ii < nangs; ++ii) {
        found = 0;
        ang0 = cpl_vector_get(angles,ii);
        for (jj = ii+1; jj < nangs; ++jj) {
            if ( cpl_vector_get(angles,jj) - ang0 >= angmin ) {
                found = jj;
                SPH_RAISE_CPL_RESET;

                break;
            }
        }
        if ( found ) {
            cpl_vector_set(indexi,nfound,ii);
            cpl_vector_set(indexj,nfound,jj);
            nfound++;
            SPH_RAISE_CPL_RESET;
        }
        else {
            for (jj = ii-1; jj >=0; --jj) {
                if ( ang0 - cpl_vector_get(angles,jj) >= angmin ) {
                    found = jj;
                    SPH_RAISE_CPL_RESET;
                    break;
                }
            }
            if ( found ) {
                cpl_vector_set(indexi,nfound,ii);
                cpl_vector_set(indexj,nfound,jj);
                nfound++;
                SPH_RAISE_CPL_RESET;
            }
            else {
                cpl_vector_set(unmatch,nunmatched,ii);
                nunmatched++;
                SPH_RAISE_CPL_RESET;
            }
        }
    }

    if ( nfound < 1 ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_DATA_NOT_FOUND,"Could not find any angle pairs.");
        cpl_vector_delete(unmatch);
        cpl_vector_delete(indexi);
        cpl_vector_delete(indexj);
        return NULL;
    }

    cpl_vector_set_size(indexi,nfound);
    cpl_vector_set_size(indexj,nfound);
    result = cpl_bivector_wrap_vectors(indexi,indexj);

    if ( waste ) {
        *waste = unmatch;
    }
    else {
        cpl_vector_delete(unmatch);
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Set region info
 * @param x     the  x pixel coordinate of center
 * @param y     the  y pixel coordinate of center
 * @param i     the index in the list of difference images
 * @param core  pointer to the core structure, will be changed on output
 *
 * @return error code
 *
 * This function sets the region fields in the core structure to contain
 * the two PSFs fully.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_andromeda_support_get_region(
        int x, int y,int i,
        sph_andromeda_core* core)
{
    double      x0      = 0.0;
    double      y0      = 0.0;
    double      decalx_p  = 0.0;
    double      decaly_p  = 0.0;
    double      decalx_n  = 0.0;
    double      decaly_n  = 0.0;
    double      parang_p  = 0.0;
    double      parang_n  = 0.0;

    double      fldecalx_p  = 0.0;
    double      fldecaly_p  = 0.0;
    double      fldecalx_n  = 0.0;
    double      fldecaly_n  = 0.0;
    int         psfsize     = 0;
    double      posright = 0;
    double      postop = 0;
    double      negright = 0;
    double      negtop = 0;
    double      posleft = 0;
    double      posbot= 0;
    double      negleft = 0;
    double      negbot = 0;

    cpl_ensure_code(core,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( i >=0 ,CPL_ERROR_NULL_INPUT);
    parang_p = cpl_vector_get(core->angles,
            cpl_vector_get(cpl_bivector_get_x(core->indices),i));
    parang_n = cpl_vector_get(core->angles,
            cpl_vector_get(cpl_bivector_get_y(core->indices),i));

    parang_p *= 1.0 * CPL_MATH_RAD_DEG;
    parang_n *= 1.0 * CPL_MATH_RAD_DEG;

    x0 = x - (core->nx-1.0)/2.0;
    y0 = y - (core->ny-1.0)/2.0;
    decalx_p = x0 * cos(parang_p) - y0 * sin(parang_p);
    decaly_p = y0 * cos(parang_p) + x0 * sin(parang_p);

    decalx_n = x0 * cos(parang_n) - y0 * sin(parang_n);
    decaly_n = y0 * cos(parang_n) + x0 * sin(parang_n);

    fldecalx_p = floor(decalx_p);
    fldecaly_p = floor(decaly_p);

    fldecalx_n = floor(decalx_n);
    fldecaly_n = floor(decaly_n);

    psfsize = sph_psfcube_get_npix_side(core->psfcube);

#ifdef SPH_ANDROMEDA_USE
    double prec = (double)(sph_psfcube_get_size(core->psfcube)-1.0);

//    core->possubx = round((decalx_p-fldecalx_p)); // * prec when indexed
//    core->possuby = round((decaly_p-fldecaly_p)); // version of psf cube
//    core->negsubx = round((decalx_n-fldecalx_n)); // has been implemented
//    core->negsuby = round((decaly_n-fldecaly_n));
#endif

    core->possubx = decalx_p-fldecalx_p;
    core->possuby = decaly_p-fldecaly_p;
    core->negsubx = decalx_n-fldecalx_n;
    core->negsuby = decaly_n-fldecaly_n;

    posleft = round((double)core->nx/2.0+fldecalx_p-(double)psfsize/2.0);
    negleft = round((double)core->nx/2.0+fldecalx_n-(double)psfsize/2.0);
    posbot = round((double)core->ny/2.0+fldecaly_p-(double)psfsize/2.0);
    negbot = round((double)core->ny/2.0+fldecaly_n-(double)psfsize/2.0);
    posright = round((double)core->nx/2.0+fldecalx_p+(double)psfsize/2.0 - 1.0);
    negright = round((double)core->nx/2.0+fldecalx_n+(double)psfsize/2.0 - 1.0);
    postop = round((double)core->ny/2.0+fldecaly_p+(double)psfsize/2.0 - 1.0);
    negtop = round((double)core->ny/2.0+fldecaly_n+(double)psfsize/2.0 - 1.0);
    core->posleft = (int)(posleft);
    core->negleft = (int)(negleft);
    core->posbot = (int)(posbot);
    core->negbot = (int)(negbot);
    core->maxx = posright > negright ? posright : negright;
    core->maxy = postop > negtop ? postop : negtop;
    core->minx = posleft < negleft ? posleft : negleft;
    core->miny = posbot < negbot ? posbot : negbot;

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create cut-out with the pos and neg PSFs
 * @param core  the andromeda core
 * @param gamma the gamma weighting
 *
 * @return the new cut-out image with PSFs
 *
 * Places (paints) the positive and the negative PSF onto
 * a cut-out image. The neg PSF is weighed by gamma.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_andromeda_support_place_psfs( sph_andromeda_core* core,
        double gamma ) {
    cpl_image*          result  = NULL;
    cpl_image*          neg  = NULL;
    sph_error_code      rerr = CPL_ERROR_NONE;
    const cpl_image*    psf;

    cpl_ensure(core,CPL_ERROR_NULL_INPUT,NULL);
    SPH_RAISE_CPL_RESET
    result = cpl_image_new( core->maxx - core->minx + 1,
            core->maxy - core->miny + 1,
            CPL_TYPE_DOUBLE);
    neg = cpl_image_new( core->maxx - core->minx + 1,
            core->maxy - core->miny + 1,
            CPL_TYPE_DOUBLE);

    if ( !result || !neg ) {
        SPH_ERROR_RAISE_ERR(SPH_ERROR_MEMORY_FAIL,
                "Out of memory trying to create image of %d ,%d",
                core->maxx - core->minx + 1,
                core->maxy - core->miny + 1);
        cpl_image_delete(result);cpl_image_delete(neg);
        return NULL;
    }

    psf = sph_psfcube_get(
            core->psfcube,
            core->possubx,
            core->possuby);
    if ( !psf ) {
        SPH_ERROR_RAISE_ERR(SPH_ERROR_PREVIOUS_OP_FAILED,
                "Could not obtain PSF from PSF cube.");
        cpl_image_delete(result);
        cpl_image_delete(neg);
        return NULL;
    }
    rerr = cpl_image_copy(result,
            psf,
            core->posleft-core->minx+1,
            core->posbot-core->miny+1);
    if ( rerr == CPL_ERROR_ACCESS_OUT_OF_RANGE ) {
        SPH_RAISE_CPL;
        cpl_image_delete(result);
        cpl_image_delete(neg);
        return NULL;
    }
    rerr = cpl_image_copy(neg,
            psf,
            core->negleft-core->minx+1,
            core->negbot-core->miny+1);
    if ( rerr == CPL_ERROR_ACCESS_OUT_OF_RANGE ) {
        SPH_RAISE_CPL;
        cpl_image_delete(result);
        cpl_image_delete(neg);
        return NULL;
    }
    cpl_image_multiply_scalar(neg, gamma);
    cpl_image_subtract(result, neg);
    cpl_image_delete(neg);
    SPH_RAISE_CPL_RESET
    return result;
}
/**@}*/
