/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include <cpl.h>
#include "sph_coord.h"
#include "sph_error.h"
#include "sph_common_keywords.h"
#include <math.h>

static
sph_error_code
sph_coord_get_ra_dec_epoch( cpl_propertylist* pl,
                            char* ra, char* dec,
                            int* epoch, unsigned int n );

/*----------------------------------------------------------------------------*/
/**
 * @brief Initialize sky calc
 *
 * @return error code
 *
 * This method intializes skycalc to be used with different times using the
 * do_skycalc function.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_coord_initialise_skycalc( cpl_propertylist* pl,
                              sph_skycalc_input* skyinp,
                              double zerotime,
                              sph_skycalc_output* skyout)
{
    char ra[20];
    char dec[20];
    int epoch = 2000;


    snprintf( ra, sizeof(ra), "NONE");
    snprintf( dec, sizeof(ra), "NONE");

    sph_coord_get_ra_dec_epoch(pl,ra,dec,&epoch,sizeof(ra));

    SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
                          "Time = %f",   zerotime );

    sph_skycalc_setup_mjd( skyinp, ra, dec, epoch,
                           zerotime );


    sph_skycalc( skyinp, skyout );

    SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
                          "Time = %f, PA = %f",
                          zerotime,
                          skyout->pa );

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Do skycalc
 *
 * @return error code
 *
 * This method calls a previously initialised skycalc with a new timestamp and
 * fills the skyput data structure with the data corresponding to that timestamp.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_coord_do_skycalc( sph_skycalc_input* skyinp,
                            double time,
                            sph_skycalc_output* skyout)
{
    sph_skycalc_set_mjd( skyinp, time );
    sph_skycalc( skyinp, skyout );
    SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO, "iTime = %f, iPA = %f", time, skyout->pa );
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*---------------------------------------------------------------------------------------------------
 -         Internal functions
 ----------------------------------------------------------------------------------------------------*/

static
sph_error_code sph_coord_get_ra_dec_epoch( cpl_propertylist* pl,
                                           char* ra,
                                           char* dec,
                                           int* epoch,
                                           unsigned int n )
{
    double raordec = 0.0;
    int hod = 0;
    int mm = 0;
    double ss = 0;

    if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_RA ) )
    {
        raordec = cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_RA);

        hod     = (int)floor(raordec / 10000.0);

        mm      = (int)floor((raordec - 10000.0*hod)/100.0);

        ss      = raordec - 10000.0*hod - 100.0*mm;

        SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO,
                             "Original RA was %.2f",raordec)
        SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO,
                             "derived seconds %.2f",ss)

        SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO,
                             "Constructing RA string to be %02d:%02d:%02.2f",
                             hod,mm,ss)

        snprintf( ra, n, "%02d:%02d:%02.2f",hod,mm,ss);
    }
    else {
        SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING,
                                "%s keyword not found!\n",
                                SPH_COMMON_KEYWORD_RA );
    }
    if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_DEC ) ) {
        raordec = cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_DEC);

        hod     = (int)floor(raordec / 10000.0);

        mm      = (int)floor((raordec - 10000.0*hod)/100.0);

        ss      = raordec - 10000.0 * hod - 100.0 * mm;

        SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO,
                             "Original DEC was %.2f",raordec)
        SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO,
                             "Constructing DEC string to be %02d:%02d:%02.2f",
                             hod,mm,ss)
        snprintf( dec, n, "%02d:%02d:%02.2f",hod,mm,ss);
    }
    else
    {
        SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING,
                                "%s keyword not found!\n",
                                SPH_COMMON_KEYWORD_DEC );
    }
    if ( cpl_propertylist_has( pl, SPH_COMMON_KEYWORD_EPOCH ) )
    {
        *epoch = cpl_propertylist_get_type(pl, SPH_COMMON_KEYWORD_EPOCH)
            == CPL_TYPE_INT
            ? cpl_propertylist_get_int(pl, SPH_COMMON_KEYWORD_EPOCH)
            : (int)cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_EPOCH);
    }
    else
    {
        SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING,
                                "%s keyword not found!",
                                SPH_COMMON_KEYWORD_EPOCH );
        SPH_INFO_MSG("Set default EPOCH = 2000");
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
