/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */
#include "sph_error.h"
#include "sph_dataset.h"
#include "sph_dataset_peak.h"
#include <cpl.h>

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new root peak
 *
 * @param dat       The dataset to create the peak root for
 *
 * @return          pointer to the new peak
 *
 * This creates a new peak root. The peak root is the root for the peak
 * tree that can be build on the dataset to find all peaks in the dataset.
 *
 */
/*----------------------------------------------------------------------------*/
sph_dataset_peak*
sph_dataset_peak_new_root( sph_dataset* dat ) {
    sph_dataset_peak*   self = NULL;

    if ( !dat ) {
        SPH_NULL_ERROR
        return NULL;
    }

    self = cpl_calloc( 1, sizeof(sph_dataset_peak));

    if ( !self ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "Could not allocate memory for peak.");
        return NULL;
    }

    self->dat = dat;
    self->minpixels = 1;
    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new sub-peak
 *
 * @param parent       The peak to create the sub peak for
 *
 * @return              pointer to the new peak
 *
 * This creates a new sub-peak root. The peak is NOT entered as a child, but
 * the parent pointer of the newly created subpeak does point to parent.
 * Also, the frac_acc value is copied from the parent
 */
/*----------------------------------------------------------------------------*/
sph_dataset_peak*
sph_dataset_peak_new_sub( sph_dataset_peak* parent, double fthreshold ) {
    sph_dataset_peak*   self = NULL;

    if ( !parent ) {
        SPH_NULL_ERROR;
        return NULL;
    }
    self = cpl_calloc( 1, sizeof(sph_dataset_peak));

    if ( !self ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "Could not allocate memory for peak.");
        return NULL;
    }
    self->parent = parent;
    self->frac_acc = parent->frac_acc;
    self->minpixels = parent->minpixels;
    self->frac_threshold = fthreshold;

    return self;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Return the number of subpeaks of a peak
 *
 * @param self      the peak to get the number of found subpeaks for
 *
 * @return number of subpeaks or -1 on error
 *
 * This call creates the whole tree of sub-peaks for this peak. This is done
 * iteratively. Before calling this, the frac_acc and frac_threshold
 * of the peak have to be set.
 *
 */
/*----------------------------------------------------------------------------*/

int sph_dataset_peak_get_number_subpeaks( sph_dataset_peak* self )
{
    sph_dataset_peak*           temppeak            = NULL;
    int                         result              = 0;

    if ( !self ) {
        SPH_NULL_ERROR
        return -1;
    }

    temppeak = self->firstchild;

    while ( temppeak )
    {
        result++;
        result = result + sph_dataset_peak_get_number_subpeaks( temppeak );
        temppeak = temppeak->next;
    }

    return result;

}

/*----------------------------------------------------------------------------*/
/**
 * @brief Find the subpeaks of a peak
 *
 * @param self      the peak to find subpeaks for
 *
 * @return error code of the operation
 *
 * This call creates the whole tree of sub-peaks for this peak. This is done
 * iteratively. Before calling this, the frac_acc and frac_threshold
 * of the peak have to be set.
 *
 */
/*----------------------------------------------------------------------------*/

int sph_dataset_peak_find_subpeaks( sph_dataset_peak* self )
{
    int             ii          = 0;
    double          y           = 0.0;
    double          x           = 0.0;
    double          norm        = 0.0;
    cpl_vector*     xvals       = NULL;
    cpl_vector*     yvals       = NULL;
    short           connected   = 0;
    short           peakcount   = 0;
    int             vecsizes    = 0;
    double          maxn        = 0.0;
    double          fthreshold  = 0;
    sph_dataset_peak* dummy     = NULL;
    sph_dataset_peak* root      = NULL;
    sph_dataset_peak* newpeak   = NULL;
     sph_dataset**   peaks       = NULL;
    sph_dataset*    cpeak       = NULL;
    cpl_vector*     cpeakx      = NULL;
    cpl_vector*     cpeaky      = NULL;
    cpl_vector*     tempvect    = NULL;
    double          tempnorm    = 0.0;
    int             minsize     = 0;


    if ( !self ) {
        SPH_NULL_ERROR
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !self->dat ) {
        SPH_NULL_ERROR
        return CPL_ERROR_NULL_INPUT;
    }
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        sph_error_raise( SPH_ERROR_PREVIOUS_OP_FAILED,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Entering the subfinding with a CPL error "
                "is a bad idea. The CPL error was: %s.",
                cpl_error_get_message());
        return SPH_ERROR_PREVIOUS_OP_FAILED;
    }
    if ( self->frac_acc == 0.0 || self->frac_threshold > 1.0 ) {
        sph_error_raise( SPH_ERROR_INCONSISTENT_INPUT,
                        __FILE__,
                        __func__,
                        __LINE__,
                        SPH_ERROR_ERROR,
                        "No frac_acc and/or "
                        "frac_threshold defined! %f, %f", self->frac_acc, self->frac_threshold);
        return SPH_ERROR_INCONSISTENT_INPUT;
    }
    fthreshold = self->frac_threshold;

    if ( fthreshold > 1.0 ) {
        return CPL_ERROR_NONE;
    }


    xvals = self->dat->xvalues;
    yvals = self->dat->yvalues;
    if ( xvals == NULL || yvals == NULL) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__,
                         __LINE__, SPH_ERROR_ERROR,
                         "No vectors defined in dataset !");
        return SPH_ERROR_GENERAL;
    }

    if ( self->dat->npoints == 0 ) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__,
                         __LINE__, SPH_ERROR_WARNING,
                         "No points in dataset to investigate!");
    }

    root = self;
    while ( root->parent ) {
        root = root->parent;
    }

    norm = cpl_vector_get_max( root->dat->yvalues );

    maxn = cpl_vector_get_max( root->dat->yvalues );

    norm = norm * fthreshold;

    minsize = self->minpixels;

    peaks = cpl_calloc( self->dat->npoints, sizeof(sph_dataset*) );

    for (ii = 0; ii < self->dat->npoints; ++ii) {
        y = cpl_vector_get( yvals, ii);
        x = cpl_vector_get( xvals, ii);
        if ( y > norm ) {
            if ( connected ) {
                vecsizes = cpl_vector_get_size( cpeakx );
                cpl_vector_set_size( cpeakx, vecsizes + 1 );
                cpl_vector_set_size( cpeaky, vecsizes + 1 );
                cpl_vector_set( cpeakx, vecsizes, x);
                cpl_vector_set( cpeaky, vecsizes, y);
            }
            else {
                cpeakx = cpl_vector_new(1);
                cpeaky = cpl_vector_new(1);
                cpl_vector_set( cpeakx, 0, x);
                cpl_vector_set( cpeaky, 0, y);
                connected = 1;
            }
        }
        else {
            if ( connected ) {
                connected = 0;
                vecsizes = cpl_vector_get_size( cpeakx );
                if ( vecsizes >= minsize ) {
                    if ( vecsizes == 1 ) { /* TODO: this is ugly. */
                        cpl_vector_set_size( cpeakx, 2 );
                        cpl_vector_set_size( cpeaky, 2 );
                        cpl_vector_set( cpeakx, 1, cpl_vector_get( cpeakx, 0 ) );
                        cpl_vector_set( cpeaky, 1, cpl_vector_get( cpeaky, 0 ) );
                        vecsizes = 2;
                    }
                    cpeak = sph_dataset_new( vecsizes,
                                             cpl_vector_get(cpeakx,0),
                                             cpl_vector_get(cpeakx,vecsizes-1)
                    );
                    cpl_vector_delete( cpeak->yvalues );cpeak->yvalues = NULL;
                    cpl_vector_delete( cpeakx ); cpeakx = NULL;
                    cpeak->yvalues = cpeaky; cpeaky = NULL;

                    peaks[ peakcount ] = cpeak;
                    cpeak = NULL;
                    peakcount++;
                }
                else {
                    cpl_vector_delete( cpeakx );cpeakx = NULL;
                    cpl_vector_delete( cpeaky );cpeaky = NULL;
                }
            }
        }
    }
    if ( connected ) {
        connected = 0;
        vecsizes = cpl_vector_get_size( cpeakx );
        if ( vecsizes >= minsize ) {
            if ( vecsizes == 1 ) { /* TODO: this is ugly. */
                cpl_vector_set_size( cpeakx, 2 );
                cpl_vector_set_size( cpeaky, 2 );
                cpl_vector_set( cpeakx, 1, cpl_vector_get( cpeakx, 0 ) );
                cpl_vector_set( cpeaky, 1, cpl_vector_get( cpeaky, 0 ) );
                vecsizes = 2;
            }
            cpeak = sph_dataset_new( vecsizes,
                    cpl_vector_get(cpeakx,0),
                    cpl_vector_get(cpeakx,vecsizes - 1)
            );
            cpl_vector_delete( cpeak->yvalues ); cpeak->yvalues = NULL;
            cpl_vector_delete( cpeakx ); cpeakx = NULL;

            cpeak->yvalues = cpeaky;


            peaks[ peakcount ] = cpeak;
            cpeak = NULL;
            peakcount++;
        }
    }


    if ( self->firstchild ) {
        sph_dataset_peak_delete( self->firstchild );
        self->firstchild = NULL;
    }

    if ( peakcount > 1  ) {
        for (ii = 0; ii < peakcount; ++ii) {
            if ( peaks[ii] ) {
                newpeak = sph_dataset_peak_new_sub( self, fthreshold );

                newpeak->dat = sph_dataset_new( peaks[ii]->npoints, cpl_vector_get_min( peaks[ii]->xvalues),
                                                cpl_vector_get_max( peaks[ii]->xvalues ) );
                cpl_vector_delete(newpeak->dat->xvalues); newpeak->dat->xvalues = NULL;
                cpl_vector_delete(newpeak->dat->yvalues); newpeak->dat->yvalues = NULL;

                newpeak->dat->xvalues = cpl_vector_duplicate( peaks[ii]->xvalues );
                newpeak->dat->yvalues = cpl_vector_duplicate( peaks[ii]->yvalues );

                newpeak->maxpos     = sph_dataset_get_max_xpos( peaks[ii] );
                newpeak->meanval    = cpl_vector_get_mean( peaks[ii]->yvalues );
                newpeak->medianval  = cpl_vector_get_mean( peaks[ii]->yvalues );
                newpeak->totval     = newpeak->meanval *
                                        cpl_vector_get_size( peaks[ii]->yvalues );

                tempvect = cpl_vector_duplicate( peaks[ii]->yvalues );
                tempnorm = cpl_vector_product( tempvect, tempvect );
                cpl_vector_power( tempvect, 2.0 );
                cpl_vector_multiply( tempvect, peaks[ii]->xvalues );
                newpeak->meanpos     = cpl_vector_get_mean( tempvect ) * cpl_vector_get_size( tempvect ) / tempnorm;

                cpl_vector_delete( tempvect );tempvect = NULL;

                if ( self->firstchild == NULL ) {
                    self->firstchild = newpeak;
                }
                else {
                    dummy = self->firstchild->next;
                    self->firstchild->next = newpeak;
                    newpeak->prev = self->firstchild;
                    if ( dummy ) {
                        dummy->prev = newpeak;
                        newpeak->next = dummy;
                    }
                }
                sph_dataset_delete( peaks[ii] );peaks[ii] = NULL;
            }
        }
    }
    else {
        self->maxpos     = sph_dataset_get_max_xpos( self->dat );
        self->meanval    = cpl_vector_get_mean( self->dat->yvalues );
        self->medianval  = cpl_vector_get_mean( self->dat->yvalues );
        self->totval     = self->meanval *
                                cpl_vector_get_size( self->dat->yvalues );

        tempvect = cpl_vector_duplicate( self->dat->yvalues );
        tempnorm = cpl_vector_product( tempvect, tempvect );
        cpl_vector_power( tempvect, 2.0 );
        cpl_vector_multiply( tempvect, self->dat->xvalues );
        self->meanpos     = cpl_vector_get_mean( tempvect ) * cpl_vector_get_size( tempvect )/ tempnorm;

        cpl_vector_delete( tempvect );
        tempvect = NULL;
        sph_dataset_delete( peaks[0] );peaks[0] = NULL;
    }

    if ( peaks ) {
        cpl_free(peaks); peaks = NULL;
    }

    if ( self->firstchild ) {
        dummy = self->firstchild;
        while ( dummy ) {
            sph_dataset_peak_find_subpeaks( dummy );
            dummy = dummy->next;
        }
    }
    else {
        self->frac_threshold = self->frac_threshold + self->frac_acc;

        if ( self->frac_threshold < 1.0 ) {
            sph_dataset_peak_find_subpeaks( self );
        }
    }
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        sph_error_raise( SPH_ERROR_CPL_ERROR,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "During execution of the function a CPL error "
                "was raised, the error was: %s.",
                cpl_error_get_message());
        return SPH_ERROR_CPL_ERROR;
    }
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the peak and all children ( but not siblings)
 *
 * @param peak      the peak to delete (along with its children)
 *
 * @return error code
 *
 * The deletes the peak and all the children below it (inlcuding children of
 * children etc.)
 * Delete a root peak will delete the whole tree.
 *
 */
/*----------------------------------------------------------------------------*/
int
sph_dataset_peak_delete( sph_dataset_peak* self ) {
    sph_dataset_peak*       dummy   =   NULL;
    if ( !self ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    if ( self->firstchild ) {
        while ( self->firstchild->prev ) {
            dummy = self->firstchild->prev->prev;
            sph_dataset_peak_delete( self->firstchild->prev );
            self->firstchild->prev = dummy;
        }
        while ( self->firstchild->next ) {
            dummy = self->firstchild->next->next;
            sph_dataset_peak_delete( self->firstchild->next );
            self->firstchild->next = dummy;
        }
        sph_dataset_peak_delete( self->firstchild );self->firstchild = NULL;
    }
    sph_dataset_delete(self->dat);self->dat = NULL;
    cpl_free( self );
    return CPL_ERROR_NONE;
}
