/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include "sph_dataset_stats.h"
#include "sph_dataset_peak.h"
#include "sph_dataset.h"
#include "sph_error.h"
#include <cpl.h>

sph_error_code SPH_DATASET_STATS_GENERAL     = SPH_DATASET_STATS_ERR_START + 0;

sph_dataset_stats*
sph_dataset_stats_new( sph_dataset* datat ) {
    sph_dataset_stats*      self        = NULL;

    self = cpl_calloc( 1, sizeof(sph_dataset_stats) );

    self->dataset = datat;
    return self;
}

int sph_dataset_stats_find_peaks_simple_method( sph_dataset_stats* self,
                                  double fthreshold )
{
    int             ii          = 0;
    double          y           = 0.0;
    double          x           = 0.0;
    double          norm        = 0.0;
    cpl_vector*     xvals       = NULL;
    cpl_vector*     yvals       = NULL;
    short           connected   = 0;
    short           peakcount   = 0;
    int             vecsizes    = 0;
    double          maxn        = 0.0;
    double          minfract    = 0.001;

    sph_dataset**   peaks       = NULL;
    sph_dataset*    cpeak       = NULL;
    cpl_vector*     cpeakx      = NULL;
    cpl_vector*     cpeaky      = NULL;


    if ( !self ) {
        SPH_NULL_ERROR
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !self->dataset ) {
        SPH_NULL_ERROR
        return CPL_ERROR_NULL_INPUT;
    }

    xvals = self->dataset->xvalues;
    yvals = self->dataset->yvalues;
    if ( xvals == NULL || yvals == NULL) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__,
                         __LINE__, SPH_ERROR_ERROR,
                         "No vectors defined in dataset !");
        return SPH_ERROR_GENERAL;
    }

    if ( self->dataset->npoints == 0 ) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__,
                         __LINE__, SPH_ERROR_WARNING,
                         "No points in dataset to investigate!");
    }
    norm = cpl_vector_get_max( self->dataset->yvalues );

    maxn = cpl_vector_get_max( self->dataset->yvalues );

    if ( fthreshold < 0.0 ) { //interpret fthreshold as fractional value
        norm = -1.0 * norm * fthreshold;
    }
    else { // interpret fthreshold as absolute value
        norm = fthreshold;
    }

    peaks = cpl_calloc( self->dataset->npoints, sizeof(sph_dataset*) );

    for (ii = 0; ii < self->dataset->npoints; ++ii) {
        y = cpl_vector_get( yvals, ii);
        x = cpl_vector_get( xvals, ii);
        if ( y > norm ) {
            if ( connected ) {
                vecsizes = cpl_vector_get_size( cpeakx );
                cpl_vector_set_size( cpeakx, vecsizes + 1 );
                cpl_vector_set_size( cpeaky, vecsizes + 1 );
                cpl_vector_set( cpeakx, vecsizes, x);
                cpl_vector_set( cpeaky, vecsizes, y);
            }
            else {
                cpeakx = cpl_vector_new(1);
                cpeaky = cpl_vector_new(1);
                cpl_vector_set( cpeakx, 0, x);
                cpl_vector_set( cpeaky, 0, y);
                connected = 1;
            }
        }
        else {
            if ( connected ) {
                connected = 0;
                vecsizes = cpl_vector_get_size( cpeakx );
                if ( vecsizes == 1 ) { /* TODO: this is ugly. */
                    cpl_vector_set_size( cpeakx, 2 );
                    cpl_vector_set_size( cpeaky, 2 );
                    cpl_vector_set( cpeakx, 1, cpl_vector_get( cpeakx, 0 ) );
                    cpl_vector_set( cpeaky, 1, 0.0);
                    vecsizes = 2;
                }
                if ( vecsizes > minfract * maxn ) {
                    cpeak = sph_dataset_new( vecsizes,
                                             cpl_vector_get(cpeakx,0),
                                             cpl_vector_get(cpeakx,vecsizes-1)
                    );
                    cpl_vector_delete( cpeak->yvalues );
                    cpl_vector_delete( cpeakx );
                    cpeak->yvalues = cpeaky;

                    cpeakx = NULL;
                    cpeaky = NULL;

                    peaks[ peakcount ] = cpeak;
                    cpeak = NULL;
                    peakcount++;
                }
                else {
                    cpl_vector_delete( cpeakx );
                    cpl_vector_delete( cpeaky );

                    cpeakx = NULL;
                    cpeaky = NULL;
                }
            }
        }
    }
    if ( connected ) {
        connected = 0;
        vecsizes = cpl_vector_get_size( cpeakx );
        if ( vecsizes == 1 ) { /* TODO: this is ugly. */
            cpl_vector_set_size( cpeakx, 2 );
            cpl_vector_set_size( cpeaky, 2 );
            cpl_vector_set( cpeakx, 1, cpl_vector_get( cpeakx, 0 ) );
            cpl_vector_set( cpeaky, 1, 0.0);
            vecsizes = 2;
        }
        if ( vecsizes > minfract * maxn ) {
            cpeak = sph_dataset_new( vecsizes,
                                     cpl_vector_get(cpeakx,0),
                                     cpl_vector_get(cpeakx,vecsizes-1)
            );
            cpl_vector_delete( cpeak->yvalues );
            cpl_vector_delete( cpeakx );
            cpeak->yvalues = cpeaky;

            cpeakx = NULL;
            cpeaky = NULL;

            peaks[ peakcount ] = cpeak;
            cpeak = NULL;
            peakcount++;
        }
        else {
            cpl_vector_delete( cpeakx );
            cpl_vector_delete( cpeaky );

            cpeakx = NULL;
            cpeaky = NULL;
        }
    }

    self->npeaks = peakcount;

    if ( peakcount > 0  ) {
        if ( self->peakloc ) {
            cpl_free( self->peakloc );
        }
        self->peakloc = cpl_calloc( peakcount, sizeof(double) );
        for (ii = 0; ii < peakcount; ++ii) {
            if ( peaks[ii] ) {
                self->peakloc[ii] = sph_dataset_get_max_xpos( peaks[ii] );
            }
        }
    }

    self->peakproduct = 1.0;
    for (ii = 0; ii < peakcount; ++ii) {
        if ( peaks[ii] ) {
            self->peakproduct = self->peakproduct * peaks[ii]->npoints;
        }
    }


    if ( peaks ) {
        for (ii = 0; ii < peakcount; ++ii) {
            if ( peaks[ii] ) {
                sph_dataset_delete( peaks[ii] );peaks[ii] = NULL;
            }
        }
        cpl_free(peaks); peaks = NULL;
    }
    if ( cpeak ) {
        sph_dataset_delete( cpeak );
    }
    self->peakthresh = norm;

    return cpl_error_get_code();
}

int sph_dataset_stats_find_peaks_tree_method( sph_dataset_stats* self,
                                  double fthreshold,
                                  int minpix )
{
    sph_error_code              rerr    = 0;
    sph_dataset_peak*           root    = NULL;
    sph_dataset_peak*           apeak   = NULL;
    int                         count   = 0;

    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        sph_error_raise( SPH_ERROR_PREVIOUS_OP_FAILED,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Entering sph_dataset_find_peaks with a CPL error "
                "is a bad idea. The CPL error was: %s.",
                cpl_error_get_message());
        return SPH_ERROR_PREVIOUS_OP_FAILED;
    }

    root = sph_dataset_peak_new_root( self->dataset );

    if ( !root ) {
        sph_error_raise( SPH_DATASET_STATS_GENERAL, __FILE__,
                         __func__, __LINE__, SPH_ERROR_ERROR,
                         "Could not create tree root for peak finding.");
        return SPH_DATASET_STATS_GENERAL;
    }
    self->rootpeak = root;

    if ( fthreshold < 0.0 ) { //interpret as fractional value
        root->frac_threshold = -1.0 * fthreshold;
        root->frac_acc = 0.001;

    }
    else { //interpret as absolute value
        root->frac_threshold = fthreshold / cpl_vector_get_max( self->dataset->yvalues );
        if ( root->frac_threshold >= 1.0 ) {
            sph_error_raise( SPH_DATASET_STATS_GENERAL,
                    __FILE__, __func__, __LINE__, SPH_ERROR_WARNING,
                    "The threshold %f was bigger than "
                    "the maximum value of the dataset: "
                    "%f. I definitely wont find any "
                    "peaks.", (float)fthreshold,
                    (float)cpl_vector_get_max( self->dataset->yvalues ) );
            root->frac_threshold = 1.0;
        }
        root->frac_acc = 0.001;
    }

    root->minpixels = minpix;

    rerr = sph_dataset_peak_find_subpeaks( root );

    if ( root->firstchild == NULL ) {
        self->peakloc = cpl_calloc( 1, sizeof(double) );
        self->npeaks = 1;
        self->peakloc[ 0 ] = root->meanpos;
    }
    else {
        apeak = root->firstchild;

        while ( apeak ) {
            self->npeaks++;
            apeak = apeak->next;
        }
        self->peakloc = cpl_calloc( self->npeaks, sizeof(double) );
        apeak = root->firstchild;
        count = 0;
        while ( apeak ) {
            self->peakloc[count] = apeak->meanpos;
            apeak = apeak->next;
            count++;
        }
    }
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        sph_error_raise( SPH_ERROR_PREVIOUS_OP_FAILED,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Entering ${enclosing_method} with a CPL error "
                "is a bad idea. The CPL error was: %s.",
                cpl_error_get_message());
        return SPH_ERROR_PREVIOUS_OP_FAILED;
    }
    sph_dataset_peak_delete(root); root = NULL;
    return CPL_ERROR_NONE;

}
int sph_dataset_stats_find_peaks( sph_dataset_stats* self,
                                  double fthreshold, int minpix )
{
    sph_error_code      rerr        = CPL_ERROR_NONE;
    int                 method      = 1;


    if ( method == 0 ) {
        rerr = sph_dataset_stats_find_peaks_simple_method( self, fthreshold );
    }
    else if ( method == 1 ) {
        rerr = sph_dataset_stats_find_peaks_tree_method( self, fthreshold, minpix );
    }
    return rerr;
}
int sph_dataset_stats_find_peaks_exact_number( sph_dataset_stats* self,
                                  int npeaks, double acc )
{
    double          fmin        = 0.0;
    double          fmax        = 1.0;
    double          goodness    = 0.0;
    double          prevgoodness    = 0.0;
    double          bestlevel   = 0.0;

    if ( !self ) {
        SPH_NULL_ERROR
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !self->dataset ) {
        SPH_NULL_ERROR
        return CPL_ERROR_NULL_INPUT;
    }

    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        sph_error_raise( SPH_ERROR_PREVIOUS_OP_FAILED,
                         __FILE__,
                         __func__,
                         __LINE__,
                         SPH_ERROR_WARNING,
                         "CPL error was set! CPL said:%s, func: %s, line: %d",
                         cpl_error_get_message(),
                         cpl_error_get_function(),
                         cpl_error_get_line() );
        cpl_error_reset();
    }

    while ( fmin < fmax ) {
        fmin += acc;
        sph_dataset_stats_find_peaks_simple_method( self, -fmin );
        if ( self->npeaks == npeaks ) {
            goodness = self->peakproduct;
            if ( goodness > prevgoodness ) {
                prevgoodness    = goodness;
                bestlevel       = fmin;
            }
        }
    }

    sph_dataset_stats_find_peaks_simple_method( self, bestlevel );

    if ( self->npeaks != npeaks ) {
        sph_error_raise( SPH_ERROR_GENERAL,
                         __FILE__,
                         __func__,
                         __LINE__,
                         SPH_ERROR_WARNING,
                         "Could not find the requested number "
                         "of peaks. Found %d instead of %d.",
                         self->npeaks, npeaks);
    }
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return the peaks as a cpl_vector
 *
 * @param self the dataset_stat structure with defined peaks
 *
 * Returns the peaks as obtained from the sph_dataset_stats_find_peaks function
 * as a cpl_vector. More precisely, a cpl_vector is created which contrains the
 * x-positions of the peaks as found using the sph_dataset_stats_find_peaks
 * function.
 *
 * @return the peaks positions as a cpl_vector
 *
 *
 *
 */
/*----------------------------------------------------------------------------*/
cpl_vector*
sph_dataset_stats_get_peaks_as_cpl_vector( sph_dataset_stats* self ) {
    cpl_vector*         result      = NULL;
    int                 ii          = 0;

    if ( !self ) {
        SPH_NULL_ERROR;
        return NULL;
    }
    if ( !self->peakloc ) {
        SPH_NULL_ERROR;
        return NULL;
    }

    result = cpl_vector_new( self->npeaks );

    for (ii = 0; ii < self->npeaks; ++ii) {
        cpl_vector_set( result, ii, self->peakloc[ii] );
    }
    cpl_vector_sort( result, 1 );
    return result;
}
int
sph_dataset_stats_delete( sph_dataset_stats* self ) {

    if ( self ) {
        if ( self->peakloc ) {
            cpl_free( self->peakloc );
        }
        cpl_free(self);
    }
    return CPL_ERROR_NONE;
}
