/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include "sph_distortion_map.h"

#include "sph_error.h"
#include "sph_point.h"
#include "sph_point_pattern.h"

#include <math.h>
#include <assert.h>

/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/

/* X/Y-Observed + X/Y-Delta-Add = X/Y-Calibration + X/Y-Residual */

#define SPH_DISTORITION_TABLE_XOBS "X_observed"
#define SPH_DISTORITION_TABLE_YOBS "Y_observed"
#define SPH_DISTORITION_TABLE_XCAL "X_Calibration"
#define SPH_DISTORITION_TABLE_YCAL "Y_Calibration"
#define SPH_DISTORITION_TABLE_XADD "X_Delta_Add"
#define SPH_DISTORITION_TABLE_YADD "Y_Delta_Add"
#define SPH_DISTORITION_TABLE_XRES "X_Residual"
#define SPH_DISTORITION_TABLE_YRES "Y_Residual"

#define SPH_DISTORITION_MAP_X       "X"
#define SPH_DISTORITION_MAP_Y       "Y"
#define SPH_DISTORITION_MAP_DX      "DX"
#define SPH_DISTORITION_MAP_DY      "DY"
#define SPH_DISTORITION_MAP_KEY     "KEY"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides ...
 */
/*----------------------------------------------------------------------------*/
/**@{*/

static cpl_table* sph_distortion_map_new_(cpl_size)
     CPL_ATTR_ALLOC;

static cpl_error_code sph_distortion_map_append(sph_distortion_map*,
                                                double, double,
                                                double, double, cpl_size)
    CPL_ATTR_NONNULL;

static void sph_distortion_map_reject(sph_distortion_map*, cpl_size)
    CPL_ATTR_NONNULL;

static void sph_distortion_map_set_(sph_distortion_map*, cpl_size,
                                    const sph_distortion_vector*)
    CPL_ATTR_NONNULL;

static void sph_distortion_map_get_(const sph_distortion_map*, cpl_size,
                                    sph_distortion_vector*)
    CPL_ATTR_NONNULL;

static void sph_distortion_map_shift_valid(sph_distortion_map*, cpl_size)
    CPL_ATTR_NONNULL;

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a map of the distortion from pattern A to B
 @param    pA  The observed pattern
 @param    pB  The reference pattern
 @return   The created object or NULL on error
 @note The observed pattern must be a subset of the reference one
 */
/*----------------------------------------------------------------------------*/
sph_distortion_map* sph_distortion_map_new(const sph_point_pattern* pA,
                                           const sph_point_pattern* pB) {
    const cpl_size pAsz = sph_point_pattern_get_size(pA);
    const cpl_size pBsz = sph_point_pattern_get_size(pB);
    sph_distortion_map* self = cpl_calloc(1, sizeof(*self));

    if (pBsz > 0 && pAsz > 0) {
        unsigned* pBc = (unsigned*)cpl_calloc(pBsz, sizeof(*pBc));
        const double*  pA0 = sph_point_pattern_get(pA, 0);
        const cpl_size j   = sph_point_pattern_find_closest(pB, pA0);
        const double*  pBj = sph_point_pattern_get(pB, j);
        cpl_size i;


        self->distvectlist = sph_distortion_map_new_(pAsz);

        pBc[j]++;
        sph_distortion_map_append(self, pBj[0], pBj[1],
                                  pBj[0] - pA0[0],
                                  pBj[1] - pA0[1], j);

        for (i = 1; i < pAsz; i++) {
            const double* pAi = sph_point_pattern_get(pA, i);
            const cpl_size k  = sph_point_pattern_find_closest(pB, pAi);
            const double* pBk = sph_point_pattern_get(pB, k);

            pBc[k]++;

            assert(self->sz == i);
            if (sph_distortion_map_append(self, pBk[0], pBk[1],
                                          pBk[0] - pAi[0],
                                          pBk[1] - pAi[1], k))
                break;
        }

        if (i < pAsz) {
            sph_distortion_map_delete(self);
            self = NULL;
            (void)cpl_error_set_where(cpl_func);
        } else {
            cpl_size k = 0;
            cpl_size nnomap = 0;
            double minsqdist = DBL_MAX;
            cpl_size minind = pBsz;

            /* Each observed point has now been mapped to its closest reference.
               Points that have been mapped to the same reference point are all
               discarded. */

            for (i = 0; i < pBsz; i++) {
                if (pBc[i] == 1) {
                    k++;
                } else {
                    const double* pBk = sph_point_pattern_get(pB, i);
                    const double sqdist = pBk[0] * pBk[0] + pBk[1] * pBk[1];

                    if (pBc[i] == 0) {
                        nnomap++;
                        cpl_msg_info(cpl_func, "Point %d/%d was not mapped: "
                                     "(%g, %g), distance: %g (%d)", (int)i,
                                     (int)pBsz, pBk[0], pBk[1], sqrt(sqdist),
                                     (int)nnomap);
                        if (sqdist < minsqdist) {
                            minsqdist = sqdist;
                            minind = i;
                        }
                    } else {
                        k += pBc[i];
                    }
                }
            }
            cpl_msg_info(cpl_func, "Of %d detected point(s) %d not mapped to "
                         "%d. Closest non-mapped (%d) to center [pixel]: %g",
                         (int)pBsz, (int)nnomap, (int)pAsz, (int)minind,
                         sqrt(minsqdist));
            assert(k == pAsz);
        }
        cpl_free(pBc);
    }

    return self;
}


/*----------------------------------------------------------------------------*/
/**
   @brief    Append the specified distortion point
   @param    self   The const distortion map to access
   @param    x      The X-coordinate
   @param    y      The Y-coordinate
   @param    dx     The X-offset
   @param    dy     The Y-offset
   @return   CPL_ERROR_NONE on success, or else the relevant CPL error code
*/
/*----------------------------------------------------------------------------*/
cpl_error_code sph_distortion_map_append_one(sph_distortion_map* self,
                                             double x, double y,
                                             double dx, double dy)
{

    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    if (self->distvectlist == NULL)
        self->distvectlist = sph_distortion_map_new_(1);

    return sph_distortion_map_append(self, x, y, dx, dy, -1)
        ? cpl_error_set_where(cpl_func)
        : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
   @brief    Get direct access to all X-coordinates
   @param    self   The const distortion map to access
   @return   A pointer to the data or NULL on error
*/
/*----------------------------------------------------------------------------*/
const double * sph_distortion_map_get_data_x(const sph_distortion_map* self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    return cpl_table_get_data_double_const(self->distvectlist,
                                           SPH_DISTORITION_MAP_X);
}

/*----------------------------------------------------------------------------*/
/**
   @brief    Get direct access to all Y-coordinates
   @param    self   The const distortion map to access
   @return   A pointer to the data or NULL on error
*/
/*----------------------------------------------------------------------------*/
const double * sph_distortion_map_get_data_y(const sph_distortion_map* self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    return cpl_table_get_data_double_const(self->distvectlist,
                                           SPH_DISTORITION_MAP_Y);
}

/*----------------------------------------------------------------------------*/
/**
   @brief    Get direct access to all X-offsets
   @param    self   The const distortion map to access
   @return   A pointer to the data or NULL on error
*/
/*----------------------------------------------------------------------------*/
const double * sph_distortion_map_get_data_dx(const sph_distortion_map* self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    return cpl_table_get_data_double_const(self->distvectlist,
                                           SPH_DISTORITION_MAP_DX);
}

/*----------------------------------------------------------------------------*/
/**
   @brief    Get direct access to all Y-offsets
   @param    self   The const distortion map to access
   @return   A pointer to the data or NULL on error
*/
/*----------------------------------------------------------------------------*/
const double * sph_distortion_map_get_data_dy(const sph_distortion_map* self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    return cpl_table_get_data_double_const(self->distvectlist,
                                           SPH_DISTORITION_MAP_DY);
}

/*----------------------------------------------------------------------------*/
/**
   @brief    Get the specified distortion point
   @param    self   The const distortion map to access
   @param    n      The element to get (0 is first element)
   @param    pv     On success, filled with the distortion point
   @return   CPL_ERROR_NONE on success, or else the relevant CPL error code
*/
/*----------------------------------------------------------------------------*/
cpl_error_code sph_distortion_map_get_one(const sph_distortion_map* self,
                                          cpl_size n,
                                          sph_distortion_vector * pv)
{
    cpl_ensure_code(self != NULL,     CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(n    >= 0,        CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(n    <  self->sz, CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(pv   != NULL,     CPL_ERROR_NULL_INPUT);

    sph_distortion_map_get_(self, n, pv);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Remove any element with a delta exceeding the given threshold
 * @param self      The distorion map to process
 * @param threshold The maximum allowed delta
 * @param do_msg    Iff true, enable messaging
 * @return The number of removed elements, zero on NULL input or non-positive
           threshold
 */
/*----------------------------------------------------------------------------*/
int sph_distortion_map_threshold(sph_distortion_map* self,
                                 double threshold,
                                 cpl_boolean do_msg)
{
    int cc = 0;

    if (self != NULL || threshold <= 0.0) {
        const cpl_size n     = sph_distortion_map_get_size(self);
        const double   sqlim = threshold * threshold;

        for (cpl_size i = 0; i < n; i++) {
            sph_distortion_vector v;
            const cpl_error_code code = 
                sph_distortion_map_get_one(self, i, &v);
            const double sqdist = v.dx * v.dx + v.dy * v.dy;

            assert(!code);

            if (sqdist > sqlim) {
                cc++;
                sph_distortion_map_reject(self, i);
                if (do_msg)
                    cpl_msg_warning(cpl_func, "Distortion %d/%d discarded: "
                                    "(%g,%g) - (%g,%g) > %g (%d)",
                                    1+(int)i, (int)n, v.x, v.y, v.dx, v.dy,
                                    threshold, cc);
            }
        }

        if (cc > 0) {
            sph_distortion_map_shift_valid(self, cc);
            assert(self->sz + cc == n);
        }
    }
    return cc;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Remove any element registered more than once
 * @param self      The distorion map to process
 * @return The number of removed elements, zero on NULL input
 */
/*----------------------------------------------------------------------------*/
int sph_distortion_map_delete_multi(sph_distortion_map* self)
{
    int k = 0;

    if (self != NULL) {
        const int n = sph_distortion_map_get_size(self);

        for (int i = 0; i < n; i++) {
            if (!cpl_table_is_valid(self->distvectlist,
                                    SPH_DISTORITION_MAP_X, i)) {

                sph_distortion_vector v;
                const cpl_error_code code = 
                    sph_distortion_map_get_one(self, i, &v);
                const double x = v.x;
                const double y = v.y;
                cpl_boolean  do_rm = CPL_FALSE;

                assert(!code);

                for (int j = i+1; j < n; j++) {
                    sph_distortion_vector temp;

                    sph_distortion_map_get_one(self, j, &temp);

                    if (temp.x != x || temp.y != y) continue;

                    sph_distortion_map_reject(self, j);

                    do_rm = CPL_TRUE;
                    k++;
                }

                if (do_rm) {
                    sph_distortion_map_reject(self, i);
                    k++;
                }
            }
        }
        cpl_msg_warning(cpl_func, "From %d-distortion map rejected %d multi-"
                        "mapped vector(s): %d", n, (int)k, (int)self->sz);
        if (k > 0) {
            sph_distortion_map_shift_valid(self, k);
            assert(self->sz + k == n);
        }
    }
    return k;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Compute and remove the mean position from the map
 * @param self      The distorion map to process
 * @param poffx     The determined and removed X-offset, or NULL
 * @param poffy     The determined and removed Y-offset, or NULL
 * @return The number of modified elements, zero on NULL input
 */
/*----------------------------------------------------------------------------*/
int sph_distortion_map_remove_offset(sph_distortion_map* self,
                                     double* poffx,
                                     double* poffy)
{
    double offx = 0.0, offy = 0.0;
    int n = 0;

    if (self != NULL) {

        n = cpl_table_count_invalid(self->distvectlist,
                                    SPH_DISTORITION_MAP_X);

        if (n > 0) {

            offx = cpl_table_get_column_mean(self->distvectlist,
                                             SPH_DISTORITION_MAP_X);

            cpl_table_subtract_scalar(self->distvectlist,
                                      SPH_DISTORITION_MAP_X, offx);

            offy = cpl_table_get_column_mean(self->distvectlist,
                                             SPH_DISTORITION_MAP_Y);

            cpl_table_subtract_scalar(self->distvectlist,
                                      SPH_DISTORITION_MAP_Y, offy);
        }
    }

    if (poffx != NULL) {
        *poffx = offx;
    }
    if (poffy != NULL) {
        *poffy = offy;
    }

    return n;
}

int sph_distortion_map_get_size(const sph_distortion_map* self)
{
    return self != NULL ? (int)self->sz : 0;
}

void sph_distortion_map_delete(sph_distortion_map* self) {

    if (self != NULL) {
        cpl_table_delete(self->distvectlist);
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Move non-invalid rows to the front of the list, decrease counter
 * @param self      The distorion map to process
 * @param nrm       The number of invalid rows to move
 * @return void
 * @note The number of invalid rows nrm is also present from within the table

   Shifts all non-invalid points to the front of the table, ignoring rows
  (invalid or not) left unused at the end of the table

 */
/*----------------------------------------------------------------------------*/
static void sph_distortion_map_shift_valid(sph_distortion_map* self,
                                           cpl_size nrm)
{
    if (self->sz == nrm) {
        self->sz = 0;
    } else if (nrm > 0) {
        const cpl_size n = self->sz;

        self->sz = 0;

        for (cpl_size i = 0; i < n; i++) {
            if (cpl_table_is_valid(self->distvectlist, SPH_DISTORITION_MAP_X,
                                   i)) {
                if (i > self->sz) {
                    sph_distortion_vector vv;

                    sph_distortion_map_get_(self, i, &vv);
                    sph_distortion_map_set_(self, self->sz, &vv);
                }
                self->sz++;
            }
        }
        assert(self->sz + nrm == n);
    }
}

/**@}*/

static cpl_table* sph_distortion_map_new_(cpl_size size)
{
    cpl_table* self = cpl_table_new(size);

    assert(self != NULL);

    cpl_table_new_column(self, SPH_DISTORITION_MAP_X,      CPL_TYPE_DOUBLE);
    cpl_table_new_column(self, SPH_DISTORITION_MAP_Y,      CPL_TYPE_DOUBLE);
    cpl_table_new_column(self, SPH_DISTORITION_MAP_DX,     CPL_TYPE_DOUBLE);
    cpl_table_new_column(self, SPH_DISTORITION_MAP_DY,     CPL_TYPE_DOUBLE);
    cpl_table_new_column(self, SPH_DISTORITION_MAP_KEY,    CPL_TYPE_INT);

    cpl_table_set_column_unit(self, SPH_DISTORITION_MAP_X,   "pixel");
    cpl_table_set_column_unit(self, SPH_DISTORITION_MAP_Y,   "pixel");
    cpl_table_set_column_unit(self, SPH_DISTORITION_MAP_DX,  "pixel");
    cpl_table_set_column_unit(self, SPH_DISTORITION_MAP_DY,  "pixel");
    cpl_table_set_column_unit(self, SPH_DISTORITION_MAP_KEY, "unitless");

    return self;
}

/*----------------------------------------------------------------------------*/
/**
   @brief    Append the specified distortion point
   @param    self   The const distortion map to access
   @param    x      The X-coordinate
   @param    y      The Y-coordinate
   @param    dx     The X-offset
   @param    dy     The Y-offset
   @return   CPL_ERROR_NONE on success, or else the relevant CPL error code
   @see sph_distortion_map_append_one()
*/
/*----------------------------------------------------------------------------*/
static cpl_error_code sph_distortion_map_append(sph_distortion_map* self,
                                                double x, double y,
                                                double dx, double dy,
                                                cpl_size key)
{
    const sph_distortion_vector vv = {x, y, dx, dy, key};

    if (self->sz == cpl_table_get_nrow(self->distvectlist)) {
        if (cpl_table_set_size(self->distvectlist, 1 + 2 * self->sz))
            return cpl_error_set_where(cpl_func);
    }

    sph_distortion_map_set_(self, self->sz, &vv);

    self->sz++;

    return CPL_ERROR_NONE;
}

static void sph_distortion_map_reject(sph_distortion_map* self, cpl_size i)
{
    cpl_table_set_invalid(self->distvectlist, SPH_DISTORITION_MAP_X,   i);
    cpl_table_set_invalid(self->distvectlist, SPH_DISTORITION_MAP_Y,   i);
    cpl_table_set_invalid(self->distvectlist, SPH_DISTORITION_MAP_DX,  i);
    cpl_table_set_invalid(self->distvectlist, SPH_DISTORITION_MAP_DY,  i);
    cpl_table_set_invalid(self->distvectlist, SPH_DISTORITION_MAP_KEY, i);
}

/*----------------------------------------------------------------------------*/
/**
   @internal
   @brief    Get the specified distortion point
   @param    self   The const distortion map to access
   @param    n      The element to get (0 is first element)
   @param    pv     On success, filled with the distortion point
   @return   CPL_ERROR_NONE on success, or else the relevant CPL error code
*/
/*----------------------------------------------------------------------------*/
static void sph_distortion_map_get_(const sph_distortion_map* self,
                                    cpl_size n,
                                    sph_distortion_vector * pv)
{
    pv->x   = cpl_table_get_double(self->distvectlist, SPH_DISTORITION_MAP_X,
                                   n, NULL);
    pv->y   = cpl_table_get_double(self->distvectlist, SPH_DISTORITION_MAP_Y,
                                   n, NULL);
    pv->dx  = cpl_table_get_double(self->distvectlist, SPH_DISTORITION_MAP_DX,
                                   n, NULL);
    pv->dy  = cpl_table_get_double(self->distvectlist, SPH_DISTORITION_MAP_DY,
                                   n, NULL);
    pv->key = cpl_table_get_int(self->distvectlist, SPH_DISTORITION_MAP_KEY,
                                n, NULL);
}

/*----------------------------------------------------------------------------*/
/**
   @brief    Set the specified distortion point
   @param    self   The const distortion map to access
   @param    n      The element to set (0 is first element)
   @param    pv     The distortion point to set from
   @return   void
   @see sph_distortion_map_append()
*/
/*----------------------------------------------------------------------------*/
static void sph_distortion_map_set_(sph_distortion_map* self,
                                    cpl_size n,
                                    const sph_distortion_vector * pv)
{
    cpl_table_set_double(self->distvectlist, SPH_DISTORITION_MAP_X,
                         n, pv->x);
    cpl_table_set_double(self->distvectlist, SPH_DISTORITION_MAP_Y,
                         n, pv->y);
    cpl_table_set_double(self->distvectlist, SPH_DISTORITION_MAP_DX,
                         n, pv->dx);
    cpl_table_set_double(self->distvectlist, SPH_DISTORITION_MAP_DY,
                         n, pv->dy);
    cpl_table_set_int(self->distvectlist, SPH_DISTORITION_MAP_KEY,
                      n, pv->key);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Add a point pattern to a distortion map base, shifting its offset
 * @param self  The point pattern to modify
 * @param pnew  The point pattern to add
 * @return Nothing
 *
 */
/*----------------------------------------------------------------------------*/
void sph_distortion_map_add(sph_distortion_map* self,
                            const sph_point_pattern* pnew)
{
    const int n = sph_point_pattern_get_size(pnew);

    for (cpl_size i = 0; i < self->sz; i++) {
        const double* xyi;
        sph_distortion_vector vv;
        double dx, dy;

        sph_distortion_map_get_(self, i, &vv);

        assert(vv.key >= 0);
        assert(vv.key < n);

        xyi = sph_point_pattern_get(pnew, vv.key);

        dx = vv.x - xyi[0];
        dy = vv.y - xyi[1];

        if ((vv.dx - dx) * (vv.dx - dx) + (vv.dy - dy) * (vv.dy - dy) > 49.0)
            cpl_msg_warning(cpl_func, "Ref-point undistorted correction(%d) "
                            "(%g,%g): (%g,%g) => (%g,%g)",
                            (int)vv.key, xyi[0], xyi[1], vv.dx, vv.dy,
                            vv.dx - dx, vv.dy - dy);

        vv.dx -= dx;
        vv.dy -= dy;

        vv.x  -= dx; /* Set to original X-reference, xyi[0] */
        vv.y  -= dy; /* Set to original Y-reference, xyi[1] */

        sph_distortion_map_set_(self, i, &vv);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a CPL table with the distortion map
 * @param self   The distortion map
 * @param dist   The distortion model
 * @param ref_pp The reference point pattern
 * @param xcen   The X-reference point, equal to 0 in the reference pattern
 * @param ycen   The Y-reference point, equal to 0 in the reference pattern
 * @return On success a created CPL table with the distortion
 *
 */
/*----------------------------------------------------------------------------*/
cpl_table *
sph_distortion_map_table_create(const sph_distortion_map              * self,
                                const sph_distortion_model * dist,
                                const sph_point_pattern               * ref_pp,
                                double xcen,
                                double ycen)
{
    const cpl_size      n        = sph_distortion_map_get_size(self);
    const cpl_size      np       = sph_point_pattern_get_size(ref_pp);
    cpl_table         * result   = cpl_table_new(n);
    sph_point_pattern * corr_pp = NULL;


    cpl_table_new_column(result, SPH_DISTORITION_TABLE_XOBS, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, SPH_DISTORITION_TABLE_YOBS, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, SPH_DISTORITION_TABLE_XCAL, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, SPH_DISTORITION_TABLE_YCAL, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, SPH_DISTORITION_TABLE_XADD, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, SPH_DISTORITION_TABLE_YADD, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, SPH_DISTORITION_TABLE_XRES, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, SPH_DISTORITION_TABLE_YRES, CPL_TYPE_DOUBLE);

    cpl_table_set_column_unit(result, SPH_DISTORITION_TABLE_XOBS, "pixel");
    cpl_table_set_column_unit(result, SPH_DISTORITION_TABLE_YOBS, "pixel");
    cpl_table_set_column_unit(result, SPH_DISTORITION_TABLE_XCAL, "pixel");
    cpl_table_set_column_unit(result, SPH_DISTORITION_TABLE_YCAL, "pixel");
    cpl_table_set_column_unit(result, SPH_DISTORITION_TABLE_XADD, "pixel");
    cpl_table_set_column_unit(result, SPH_DISTORITION_TABLE_YADD, "pixel");
    cpl_table_set_column_unit(result, SPH_DISTORITION_TABLE_XRES, "pixel");
    cpl_table_set_column_unit(result, SPH_DISTORITION_TABLE_YRES, "pixel");

    corr_pp = sph_distortion_model_apply_pp(dist, ref_pp);

    for (cpl_size i = 0; i < n; i++) {
        sph_distortion_vector vv;
        const double* ref_xyi;
        const double* corr_xyi;
        double xobs, yobs, dx, dy, xres, yres;

        sph_distortion_map_get_(self, i, &vv);

        assert(vv.key >= 0);
        assert(vv.key < np);

        ref_xyi  = sph_point_pattern_get(ref_pp,  vv.key);
        corr_xyi = sph_point_pattern_get(corr_pp, vv.key);

        xobs = vv.x - vv.dx;
        yobs = vv.y - vv.dy;

        dx = ref_xyi[0] - corr_xyi[0];
        dy = ref_xyi[1] - corr_xyi[1];

        xres = xobs - (ref_xyi[0] - dx);
        yres = yobs - (ref_xyi[1] - dy);

        if (vv.x != ref_xyi[0] || vv.y != ref_xyi[1]) {
            cpl_msg_warning(cpl_func, "TABLE(%d/%d): %g %g (%d)", 1+(int)i,
                            (int)n,
                            vv.x - ref_xyi[0], vv.y - ref_xyi[1], (int)vv.key);
        }

        cpl_table_set_double(result, SPH_DISTORITION_TABLE_XOBS, i, xobs + xcen);
        cpl_table_set_double(result, SPH_DISTORITION_TABLE_YOBS, i, yobs + ycen);
        cpl_table_set_double(result, SPH_DISTORITION_TABLE_XCAL, i, ref_xyi[0] + xcen);
        cpl_table_set_double(result, SPH_DISTORITION_TABLE_YCAL, i, ref_xyi[1] + ycen);
        cpl_table_set_double(result, SPH_DISTORITION_TABLE_XADD, i, dx);
        cpl_table_set_double(result, SPH_DISTORITION_TABLE_YADD, i, dy);
        cpl_table_set_double(result, SPH_DISTORITION_TABLE_XRES, i, xres);
        cpl_table_set_double(result, SPH_DISTORITION_TABLE_YRES, i, yres);

    }
    sph_point_pattern_delete(corr_pp);

    return result;
}
