/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include "sph_frame_validator.h"
#include "sph_keyword_manager.h"
#include "sph_cube.h"
#include "sph_fits.h"
#include <cpl.h>
#include <math.h>
#include <strings.h>


static cpl_type        static_current_type    = CPL_TYPE_DOUBLE;

/*----------------------------------------------------------------------------*/
/**
 * @brief Validate that two frames are identical
 * @param is        the frame
 * @param shouldbe    the frame to compare with
 *
 * @return SPH_FRAME_VALIDATOR_VALIDATED if true (identical),
 * SPH_FRAME_VALIDATOR_NOT_VALIDATED if not, SPH_FRAME_VALIDATOR_ERROR on error.
 *
 * The two frames are compared with and matched exactly on these points:
 * <ul>
 * <li>Frame tag</li>
 * <li>Frame type</li>
 * <li>Number extensions</li>
 * <li>Extension types( plane / table )</li>
 * <li>Number planes</li>
 * <li>Image size</li>
 * <li>Pixel values</li>
 * </ul>
 */
/*----------------------------------------------------------------------------*/
sph_frame_validator_result
sph_frame_validator_validate_identical( const cpl_frame* is,
                                        const cpl_frame* shouldbe ) {
    sph_frame_validator_result result = SPH_FRAME_VALIDATOR_VALIDATED;
    int            next        = 0;

    if ( is == NULL || shouldbe == NULL ) {
        return SPH_FRAME_VALIDATOR_ERROR;
    }
    next = cpl_frame_get_nextensions( shouldbe ) ;

    if ( cpl_frame_get_nextensions( is ) != next )
        return SPH_FRAME_VALIDATOR_NOT_VALIDATED;

    if ( strcasecmp( cpl_frame_get_tag( is ),
                     cpl_frame_get_tag( shouldbe ) ) != 0 )
        return SPH_FRAME_VALIDATOR_NOT_VALIDATED;

    if ( cpl_frame_get_type( is ) != cpl_frame_get_type(shouldbe) )
        return SPH_FRAME_VALIDATOR_NOT_VALIDATED;

    for (int cext = 0; cext < next; ++cext) {
        const int nplanes =
            sph_fits_get_nplanes( cpl_frame_get_filename(shouldbe), cext );

        if ( sph_fits_get_nplanes( cpl_frame_get_filename(is), cext ) != nplanes )
            return SPH_FRAME_VALIDATOR_NOT_VALIDATED;

        for ( int cplane = 0;  cplane < nplanes; ++cplane) {
            
            cpl_image* should_be_image = NULL;
            cpl_image* is_image        = NULL;

            cpl_error_reset(); /* FIXME: remove */

            should_be_image = cpl_image_load( cpl_frame_get_filename(shouldbe),
                                              static_current_type, cplane, cext  );
            if ( !should_be_image ) return SPH_FRAME_VALIDATOR_ERROR;

            is_image = cpl_image_load( cpl_frame_get_filename(is),
                                       static_current_type, cplane, cext  );
            if ( !is_image ) {
                cpl_image_delete( should_be_image ); should_be_image = NULL;
                return SPH_FRAME_VALIDATOR_ERROR;
            }

            if ( cpl_image_get_size_x( is_image ) !=
                 cpl_image_get_size_x( should_be_image ) ) {
                result = SPH_FRAME_VALIDATOR_NOT_VALIDATED;
            } else if ( cpl_image_get_size_y( is_image ) !=
                        cpl_image_get_size_y( should_be_image ) ) {
                result = SPH_FRAME_VALIDATOR_NOT_VALIDATED;
            } else {
                cpl_image_subtract( is_image, should_be_image );
                if ( cpl_image_get_absflux( is_image ) != 0 ) {
                    result = SPH_FRAME_VALIDATOR_NOT_VALIDATED;
                }
            }
            cpl_image_delete( is_image ); is_image = NULL;
            cpl_image_delete( should_be_image ); should_be_image = NULL;
        }
    }
    return result;
}
#ifdef SPH_FRME_VALIDATOR_UNUSED
/*----------------------------------------------------------------------------*/
/**
 * @brief Validate that two frames are identical
 * @param is        the frame
 * @param shouldbe    the frame to compare with
 *
 * @return SPH_FRAME_VALIDATOR_VALIDATED if true (identical),
 * SPH_FRAME_VALIDATOR_NOT_VALIDATED if not, SPH_FRAME_VALIDATOR_ERROR on error.
 *
 * The two frames are compared with and matched exactly on these points:
 * <ul>
 * <li>Frame tag</li>
 * <li>Frame type</li>
 * <li>Number extensions</li>
 * <li>Extension types( plane / table )</li>
 * <li>Number planes</li>
 * <li>Image size</li>
 * <li>Pixel values</li>
 * </ul>
 */
/*----------------------------------------------------------------------------*/
sph_frame_validator_result
sph_frame_validator_validate_identical_image_ignore_bads( cpl_frame* is, cpl_frame* shouldbe ) {
    sph_frame_validator_result result = SPH_FRAME_VALIDATOR_VALIDATED;
    int            next        = 0;
    int            nplanes        = 0;
    int            cext        = 0;
    int            cplane        = 0;

    if ( is == NULL || shouldbe == NULL ) {
        return SPH_FRAME_VALIDATOR_ERROR;
    }
    next = cpl_frame_get_nextensions( shouldbe ) ;

    if ( cpl_frame_get_nextensions( is ) != next  ) return SPH_FRAME_VALIDATOR_NOT_VALIDATED;
    if ( strcasecmp( cpl_frame_get_tag( is ), cpl_frame_get_tag( shouldbe ) ) != 0 ) return SPH_FRAME_VALIDATOR_NOT_VALIDATED;
    if ( cpl_frame_get_type( is ) != cpl_frame_get_type(shouldbe) ) return SPH_FRAME_VALIDATOR_NOT_VALIDATED;

    for (cext = 0; cext < next; ++cext) {
        nplanes = sph_fits_get_nplanes( cpl_frame_get_filename(shouldbe), cext );
        if ( sph_fits_get_nplanes( cpl_frame_get_filename(is), cext ) != nplanes ) return SPH_FRAME_VALIDATOR_NOT_VALIDATED;
    }
    cext = SPH_CUBE_IMAGE_EXT;
    nplanes = sph_fits_get_nplanes( cpl_frame_get_filename(shouldbe), cext );
    for ( cplane = 0;  cplane < nplanes; ++cplane) {
        if ( cpl_table_load( cpl_frame_get_filename(shouldbe), cext, 0 ) != NULL ) {
            if ( cpl_table_load( cpl_frame_get_filename(shouldbe), cext, 0 ) != NULL ) return SPH_FRAME_VALIDATOR_NOT_VALIDATED;
        }
        else {
            cpl_image*    should_be_image    = NULL;
            cpl_image*    is_image    = NULL;
            cpl_image*    bpixim        = NULL;
            cpl_mask*    bpixmask    = NULL;
            double         mean        = 0.0;
            cpl_error_reset();

            should_be_image = cpl_image_load(cpl_frame_get_filename(shouldbe),
                                             static_current_type, cplane, cext);
            if ( !should_be_image ) return SPH_FRAME_VALIDATOR_ERROR;
            is_image = cpl_image_load( cpl_frame_get_filename(is),
                                       static_current_type, cplane, cext );
            if ( !is_image ) {
                cpl_image_delete(should_be_image);
                return SPH_FRAME_VALIDATOR_ERROR;
            }
            if ( cpl_image_get_size_x( is_image ) != cpl_image_get_size_x( should_be_image ) ) {
                cpl_image_delete( is_image ); is_image = NULL;
                cpl_image_delete( should_be_image ); should_be_image = NULL;
                return SPH_FRAME_VALIDATOR_NOT_VALIDATED;
            }
            if ( cpl_image_get_size_y( is_image ) != cpl_image_get_size_y( should_be_image ) ) {
                cpl_image_delete( is_image ); is_image = NULL;
                cpl_image_delete( should_be_image ); should_be_image = NULL;
                return SPH_FRAME_VALIDATOR_NOT_VALIDATED;
            }
            bpixim = cpl_image_load( cpl_frame_get_filename(shouldbe),
                                     static_current_type, cplane,
                                     SPH_CUBE_BADPIX_EXT );
            bpixmask = cpl_mask_threshold_image_create( bpixim, 0.5, 10000000.5 );
            cpl_image_reject_from_mask( is_image, bpixmask );
            cpl_image_reject_from_mask( should_be_image, bpixmask );
            mean = cpl_image_get_mean( is_image );
            mean = mean / cpl_image_get_mean( should_be_image );
            cpl_image_multiply_scalar( should_be_image, mean );
            cpl_image_subtract( is_image, should_be_image );
            cpl_image_save( is_image, "tttt.fits", CPL_TYPE_DOUBLE, NULL,
                            CPL_IO_DEFAULT );
            mean = cpl_image_get_mean( is_image );
            if ( fabs(mean) > 1.0e-14 ) {
                result = SPH_FRAME_VALIDATOR_NOT_VALIDATED;
            }
            mean = cpl_image_get_stdev( is_image );
            if ( fabs(mean) > 1.0e-6 ) {
                result = SPH_FRAME_VALIDATOR_NOT_VALIDATED;
            }
            cpl_image_delete( is_image ); is_image = NULL;
            cpl_image_delete( should_be_image ); should_be_image = NULL;
            cpl_image_delete( bpixim ); bpixim = NULL;
            cpl_mask_delete( bpixmask ); bpixmask = NULL;
        }
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Validate that two frames are identical
 * @param is        the frame
 * @param shouldbe    the frame to compare with
 *
 * @return SPH_FRAME_VALIDATOR_VALIDATED if true (identical),
 * SPH_FRAME_VALIDATOR_NOT_VALIDATED if not, SPH_FRAME_VALIDATOR_ERROR on error.
 *
 * The two frames are compared with and matched exactly on these points:
 * <ul>
 * <li>Frame tag</li>
 * <li>Frame type</li>
 * <li>Number extensions</li>
 * <li>Extension types( plane / table )</li>
 * <li>Number planes</li>
 * <li>Image size</li>
 * <li>Pixel values</li>
 * </ul>
 */
/*----------------------------------------------------------------------------*/
sph_frame_validator_result
sph_frame_validator_validate_identical_nearest_int( cpl_frame* is, cpl_frame* shouldbe ) {
    sph_frame_validator_result    result = SPH_FRAME_VALIDATOR_NOT_VALIDATED;
    static_current_type = CPL_TYPE_INT;
    result = sph_frame_validator_validate_identical( is, shouldbe );
    static_current_type = CPL_TYPE_DOUBLE;

    return result;
}

sph_frame_validator_result
sph_frame_validator_validate_identical_image_nearest_int_ignore_bads( cpl_frame* is, cpl_frame* shouldbe ) {
    sph_frame_validator_result    result = SPH_FRAME_VALIDATOR_NOT_VALIDATED;
    static_current_type = CPL_TYPE_INT;
    result = sph_frame_validator_validate_identical_image_ignore_bads(is, shouldbe );
    static_current_type = CPL_TYPE_DOUBLE;

    return result;
}
#endif
/*----------------------------------------------------------------------------*/
/**
 * @brief check that the values for a specific keyword for a whole frameset
 *        are all within a given tolerance
 *
 * @param inframes        the input frameset
 * @param ext             extension to look for keyword in
 * @param keyname         the keyword name
 * @param tolerance       the allowed tolerance
 * @param badoutlist      frameset to store frames that were out (maybe NULL)
 *
 * @return SPH_FRAME_VALIDATOR_VALIDATED if all frames are in bounds,
 * SPH_FRAME_VALIDATOR_NOT_VALIDATED if not, SPH_FRAME_VALIDATOR_ERROR on error.
 *
 * The frames keywords are loaded using the keyword manager (and are hence
 * converted into software representations) and all frames that have the
 * specified keyword are sorted into a temporary list. The median value of the
 * keyword is then stored and all frames are checked to be within tol of
 * this median. Those frames that are not are stored in the badoutlist if
 * this frameset is provided.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_frame_validator_result
sph_frame_validator_warn_keyword_tolerance( const cpl_frameset* inframes,
        int ext, const char* keyname, double tol, cpl_frameset* badoutlist )
{
    const cpl_frame*                   aframe = NULL;
    int                          ii      = 0;
    sph_frame_validator_result   result  = SPH_FRAME_VALIDATOR_VALIDATED;
    cpl_vector*                  vals = NULL;
    int                          cc = 0;
    cpl_propertylist*            pl = NULL;
    double                       val = 0.0;
    double                       median = 0.0;
    cpl_frameset*                frames_wkey = NULL;

    SPH_RAISE_CPL;
    cpl_ensure(cpl_error_get_code() == CPL_ERROR_NONE,cpl_error_get_code(),
            SPH_FRAME_VALIDATOR_ERROR);
    cpl_ensure(inframes,CPL_ERROR_NULL_INPUT,SPH_FRAME_VALIDATOR_ERROR);

    frames_wkey = cpl_frameset_new();
    vals = cpl_vector_new(1);
    for (ii = 0; ii < cpl_frameset_get_size(inframes); ++ii) {
        aframe = cpl_frameset_get_position_const(inframes,ii);
        pl = cpl_propertylist_load(cpl_frame_get_filename(aframe),ext);
        if ( pl ) {
            if ( cpl_propertylist_has(pl,keyname)) {
                val = cpl_propertylist_get_double(pl,keyname);
                cc  = cpl_vector_get_size(vals);
                cpl_vector_set(vals,cc-1,val);
                cpl_vector_set_size(vals,cc+1);
                cpl_frameset_insert(frames_wkey,cpl_frame_duplicate(aframe));
            }
            cpl_propertylist_delete(pl); pl = NULL;
        }
        else {
            result = SPH_FRAME_VALIDATOR_ERROR;
            break;
        }
    }

    if ( cpl_frameset_get_size(frames_wkey) < 1 ) {
        result = SPH_FRAME_VALIDATOR_ERROR;
    }
    if ( result != SPH_FRAME_VALIDATOR_ERROR ) {
        cpl_vector_set_size(vals,cpl_vector_get_size(vals)-1);
        median = cpl_vector_get_median(vals);
        for (ii = 0; ii < cpl_frameset_get_size(frames_wkey); ++ii) {
            aframe = cpl_frameset_get_position_const(frames_wkey,ii);
            pl = cpl_propertylist_load(cpl_frame_get_filename(aframe),ext);
            if ( pl ) {
                if ( cpl_propertylist_has(pl,keyname)) {
                    val = cpl_propertylist_get_double(pl,keyname);
                    if ( fabs(val - median) > tol ) {
                        if ( badoutlist ) {
                            cpl_frameset_insert(badoutlist,cpl_frame_duplicate(aframe));
                        }
                        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                                             "Val was: %f, median: %f",
                                             val, median);
                        result = SPH_FRAME_VALIDATOR_NOT_VALIDATED;
                    }
                }
                cpl_propertylist_delete(pl); pl = NULL;
            }
            else {
                result = SPH_FRAME_VALIDATOR_ERROR;
                break;
            }
        }
    }
    cpl_vector_delete(vals); vals = NULL;
    cpl_frameset_delete(frames_wkey); frames_wkey = NULL;
    SPH_RAISE_CPL;
    cpl_ensure(cpl_error_get_code() == CPL_ERROR_NONE,cpl_error_get_code(),
            SPH_FRAME_VALIDATOR_ERROR);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Validate that a given frame contains the properties in list
 *             possibly also testing if they have identical values
 * @param is        the frame
 * @param ext        the extension in the frame to use for propertylist extraction
 * @param plist        the propertylist to compare with
 * @param dont_care_about_value        set to 1 if no value matching needed.
 *
 * @return SPH_FRAME_VALIDATOR_VALIDATED if true (identical),
 * SPH_FRAME_VALIDATOR_NOT_VALIDATED if not, SPH_FRAME_VALIDATOR_ERROR on error.
 *
 * The frames keywords are loaded using the keyword manager (and are hence
 * converted into software representations) and for each property in the plist
 * propertylist passed to this function it is checked that the frames propertylist
 * contains the property and has matching type. If the plist property contains
 * a comment, a matching comment in the propertylist from the frame is required
 * for a match. If care_about_value is true and identical value is required
 * for a match.
 */
/*----------------------------------------------------------------------------*/
sph_frame_validator_result
sph_frame_validator_validate_contains_keywords_soft( const cpl_frame* is,
                                                     int ext,
                                                     const cpl_propertylist* plist,
                                                     short care_about_value ) {
    cpl_propertylist*    pl        = NULL;
    int                  bad        = 0;

    if ( is == NULL || plist == NULL ) {
        return -1;
    }
    pl = sph_keyword_manager_load_properties( cpl_frame_get_filename( is ), ext );
    if ( !pl ) return SPH_FRAME_VALIDATOR_ERROR;

    for (int ii = 0; ii < cpl_propertylist_get_size( plist ); ++ii) {
        const cpl_property* prop  = cpl_propertylist_get_const( plist, ii );
        const char*         pname = cpl_property_get_name(prop);

        if ( cpl_propertylist_has( pl, pname ) ) {
            const cpl_type ptype = cpl_property_get_type(prop);
            const cpl_property*  prop_is = 
                cpl_propertylist_get_property_const( pl, pname );
            if ( cpl_property_get_type( prop_is ) != ptype &&
                 ptype != CPL_TYPE_CHAR &&
                 ptype != CPL_TYPE_FLOAT &&
                 ptype != CPL_TYPE_LONG ) {
                bad++;
            }
            else {
                if ( cpl_property_get_comment(prop) ) {
                    if ( !cpl_property_get_comment(prop_is) ) {
                        bad++;
                    }
                    else {
                        if ( strcasecmp( cpl_property_get_comment(prop_is ),
                                         cpl_property_get_comment(prop)) != 0  ) {
                            bad++;
                        }
                    }
                }
                cpl_error_reset();
                if ( care_about_value ) {
                    if ( ptype == CPL_TYPE_BOOL ) {
                        if ( cpl_property_get_bool( prop_is) != cpl_property_get_bool( prop) ) bad++;
                    }
                    if ( ptype == CPL_TYPE_CHAR ) {
                        if ( cpl_property_get_char( prop_is) != cpl_property_get_char( prop) ) {
                            if ( cpl_property_get_type( prop_is ) != CPL_TYPE_STRING ) bad++;
                        }
                    }
                    if ( ptype == CPL_TYPE_DOUBLE ) {
                        if ( cpl_property_get_double( prop_is) != cpl_property_get_double( prop) ) bad++;
                    }
                    if ( ptype == CPL_TYPE_FLOAT ) {
                        if ( cpl_property_get_float( prop_is) != cpl_property_get_float( prop) &&
                                cpl_property_get_double( prop_is) != cpl_property_get_float( prop) ) bad++;
                    }
                    if ( ptype == CPL_TYPE_INT ) {
                        if ( cpl_property_get_int( prop_is) != cpl_property_get_int( prop) ) bad++;
                    }
                    if ( ptype == CPL_TYPE_LONG ) {
                        sph_error_raise( CPL_ERROR_INVALID_TYPE, __FILE__, __func__, __LINE__, SPH_ERROR_WARNING,
                                "CPL does not support loading LONG type keywords and so can not verify the"
                                "presence of such keywords. Will ignore %s it for validation.",
                                pname );
                    }
                    if ( ptype == CPL_TYPE_STRING ) {
                        if ( strcasecmp( cpl_property_get_string(prop_is ), cpl_property_get_string(prop) ) != 0 ) bad++;
                    }
                }
            }
        }
        else {
            bad++;
        }
    }
    cpl_propertylist_delete(pl); pl = NULL;

    return  bad > 0
        ? SPH_FRAME_VALIDATOR_NOT_VALIDATED
        : SPH_FRAME_VALIDATOR_VALIDATED;
}
