/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */
#include "sph_framesorter.h"
#include "sph_keyword_manager.h"
#include "cpl.h"
#include <math.h>

const int SPH_FRAMESORT_MAX_SORT = 10;

sph_framebag* sph_framebag_new( int nsort ) {
    sph_framebag*    bag    = NULL;
    if ( nsort < 1 || nsort > SPH_FRAMESORT_MAX_SORT )
        return NULL;
    bag = cpl_calloc( 1, sizeof(sph_framebag ) );
    bag->maxs = cpl_calloc( nsort, sizeof( double ) );
    bag->types = cpl_calloc( nsort, sizeof( cpl_type ) );
    bag->mins = cpl_calloc( nsort, sizeof( double ) );
    bag->frameset = cpl_frameset_new( );
    return bag;
}

void sph_framebag_delete_all( sph_framebag* self ) {
    if ( self ) {
        if ( self->next ) {
            sph_framebag_delete_all( self->next );
        }
        sph_framebag_delete( self );
    }
}

sph_framebag* sph_framebag_get_last( sph_framebag* self ) {
    sph_framebag*    result = self;
    if ( !self ) return NULL;
    if ( self->next ) {
        result = sph_framebag_get_last( self->next );
    }
    return result;
}

sph_framebag* sph_framebag_search( sph_framebag* self, const cpl_vector* svect ) {
    int            ii        = 0;
    double        val        = 0.0;
    int            found    = 1;
    for (ii = 0; ii < cpl_vector_get_size( svect ); ++ii) {
        val = cpl_vector_get( svect, ii );
        if ( val < self->mins[ii] || val > self->maxs[ii] ) {
            found = 0;
        }
    }
    if ( found ) {
        return self;
    }
    else {
        if ( self->next ) {
            return sph_framebag_search( self->next, svect );
        }
        else {
            return NULL;
        }
    }
}

void sph_framebag_delete( sph_framebag* self ) {
    if ( self ) {
        if ( self->frameset ) {
            cpl_frameset_delete( self->frameset );
            self->frameset = NULL;
        }
        if ( self->maxs ) {
            cpl_free( self->maxs );
            self->maxs = NULL;
        }
        if ( self->mins ) {
            cpl_free( self->mins );
            self->mins = NULL;
        }
        if ( self->types ) {
            cpl_free( self->types );
            self->types = NULL;
        }
        cpl_free(self);
    }
}


sph_framesorter*
sph_framesorter_new( cpl_frameset* inframes ) {
    sph_framesorter*        self        = NULL;

    if ( !inframes ) {
        return NULL;
    }
    self = cpl_calloc( 1, sizeof(sph_framesorter) );
    if ( !self ) {
        return NULL;
    }
    self->inframeset = inframes;
    self->sort_criteria = NULL;
    self->framebag_list = NULL;
    self->nsets = 0;
    return self;
}

sph_error_code
sph_framesorter_check_set( sph_framesorter* self, cpl_frame* inframe, int inpos ) {
    cpl_frame*            dupframe    = NULL;
    cpl_propertylist*    plist        = NULL;
    cpl_property*        prop        = NULL;
    cpl_parameter*        paramsort    = NULL;
    cpl_parameter*        paramtol    = NULL;
    cpl_parameter*        paramrange    = NULL;
    sph_framebag*        bag            = NULL;
    cpl_type            proptype    = CPL_TYPE_UNSPECIFIED;
    char                sstr[256];
    char                tolstr[256];
    char                rangestr[256];
    int                    ii            = 0;
    int                    ncriteria    = 0;
    double                dval        = 0.0;
    cpl_vector*            validvals    = NULL;
    int                    count        = 0;

    if ( !self ) return CPL_ERROR_NULL_INPUT;
    if ( !self->sort_criteria ) return CPL_ERROR_NULL_INPUT;

    plist = sph_keyword_manager_load_properties( cpl_frame_get_filename( inframe ), inpos );

    sprintf( sstr, "recipe.dependency.%d.keyname", ncriteria + 1 );
    paramsort = cpl_parameterlist_find( self->sort_criteria, sstr );
    while ( paramsort ) {
        ncriteria++;
        sprintf( sstr, "recipe.dependency.%d.keyname", ncriteria + 1 );
        paramsort = cpl_parameterlist_find( self->sort_criteria, sstr );
    }

    validvals = cpl_vector_new( ncriteria );

    for (ii = 0; ii < ncriteria; ++ii)
    {
        sprintf( sstr, "recipe.dependency.%d.keyname", ii + 1 );
        sprintf( tolstr, "recipe.dependency.%d.tolerance", ii + 1 );
        sprintf( rangestr, "recipe.dependency.%d.range", ii + 1 );

        paramsort = cpl_parameterlist_find( self->sort_criteria, sstr );
        if ( !paramsort ) {
            goto ERROR_EXIT;
        }
        paramtol = cpl_parameterlist_find( self->sort_criteria, tolstr );
        paramrange = cpl_parameterlist_find( self->sort_criteria, rangestr );

        /* Check if specified sorting criteria
         * is in propertylist */
        if ( cpl_propertylist_has( plist, cpl_parameter_get_string(paramsort) ) ) {
            /* Yes, so check that type of property is ok */

            prop = cpl_propertylist_get_property( plist, cpl_parameter_get_string(paramsort) );
            proptype = cpl_property_get_type( prop );
            if ( proptype == CPL_TYPE_STRING || proptype == CPL_TYPE_CHAR ) {
                goto ERROR_EXIT;
            }
            if ( !paramrange || !paramtol ) {
                goto ERROR_EXIT;
            }
            /* OK type so now take values */
            if ( proptype == CPL_TYPE_INT  )
            {
                dval = (double)cpl_property_get_int( prop );
            }
            if ( proptype == CPL_TYPE_BOOL ) {
                dval = (double)cpl_property_get_bool( prop );
            }
            if ( proptype == CPL_TYPE_FLOAT ) {
                dval = (double)cpl_property_get_float( prop );
            }
            if ( proptype == CPL_TYPE_DOUBLE ) {
                    dval = cpl_property_get_double( prop );
            }
            /* Check that value is in ok range */
            if ( cpl_parameter_get_type(paramrange) == CPL_TYPE_DOUBLE  ) {
                if ( dval >= cpl_parameter_get_range_min_double( paramrange )
                        && dval <= cpl_parameter_get_range_max_double( paramrange ) )
                {
                    /* Value is allowed so add a new set */
                    cpl_vector_set( validvals, ii, dval );
                    count++;
                }
            }
            else if ( cpl_parameter_get_type(paramrange) == CPL_TYPE_INT  ) {
                if ( dval >= (double)cpl_parameter_get_range_min_int( paramrange )
                        && dval <= (double)cpl_parameter_get_range_max_int( paramrange ) )
                {
                    /* Value is allowed so add a new set */
                    cpl_vector_set( validvals, ii, dval );
                    count++;
                }
            }

        }
    }

    if ( count != ncriteria ) {
        goto ERROR_EXIT;
    }
    dupframe = cpl_frame_duplicate( inframe );

    if ( self->framebag_list ) {
        bag = sph_framebag_search( self->framebag_list, validvals );
    }
    if ( bag ) {
        cpl_frameset_insert( bag->frameset, dupframe );
    }
    else {
        bag = sph_framesort_new_bag( self, validvals );
        if ( bag ) {
            cpl_frameset_insert( bag->frameset, dupframe );
            if ( self->framebag_list ) {
                sph_framebag_get_last( self->framebag_list )->next = bag;
            }
            else {
                self->framebag_list = bag;
            }
        }
    }
    cpl_vector_delete(validvals); validvals = NULL;
    cpl_propertylist_delete(plist); plist = NULL;
    return CPL_ERROR_NONE;

ERROR_EXIT:
    if ( validvals ) {
        cpl_vector_delete( validvals );
    }
    if ( plist ) {
        cpl_propertylist_delete( plist );
    }
    return CPL_ERROR_ILLEGAL_INPUT;
}
sph_framebag* sph_framesort_new_bag( sph_framesorter* self, cpl_vector* validvals ) {
    int                ii            = 0;
    int                ncriteria    = 0;
    char            sstr[256];
    char            tolstr[256];
    char            rangestr[256];
    cpl_parameter*    paramsort    = NULL;
    cpl_parameter*    paramtol    = NULL;
    cpl_parameter*    paramrange    = NULL;
    double            dval        = 0.0;
    double             tolval        = 0.0;
    double            range        = 0.0;
    int                ival        = 0;
    sph_framebag*    bag            = NULL;

    sprintf( sstr, "recipe.dependency.%d.keyname", ncriteria + 1 );
    paramsort = cpl_parameterlist_find( self->sort_criteria, sstr );
    while ( paramsort ) {
        ncriteria++;
        sprintf( sstr, "recipe.dependency.%d.keyname", ncriteria + 1 );
        paramsort = cpl_parameterlist_find( self->sort_criteria, sstr );
    }
    cpl_error_reset();
    bag = sph_framebag_new( ncriteria );

    for (ii = 0; ii < ncriteria; ++ii) {
        sprintf( sstr, "recipe.dependency.%d.keyname", ii + 1 );
        sprintf( tolstr, "recipe.dependency.%d.tolerance", ii + 1 );
        sprintf( rangestr, "recipe.dependency.%d.range", ii + 1 );
        paramsort = cpl_parameterlist_find( self->sort_criteria, sstr );
        paramtol = cpl_parameterlist_find( self->sort_criteria, tolstr );
        paramrange = cpl_parameterlist_find( self->sort_criteria,rangestr );
        bag->types[ ii ] = cpl_parameter_get_type( paramrange );
        dval = cpl_vector_get( validvals, ii );
        if ( bag->types[ii] == CPL_TYPE_DOUBLE ) {
        	tolval = cpl_parameter_get_double( paramtol );
        }
        else if ( bag->types[ii] == CPL_TYPE_INT ) {
            tolval = (double)cpl_parameter_get_int( paramtol );
        }
        if ( tolval ) {
            range = ( dval -  cpl_parameter_get_range_min_double( paramrange ) )/ tolval ;
            ival = floor(range);
            bag->mins[ ii ] = ival * tolval;
            bag->maxs[ ii ] = ival * tolval + tolval;
        }
        else {
            bag->mins[ ii ] = dval;
            bag->maxs[ ii ] = dval;
        }
    }
    return bag;
}

sph_error_code
sph_framesorter_sort( sph_framesorter* self,
        cpl_parameterlist* dependencies )
{
    int                ii            = 0;
    sph_error_code    rerr        = CPL_ERROR_NONE;
    if ( !self ) return CPL_ERROR_NULL_INPUT;
    if ( !dependencies ) return CPL_ERROR_NULL_INPUT;
    if ( self->sort_criteria ) cpl_parameterlist_delete(self->sort_criteria);
    self->sort_criteria = dependencies;
    if ( !self->inframeset ) return CPL_ERROR_NULL_INPUT;
    if ( cpl_frameset_get_size( self->inframeset ) < 1  )
        return CPL_ERROR_NULL_INPUT;
    if ( self->framebag_list ) {
        sph_framebag_delete_all( self->framebag_list);
        self->framebag_list = NULL;
    }
    for (ii = 0; ii < cpl_frameset_get_size( self->inframeset ); ++ii) {
        rerr = sph_framesorter_check_set( self, cpl_frameset_get_position( self->inframeset, ii), 0 );
        if ( rerr ) break;
    }
    return rerr;
}

cpl_frameset*
sph_framesorter_get_frameset( sph_framesorter* self,
        int bagid )
{
    cpl_frameset*    result  = NULL;
    sph_framebag*    fb        = NULL;
    int                ii        = 0;

    if ( !self ) return NULL;
    if ( bagid < 0 ) return NULL;
    if ( bagid > self->nsets - 1) return NULL;
    fb = self->framebag_list;
    for ( ii = 0; ii < bagid; ++ii ) {
        if ( !fb->next ) {
            return NULL;
        }
        result = fb->next->frameset;
    }
    return result;
}

void sph_framesorter_delete( sph_framesorter* self ) {

    if ( self ) {
        if ( self->framebag_list ) {
            sph_framebag_delete_all( self->framebag_list );
            self->framebag_list = NULL;
            self->nsets = 0;
        }
        if ( self->sort_criteria ) {
        	cpl_parameterlist_delete(self->sort_criteria);
        }
        self->inframeset = NULL;
        cpl_free(self);
    }
}
