/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_interpolation.h"

#include <cpl.h>
#include <math.h>
#include <assert.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

sph_interpolation* sph_interpolation_new(cpl_array* xpositions,
                     cpl_array* ypositions,
                     int n) {
    sph_interpolation* self = NULL;
    const cpl_size  asize   = cpl_array_get_size(xpositions);
    
    cpl_ensure(xpositions != NULL, CPL_ERROR_NULL_INPUT,   NULL);
    cpl_ensure(ypositions != NULL, CPL_ERROR_NULL_INPUT,   NULL);
    cpl_ensure(n          >  0,   CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(cpl_array_get_size(ypositions) == asize,
           CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
    
    self = (sph_interpolation*)cpl_calloc(1,sizeof(*self));
    self->xpositions = xpositions;
    self->ypositions = ypositions;
    self->n = n;
    self->indices = cpl_calloc(asize, sizeof(int));
    self->kdtree = sph_kd_create(2);
    
    for (int ii = 0; ii < asize; ++ii) {
      const double xpos = cpl_array_get(xpositions, ii, NULL);
      const double ypos = cpl_array_get(ypositions, ii, NULL);
      const double pos[2] = {xpos, ypos};

      self->indices[ii] = ii;
      sph_kd_insert(self->kdtree, pos, &self->indices[ii]);
    }
    
    return self;
}

void
sph_interpolation_delete( sph_interpolation* self) {
    if ( self ) {
        cpl_array_delete( self->xpositions );
        cpl_array_delete( self->ypositions );
        cpl_free(self->indices);
        sph_kd_free(self->kdtree);
        cpl_imagelist_delete(self->index_imagelist);
        cpl_imagelist_delete(self->distances_imagelist);
        cpl_free(self);
    }
}

cpl_vector* sph_interpolation_find_closest_points(sph_interpolation* self,
                          double xpos,
                          double ypos)
{
    cpl_vector*     result = NULL;
    const double    pos[2] = {xpos, ypos};
    sph_kdres*      kdresult;
    int             ii;

    cpl_ensure(self    != NULL, CPL_ERROR_NULL_INPUT,    NULL);
    cpl_ensure(self->n >  0,    CPL_ERROR_ILLEGAL_INPUT, NULL);

    kdresult = sph_kd_nearest(self->kdtree, pos, 0.0);
      
    result = cpl_vector_new(self->n);

    for (ii = 0; kdresult != NULL && ii < self->n; ++ii) {
      const int*   intp = (int*)kdresult->riter->item->data;
      const double dist = kdresult->riter->dist_sq;
    
      cpl_vector_set(result, ii, *intp);
    
      sph_kd_res_free(kdresult);
      kdresult = sph_kd_nearest(self->kdtree, pos, dist + 0.001);
    
    }
      
    if (ii < self->n) {
      cpl_vector_delete(result);
      result = NULL;
    } else {
      sph_kd_res_free(kdresult);
    }
    
    return result;
}

cpl_image* sph_interpolation_interpolate(const sph_interpolation* self,
                                         const cpl_array*         values)
{
    cpl_image*      result = NULL;

    if (self == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (self->index_imagelist == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (self->distances_imagelist == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (values == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (cpl_array_get_size( values ) !=
               cpl_array_get_size(self->xpositions)) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT);
    } else if (cpl_array_get_type( values ) != CPL_TYPE_DOUBLE) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    } else if (cpl_imagelist_get_size( self->index_imagelist ) != self->n) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT);
    } else if (cpl_imagelist_get_size(self->distances_imagelist ) != self->n) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT);
    } else {
        const cpl_image* imfirst = cpl_imagelist_get_const
            (self->distances_imagelist, 0);
        const cpl_size   nx  = cpl_image_get_size_x(imfirst);
        const cpl_size   ny  = cpl_image_get_size_y(imfirst);
        const size_t     nxy = (size_t)(nx * ny);
        const double*    pvalue = cpl_array_get_data_double_const(values);

        cpl_image* imweight;
        double*    pimweight;
        double*    presult;

        cpl_msg_info(cpl_func, "Interpolation %d X %d X %d",
                     (int)nx, (int)ny, self->n);

        cpl_ensure(cpl_image_get_type(imfirst) == CPL_TYPE_DOUBLE,
                   CPL_ERROR_ILLEGAL_INPUT, NULL);
        cpl_ensure(cpl_image_get_type(cpl_imagelist_get_const
                                      (self->index_imagelist, 0))
                   == CPL_TYPE_INT, CPL_ERROR_ILLEGAL_INPUT, NULL);
        cpl_ensure(cpl_image_get_size_x(cpl_imagelist_get_const
                                        (self->index_imagelist, 0)) == nx,
                   CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
        cpl_ensure(cpl_image_get_size_y(cpl_imagelist_get_const
                                        (self->index_imagelist, 0)) == ny,
                   CPL_ERROR_INCOMPATIBLE_INPUT, NULL);

    
        imweight = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        pimweight = cpl_image_get_data_double(imweight);

        result    = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        presult   = cpl_image_get_data_double(result);

        for (cpl_size pp = 0; pp < (cpl_size)self->n; ++pp) {
            const cpl_image* imdist = cpl_imagelist_get_const
                (self->distances_imagelist, pp);
            const cpl_image* imindices = cpl_imagelist_get_const
                (self->index_imagelist, pp);
            const double* pdist  = cpl_image_get_data_double_const(imdist);
            const int*    pindex = cpl_image_get_data_int_const(imindices);

            assert(pdist  != NULL);
            assert(pindex != NULL);
    
            for (size_t xy = 0; xy < nxy; ++xy) {
                if (pdist[xy] > 0.0000001 ) { /* FIXME: Random tol ? */
                    presult[xy]   += pvalue[pindex[xy]] / pdist[xy];
                    pimweight[xy] += 1.0 / pdist[xy];
                }

                if (pp == (cpl_size)self->n - 1 && pimweight[xy] != 0.0) {
                    presult[xy] /= pimweight[xy];
                }
            }
        }
        cpl_image_delete(imweight);
    }
    return result;       
}

sph_error_code sph_interpolation_prepare_index_assocs(sph_interpolation* self,
                                                      int                nx,
                                                      int                ny)
{
    double              xpos = 0.0;
    double              ypos = 0.0;
    int                 xx = 0;
    int                 yy = 0;
    int                 pp = 0;
    cpl_image*          im = NULL;
    cpl_mask*           m = NULL;
    cpl_vector*         indices = NULL;
    int                 index = 0;
    double              sq_dist = 0.0;
    double              dx = 0.0;
    double              dy = 0.0;
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);

    if ( self->index_imagelist ) {
        cpl_imagelist_delete(self->index_imagelist);
        self->index_imagelist = NULL;
    }
    if ( self->distances_imagelist ) {
        cpl_imagelist_delete(self->distances_imagelist);
        self->distances_imagelist = NULL;
    }
    self->index_imagelist = cpl_imagelist_new();
    self->distances_imagelist = cpl_imagelist_new();
    for (pp = 0; pp < self->n; ++pp) {
        im = cpl_image_new(nx,ny,CPL_TYPE_INT);
        m = cpl_image_get_bpm(im);
        cpl_mask_not(m);
        cpl_imagelist_set(
                self->index_imagelist,
                im,pp);
        im = cpl_image_new(nx,ny,CPL_TYPE_DOUBLE);
        m = cpl_image_get_bpm(im);
        cpl_mask_not(m);
        cpl_imagelist_set(self->distances_imagelist,
                im,pp);
    }

    for (yy = 0; yy < ny; ++yy) {
        for (xx = 0; xx < nx; ++xx) {
            xpos = xx + 0.5; // pixel midpoint
            ypos = yy + 0.5;
            indices = sph_interpolation_find_closest_points(
                    self,xpos,ypos);
            if ( indices ) {
                if ( cpl_vector_get_size(indices) == self->n ) {
                    for (pp = 0; pp < self->n; ++pp) {
                        index = cpl_vector_get( indices, pp);
                        if ( index >=0 ) {
                            im = cpl_imagelist_get(
                                    self->index_imagelist,pp);
                            cpl_image_accept(im,xx+1,yy+1);
                            cpl_image_set(im,
                                    xx + 1, yy + 1,index);
                            im = cpl_imagelist_get(
                                    self->distances_imagelist,
                                    pp);
                            cpl_image_accept(im,xx+1,yy+1);
                            dx = cpl_array_get(
                                    self->xpositions,
                                    index,NULL) - xpos;
                            dy = cpl_array_get(
                                    self->ypositions,
                                    index,NULL) - ypos;
                            sq_dist = dx * dx + dy * dy;
                            cpl_image_set(im,
                                    xx + 1, yy + 1,sq_dist);

                        }
                    }
                    cpl_vector_delete(indices); indices = NULL;
                }
                else {
                    SPH_ERROR_RAISE_ERR(
                            SPH_ERROR_GENERAL,
                            "Could not find %d neigbours for %d,%d",self->n,xx,yy);
                }
            }
            else {
                SPH_ERROR_RAISE_ERR(
                        SPH_ERROR_GENERAL,
                        "Could not find neigbours for %d,%d",xx,yy);
            }
        }
    }
    return cpl_error_get_code();
}
/**@}*/
