/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "sph_error.h"
#include "sph_pixel_polyfit_table.h"
#include "sph_master_frame.h"
#include "sph_common_keywords.h"
#include "sph_utils.h"
#include "sph_fitting.h"
#include <time.h>
#include <math.h>
#include "sph_cube.h"
#include "sph_fits.h"
#include "sph_keyword_manager.h"
#include "sph_shared_irdifs.h"
#include "sph_common_science.h"
#include "sph_filemanager.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides ...
 */
/*----------------------------------------------------------------------------*/
/**@{*/
static
double sph_irdifs_detector_persistence_get_exptime(cpl_frame* aframe) {
    cpl_propertylist* pl = NULL;
    double exptime = 0.0;
    pl = sph_keyword_manager_load_properties(cpl_frame_get_filename(aframe), 0);
    exptime = cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_SEQ1DIT);
    cpl_propertylist_delete(pl);
    return exptime;
}
static
double sph_irdifs_detector_persistence_get_date(cpl_frame* aframe,
        double* atime0) {
    cpl_propertylist* pl = NULL;
    struct tm timestruct;
    time_t atime = -1;
    const char* date = NULL;
    double fftime = 0.0;
    double rettime = 0.0;

    pl = sph_keyword_manager_load_properties(cpl_frame_get_filename(aframe), 0);
    date = cpl_propertylist_get_string(pl, SPH_COMMON_KEYWORD_DATE);
    if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND) {
        cpl_error_reset();
        pl = cpl_propertylist_load(cpl_frame_get_filename(aframe), 0);
        date = cpl_propertylist_get_string(pl, SPH_COMMON_KEYWORD_DATE);
    }SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Read date %s from %s.", date, cpl_frame_get_filename(aframe));
    if (date) {
        sscanf(date, "%d-%d-%dT%d:%d:%d.%lf", &timestruct.tm_year,
                &timestruct.tm_mon, &timestruct.tm_mday, &timestruct.tm_hour,
                &timestruct.tm_min, &timestruct.tm_sec, &fftime);
        atime = mktime(&timestruct);
        sscanf(date, "%d-%d-%dT%d:%d:%d.%lf", &timestruct.tm_year,
                &timestruct.tm_mon, &timestruct.tm_mday, &timestruct.tm_hour,
                &timestruct.tm_min, &timestruct.tm_sec, &fftime);
        fftime /= 10000.0;
        //printf("Seconds: %d + %f\n", timestruct.tm_sec,fftime);
        atime = mktime(&timestruct);
        if (*atime0 == -1.0)
            *atime0 = ((double) atime) + fftime;
        rettime = ((double) atime) + fftime - *atime0;
        //printf("Atime: %d, Time0: %f, Time: %f\n", (int)mktime(&timestruct), *atime0,rettime);
    }
    cpl_propertylist_delete(pl);
    return rettime;
}
cpl_vector*
sph_irdifs_detector_persistence_get_timing(cpl_frameset** prawframes,
        double atime0) {
    cpl_vector* times = NULL;
    int nims = 0;
    int ff = 0;
    int kk = 0;
    cpl_frame* aframe = NULL;
    cpl_frameset* timesorted = NULL;
    cpl_vector* already_used = NULL;
    double atime = 0.0;
    double mintime = FLT_MAX;
    int minff = -1;
    cpl_frameset* rawframes = NULL;
    int nplanes = 0;
    double exptime = 0.0;
    int pp = 0;
    int nn = 0;

    cpl_ensure(prawframes, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(*prawframes, CPL_ERROR_NULL_INPUT, NULL);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    rawframes = *prawframes;
    already_used = cpl_vector_new(cpl_frameset_get_size(rawframes));
    cpl_vector_fill(already_used, 0.0);
    timesorted = cpl_frameset_new();

    for (kk = 0; kk < cpl_frameset_get_size(rawframes); ++kk) {
        minff = -1;
        mintime = FLT_MAX;

        for (ff = 0; ff < cpl_frameset_get_size(rawframes); ++ff) {
            if (cpl_vector_get(already_used, ff) < 1) {
                aframe = cpl_frameset_get_position(rawframes, ff);
                atime = -1.0;
                sph_irdifs_detector_persistence_get_date(aframe, &atime);
                if (atime <= mintime) {
                    mintime = atime;
                    minff = ff;
                }
            }
        }
        if (minff >= 0) {
            cpl_vector_set(already_used, minff, 2);
            cpl_frameset_insert(
                    timesorted,
                    cpl_frame_duplicate(
                            cpl_frameset_get_position(rawframes, minff)));
        }
    }

    cpl_frameset_delete(rawframes);
    rawframes = NULL;
    rawframes = cpl_frameset_duplicate(timesorted);
    cpl_frameset_delete(timesorted);
    timesorted = NULL;
    *prawframes = rawframes;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    nims = sph_common_science_get_nraws(rawframes, 0);

    times = cpl_vector_new(nims);

    nn = 0;
    for (ff = 0; ff < cpl_frameset_get_size(rawframes); ++ff) {
        aframe = cpl_frameset_get_position(rawframes, ff);
        if (sph_fits_test_iscube(cpl_frame_get_filename(aframe), 0) == 1) {
            nplanes = sph_fits_get_nplanes(cpl_frame_get_filename(aframe), 0);
        } else {
            nplanes = 1;
        }
        exptime = sph_irdifs_detector_persistence_get_exptime(aframe);
        exptime /= nplanes;
        atime = sph_irdifs_detector_persistence_get_date(aframe, &atime0);
        //atime = atime + 0.5 * exptime;
        for (pp = 0; pp < nplanes; ++pp) {
            cpl_vector_set(times, nn, 1.0 / (atime + pp * exptime));
            nn++;
        }
    }
    if (nn != nims) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_INCOMPATIBLE_INPUT,
                "Have mismatch in frame count.");
    }
    cpl_vector_delete(already_used);
    already_used = NULL;
    return times;
}

cpl_error_code sph_irdifs_detector_persistence_run(
        cpl_frame* static_badpixel_frame, cpl_frame* master_dark_frame,
        cpl_frameset* satframes, cpl_frameset* unsatframes,
        sph_collapse_algorithm coll_alg,
        cpl_parameterlist* framecomb_parameterlist, cpl_frameset* rawframes,
        cpl_frameset* inframes, cpl_parameterlist * inparams,
        double threshold_dark, double threshold_ill, double fitorder,
        const char* outfilename, const char* tag, const char* recipe,
        const char* pipeline) {
    sph_master_frame* dark = NULL;
    sph_master_frame* on_sat = NULL;
    sph_master_frame* on_unsat = NULL;
    cpl_image* image = NULL;
    cpl_image* hot_image = NULL;
    cpl_mask* hot_mask = NULL;
    cpl_mask* ill_mask = NULL;
    cpl_mask* dark_mask = NULL;
    int ff = 0;
    cpl_size p = 1;
    cpl_vector* timev = NULL;
    cpl_vector* meanill = NULL;
    cpl_vector* meandark = NULL;
    cpl_vector* meanillerr = NULL;
    cpl_vector* meandarkerr = NULL;
    cpl_vector* coefferrs = NULL;
    cpl_vector* evalt = NULL;
    cpl_polynomial* poly = NULL;
    double redchi = 0.0;
    double illum = 0.0;
    double illum_err = 0.0;
    double unsat_t = 0.0;
    double sat_t = 0.0;
    double atime_sat = -1.0;
    cpl_propertylist* pl = NULL;
    int nimages = 0;
    sph_master_frame* tempmaster = NULL;
    cpl_vector* binpos = NULL;
    cpl_vector* binvals = NULL;
    double mean = 0.0;
    double stdev = 0.0;
    double darkcount = 0.0;
    double dark_err = 0.0;

    if (static_badpixel_frame) {
        hot_image = cpl_image_load(
                cpl_frame_get_filename(static_badpixel_frame), CPL_TYPE_INT, 0,
                0);
    }
    if (master_dark_frame) {
        dark = sph_master_frame_load_(master_dark_frame,
                0);
        if (dark) {
            if (hot_image) {
                cpl_image_add(hot_image, dark->badpixelmap);
            } else {
                hot_image = sph_master_frame_get_badpixelmap(dark);
            }
        }
        sph_master_frame_delete(dark);
        dark = NULL;
    }
    if (hot_image) {
        hot_mask = cpl_mask_threshold_image_create(hot_image, 1.0, FLT_MAX);
    }

    on_unsat = sph_framecombination_master_frame_from_cpl_frameset(unsatframes,
            coll_alg, framecomb_parameterlist);

    cpl_ensure_code(on_unsat != NULL, CPL_ERROR_INCOMPATIBLE_INPUT);

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Created unsaturated "
    "calibrated frames...");

    unsat_t = sph_irdifs_detector_persistence_get_exptime(
            cpl_frameset_get_first(unsatframes));
    sat_t = sph_irdifs_detector_persistence_get_exptime(
            cpl_frameset_get_first(satframes));

    if (sat_t <= 0 || unsat_t <= 0) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "Zero exposure time in a frame. "
                "Thats bad. Is this an ESO conform frame?");
        cpl_ensure_code(0, CPL_ERROR_INCOMPATIBLE_INPUT);
    }SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Respective exposure times were %f for "
            "saturated and %f for unsaturated frame.", sat_t, unsat_t);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_ERROR_RAISE_WARNING(SPH_ERROR_GENERAL,
                "Could not calculate the ratio of exposure times.");
        cpl_error_reset();
    }

    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Multiplying unsaturated frame by %f and then apply "
            "thresholding of %f to determine saturated pixels.",
            sat_t / unsat_t, threshold_ill);

    sph_master_frame_multiply_double(on_unsat, sat_t / unsat_t);

    sph_master_frame_mask_tolerance(on_unsat, threshold_ill, FLT_MAX);

    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Found %lld pixels in "
            "unsaturated image with threshold above %f.", cpl_image_get_size_x(on_unsat->image) * cpl_image_get_size_y(on_unsat->image) - sph_master_frame_get_nbads(on_unsat), threshold_ill);

    illum = sph_master_frame_get_mean(on_unsat, &illum_err);
    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Calculated mean in illuminated regions on unsaturated image of %f.", illum);

    ill_mask = sph_master_frame_get_badpixelmask(on_unsat);

    sph_master_frame_accept_all(on_unsat);
    sph_master_frame_mask_tolerance(on_unsat, threshold_dark, FLT_MAX);
    darkcount = sph_master_frame_get_mean(on_unsat, &dark_err);

    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Calculated mean in dark regions on unsaturated image of %f.", darkcount);

    dark_mask = sph_master_frame_get_badpixelmask(on_unsat);
    cpl_mask_not(dark_mask);

    if (hot_mask) {
        cpl_mask_or(ill_mask, hot_mask);
        cpl_mask_or(dark_mask, hot_mask);
    }

    sph_irdifs_detector_persistence_get_date(
            cpl_frameset_get_position(satframes,
                    cpl_frameset_get_size(satframes) - 1), &atime_sat);
    atime_sat += sat_t;
    timev = sph_irdifs_detector_persistence_get_timing(&rawframes, atime_sat);

    nimages = sph_common_science_get_nraws(rawframes, 0);

    meanill = cpl_vector_new(nimages);
    meandark = cpl_vector_new(nimages);
    meanillerr = cpl_vector_new(nimages);
    meandarkerr = cpl_vector_new(nimages);

    image = sph_common_science_get_next_raw(rawframes, 0);
    ff = 0;
    binpos = cpl_vector_new(1000);
    binvals = cpl_vector_new(1000);
    while (image) {
        SPH_RAISE_CPL_RESET
        cpl_image_reject_from_mask(image, ill_mask);
        tempmaster = sph_master_frame_new_from_cpl_image(image);
        sph_master_frame_set_bads_from_mask(tempmaster, ill_mask);
        mean = sph_master_frame_get_mean(tempmaster, &stdev);
        sph_master_frame_fit_gauss_hist(tempmaster, binpos, binvals,
                mean * 0.01, mean * 2, &mean, &stdev, &redchi);
        if (cpl_error_get_code() == CPL_ERROR_CONTINUE)
            cpl_error_reset();
        cpl_vector_set(meanill, ff, mean);
        cpl_vector_set(meanillerr, ff, stdev);
        sph_master_frame_accept_all(tempmaster);
        sph_master_frame_set_bads_from_mask(tempmaster, dark_mask);
        mean = sph_master_frame_get_mean(tempmaster, &stdev);
        sph_master_frame_fit_gauss_hist(tempmaster, binpos, binvals,
                mean * 0.01, mean * 2, &mean, &stdev, &redchi);
        if (cpl_error_get_code() == CPL_ERROR_CONTINUE)
            cpl_error_reset();
        cpl_vector_set(meandark, ff, mean);
        cpl_vector_set(meandarkerr, ff, stdev);
        cpl_image_accept_all(image);
        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_GENERAL,
                "Measured a mean count of %f+/-%f in the dark regions, "
                "and a mean count of %f+/-%f on the illuminated regions "
                "at %f seconds", cpl_vector_get(meandark,ff), cpl_vector_get(meandarkerr,ff), cpl_vector_get(meanill,ff), cpl_vector_get(meanillerr,ff), 1.0/cpl_vector_get(timev,ff));
        SPH_RAISE_CPL_RESET
        cpl_image_delete(image);
        image = NULL;
        image = sph_common_science_get_next_raw(rawframes, 0);
        cpl_error_reset();
        sph_master_frame_delete(tempmaster);
        tempmaster = NULL;
        ff++;
    }

    cpl_vector_power(meandarkerr, 2.0);
    cpl_vector_power(meanillerr, 2.0);
    cpl_vector_add(meanillerr, meandarkerr);
    cpl_vector_sqrt(meanillerr);
    cpl_vector_subtract(meanill, meandark);

    cpl_vector_divide_scalar(meanillerr, illum);
    cpl_vector_divide_scalar(meanill, illum);
    poly = sph_fitting_fit_poly1d(timev, meanill, meanillerr, 0, fitorder, 0,
            0.0, &redchi);

    coefferrs = sph_fitting_estimate_error(poly, timev, meanill, meanillerr,
            NULL);

    pl = cpl_propertylist_new();
    evalt = cpl_vector_new(1);
    cpl_vector_set(evalt, 0, 1.0);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_QC_PERSISTENCE_ONES,
            cpl_polynomial_eval(poly, evalt));
    cpl_vector_set(evalt, 0, 0.5);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_QC_PERSISTENCE_FIVES,
            cpl_polynomial_eval(poly, evalt));
    cpl_vector_set(evalt, 0, 0.1);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_QC_PERSISTENCE_TENS,
            cpl_polynomial_eval(poly, evalt));

    for (p = 0; p <= fitorder; ++p) {
        sph_keyword_manager_update_double1(pl,
                SPH_COMMON_KEYWORD_QC_PERSISTENCE_COEFF, p,
                cpl_polynomial_get_coeff(poly, &p));
        sph_keyword_manager_update_double1(pl,
                SPH_COMMON_KEYWORD_QC_PERSISTENCE_COEFF_RMS, p,
                cpl_vector_get(coefferrs, p));
    }

    sph_master_frame_save_dfs(on_unsat, outfilename, inframes, NULL, inparams,
            tag, recipe, pipeline, pl);

    cpl_propertylist_delete(pl);
    pl = NULL;
    cpl_polynomial_delete(poly);
    poly = NULL;
    cpl_vector_delete(evalt);
    evalt = NULL;
    cpl_vector_delete(meandark);
    meandark = NULL;
    cpl_vector_delete(meandarkerr);
    meandarkerr = NULL;
    cpl_vector_delete(meanill);
    meanill = NULL;
    cpl_vector_delete(meanillerr);
    meanillerr = NULL;
    cpl_vector_delete(timev);
    timev = NULL;
    cpl_vector_delete(coefferrs);
    coefferrs = NULL;
    cpl_image_delete(hot_image);
    hot_image = NULL;
    cpl_mask_delete(hot_mask);
    hot_mask = NULL;
    cpl_mask_delete(ill_mask);
    ill_mask = NULL;
    cpl_mask_delete(dark_mask);
    dark_mask = NULL;
    sph_master_frame_delete(on_sat);
    on_sat = NULL;
    sph_master_frame_delete(on_unsat);
    on_unsat = NULL;
    return CPL_ERROR_NONE;
}

/**@}*/
