/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "sph_master_frame_interpolate.c"

#include "sph_smart_imagelist.h"
#include "sph_error.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_fft.h"
#include "sph_utils.h"
#include "sph_version.h"

#include <cpl.h>
#include <math.h>
#include <string.h>


/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/
sph_error_code SPH_MASTER_FRAME_GENERAL        = SPH_MASTER_FRAME_ERR_START + 0;
sph_error_code SPH_MASTER_FRAME_BAD_ALGORITHM  = SPH_MASTER_FRAME_ERR_START + 1;
sph_error_code SPH_MASTER_FRAME_NO_ALGORITHM   = SPH_MASTER_FRAME_ERR_START + 2;
const int SPH_MASTER_FRAME_IMAGE_EXT = 0;
const int SPH_MASTER_FRAME_BPIX_EXT = 1;
const int SPH_MASTER_FRAME_NC_EXT = 2;
const int SPH_MASTER_FRAME_RMS_EXT = 3;

static void sph_master_frame_reject_(const sph_master_frame*, cpl_boolean)
    CPL_ATTR_NONNULL;
static sph_error_code
sph_master_frame_div_or_mul_master_frame( sph_master_frame* self,
        const sph_master_frame* mframe, const cpl_boolean is_mul);

static
sph_error_code sph_master_frame_save_(const sph_master_frame* self,
                                      const char* czFilename,
                                      const cpl_propertylist* pli,
                                      unsigned int mode )
{
    sph_error_code            rerr            = CPL_ERROR_NONE;
    cpl_propertylist*        mainpl            = NULL;
    cpl_propertylist*        tpli            = NULL;
    cpl_propertylist*        extpl            = NULL;
    int                        ii                = 0;
    if ( !self ) {
        SPH_NO_SELF
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !self->image || !self->badpixelmap || !self->ncombmap || !self->rmsmap ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    if ( self->properties ) {
        mainpl = sph_keyword_manager_trans_keywords( self->properties );
        extpl = sph_keyword_manager_trans_keywords( self->properties );
    }
    else {
        mainpl = cpl_propertylist_new();
    }
    if ( pli ) {
        tpli = sph_keyword_manager_trans_keywords( pli );
    }
    if ( tpli && mainpl ) {
        for (ii = 0; ii < cpl_propertylist_get_size( tpli ); ++ii) {
            if ( !cpl_propertylist_has( mainpl, cpl_property_get_name( cpl_propertylist_get( tpli, ii) ) ) ) {
                rerr = cpl_propertylist_append_property( mainpl, cpl_propertylist_get( tpli, ii) );
            }
        }
    }
    if ( self->qclist ) {
        rerr = cpl_propertylist_append( mainpl, self->qclist );
    }
    cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_MASTER_FRAME );
    cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_IMAGE_EXTNAME );
    cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPNAME, SPH_QC_COMPNAME );
    cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPVERN, SPH_QC_COMPVERN );
    cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPREVN, SPH_QC_COMPREVN );
    cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPDATE, SPH_QC_COMPDATE );

    /* Update the headers if required */
    sph_utils_update_header(mainpl);
    sph_utils_update_header(extpl);

    rerr = cpl_image_save( self->image, czFilename, CPL_TYPE_FLOAT, mainpl, mode );
    cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_BADPIXMAP_EXTNAME );
    rerr = cpl_image_save( self->badpixelmap, czFilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_NCOMBMAP_EXTNAME );
    rerr = cpl_image_save( self->ncombmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_RMSMAP_EXTNAME );
    rerr = cpl_image_save( self->rmsmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    if ( mainpl ) {
        cpl_propertylist_delete( mainpl );
    }
    if ( tpli ) {
        cpl_propertylist_delete( tpli );
    }
    if ( extpl ) {
        cpl_propertylist_delete( extpl );
    }
    SPH_RAISE_CPL;
    return rerr;
}


/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_master_frame  Master Frame Object
 *
 * @par Descirption:
 * This object represents a master frame in the SPHERE DRH. Master Frames
 * witin the SPHERE DRH are reduced images (single plane) with associated
 * statistical information relating to their creation and properties. This
 * statistical information is:
 * <ol>
 * <li> RMSMAP. A map of the same dimension and number of pixels as the
 *      image giving the inverse of the RMS of the image pixel</li>
 * <li> NCombmap. A map of the same dimension and number of pixels as the
 *      image giving the number of pixels that were used to obtain the image</li>
 * <li> Badpixelmap. A map of the same dimension and number of pixels as the
 *      image flagging pixels that were impossible to reduce from the raw frames
 *      either because all input raw pixels at that position were bad or because
 *      the statistics is not calculable.</li>
 * <li> Quality Control Parameter list. A list of quality control parameters.
 *      What quality parameters are listed depends on the recipe used to create
 *      the frame.
 * </ol>
 *
 * At the moment the relation to a CPL Frame is that the master frame contains
 * a pointer to an associated frame. Ths is probably not the best solution --
 * a cpl frame should rather point to an associated master frame. This aspect
 * will have to undergo some revision.
 *
 * @par Synopsis:
 * @code
 * typedef struct _sph_master_frame_
 * {
 *   cpl_frame* frame;
 *   cpl_image* image;
 *   cpl_image* badpixelmap;
 *   cpl_image* rmsmap;
 *   cpl_image* ncombmap;
 *   sph_coll_alg collapsealgorithm;
 *   cpl_propertylist* properties;
 *   cpl_propertylist* qclist;
 * } sph_master_frame;
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor fucntion for sph_master_frame.
  @return   a pointer to the newly created sph_master_frame or NULL if
              unsuccessfull.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_master_frame
 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_master_frame_new_empty(void) {
    sph_master_frame*     spmi         = NULL;
    sph_error_code        rerr         = CPL_ERROR_NONE;

    spmi = cpl_calloc( 1, sizeof(sph_master_frame) );
    spmi->qclist = cpl_propertylist_new();
    spmi->properties = cpl_propertylist_new();

    rerr = cpl_propertylist_append_string( spmi->properties,
                                           SPH_COMMON_KEYWORD_SPH_TYPE,
                                           SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_MASTER_FRAME );
    if ( rerr != CPL_ERROR_NONE )
    {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Could not allocate master frame" );
        return NULL;
    }
    return spmi;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new master frame as a subsection of another.
  @param    ll_x  	the lower left corner x coordinate (first pixel = 1)
  @param    ll_y    the lower left corner y coordinate (first pixel = 1)
  @param    ur_x    the upper right corner x coordinate (first pixel = 1)
  @param    ur_y    the upper right corner y coordinate (first pixel = 1)
  @return   a pointer to the newly created sph_master_frame or NULL if
              unsuccessfull.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_master_frame by extracting a window of another
  master frame. The coordinates are using the same conventions as for
  the usual cpl_image functions, with pixel coundting starting at 1,1.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_master_frame_new_extract( const sph_master_frame* mframe,
		int ll_x, int ll_y, int ur_x, int ur_y )
{
    sph_master_frame*     spmi         = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    spmi = sph_master_frame_new_empty();
    spmi->image = cpl_image_extract( mframe->image,
            ll_x,
            ll_y,
            ur_x,
            ur_y);
    spmi->badpixelmap = cpl_image_extract( mframe->badpixelmap,
            ll_x,
            ll_y,
            ur_x,
            ur_y);
    spmi->rmsmap = cpl_image_extract( mframe->rmsmap,
            ll_x,
            ll_y,
            ur_x,
            ur_y);
    spmi->ncombmap = cpl_image_extract( mframe->ncombmap,
            ll_x,
            ll_y,
            ur_x,
            ur_y);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return spmi;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_master_frame.

  @param nx     the number of pixels in x
  @param ny     the number of pixels in y

  @return   a pointer to the newly created sph_master_frame or NULL if
            unsuccessfully.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_master_frame of the give n pixel size.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_master_frame_new( int nx, int ny ) {
    sph_master_frame* spmi;

    cpl_ensure(nx > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(ny > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    spmi =sph_master_frame_new_empty();
    spmi->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    spmi->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    spmi->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    spmi->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);

    return spmi;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_master_frame from a cpl_image.

  @param self     the cpl_image to create the master frame from

  @return   a pointer to the newly created sph_master_frame or NULL if
            unsuccessfully.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_master_frame from the given cpl_image.
  No resources are shared between master_frame and input cpl_image
  so it is safe for example to call cpl_image_delete on the input
  image after creating the master_frame.

 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_master_frame_new_from_cpl_image( const cpl_image* self )
{
    sph_master_frame* result = NULL;

    if (self == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {

        const cpl_size  nx      = cpl_image_get_size_x(self);
        const cpl_size  ny      = cpl_image_get_size_y(self);

        const cpl_mask* badmask = cpl_image_get_bpm_const(self);

        cpl_msg_info(cpl_func, "Creating %d X %d master frame %s bad pixel map",
                     (int)nx, (int)ny, badmask ? "with" : "without");

        result = sph_master_frame_new_empty();

        result->image = cpl_image_cast(self, CPL_TYPE_DOUBLE);

        assert( result->image != NULL);

        result->ncombmap = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
#if 0
        if (badmask != NULL)
            cpl_image_reject_from_mask(result->ncombmap, badmask);
        cpl_image_add_scalar(result->ncombmap, 1.0);
#endif

        result->rmsmap   = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
#if 0
        if (badmask != NULL)
            cpl_image_reject_from_mask(result->rmsmap, badmask);
#endif

        result->badpixelmap = badmask
            ? cpl_image_new_from_mask(badmask)
            : cpl_image_new(nx, ny, CPL_TYPE_INT);
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Copy of the madpixelmap from the given sph_master_frame.

  @param self   the master frame which badpixelmap is to retrieve

  @return a pointer to the newly copied badpixelmap or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Create/return a new badpixelmap as copy of the badpixelmap from the
  input master frame.

 */
/*----------------------------------------------------------------------------*/
cpl_image* sph_master_frame_get_badpixelmap( const sph_master_frame* self) {

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);

    return self->badpixelmap ? cpl_image_duplicate( self->badpixelmap ) : NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Copy of the badpixelmap from the given sph_master_frame and return as float
            image.

  @param self   the master frame which badpixelmap is to retrieve

  @return a pointer to the newly copied badpixelmap or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Create/return a new badpixelmap as copy of the badpixelmap from the
  input master frame and return it as a CPL_FLOAT image.

 */
/*----------------------------------------------------------------------------*/
cpl_image* sph_master_frame_get_badpixelmap_float( const sph_master_frame* self) {

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);

    return self->badpixelmap 
        ? cpl_image_cast( self->badpixelmap, CPL_TYPE_FLOAT ) : NULL;

}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the number of badpixels

  @param self the master frame

  @return The number of badpixels or -1 on error

  @note     A CPL error is raised in case there was a problem.

  Create/return the number of badpixels.
 */
/*----------------------------------------------------------------------------*/
int sph_master_frame_get_nbads( const sph_master_frame* self) {

    int         nbads = 0;

    cpl_ensure(cpl_error_get_code()== CPL_ERROR_NONE, cpl_error_get_code(), -1);

    if ( self->badpixelmap ) {
    	// TODO:
    	// The below switch between INT and DOUBLE beahviour is a
    	// temporary hack - even though the badpixelmap type should be INT
    	// (presumably that was what llundin patch revision 2923 was about)
    	// various manipulations of master
    	// frame can change the badpixel map type to double
    	if ( cpl_image_get_type(self->badpixelmap) == CPL_TYPE_INT ) {
    		const int*   pim = cpl_image_get_data_int_const(self->badpixelmap);
    		if ( pim ) {
    			const size_t nxy = (size_t)(cpl_image_get_size_x(self->badpixelmap) *
    					cpl_image_get_size_y(self->badpixelmap));
    			size_t i;

    			for (i = 0; i < nxy; i++) {
    				if (pim[i] > 0) nbads++;
    			}
    		}
    	}
    	if ( cpl_image_get_type(self->badpixelmap) == CPL_TYPE_DOUBLE ) {
    		const double*   pim = cpl_image_get_data_double_const(self->badpixelmap);
    		if ( pim ) {
    			const size_t nxy = (size_t)(cpl_image_get_size_x(self->badpixelmap) *
    					cpl_image_get_size_y(self->badpixelmap));
    			size_t i;

    			for (i = 0; i < nxy; i++) {
    				if (pim[i] > 0.0) nbads++;
    			}
    		}
    	}
    }

    return nbads;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Interpolate the badpixels

  @param self the master frame

  @return Error code

  This function interpolates the bad pixels in the master frame image
  extension, rmsmap and ncombmap.
  The badpixel map of the master frame is constant to zero
  after successfull completion of this operation.

 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_master_frame_interpolate_bpix( sph_master_frame* self) {

    const cpl_size nxy = cpl_image_get_size_x(self->image) *
        cpl_image_get_size_y(self->image);
 
    const cpl_size nset = sph_master_frame_interpolate(self);

    cpl_msg_info(cpl_func, "Interpolated master frame bad pixel(s): %u/%u",
                 (unsigned)nset, (unsigned)nxy);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    return nset != nxy ? CPL_ERROR_NONE :
        cpl_error_set(cpl_func, CPL_ERROR_DATA_NOT_FOUND);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the badpixels as a cpl_mask

  @param self   the master frame which mask is to retrieve

  @return a pointer to the newly created mask NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Create/return a new mask as copy of the badpixelmap from the
  input master frame. In the mask

 */
/*----------------------------------------------------------------------------*/
cpl_mask* sph_master_frame_get_badpixelmask( const sph_master_frame* self) {

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);

    /* FIXME: 2048 ? */

    return self->badpixelmap
        ? cpl_mask_threshold_image_create(self->badpixelmap, 0.5, 2048.0 )
        : NULL;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Resets the badpixels and destroys rms info
 * @param self    the master frame
 *
 * @return  error code of the operation.
 *
 * Set all bixels and rms to 0 .
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_master_frame_accept_all( sph_master_frame* self ) {

    const size_t nxy = (size_t)(cpl_image_get_size_x(self->image) *
                                cpl_image_get_size_y(self->image));

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    cpl_image_accept_all( self->image);
    (void)memset(cpl_image_get_data_int(self->badpixelmap), 0,
                 nxy * sizeof(int));
    (void)memset(cpl_image_get_data_double(self->rmsmap), 0,
                 nxy * sizeof(double));

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set bad pixels from an input mask
 * @param self    the master frame
 * @param mask    the mask containing the bad pixels
 *
 * @return  error code of the operation.
 *
 * Set all bixels marked in the mask as badpixels.
 * This is an or combination to the already existing badpixels.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_master_frame_set_bads_from_mask( sph_master_frame* self,
                                     const cpl_mask* mask ) {

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    if (self == NULL) {
        return cpl_error_set(cpl_func,  CPL_ERROR_NULL_INPUT);
    } else if (mask == NULL) {
        return cpl_error_set(cpl_func,  CPL_ERROR_NULL_INPUT);
    } else if (cpl_image_get_type(self->badpixelmap) == CPL_TYPE_INT) {
        const size_t      nx   = (size_t)cpl_mask_get_size_x(mask);
        const size_t      ny   = (size_t)cpl_mask_get_size_y(mask);
        const cpl_binary* pmask = cpl_mask_get_data_const(mask);
        double*           prms  = cpl_image_get_data_double(self->rmsmap);
        int*              pbad  = self->badpixelmap == NULL ? NULL
            : cpl_image_get_data_int(self->badpixelmap);
        cpl_binary*       pbpm  =
            cpl_mask_get_data(cpl_image_get_bpm(self->image));
        /* FIXME: Make sure that no one sets a non-standard value in the mask */
        const cpl_binary* found =
            memchr(pmask, CPL_BINARY_1, nx * ny * sizeof(*pmask));


        cpl_ensure_code( (size_t)cpl_image_get_size_x(self->image) == nx,
                         CPL_ERROR_ILLEGAL_INPUT);
        cpl_ensure_code( (size_t)cpl_image_get_size_y(self->image) == ny,
                         CPL_ERROR_ILLEGAL_INPUT);

        while (found != NULL) {
            const size_t ij = found - pmask;

            pbpm[ij] = CPL_BINARY_1;
            if (pbad != NULL) pbad[ij] = 1;
            prms[ij] = SPH_MASTER_FRAME_BAD_RMS;

            found = memchr(found+1, CPL_BINARY_1,
                           (nx * ny - ij - 1) * sizeof(*pmask));
        }
    } else {
        /* FIXME: Oh, no who is using a non-integer cpl_image for bad pixels? */
        const cpl_size  nx = cpl_image_get_size_x(self->image);
        const cpl_size  ny = cpl_image_get_size_y(self->image);

        cpl_msg_warning(cpl_func, "Using-non-integer %s BPM (%d x %d) at %p",
                        cpl_type_get_name(cpl_image_get_type(self->badpixelmap)),
                        (int)nx, (int)ny,
                        (const void*)cpl_image_get_bpm_const(self->image));

        for (int xx = 0; xx < nx; ++xx) {
            for (int yy = 0; yy < ny; ++yy) {
                if ( cpl_mask_get(mask, xx + 1, yy + 1) ) {
                    cpl_image_set( self->badpixelmap, xx + 1, yy + 1, 1 );
                    cpl_image_reject( self->image, xx + 1, yy + 1);
                    cpl_image_set( self->rmsmap, xx + 1, yy + 1,
                                   SPH_MASTER_FRAME_BAD_RMS);
                }
            }
        }
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set bad pixels from an input mask
 * @param self    the master frame
 * @param mask    the mask containging the bad pixels
 *
 * @return  error code of the operation.
 *
 * Set all bixels marked in the mask as badpixels.
 * This is an or combination to the already existing badpixels.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_master_frame_set_bads_from_image( sph_master_frame* self,
                                      const cpl_image* mask ) {

   SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    if (self == NULL) {
        return cpl_error_set(cpl_func,  CPL_ERROR_NULL_INPUT);
    } else if (mask == NULL) {
        return cpl_error_set(cpl_func,  CPL_ERROR_NULL_INPUT);
    } else if (cpl_image_get_type(mask) == CPL_TYPE_INT) {
        const size_t    nx    = (size_t)cpl_image_get_size_x(mask);
        const size_t    ny    = (size_t)cpl_image_get_size_y(mask);

        const int*      pmask = cpl_image_get_data_int_const(mask);
        const cpl_mask* pmbpm = cpl_image_get_bpm_const(mask);

        double*         prms  = cpl_image_get_data_double(self->rmsmap);
        /* FIXME: Is a NULL bpm in Recipes/IFS/cutest_ifs_science a bug? */
        int*              pbad_int  = NULL;
        double*           pbad_double = NULL;
        cpl_binary*     pbpm  =
            cpl_mask_get_data(cpl_image_get_bpm(self->image));
        size_t ij;


        cpl_ensure_code( (size_t)cpl_image_get_size_x(self->image) == nx,
                         CPL_ERROR_ILLEGAL_INPUT);
        cpl_ensure_code( (size_t)cpl_image_get_size_y(self->image) == ny,
                         CPL_ERROR_ILLEGAL_INPUT);

        if ( cpl_image_get_type(self->badpixelmap) == CPL_TYPE_INT ) {
        	pbad_int = self->badpixelmap == NULL ? NULL
                : cpl_image_get_data_int(self->badpixelmap);
        }
        else { // badpixelmap either int or double type
        	assert(cpl_image_get_type(self->badpixelmap) == CPL_TYPE_DOUBLE);
        	pbad_double = self->badpixelmap == NULL ? NULL
                : cpl_image_get_data_double(self->badpixelmap);
        }
        if (pmbpm == NULL) {
            for (ij = 0; ij < nx * ny; ij++) {
                if (pmask[ij]) {
                    pbpm[ij] = CPL_BINARY_1;
                    if (pbad_int != NULL) pbad_int[ij] = 1;
                    if (pbad_double != NULL) pbad_double[ij] = 1;
                    prms[ij] = SPH_MASTER_FRAME_BAD_RMS;
                }
            }
        } else {
            /* FIXME: The mask image has a bpm! Is this even allowed???? */

            const cpl_binary* pmbin = cpl_mask_get_data_const(pmbpm);

            for (ij = 0; ij < nx * ny; ij++) {
                if (pmask[ij] || pmbin[ij]) {
                    pbpm[ij] = CPL_BINARY_1;
                    if (pbad_int != NULL) pbad_int[ij] = 1;
                    if (pbad_double != NULL) pbad_double[ij] = 1;
                    prms[ij] = SPH_MASTER_FRAME_BAD_RMS;
                }
            }
        }
    } else {
        /* FIXME: Oh, no who is using a non-integer cpl_image for bad pixels? */
        const cpl_size  nx = cpl_image_get_size_x(self->image);
        const cpl_size  ny = cpl_image_get_size_y(self->image);

        int xx = 0, yy = 0;
        int                 bp = 0;
        double              val = 0.0;

        cpl_msg_warning(cpl_func, "Using-non-integer %s BPM (%d x %d) at %p",
                        cpl_type_get_name(cpl_image_get_type(mask)),
                        (int)cpl_image_get_size_x(mask),
                        (int)cpl_image_get_size_y(mask),
                        (const void*)cpl_image_get_bpm_const(mask));

        if ( cpl_image_get_size_x(mask) != nx )
            return CPL_ERROR_ILLEGAL_INPUT;
        if ( cpl_image_get_size_y(mask) != ny )
            return CPL_ERROR_ILLEGAL_INPUT;

        for (yy = 0; yy < ny; ++yy) {
            for (xx = 0; xx < nx; ++xx) {
                val = cpl_image_get(mask, xx + 1, yy + 1, &bp);
                if (  val > 0|| bp > 0 ) {
                    cpl_image_set( self->badpixelmap, xx + 1, yy + 1, 1 );
                    cpl_image_reject( self->image, xx + 1, yy + 1);
                    cpl_image_set( self->rmsmap, xx + 1, yy + 1,
                                   SPH_MASTER_FRAME_BAD_RMS);
                }
            }
        }
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Set the the value of any bad pixel
 * @param self    The master frame
 * @param val     The value to set the bad pixels to
 *
 * @return the number of pixels reset, -1 on error.
 *
 * Set all values for bad pixels to the given value.
 * Also the bpm for the actual image is set (FIXME: Why ?)
 *
 */
/*----------------------------------------------------------------------------*/
int
sph_master_frame_set_bads( sph_master_frame* self, double val ) {

    const size_t      nxy     = (size_t)(cpl_image_get_size_x(self->image) *
                                         cpl_image_get_size_y(self->image));
    double*           pim     = cpl_image_get_data_double(self->image);
    const int*        pbad    = cpl_image_get_data_int_const(self->badpixelmap);
    const cpl_binary* pbadbpm = cpl_image_get_bpm_const(self->badpixelmap)
        ? cpl_mask_get_data_const(cpl_image_get_bpm_const(self->badpixelmap))
        : NULL;
    cpl_binary*       pbpmset = cpl_image_get_bpm_const(self->image)
        ? cpl_mask_get_data(cpl_image_get_bpm(self->image)) : NULL;
    const cpl_binary* pbpm = pbpmset; /* If NULL then stays so also w. cc > 0 */
    size_t i;
    int    cc = 0;

    if (pbad == NULL) {
        (void)cpl_error_set_where(cpl_func);
        return -1;
    }

    SPH_RAISE_CPL_RESET;

    /* Cannot fail now */
    for (i = 0; i < nxy; i++) {
        if (pbad[i] != 0 || (pbadbpm && pbadbpm[i]) || (pbpm && pbpm[i])) {
            pim[i] = val;
            if (pbpmset == NULL)
                pbpmset = cpl_mask_get_data(cpl_image_get_bpm(self->image));
            pbpmset[i] = CPL_BINARY_1;
            cc++;
        }
    }

    return cc;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Copy of the image from the given sph_master_frame.

  @param self   the master frame which badpixelmap is to retrieve
  @param rej_flag a flag to control if badpixels from the badpixel map should
                  flagged as rejected
  @return a pointer to the newly copied badpixelmap or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Create/return a new image as copy of the image from the
  input master frame. If rej_flag is one, badpixels will be
  marked rejected in the cpl_image returned.

 */
/*----------------------------------------------------------------------------*/
cpl_image* sph_master_frame_extract_image(const sph_master_frame* self,
                                          int rej_flag)
{
    cpl_image* image = NULL;

    cpl_ensure( self, CPL_ERROR_NULL_INPUT, NULL);

    if (self->image != NULL) {
        image = cpl_image_duplicate(self->image);
        if (self->badpixelmap && rej_flag) {
            cpl_mask* mask = sph_master_frame_get_badpixelmask(self);

            cpl_image_reject_from_mask(image, mask);
            cpl_mask_delete(mask);
        }
    }

    return image;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the RMS map to the poisson error from image

  @param self   the master frame
  @param rms_zeros  the rms value to assign for zero or neg. value pixels
  @param bad_zeros  if  not 0, zero or neg. values pixels are set as bad

  @return error code of the operation

  @note     A CPL error is raised in case there was a problem.

  This calculates the RMS map of the master frame as the poisson noise
  from the image. Specifically the rms for a pixels is set to the square root
  of the pixel value. Pixels with value 0 or negative values get assigned the
  value rms_zeros. The old RMS map is overwritten.
  If bad_zeros is set then any pixels which have zero or neg. values are set
  as bad pixels in this master_frame.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_master_frame_set_rms_poisson(sph_master_frame* self,
                                                double rms_zeros,
                                                int bad_zeros)
{
    if (self == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        const cpl_size  nx = cpl_image_get_size_x(self->image);
        const cpl_size  ny = cpl_image_get_size_y(self->image);
        int         xx  = 0;
        int         yy  = 0;
        double      val = 0;
        int         bpix = 0;

        for (yy = 0; yy < ny; ++yy) {
            for (xx = 0; xx < nx; ++xx) {
                bpix = 0;
                val = cpl_image_get( self->image, xx + 1, yy + 1,  &bpix);
                if ( val > 1.0 )
                    {
                        cpl_image_set(self->rmsmap, xx + 1, yy + 1, sqrt(val));
                    }
                else {
                    cpl_image_set(self->rmsmap, xx + 1, yy + 1, rms_zeros);
                    if ( bad_zeros ) {
                        cpl_image_set(self->badpixelmap, xx + 1, yy + 1, 1);
                        cpl_image_reject(self->image, xx + 1, yy + 1);
                    }
                }
            }
        }
    }
    return SPH_RAISE_CPL;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Sets the RMS from a weightmap.
 * @param self      the master frame
 * @param weightmap the weightmap
 * @param bad_zeros flag, set to 1 to mark all pixels with weight <=0 as bad
 *
 * @return error code
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_master_frame_set_rms_from_weightmap(sph_master_frame* self,
                                                       const cpl_image* wim,
                                                       int bad_zeros)
{
    if (self == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        const size_t  nx = cpl_image_get_size_x(self->image);
        const size_t  ny = cpl_image_get_size_y(self->image);
        const double* pwim = cpl_image_get_data_double_const(wim);
        double*       prms = cpl_image_get_data_double(self->rmsmap);
        int*          pbad = cpl_image_get_data_int(self->badpixelmap);
        cpl_binary*   pbpm = NULL;
        size_t        ii;

        SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

        cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
        cpl_ensure_code(wim,  CPL_ERROR_NULL_INPUT);
        cpl_ensure_code((size_t)cpl_image_get_size_x(wim) == nx,
                        CPL_ERROR_ILLEGAL_INPUT);
        cpl_ensure_code((size_t)cpl_image_get_size_y(wim) == ny,
                        CPL_ERROR_ILLEGAL_INPUT);

        for (ii = 0; ii < nx * ny; ii++) {
            const double val = pwim[ii];

            if (val > 0.0) {
                prms[ii] = sqrt(1.0 / val);
            } else {
                prms[ii] = SPH_MASTER_FRAME_BAD_RMS;
                if ( bad_zeros ) {
                    pbad[ii] = 1;
                    if (pbpm == NULL) {
                        pbpm = cpl_mask_get_data(cpl_image_get_bpm(self->image));
                    }
                    pbpm[ii] = CPL_BINARY_1;
                }
            }
        }
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The pixel RMS (standard deviation) map of the master frame.

  @param self   the master frame
  @return a pointer to the newly created RMS map or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Create/return a new rms image from the
  input master frame.

 */
/*----------------------------------------------------------------------------*/
cpl_image* sph_master_frame_get_rms( const sph_master_frame* self) {

    cpl_ensure( self, CPL_ERROR_NULL_INPUT, NULL);

    return self->rmsmap ? cpl_image_duplicate( self->rmsmap ) : NULL;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    The pixel variance map of the master frame.

  @param self   the master frame
  @return a pointer to the newly created RMS map or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Create/return a new rms image from the
  input master frame.

 */
/*----------------------------------------------------------------------------*/
cpl_image* sph_master_frame_get_variance( const sph_master_frame* self) {
    cpl_image* tmpim = NULL;
    cpl_image* tmpim2 = NULL;

    if ( self->rmsmap ) {
        tmpim = cpl_image_subtract_scalar_create(self->ncombmap,1.5);
        tmpim2 = cpl_image_subtract_scalar_create(self->ncombmap,1.0);
        cpl_image_multiply(tmpim,self->rmsmap);
        cpl_image_multiply(tmpim,self->rmsmap);
        cpl_image_divide(tmpim,tmpim2);
        cpl_image_delete(tmpim2);
    }
    return tmpim;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    The pixel weight map of the master frame.

  @param self   the master frame
  @return a pointer to the newly created weightmap or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Create/return a new weight image from the
  input master frame.

 */
/*----------------------------------------------------------------------------*/
cpl_image* sph_master_frame_get_weights( const sph_master_frame* self) {
    cpl_image* image = NULL;
    int         badrms_flag = 0;

    if ( self->rmsmap ) {
        const cpl_size nx = cpl_image_get_size_x(self->rmsmap);
        const cpl_size ny = cpl_image_get_size_y(self->rmsmap);
        int         xx  = 0;
        int         yy  = 0;
        double      rms = 0;
        int         bpix = 0;

        image = cpl_image_duplicate( self->rmsmap );

        for (yy = 0; yy < ny; ++yy) {
            for (xx = 0; xx < nx; ++xx) {
                bpix = 0;
                if ( cpl_image_get( self->badpixelmap, xx + 1, yy  + 1, &bpix) == 0 )
                {
                    rms = cpl_image_get( self->rmsmap, xx + 1, yy  + 1, &bpix);
                    if ( rms ) {
                        cpl_image_set(image, xx + 1, yy + 1, 1.0/(rms * rms));
                    }
                    else {
                        badrms_flag = 1;
                    }
                }
                else {
                    cpl_image_set(image, xx + 1, yy + 1, 0.0);
                }
            }
        }
    }
    if (badrms_flag) {
        cpl_image_delete(image);
        image = NULL;
        (void)cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    }

    return image;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Copy constructor function for sph_master_frame.

  @param self   the master frame to duplicate

  @return   a pointer to the newly created sph_master_frame or NULL if
            unsuccessfully.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_master_frame as copy of the input master frame.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_master_frame_duplicate( const sph_master_frame* self ) {

    sph_master_frame* spmi = sph_master_frame_new_empty();

    if ( self->image ) {
        spmi->image = cpl_image_duplicate( self->image );
    }
    if ( self->badpixelmap ) {
        spmi->badpixelmap = cpl_image_duplicate( self->badpixelmap );
    }
    if ( self->ncombmap ) {
        spmi->ncombmap = cpl_image_duplicate( self->ncombmap );
    }
    if ( self->rmsmap ) {
        spmi->rmsmap = cpl_image_duplicate( self->rmsmap );
    }
    if ( self->alg_parlist ) {
        spmi->alg_parlist = self->alg_parlist;
    }
    if ( self->qclist ) {
        if ( spmi->qclist ) {
            cpl_propertylist_delete( spmi->qclist );
        }
        spmi->qclist = cpl_propertylist_duplicate( self->qclist );
    }
    if ( self->properties ) {
        if ( spmi->properties ) {
            cpl_propertylist_delete( spmi->properties );
        }
        spmi->properties = cpl_propertylist_duplicate( self->properties );
    }
    return spmi;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief        Save the master frame as DFS product
 *
 * @param        self        the master frame to save
 * @param       czFilename  the filename to save under
 * @param       allframes    the frameset used to construct the product
 * @param       usedframes    the frameset used to construct the product
 * @param       template_frame  the frame to use as template for header
 * @param       param       the parameterlist used for product
 * @param       tag         the tag of the product
 * @param       recipe      the recipe creating this product
 * @param       pipename    the pipeline creating this product
 * @param        pl            a propertylist to append to main header (can be NULL)
 * @return        the error code
 *
 * Save the master frame as a FITS file according to ESO DFS.
 * This saves the master frame as a FITS
 * file with 4 extensions, one for the image, one for the badpixel map, one
 * for the number of pixels map and one for the rmsmap.
 * The propertylist pli is added to the propertylist of the main extension.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_master_frame_save_dfs(const sph_master_frame* self,
                                         const char* outfilename,
                                         cpl_frameset* allframes,
                                         const cpl_frame* template_frame,
                                         const cpl_parameterlist* params,
                                         const char* tag,
                                         const char* recipe,
                                         const char* pipename,
                                         const cpl_propertylist* plist)
{
    sph_error_code            rerr            = CPL_ERROR_NONE;
    cpl_propertylist*        mainpl            = NULL;
    cpl_propertylist*        tpli            = NULL;
    cpl_propertylist*        extpl            = NULL;
    int                        ii                = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    if ( !self ) {
        SPH_NO_SELF
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !self->image || !self->badpixelmap || !self->ncombmap || !self->rmsmap ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    if ( self->properties ) {
        mainpl = sph_keyword_manager_trans_keywords( self->properties );
        //extpl = sph_keyword_manager_trans_keywords( self->properties );
    }
    else {
        mainpl = cpl_propertylist_new();
        //extpl = cpl_propertylist_new();
    }

    if ( plist ) {
        tpli = sph_keyword_manager_trans_keywords( plist );
    }
    if ( tpli && mainpl ) {
        sph_utils_remove_wcs_3d(tpli);

        for (ii = 0; ii < cpl_propertylist_get_size( tpli ); ++ii) {
            if ( !cpl_propertylist_has( mainpl, cpl_property_get_name( cpl_propertylist_get( tpli, ii) ) ) ) {
                rerr = cpl_propertylist_append( mainpl, tpli );
            }
        }
    }
    if ( self->qclist ) {
        rerr = cpl_propertylist_append( mainpl, self->qclist );
    }
    rerr = cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_PRO_CATG, tag );
    cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_MASTER_FRAME );
    cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_IMAGE_EXTNAME );
    cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPNAME, SPH_QC_COMPNAME );
    cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPVERN, SPH_QC_COMPVERN );
    cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPREVN, SPH_QC_COMPREVN );
    cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPDATE, SPH_QC_COMPDATE );

    /* Update header if required */
    sph_utils_update_header(mainpl);

    rerr = cpl_dfs_save_image( allframes, NULL, params,
                               allframes, template_frame, self->image,
                               CPL_TYPE_FLOAT, recipe, mainpl, NULL, pipename,
                               outfilename );

    if ( rerr == CPL_ERROR_NONE ) {
    	if (extpl == NULL) 	extpl = cpl_propertylist_new();
        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_BADPIXMAP_EXTNAME );
        rerr = cpl_image_save( self->badpixelmap, outfilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );
        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_NCOMBMAP_EXTNAME );
        rerr = cpl_image_save( self->ncombmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_RMSMAP_EXTNAME );
        rerr = cpl_image_save( self->rmsmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    } else {
      	 SPH_ERR("Error raised: saving the first image from the master frame using cpl_dfs_save_image");
    }
    if ( mainpl ) {
        cpl_propertylist_delete( mainpl );
    }
    if ( tpli ) {
        cpl_propertylist_delete( tpli );
    }
    if ( extpl ) {
        cpl_propertylist_delete( extpl );
    }
    SPH_RAISE_CPL;
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        Save the master frame
 *
 * @param        self        the master frame to save
 * @param       czFilename  the filename to save under
 * @param        pl            a propertylist to append to main header (can be NULL)
 * @return        the error code
 *
 * Save the master frame as a FITS file. This saves the master frame as a FITS
 * file with 4 extensions, one for the image, one for the badpixel map, one
 * for the number of pixels map and one for the rmsmap.
 * The propertylist pli is added to the propertylist of the main extension.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_master_frame_save( const sph_master_frame* self,
                       const char* czFilename,
                       const cpl_propertylist* pli )
{
    return sph_master_frame_save_(self, czFilename, pli, CPL_IO_DEFAULT)
        ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief        Save the master frame, extending FITS file
 *
 * @param        self        the master frame to save
 * @param       czFilename  the filename to save under
 * @param        pl            a propertylist to append to main header (can be NULL)
 * @return        the error code
 *
 * Save the master frame, extending a FITS file. This saves the master frame as a FITS
 * file with 4 extensions, one for the image, one for the badpixel map, one
 * for the number of pixels map and one for the rmsmap. The FITS file
 * is assumed to exist and is extended.
 * The propertylist pli is added to the propertylist of the first new extension.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_master_frame_save_extend( const sph_master_frame* self,
                              const char* czFilename,
                              const cpl_propertylist* pli )
{
    return sph_master_frame_save_(self, czFilename, pli, CPL_IO_EXTEND)
        ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief load a master frame from a file
 *
 * @param    filename        the filename to load from
 * @param     plane            the plane to laod from ( 0..Nplanes)
 *
 * This function creates a new master frame and loads the data from a FITS
 * file with at least one plane and at least 4 extensions (the first four
 * extensions are read). This function does not check the format is actually
 * a sph_master_frame.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_master_frame_load( const char* czFilename, int plane ) {
    sph_master_frame*        result        = NULL;
    int                     nx          = 0;
    int                     ny          = 0;

    result = sph_master_frame_new_empty();

    if ( !result ) {
        return NULL;
    }
    result->image = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 0 );
    if ( !result->image ) {
        sph_master_frame_delete( result );
        return NULL;
    }
    if ( result->properties ) {
        cpl_propertylist_delete( result->properties ); result->properties = NULL;
    }
    result->properties = sph_keyword_manager_load_properties( czFilename, 0 );
    if ( result->qclist ) {
        cpl_propertylist_delete( result->qclist ); result->qclist = NULL;
    }
    result->qclist = cpl_propertylist_load_regexp( czFilename, 0, ".*QC.*", 0 );
    if ( result->qclist == NULL ) {
        cpl_error_reset();
    }
    nx = cpl_image_get_size_x( result->image );
    ny = cpl_image_get_size_y( result->image );
    result->badpixelmap = cpl_image_load( czFilename, CPL_TYPE_INT, plane, 1 );
    if ( !result->badpixelmap ) {
        cpl_error_reset();
        result->badpixelmap = cpl_image_new( nx, ny, CPL_TYPE_INT );
    }
    result->ncombmap = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 2 );
    if ( !result->ncombmap ) {
        cpl_error_reset();
        result->ncombmap = cpl_image_new( nx, ny, CPL_TYPE_DOUBLE );
    }
    result->rmsmap = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 3 );
    if ( !result->rmsmap ) {
        cpl_error_reset();
        result->rmsmap = cpl_image_new( nx, ny, CPL_TYPE_DOUBLE );
    }

    return result;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief load a master frame from a CPL frame
 * @param    self         The CPL frame to load from
 * @param    plane        The plane to laod from ( 0..Nplanes)
 *
 * This function creates a new master frame and loads the data from a FITS
 * file with at least one plane and at least 4 extensions (the first four
 * extensions are read). This function does not check the format is actually
 * a sph_master_frame.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_master_frame_load_( const cpl_frame* self, int nplane)
{
    const char*       file   = cpl_frame_get_filename(self);
    const char*       tag    = cpl_frame_get_tag(self);
    sph_master_frame* result;

    cpl_msg_info(cpl_func, "Loading plane %d from %s frame: %s", nplane,
                 tag, file);

    result = sph_master_frame_load(file, nplane);

    if (result == NULL) {
        (void)cpl_error_set_where(cpl_func);
    }

    return result;

}

/*----------------------------------------------------------------------------*/
/**
 *    @brief        get the median from a master_frame
 *
 *    @param        self the master_frame
 *    @param        stdev    a pointer to a double that when not NULL will be
 *                filled with the error on the median.
 *
 *    @return     the median of the master frame, taking badpixels into
 *                account
 *
 *    This function returns the median of the image, taking the badpixels into
 *    account.
 *
 */
/*----------------------------------------------------------------------------*/
double
sph_master_frame_get_median( const sph_master_frame* self, double* pstdev ) {
    double result = 0.0;

    if (self == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (self->image == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        sph_master_frame_reject_(self, CPL_FALSE);

        result = pstdev ?
            cpl_image_get_median_dev(self->image, pstdev) :
            cpl_image_get_median(self->image);
    }

    return result;
}


/*----------------------------------------------------------------------------*/
/**
 *    @brief        get the mean from a master_frame
 *
 *    @param        self the master_frame
 *    @param        stdev    a pointer to a double that when not NULL will be
 *                filled with the error on the mean.
 *
 *    @return     the mean of the master frame, taking badpixels into
 *                account
 *
 *    This function returns the mean of the image, taking the badpixels into
 *    account.
 *
 */
/*----------------------------------------------------------------------------*/
double
sph_master_frame_get_mean( const sph_master_frame* self, double* pstdev ) {
    double result = 0.0;

    if (self == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (self->image == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        sph_master_frame_reject_(self, CPL_FALSE);

        if (pstdev != NULL) {
            cpl_stats * imstat = cpl_stats_new_from_image(self->image,
                                                          CPL_STATS_MEAN |
                                                          CPL_STATS_STDEV);
            result = cpl_stats_get_mean(imstat);
            *pstdev = cpl_stats_get_stdev(imstat);
            cpl_stats_delete(imstat);
        } else {
            result = cpl_image_get_mean( self->image );
        }
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 *    @brief        get the mean RMS from a master_frame
 *
 *    @param        self the master_frame
 *    @param        stdev    a pointer to a double that when not NULL will be
 *                filled with the error on the mean RMS.
 *
 *    @return     the mean RMS of the master frame, taking badpixels into
 *                account
 *
 *    This function returns the mean of the rms map, taking the badpixels into
 *    account.
 *
 */
/*----------------------------------------------------------------------------*/
double
sph_master_frame_get_mean_variance( const sph_master_frame* self,
                                    double* pstdev ) {
    double            result            = 0;
    cpl_image*       tmpim = NULL;
    cpl_image*       tmpim2 = NULL;
    if ( !self ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !self->rmsmap ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    sph_master_frame_reject_(self, CPL_TRUE);

    tmpim = cpl_image_subtract_scalar_create(self->ncombmap,1.5);
    tmpim2 = cpl_image_subtract_scalar_create(self->ncombmap,1.0);
    cpl_image_multiply(tmpim,self->rmsmap);
    cpl_image_multiply(tmpim,self->rmsmap);
    cpl_image_divide(tmpim,tmpim2);
    result = cpl_image_get_mean( tmpim );
    if ( pstdev ) {
        *pstdev = cpl_image_get_stdev( self->rmsmap );
    }
    cpl_image_delete(tmpim);
    cpl_image_delete(tmpim2);
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 *    @brief        get the median RMS from a master_frame
 *
 *    @param        self the master_frame
 *    @param        stdev    a pointer to a double that when not NULL will be
 *                filled with the error on the mean RMS.
 *
 *    @return     the median RMS of the master frame, taking badpixels into
 *                account
 *
 *    This function returns the median of the rms map, taking the badpixels into
 *    account.
 *
 */
/*----------------------------------------------------------------------------*/
double
sph_master_frame_get_median_variance( const sph_master_frame* self,
                                      double* pstdev ) {
    double           result          = 0;
    double           ncomb           = 1.0;
    cpl_image*       tmpim = NULL;
    cpl_image*       tmpim2 = NULL;
    cpl_error_code   err;

    if ( !self ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !self->rmsmap ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    sph_master_frame_reject_(self, CPL_TRUE);

    tmpim = cpl_image_subtract_scalar_create(self->ncombmap,1.5);
    tmpim2 = cpl_image_subtract_scalar_create(self->ncombmap,1.0);
    ncomb = cpl_image_get_mean(self->ncombmap);
    err=cpl_image_multiply(tmpim,self->rmsmap);
    err=cpl_image_multiply(tmpim,self->rmsmap);
    err=cpl_image_divide(tmpim,tmpim2);
    if(err){
    	cpl_msg_error(cpl_func,"Failed to compute RMS!");
    	if (ncomb <2)
    		cpl_msg_error(cpl_func,"Cannot work on single frame files, NDIT must be > 1!");
    	}
    else {

    	result = cpl_image_get_median( tmpim );
    	if ( pstdev ) {
    		result = cpl_image_get_median_dev( tmpim, pstdev );
    	}
    	else {
    		result = cpl_image_get_median( tmpim );
    	}
    }
    cpl_image_delete(tmpim);
    cpl_image_delete(tmpim2);
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 *    @brief        get the mean RMS from a master_frame
 *
 *    @param        self the master_frame
 *    @param        stdev    a pointer to a double that when not NULL will be
 *                filled with the error on the mean RMS.
 *
 *    @return     the mean RMS of the master frame, taking badpixels into
 *                account
 *
 *    This function returns the mean of the rms map, taking the badpixels into
 *    account.
 *
 */
/*----------------------------------------------------------------------------*/
double
sph_master_frame_get_mean_rms( const sph_master_frame* self, double* pstdev ) {
    double result = 0.0;

    if (self == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (self->rmsmap == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        sph_master_frame_reject_(self, CPL_TRUE);

        if (pstdev != NULL) {
            cpl_stats * imstat = cpl_stats_new_from_image(self->rmsmap,
                                                          CPL_STATS_MEAN |
                                                          CPL_STATS_STDEV);
            result = cpl_stats_get_mean(imstat);
            *pstdev = cpl_stats_get_stdev(imstat);
            cpl_stats_delete(imstat);
        } else {
            result = cpl_image_get_mean( self->rmsmap );
        }
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 *    @brief        get the absolute flux from a master_frame
 *
 *    @param        self the master_frame
 *    @param        stdev    a pointer to a double that when not NULL will be
 *                filled with the error on the absflux.
 *
 *    @return     the absolute flux of the master frame, taking badpixels into
 *                account
 *
 *    This function returns the absolute flux of the image, taking the
 *
 */
/*----------------------------------------------------------------------------*/
double
sph_master_frame_get_absflux( const sph_master_frame* self, double* pstdev ) {
    double            result            = 0;
    if ( !self ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !self->image ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }

    sph_master_frame_reject_(self, CPL_FALSE);
    if (self->rmsmap && pstdev) {
        assert(cpl_image_get_bpm_const(self->image) != NULL);

        cpl_image_reject_from_mask(self->rmsmap,
                                   cpl_image_get_bpm_const(self->image));
    }

    SPH_RAISE_CPL
    result = cpl_image_get_absflux( self->image );
    SPH_RAISE_CPL
    if (self->rmsmap && pstdev) {
        cpl_image* timage = cpl_image_multiply_create(self->rmsmap,
                                                      self->rmsmap);
        *pstdev = sqrt( cpl_image_get_absflux( timage ) );
        cpl_image_delete( timage );
    }

    return result;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief        multiply a master frame with another master_frame
 *
 * @param        self        the master frame to multiply
 * @param        mframe      the master frame to multiply with
 *
 * @return        error code of the operation
 *
 * This function multiplies self with master frame mframe. Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_master_frame_multiply_master_frame( sph_master_frame* self,
        const sph_master_frame* mframe ){
	return sph_master_frame_div_or_mul_master_frame(self,
			mframe, CPL_TRUE);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        divide a master frame with another master_frame
 *
 * @param        self        the master frame to divide
 * @param        mframe      the master frame to divide with
 *
 * @return        error code of the operation
 *
 * This function divides self by master frame mframe. Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_master_frame_divide_master_frame( sph_master_frame* self,
        const sph_master_frame* mframe ){

	return sph_master_frame_div_or_mul_master_frame(self,
			mframe, CPL_FALSE);
}

sph_error_code sph_master_frame_multiply_double( sph_master_frame* self,
                                                 double value ) {
    sph_error_code    rerr;

    if ( self == NULL) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if ( self->image == NULL) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        rerr = cpl_image_multiply_scalar( self->image, value );
    }

    if ( rerr == CPL_ERROR_NONE && self->rmsmap ) {
        rerr = cpl_image_multiply_scalar( self->rmsmap, fabs(value) );
    }
    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}


sph_error_code
sph_master_frame_divide_double( sph_master_frame* self,
                                double value ) {
    sph_error_code    rerr;

    if ( self == NULL) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if ( self->image == NULL) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if ( value != 0.0 ) {
        rerr = cpl_image_divide_scalar( self->image, value );
    } else {
        return cpl_error_set(cpl_func, CPL_ERROR_DIVISION_BY_ZERO);
    }

    if ( rerr == CPL_ERROR_NONE && self->rmsmap ) {
        rerr = cpl_image_divide_scalar( self->rmsmap, fabs(value) );
    }
    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

sph_error_code sph_master_frame_sqrt( sph_master_frame* self ) {

    if ( !self || !self->image) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }

    sph_error_code    rerr            = CPL_ERROR_NONE;

	if ( !self ) {
		SPH_NULL_ERROR;
		return CPL_ERROR_NULL_INPUT;
	}

	const cpl_size nx = cpl_image_get_size_x(self->image);
	const cpl_size ny = cpl_image_get_size_y(self->image);

	for (int yy = 0; yy < ny; ++yy) {
		for (int xx = 0; xx < nx; ++xx) {

			int rej;
			double newval = cpl_image_get(self->image, xx + 1, yy + 1, &rej);
			int newbad = cpl_image_get(self->badpixelmap, xx + 1, yy + 1, &rej);
			double newrms = cpl_image_get(self->rmsmap, xx + 1, yy + 1, &rej);

			if(newbad) continue;
			/* newval == 0 generates valid value but not valid error */
			if ( newval > 0 ) {
				newval = sqrt(newval);
				newrms /=  2.0 * newval;
			}
			else {
				newval = 0.0; newbad = 1;
				newrms = SPH_MASTER_FRAME_BAD_RMS;
			}
			cpl_image_set(self->image, xx + 1, yy + 1, newval);
			cpl_image_set(self->badpixelmap, xx + 1, yy + 1, newbad);
			cpl_image_set(self->rmsmap, xx + 1, yy + 1, newrms);
		}
	}

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        add a master frame from another master_frame
 *
 * @param        self        the master frame to add to
 * @param        mframe        the master frame to add
 *
 * @return        error code of the operation
 *
 * This function adds master frame mframe to self. Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_master_frame_add_master_frame( sph_master_frame* self,
        const sph_master_frame* mframe )
{

    if ( !mframe || !self ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }

    if ( mframe->image && self->image ) {
        cpl_image_add( self->image, mframe->image );
    }
    if (mframe->badpixelmap && self->badpixelmap ) {
        cpl_image_add( self->badpixelmap, mframe->badpixelmap );
        cpl_image_threshold(self->badpixelmap, 0, 1, 0, 1);
    }
    if ( mframe->ncombmap && self->ncombmap ) {
        cpl_image_add( self->ncombmap, mframe->ncombmap );
    }
    if ( mframe->rmsmap && self->rmsmap ) {
        cpl_image_hypot(self->rmsmap, NULL, mframe->rmsmap);
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        The the log10 of a master frame
 *
 * @param        self        the master frame
 *
 * @return        error code of the operation
 *
 * This function takes the log10 of self. Self is changed
 * in place. Any value of which a log10 can not be taken either because
 * they are bad or negative or 0 will be set to 0 and marked
 * as bad.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_master_frame_take_log10( sph_master_frame* self )
{
    sph_error_code    rerr            = CPL_ERROR_NONE;
    int                xx                = 0;
    int                yy                = 0;
    double            newval            = 0.0;
    int                newbad            = 0;
    double             newrms            = 0;
    double            newweight        = 0;
    int                dum                = 0;
    cpl_size           nx, ny;

    if ( !self ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    nx = cpl_image_get_size_x(self->image);
    ny = cpl_image_get_size_y(self->image);

    for (yy = 0; yy < ny; ++yy) {
        for (xx = 0; xx < nx; ++xx) {
            newval = cpl_image_get(self->image, xx + 1, yy + 1, &dum);
            newbad = cpl_image_get(self->badpixelmap, xx + 1, yy + 1, &dum);
            newrms = cpl_image_get(self->rmsmap, xx + 1, yy + 1, &dum);
            newweight = cpl_image_get(self->ncombmap, xx + 1, yy + 1, &dum);

            if(newbad) continue;

            newrms = newrms / newval;
            if ( newval > 0 ) {
                newval = log10(newval);
                newrms =  newrms / CPL_MATH_LN10;
            }
            else {
                newval = 0.0; newbad = 1;
                newrms = SPH_MASTER_FRAME_BAD_RMS;
            }
            cpl_image_set(self->image, xx + 1, yy + 1, newval);
            cpl_image_set(self->badpixelmap, xx + 1, yy + 1, newbad);
            cpl_image_set(self->rmsmap, xx + 1, yy + 1, newrms);
            cpl_image_set(self->ncombmap, xx + 1, yy + 1, newweight);
        }
    }
    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief        Create a new master frame that is 1 over the old one
 *
 * @param        self        the master frame to invert
 *
 * @return        new master frame or NULL
 *
 * This create a new inverse of a master frame.
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_master_frame_inverse( const sph_master_frame* mframe )
{
    sph_master_frame*               self = NULL;
    int                             xx  = 0;
    int                             yy = 0;
    int                             bpix = 0;
    double                          val = 0.0;
    double                          rms = 0.0;
    cpl_size                        nx, ny;

    cpl_ensure(mframe,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(mframe->image,CPL_ERROR_ILLEGAL_INPUT,NULL);
    self = sph_master_frame_duplicate(mframe);
    cpl_ensure(self,CPL_ERROR_ILLEGAL_OUTPUT,NULL);
    cpl_ensure(self->image,CPL_ERROR_ILLEGAL_OUTPUT,NULL);

    nx = cpl_image_get_size_x(self->image);
    ny = cpl_image_get_size_y(self->image);

    for (yy = 0; yy < ny; ++yy) {
        for (xx = 0; xx < nx; ++xx) {
            if ( cpl_image_get(mframe->badpixelmap, xx + 1, yy + 1, &bpix) == 0) {
                val = cpl_image_get(mframe->image, xx + 1, yy + 1, &bpix);
                const double ncomb = cpl_image_get(mframe->ncombmap, xx + 1, yy + 1, &bpix);
                if ( val == 0.0 ) {
                    cpl_image_set(self->image, xx + 1, yy + 1, 0.0);
                    cpl_image_set(self->badpixelmap, xx + 1, yy + 1, 1);
                    cpl_image_set(self->rmsmap, xx + 1, yy + 1, SPH_MASTER_FRAME_BAD_RMS);
                    cpl_image_set(self->ncombmap, xx + 1, yy + 1, 0.0);
                }
                else {
                    rms = cpl_image_get(mframe->rmsmap, xx + 1, yy + 1, &bpix);
                    rms = rms / val;
                    cpl_image_set(self->image, xx + 1, yy + 1, 1.0/val);
                    cpl_image_set(self->rmsmap, xx + 1, yy + 1, 1.0/val * rms);
                    cpl_image_set(self->ncombmap, xx + 1, yy + 1, ncomb);
                }
            }
            else  {
                cpl_image_set(self->image, xx + 1, yy + 1, 0.0);
                cpl_image_set(self->badpixelmap, xx + 1, yy + 1, 1);
                cpl_image_set(self->rmsmap, xx + 1, yy + 1, SPH_MASTER_FRAME_BAD_RMS);
                cpl_image_set(self->ncombmap, xx + 1, yy + 1, 0.0);
            }
        }
    }

    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        Add a master frame to another master_frame creating new
 *
 * @param        self        the master frame to add to
 * @param        mframe        the master frame to add
 *
 * @return        new master frame or NULL
 *
 * This function adds master frame mframe to self. A new
 * master frame is created an returned. All inputs are unchanged.
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_master_frame_add_master_frame_create( const sph_master_frame* in,
        const sph_master_frame* mframe )
{
    sph_master_frame*                self = NULL;

    if ( !mframe || !in ) {
        SPH_NULL_ERROR;
        return NULL;
    }

    self = sph_master_frame_duplicate(in);

    if ( mframe->image && self->image ) {
        cpl_image_add( self->image, mframe->image );
    }
    if ( mframe->badpixelmap && self->badpixelmap ) {
        cpl_image_add( self->badpixelmap, mframe->badpixelmap );
    }
    if ( mframe->ncombmap && self->ncombmap ) {
        cpl_image_add( self->ncombmap, mframe->ncombmap );
    }
    if ( mframe->rmsmap && self->rmsmap ) {
        cpl_image_hypot(self->rmsmap, NULL, mframe->rmsmap);
    }
    return self;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief        subtract a master frame from another master_frame creating new
 *
 * @param        in        the master frame to subtract from
 * @param        mframe    the master frame to subtract
 *
 * @return        new master frame or NULL
 *
 * This function subtracts master frame mframe from self. A new
 * master frame is created an returned. All inputs are unchanged.
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_master_frame_subtract_master_frame_create( const sph_master_frame* in,
        const sph_master_frame* mframe )
{
    sph_master_frame*  self = NULL;

    if ( !mframe || !in ) {
        SPH_NULL_ERROR;
        return NULL;
    }

    self = sph_master_frame_duplicate(in);

    if ( mframe->image && self->image ) {
        cpl_image_subtract( self->image, mframe->image );
    }
    if ( mframe->badpixelmap && self->badpixelmap ) {
        cpl_image_add( self->badpixelmap, mframe->badpixelmap );
    }
    if ( mframe->ncombmap && self->ncombmap ) {
        cpl_image_add( self->ncombmap, mframe->ncombmap );
    }
    if ( mframe->rmsmap && self->rmsmap ) {
        cpl_image_hypot(self->rmsmap, NULL, mframe->rmsmap);
    }
    return self;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief        subtract a master frame from another master_frame
 *
 * @param        self        the master frame to subtract from
 * @param        mframe        the master frame to subtract
 *
 * @return        error code of the operation
 *
 * This function subtracts master frame mframe from self. Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_master_frame_subtract_master_frame( sph_master_frame* self,
                                        const sph_master_frame* mframe )
{

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    if ( !mframe || !self ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }

    if ( mframe->image && self->image ) {
        cpl_image_subtract( self->image, mframe->image );
    }
    if ( mframe->badpixelmap && self->badpixelmap ) {
        cpl_image_add( self->badpixelmap, mframe->badpixelmap );
    }
    if ( mframe->ncombmap && self->ncombmap ) {
        cpl_image_add( self->ncombmap, mframe->ncombmap );
    }
    if ( mframe->rmsmap && self->rmsmap ) {
        cpl_image_hypot(self->rmsmap, NULL, mframe->rmsmap);
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Smooth the master frame with a Gaussian of a specified width
 * @param self The master frame to smoothe in place
 * @param fwhm The Full Width at Half Maximum of the Gaussian to smoothe with
 * @return CPL_ERROR_NONE on success, otherwise the relevant CPL error code
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_master_frame_smooth( sph_master_frame* self, double fwhm) {

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    if (self == NULL) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (fwhm <= 0.0) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "FWHM = %g <= 0", fwhm);
    } else {
        const int      nbads = sph_master_frame_get_nbads(self);

        cpl_msg_info(cpl_func, "Smoothing image with %d bad pixel(s)",
                     nbads);

        if (sph_fft_smoothe_image(self->image, NULL, fwhm)) {
            return cpl_error_set_where(cpl_func);
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        extract 2 sub-master frames from another ZIMPOL master_frame
 *
 * @param        self        the master frame to extract from
 *
 * @return        an array of 2 master frames -- the subframes or NULL on error.
 *
 * This function extracts 2 master frames from the master frame self and
 * returns these as the two elements in an array of master frames. The
 * extraction is perfromed according the the method specified in the ZIMPOL
 * FDR document: even columns are in the first frame, odd ones in the second.
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame**
sph_master_frame_zpl_split( const sph_master_frame* self )
{
    sph_master_frame**    result        = NULL;
    int                nx                = 0;
    int             ny                 = 0;
    int                xx                = 0;
    int                yy                = 0;
    int                xnew            = 0;
    int                ynew            = 0;
    int                badpix            = 0;
    double            val                = 0.0;
    sph_master_frame* m                = NULL;

    result = cpl_calloc( 2, sizeof(sph_master_frame*) );

    if ( !self || !result ) {
        SPH_NULL_ERROR;
        return NULL;
    }
    if ( !self->image ) {
        SPH_NULL_ERROR;
        return NULL;
    }
    nx = cpl_image_get_size_x( self->image );
    ny = cpl_image_get_size_y( self->image );
    result[0] = sph_master_frame_new( nx / 2, ny );
    result[1] = sph_master_frame_new( nx / 2, ny );
    for ( yy = 0; yy < ny; ++yy ) {
        for ( xx = 0; xx < nx; ++xx ) {
            if ( xx % 2 == 0 ) {
                m = result[0];
                xnew = xx / 2;
                ynew = yy;
            }
            else {
                m = result[1];
                xnew = xx / 2;
                ynew = yy;
            }
            if ( self->image ) {
                val = cpl_image_get( self->image, xx + 1, yy + 1, &badpix );
                cpl_image_set( m->image, xnew + 1, ynew + 1, val );
            }
            if ( self->badpixelmap ) {
                val = cpl_image_get( self->badpixelmap, xx + 1, yy + 1, &badpix );
                cpl_image_set( m->badpixelmap, xnew + 1, ynew + 1, val );
            }
            if ( self->ncombmap ) {
                val = cpl_image_get( self->ncombmap, xx + 1, yy + 1, &badpix );
                cpl_image_set( m->ncombmap, xnew + 1, ynew + 1, val );
            }
            if ( self->rmsmap ) {
                val = cpl_image_get( self->rmsmap, xx + 1, yy + 1, &badpix );
                cpl_image_set( m->rmsmap, xnew + 1, ynew + 1, val );
            }

        }
    }

    return result;

}


/*----------------------------------------------------------------------------*/
/**
  @brief    Obtain mean and RMS by fitting a gaussian to value histogram.
  @param    self        the master frame
  @param    cpl_vector  a vector to fill with the bin positions
  @param    cpl_vecvtor a vector to fill with the bin values
  @param    minbin      minimum value to bin
  @param    maxbin      maximum value to bin
  @param    mean_guess  a guess value for the mean
  @param    mean        the mean (output)
  @param    rms         the rms (output)
  @param    redchi      the red. chisq (output)
  @return   the error code

  @note     A CPL error is raised in case there was a problem.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_master_frame_fit_gauss_hist( const sph_master_frame* self,
                                                cpl_vector* binpos,
                                                cpl_vector* binvals,
                                                double  minbin,
                                                double  maxbin,
                                                double* mean,
                                                double* rms,
                                                double* redchi )
{
    double      area = 0.0;
    double      offset = 0.0;
    double      mse = 0.0;
    int         xx  = 0;
    int         yy = 0;
    double      dx = 0.0;
    double      val = 0.0;
    double      ov = 0.0;
    int         bin = 0;
    int         bp = 0;
    int         nb = 0;
    cpl_size nx, ny;
    cpl_vector* sigmas = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    cpl_ensure_code(binpos && binvals && mean && rms, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( self->image, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( self->badpixelmap, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(minbin < maxbin, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(cpl_vector_get_size(binpos) == cpl_vector_get_size(binvals),
            CPL_ERROR_ILLEGAL_INPUT);

    nx = cpl_image_get_size_x(self->image);
    ny = cpl_image_get_size_y(self->image);

    nb = cpl_vector_get_size(binvals);
    dx = (maxbin - minbin) / nb;
    sigmas = cpl_vector_new(nb);
    for (xx = 0; xx < cpl_vector_get_size(binpos); ++xx) {
        cpl_vector_set(binpos,xx,dx * (xx + 0.5) + minbin );
        cpl_vector_set(binvals,xx,0.0 );
        cpl_vector_set(sigmas,xx,0.0 );
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    for (yy = 0; yy < ny; ++yy) {
        for (xx = 0; xx < nx; ++xx) {
            if ( cpl_image_get(self->badpixelmap, xx + 1, yy + 1, &bp) < 1 ) {
                val = cpl_image_get(self->image, xx + 1, yy + 1, &bp);
                if ( val >= minbin && val < maxbin ) {
                    bin = (int)floor(( val - minbin ) / dx);
                    if ( bin >= 0 && bin < nb ) {
                        ov = cpl_vector_get(binvals,bin);
                        cpl_vector_set(binvals,bin,ov+1.0);
                    }
                }
            }
        }
    }
    for (xx = 0; xx < cpl_vector_get_size(binpos); ++xx) {
        if ( cpl_vector_get(binvals,xx) >= 0.0 ) {
            cpl_vector_set(sigmas,xx,sqrt(cpl_vector_get(binvals,xx) + 1.0) );
        }
    }

    cpl_vector_fit_gaussian( binpos, NULL,  binvals, sigmas,
            CPL_FIT_ALL,
            mean, rms, &area, &offset, &mse, redchi, NULL );
    cpl_vector_delete(sigmas); sigmas = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Perform a quality control on the master frame.
  @param    self        the master frame
  @return   the error code

  @note     A CPL error is raised in case there was a problem.

  Performs a quality control calculation on the master frame. This means, that
  certain statistics are calculated and added to the qclist of the master frame.
  Currently these statistics are: Mean, median and rms of the image.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_master_frame_quality_check(sph_master_frame* self) {

    int        nbads;
    double     mean, median, stdev;
    cpl_stats* imstats;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    sph_master_frame_reject_(self, CPL_FALSE);

    nbads = sph_master_frame_get_nbads(self);

    imstats = cpl_stats_new_from_image(self->image,
                                       CPL_STATS_MEAN |
                                       CPL_STATS_MEDIAN |
                                       CPL_STATS_STDEV);

    mean   = cpl_stats_get_mean(imstats);
    median = cpl_stats_get_median(imstats);
    stdev  = cpl_stats_get_stdev(imstats);
    cpl_stats_delete(imstats);

    cpl_msg_info(cpl_func, "Quality check: mean=%g, median=%g, stdev=%g, "
                 "%d bad", mean, median, stdev, nbads);

    if (cpl_error_get_code()) {
        return cpl_error_set_where(cpl_func);
    }

    if (cpl_propertylist_update_double(self->qclist,
                                       SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME,
                                       mean)) {
        return cpl_error_set_where(cpl_func);
    }

    if (cpl_propertylist_update_double(self->qclist,
                                       SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME,
                                       median)) {
        return cpl_error_set_where(cpl_func);
    }

    if (cpl_propertylist_update_double(self->qclist,
                                       SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME,
                                       stdev)) {
        return cpl_error_set_where(cpl_func);
    }

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Create the bad pixel map for the master frame.
  @param    self        the master frame
  @param    lowtol      the lower bound for masking
  @param    uptol       the upper bound for masking
  @return   the error code

  @note     A CPL error is raised in case there was a problem.

  Create (calculate) the bad pixel map for the master frame.
  This is currently a very simple algorithm that sets the bad pixels
  to be all those that are above uptol or below lowtol.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_master_frame_mask_tolerance( sph_master_frame* self,
                                                double lowtol,
                                                double uptol )
{
    cpl_mask*            bad_mask    = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    bad_mask = cpl_mask_threshold_image_create( self->image,
                                                 lowtol,
                                                 uptol );
    cpl_ensure_code(bad_mask,cpl_error_get_code());
    cpl_mask_not(bad_mask);
    sph_master_frame_set_bads_from_mask(self,bad_mask);
    cpl_mask_delete(bad_mask); bad_mask = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Create the bad pixel map for the master frame.
  @param    self        the master frame
  @param    sigma       the sigma level for masking
  @return   the error code

  @note     A CPL error is raised in case there was a problem.

  Create (calculate) the bad pixel map for the master frame.
  This is currently a very simple algorithm that sets the bad pixels
  to be all those that are further than sigma * the total RMS of the whole
  master frame away from the median of the master frame.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_master_frame_mask_sigma( sph_master_frame* self,
                                            double sigma )
{
    int                     rerr         = CPL_ERROR_NONE;
    double                median        = 0.0;
    double                 rms            = 0.0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    rerr = cpl_image_accept_all( self->image );

    median     = cpl_propertylist_get_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME );
    rms        = cpl_propertylist_get_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME );
    if ( cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND ) {
        cpl_error_reset();
        sph_master_frame_quality_check(self);
        median     = cpl_propertylist_get_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME );
        rms        = cpl_propertylist_get_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME );
        SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    }
    if ( sigma == 0 || rms == 0 ) {
        sph_error_raise( SPH_ERROR_GENERAL,
                __FILE__, __func__, __LINE__, SPH_ERROR_WARNING,
                "Masking with a sigma of zero or on a perfectly flat image"
                "will mask all pixels as bad. "
                "I doubt thats whats intended -- so i will not mask any"
                "instead.");
        return CPL_ERROR_NONE;
    }

    rerr = sph_master_frame_mask_tolerance( self, median - sigma * rms,
                                           median + sigma * rms );
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Calculate a signal to noise map.
  @param    self   the master frame to calculate the SN map from.

  @return   pointer to the new cpl_image, or NULL on error.

  @note     A CPL error is raised in case there was a problem.

  This function calculate an image where every pixel gives the signal to noise
  at the given position, which is calculated from the image and RMSMAP parts
  of the sph_master_frame. \f$S_i = I_i / W_i\f$. Where the weight is zero, the
  resulting image gets assigned the value 0 but the badpixel flag is set in the
  cpl_image badpixel map.
 */
/*----------------------------------------------------------------------------*/
cpl_image* sph_master_frame_sn_map( const sph_master_frame* self)
{
    cpl_image*      result = NULL;
    cpl_mask*       mask   = NULL;
    int             rerr   = CPL_ERROR_NONE;
    if ( !self ) {
        SPH_NO_SELF;
        return NULL;
    }

    if ( !self->rmsmap || !self->image ) {
        sph_error_raise( CPL_ERROR_ILLEGAL_INPUT,
                         __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR, "The image or RMSMAP in "
                                 "the master frame was not set -- so "
                                 "cant calculate the SN map.");
        return NULL;
    }

    mask = cpl_mask_threshold_image_create(self->rmsmap, 0.0, 0.0);

    if ( !mask ) {
        sph_error_raise(cpl_error_get_code(), __FILE__, __func__, __LINE__, SPH_ERROR_ERROR, "Could not create the mask "
                "-- so bad pixels may"
                "not be correctly identified.\nCPL says: %s", cpl_error_get_message());
    }

    result = cpl_image_divide_create(self->image, self->rmsmap);
    if ( !result ) {
        sph_error_raise(cpl_error_get_code(), __FILE__, __func__, __LINE__, SPH_ERROR_ERROR, "Could not create the image "
                "-- CPL says: %s", cpl_error_get_message() );
        return NULL;
    }

    if ( mask ) {
        rerr = cpl_image_reject_from_mask(result, mask);
        if ( rerr ) {
            sph_error_raise(cpl_error_get_code(), __FILE__, __func__, __LINE__, SPH_ERROR_ERROR, "Could not use mask "
                    "-- CPL says: %s", cpl_error_get_message() );
            return NULL;
        }
    }

    return result;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete the master image.
  @param    self   The master image to delete or NULL to do nothing

  Deallocate the master image. The associated image, bad pixel map, RMSMAP, QC list,
  number of comb. frames map all get deallocated with it. The algorithm parameterlist
  does NOT get deallocated.
 */
/*----------------------------------------------------------------------------*/
void sph_master_frame_delete( sph_master_frame* self ) {
    if ( self != NULL ) {
        cpl_image_delete(self->badpixelmap);
        cpl_image_delete(self->rmsmap);
        cpl_image_delete(self->ncombmap);
        cpl_propertylist_delete(self->qclist);
        cpl_propertylist_delete(self->properties);
        cpl_image_delete(self->image);
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief  Use the badpixel image to reject pixels in the specified image member
  @param   self    The master frame to modify
  @param   is_rms  Whether to modify the rmsmap, otherwise the main image
  @note FIXME: This operation should never be necessary! Also, const violation!
 */
/*----------------------------------------------------------------------------*/
static void sph_master_frame_reject_(const sph_master_frame* self,
                                     cpl_boolean is_rms)
{
    cpl_image* imuse = is_rms ? self->rmsmap : self->image;

    if (self->badpixelmap != NULL && imuse != NULL) {
        const cpl_size nx = cpl_image_get_size_x(self->badpixelmap);
        const cpl_size ny = cpl_image_get_size_y(self->badpixelmap);
        cpl_mask * bpm = cpl_image_get_bpm_const(imuse)
            ? cpl_image_unset_bpm(imuse)
            : cpl_mask_wrap(nx, ny, cpl_malloc(nx * ny *
                                               sizeof(cpl_binary)));
#ifndef NDEBUG
        const cpl_error_code code =
#endif
            cpl_mask_threshold_image(bpm, self->badpixelmap, -0.5, 0.5,
                                     CPL_BINARY_0);

        assert(bpm != NULL);
        assert(cpl_image_get_size_x(imuse) == nx);
        assert(cpl_image_get_size_y(imuse) == ny);
        assert(cpl_image_get_bpm_const(imuse) == NULL);
        assert(!code); /* Cannot fail */

        bpm = cpl_image_set_bpm(imuse, bpm);
        assert( bpm == NULL);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        divide/multiply a master frame with another master_frame
 *
 * @param        self        the master frame to divide/multiply
 * @param        mframe      the master frame to divide/multiply by
 * @param        is_mul      if TRUE a multiplication is performed;
 * 							 division otherwise
 *
 * @return        error code of the operation
 *
 * This function divides/multiplies self with master frame mframe. Self is changed
 * in place. This function centralizes the error propagation logic for * and / .
 *
 */
/*----------------------------------------------------------------------------*/
static sph_error_code
sph_master_frame_div_or_mul_master_frame( sph_master_frame* self,
        const sph_master_frame* mframe, const cpl_boolean is_mul)
{

	cpl_error_code (*op)(cpl_image *, const cpl_image * ) = cpl_image_multiply;

	if(!is_mul)
		op = cpl_image_divide;

    cpl_size nx, ny;
    cpl_error_code code = CPL_ERROR_NONE;

    if ( !mframe || !self ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }

    nx = cpl_image_get_size_x(self->image);
    ny = cpl_image_get_size_y(self->image);

    cpl_ensure_code(cpl_image_get_size_x(mframe->image) == nx,
                    CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_ensure_code(cpl_image_get_size_y(mframe->image) == ny,
                    CPL_ERROR_INCOMPATIBLE_INPUT);

    if ( mframe->badpixelmap && self->badpixelmap ) {
        code |= cpl_image_add( self->badpixelmap, mframe->badpixelmap );
    }
    if ( mframe->ncombmap && self->ncombmap ) {
         code |= cpl_image_add( self->ncombmap, mframe->ncombmap );
    }
    if ( mframe->rmsmap && self->rmsmap ) {

        cpl_mask * tmpmask;

        cpl_mask * bpmrmsmap;
        cpl_mask * bpmrmsmap2;
        cpl_mask * bpmbadmap = NULL;
        cpl_mask * bpm;

        cpl_image * rmsmap2;


        /* Perform the pre-division part of the error propagation
           - the BPMs may inherit bad pixels and get new ones from 0-division */
        code |= cpl_image_divide(self->rmsmap, self->image);
        rmsmap2 = cpl_image_divide_create(mframe->rmsmap, mframe->image);

        code |= cpl_image_hypot(self->rmsmap, NULL, rmsmap2);

        /* Perform the actual division / multiplication*/
        code |= op(self->image, mframe->image);

        /* Perform the post-division part of the error propagation */
        code |= cpl_image_multiply(self->rmsmap, self->image);

        code |= cpl_image_abs(self->rmsmap);

        /* Update the badpixelmap. FIXME: what a pain:
           Need to or with the self badpixelmap (thus also of the mframe one)
           Need to or with each of the two rmsmaps's (and thus also the images)
           Need to clear the BPM's ???
        */

        /* First: collect any additional bad pixels (from 0-division)
           onto any pre-existing BPM - or a created one */
        bpmrmsmap = cpl_image_get_bpm_const(self->rmsmap) == NULL ? NULL
            : cpl_image_get_bpm(self->rmsmap);
        bpmrmsmap2 = cpl_image_get_bpm_const(rmsmap2) == NULL ? NULL
            : cpl_image_get_bpm(rmsmap2);

        bpm = bpmrmsmap ? bpmrmsmap : (bpmrmsmap2 ? bpmrmsmap2 : NULL);

        if (bpm == NULL) {
            bpm = bpmbadmap = cpl_mask_new(nx, ny);
        }

        cpl_mask * bpm_from_map = cpl_mask_new(nx, ny);
        code |= cpl_mask_threshold_image(bpm_from_map, self->badpixelmap,
                                         0.5, 2048.0, CPL_BINARY_1);
        cpl_mask_or(bpm, bpm_from_map);
        cpl_mask_delete(bpm_from_map);

        if (bpmrmsmap != NULL && bpm != bpmrmsmap)
            code |= cpl_mask_or(bpm, bpmrmsmap);
        if (bpmrmsmap2 != NULL && bpm != bpmrmsmap2)
            code |= cpl_mask_or(bpm, bpmrmsmap2);

        /* Second: On any bad pixel set the bad-pixel-value (1) on the BPM */
        tmpmask = cpl_image_set_bpm(self->badpixelmap, bpm);
        code |= cpl_image_fill_rejected(self->badpixelmap, 1.0);
        (void)cpl_image_set_bpm(self->badpixelmap, tmpmask);

        /* Second: On any bad pixel set the bad-pixel-value on the RMS map */
        tmpmask = cpl_image_set_bpm(self->rmsmap, bpm);
        (void)cpl_image_set_bpm(self->rmsmap, bpm);
        code |= cpl_image_fill_rejected(self->rmsmap, SPH_MASTER_FRAME_BAD_RMS);
        (void)cpl_image_set_bpm(self->rmsmap, tmpmask);

        cpl_mask_delete(bpmbadmap);

        cpl_image_delete(rmsmap2);

#if 0
        /* BPM checking:
           1) badpixelmap (of self and redundantly mframe)
           2) bpm of image (but only of mframe, not of self!)
           3) A near-0-division with mframe image
           4) A 0-division with mframe image (no BPM, just use BAD_RMS value)
           5) A 0-division with self image - in this case the RMS is _not_ updated
        */
        for (yy = 0; yy < cpl_image_get_size_y(self->rmsmap); ++yy) {
            for (xx = 0; xx < cpl_image_get_size_x(self->rmsmap); ++xx) {
                bpix = 0;
                isbad = 0;
                rms = rms2 = 0.0;
                val = cpl_image_get(self->badpixelmap, xx + 1, yy + 1, &bpix);
                if ( val >= 1 ) isbad = 1;
                val = cpl_image_get(mframe->badpixelmap, xx + 1, yy + 1, &bpix);
                if ( val >= 1 ) isbad = 1;
                val2 = cpl_image_get(mframe->image, xx + 1, yy + 1, &bpix);
                if ( val2 < FLT_MIN && val2 > -FLT_MIN ) isbad = 1;
                if ( bpix || isbad ) {
                    cpl_image_set(self->badpixelmap, xx + 1, yy + 1, 1);
                    cpl_image_set(self->rmsmap, xx + 1, yy + 1, SPH_MASTER_FRAME_BAD_RMS);
                    cpl_image_set(self->image, xx + 1, yy + 1, 0.0);
                }
                else {
                    rms = cpl_image_get(self->rmsmap, xx + 1, yy + 1, &bpix);
                    val = cpl_image_get(self->image, xx + 1, yy + 1, &bpix);
                    rms2 = cpl_image_get(mframe->rmsmap, xx + 1, yy + 1, &bpix);
                    if ( val != 0.0 ) {
                        rms = rms / val;
                        if (val2 != 0.0 ) rms2 = rms2 / val2;
                        else rms2 = SPH_MASTER_FRAME_BAD_RMS;
                        cpl_image_set(self->rmsmap, xx + 1, yy + 1, sqrt(rms * rms + rms2 * rms2) * val/val2);
                        cpl_image_set(self->image, xx + 1, yy + 1, val/val2);
                    }
                }
            }
        }
#endif
    } else {
        code |= op(self->image, mframe->image);
    }

    return code ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;

}


/*----------------------------------------------------------------------------*/
/**
 * @brief    Median filter a master frame
 * @param    self   The master frame to filter
 * @param    radius The non-negative radius, window size is 1 + 2 times radius
 * @return   On success CPL_ERROR_NONE, otherwise the relevant CPL error code
 * @note The RMS map et al must be filtered too, see HDRL manual
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_master_frame_filter_median(sph_master_frame* self,
                                              cpl_size radius)
{
    cpl_error_code code;
    const cpl_size wsize = 1 + 2 * radius;
    cpl_mask * kernel = NULL;
    cpl_image* other = NULL;

    cpl_ensure_code(self != NULL,        CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->image != NULL, CPL_ERROR_INVALID_TYPE);
    cpl_ensure_code(radius >= 0,         CPL_ERROR_ILLEGAL_INPUT);

    kernel = cpl_mask_new(wsize, wsize);
    code = cpl_mask_not(kernel);
    if (!code) {

        other = cpl_image_duplicate(self->image);

        code = cpl_image_filter_mask(other, self->image, kernel, CPL_FILTER_MEDIAN,
                                     CPL_BORDER_FILTER);

        if (!code) {
            code = cpl_image_subtract(self->image, other);

            if (!code)
                code =
                    cpl_image_threshold(self->image, 0.0, DBL_MAX, 0.0, DBL_MAX);
        }
    }

    cpl_mask_delete(kernel);
    cpl_image_delete(other);

    return code ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/**@}*/
