/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/*
 * sph_phot.c
 *
 *  Created on: Jul 16, 2016
 *      Author: pavlov
 */


/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_phot.h"
#include "sph_strehl.h"
#include "sph_common_keywords.h"

#include "irplib_stdstar.h"
#include "irplib_utils.h"

#include <assert.h>
#include <string.h>
#include <math.h>

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/* Neutral Density filter name */
#define SPH_PHOT_ND_FILTER_IRDIS_KEY "ESO INS4 FILT2 NAME"
/* Common filter for ZIMPOL that can act as a Neutral density filter
 * when NOT in Open state*/
#define SPH_PHOT_ND_FILTER_ZIMPOL_KEY "ESO INS3 OPTI2 NAME"

/* A tolerence of 10 Arcseconds was suggested by S. Moehler */
#ifndef SPH_PHOT_STDSTAR_TABLE_MAX_DIST
#define SPH_PHOT_STDSTAR_TABLE_MAX_DIST 10.0
#endif

#define SPH_ZPL_V_FILTER "V"
#define SPH_ZPL_I_FILTER "N_I"
#define SPH_ZPL_R_FILTER "N_R"

/*-----------------------------------------------------------------------------
                                   Data Types
 -----------------------------------------------------------------------------*/
typedef struct {

	double zpoint;
	double zpoint_corrected;

}sph_phot_zeropoint;
/*-----------------------------------------------------------------------------
                                   Function declarations
 -----------------------------------------------------------------------------*/

static cpl_error_code
sph_phot_extract_nd_corr(const char * nd_string,
		const cpl_frame * filterframe,
		const char * tfilter_key, double *zpcorr);

static cpl_error_code
sph_phot_correct_nd(double zmag,
					const char* tfilter_key,
					const char* nd_string,
					const cpl_frame* filterframe,
					double * out_zmag_corrected);

static
cpl_error_code sph_phot_read_header(const cpl_propertylist * pl,
									const sph_arm            arm,
									const char             * dit_key,
									const char            ** tfilter_key,
									const char            ** tstdstar_key,
									double                 * dit,
									double                 * ra,
									double                 * dec );

static cpl_error_code
sph_phot_no_list_allocation(double flux,
		const cpl_propertylist* readpl,
		const cpl_frame* starframe,
		const cpl_frame* filterframe,
		const sph_arm    arm,
	    const char * dit_key,
		sph_phot_zeropoint * zp_calculated);

static const char *
get_nd_filter_key(const sph_arm arm);

static cpl_error_code
sph_phot_core(const double flux,
   			  const cpl_propertylist* pl_frame,
			  const cpl_frame* inframe,
			  const cpl_frame* starframe,
			  const cpl_frame* filterframe,
			  const sph_arm    arm,
			  const char * dit_key,
			  sph_phot_zeropoint * zp_calculated);

static void
sph_phot_write_qc(cpl_propertylist * plist, const sph_phot_zeropoint * zp,
			const char * zp_key, const char * zp_comment,
			const char * zp_corr_key, const char * zp_corr_comment);

static const char * sph_phot_read_arm_filter(
								const cpl_propertylist* pl,
								const sph_arm arm);

static cpl_error_code sph_phot_compute_zero_point(const double flux,
												  const double starmag,
												  const double dit,
                                                  double * zmag);

static cpl_error_code
sph_phot_read_airmass(const cpl_propertylist* pl_strehl, double * ms);

static cpl_error_code
sph_phot_correct_for_airmass(const char * filter_key,
					         const sph_arm arm,
							 const double airmass,
							 double * zmag);

static const char*
sph_phot_read_nd_header(const cpl_propertylist * pl,
						const sph_arm            arm);
/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief  Get the star magnitude from the catalogue, using its name or position
 @param  stdcat_frame The standard star catalogue frame
 @param  target_name  The name of the star
 @param  filter_name  The name of the filter
 @param  starmag      The star magnitude to look up
 @param  ra           Right ascension, or negative if absent [degrees]
 @param  dec          Declination [degrees]
 @return CPL_ERROR_NONE on success, or else the relevant CPL error

 */
/*----------------------------------------------------------------------------*/
static cpl_error_code sph_phot_read_star( const cpl_frame* stdcat_frame,
                                          const char* target_name,
                                          const char* filter_name,
                                          double* starmag,
                                          double ra, double dec) {

    cpl_errorstate okstate      = cpl_errorstate_get();
    const char   * stdcat_file  = cpl_frame_get_filename(stdcat_frame);
    cpl_table    * stdcat_table = cpl_table_load(stdcat_file, 1, 0);
    const cpl_size nrow         = cpl_table_get_nrow(stdcat_table);
    /* select rows not matching the target_name */
    const cpl_size nsel         =
        cpl_table_and_selected_string( stdcat_table,
                                       SPH_PHOT_STDSTAR_TABLE_TARGET_KEY,
                                       CPL_NOT_EQUAL_TO, target_name );

    /* Matching row by erasing all not matching ones */
    const cpl_size nunsel       = nrow - nsel;

    cpl_ensure_code(stdcat_frame != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(target_name  != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filter_name  != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(starmag      != NULL, CPL_ERROR_NULL_INPUT);
    if (stdcat_table == NULL) return cpl_error_set_where(cpl_func);

    if (!cpl_table_has_column(stdcat_table, filter_name)) {
        cpl_table_delete(stdcat_table);
        return cpl_error_set_message(cpl_func, CPL_ERROR_UNSUPPORTED_MODE,
                                     "Catalogue %s does not have a column for "
                                     "filter: %s", stdcat_file, filter_name);
    }

    if (nsel < 0) {
        cpl_table_delete(stdcat_table);
        return cpl_error_set_where(cpl_func);
    }

    if (nunsel < 1) {
        int inear = -1;

        if (ra >= 0.0) { /* RA could be absent, indicated by negative value */
            const double max_dist = SPH_PHOT_STDSTAR_TABLE_MAX_DIST / 3600.0;
            int nnear;

            assert( nsel == nrow ); /* No rows were deselected by target_name */

            if (irplib_stdstar_select_stars_dist(stdcat_table, ra, dec,
                                                 max_dist)) {
                cpl_table_delete(stdcat_table);
                return cpl_error_set(cpl_func, CPL_ERROR_UNSPECIFIED);
            }
            nnear = (int)cpl_table_count_selected(stdcat_table);

            cpl_msg_warning(cpl_func, "starfile=%s (%d row(s)). target_name=%s:"
                            " no match, trying %d closest to (%g,%g) within "
                            "[Arcseconds]: %g", stdcat_file, (int)nrow,
                            target_name, nnear, ra, dec,
                            SPH_PHOT_STDSTAR_TABLE_MAX_DIST);

            inear = irplib_stdstar_find_closest(stdcat_table, ra, dec);
        }
        if (inear < 0) {
            cpl_table_delete(stdcat_table);
            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                  "starfile=%s. target_name=%s: no match",
                                  stdcat_file, target_name);
        }

        cpl_table_select_all(stdcat_table);
        cpl_table_unselect_row(stdcat_table, inear);
    } else if (nunsel > 1) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "starfile=%s. target_name=%s.:%u of %u "
                                     "rows match. star should be unique!",
                                     stdcat_file, target_name, (unsigned)nunsel,
                                     (unsigned)nrow);
    }

    if (cpl_table_erase_selected(stdcat_table)) {
        cpl_table_delete(stdcat_table);
        return cpl_error_set_where(cpl_func);
    }

    /* retrieve magnitude of the star for given filter_name */
    *starmag = cpl_table_get_double( stdcat_table, filter_name, 0, NULL );
    cpl_table_delete(stdcat_table);

    if (!cpl_errorstate_is_equal(okstate)) return cpl_error_set_where(cpl_func);

    if (!isnormal(*starmag)) return
        cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND, "Catalog %s "
                              "with %d row(s) is missing a star magnitude (%g) "
                              "for star=%s at (%g,%g) for filter: %s",
                              stdcat_file, (int)nrow, *starmag, target_name,
                              ra, dec, filter_name);
    if (*starmag <= 0.0) return
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Catalog %s "
                              "with %d row(s) has a bad star magnitude (%g) "
                              "for star=%s at (%g,%g) for filter: %s",
                              stdcat_file, (int)nrow, *starmag, target_name,
                              ra, dec, filter_name);

    return CPL_ERROR_NONE;
}

#ifdef SPH_PHOT_READ_STAR_CHECK
static double sph_phot_read_star_check( const cpl_frame* starframe,
                                          const char* target_name,
                                          const char* filter_name ){

    cpl_errorstate okstate  = cpl_errorstate_get();
    const char   * filename = cpl_frame_get_filename(starframe);
    cpl_table    * table    = cpl_table_load(filename, 1, 0);
    const cpl_size nrow     = cpl_table_get_nrow(table);
    cpl_size       nsel;
    double starmag = -999.0;

    cpl_ensure_code(starframe    != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(target_name  != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filter_name  != NULL, CPL_ERROR_NULL_INPUT);

    if (table == NULL) {
        cpl_error_set_where(cpl_func);
        return starmag;
    }

    /* select rows not matching the target_name */
    nsel = cpl_table_and_selected_string( table, SPH_PHOT_STDSTAR_TABLE_TARGET_KEY, CPL_NOT_EQUAL_TO, target_name );
    if (nsel < 0) {
        cpl_error_set_where(cpl_func);
        return starmag;
    }

    /* Matching row by erasing all not matching ones */
    nsel = nrow-nsel;

    if (nsel < 1) {
        cpl_msg_info(cpl_func, "starfile=%s. "
                              "target_name=%s: no matching", filename, target_name);
        return starmag;
    }
    if (nsel > 1) {
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "starfile=%s. target_name=%s.:"
                              "%u of %u rows match. star should be unique!",
                              filename, target_name, (unsigned)nsel, (unsigned)nrow);
        return starmag;
    }
    if (cpl_table_erase_selected(table)){
        cpl_error_set_where(cpl_func);
        return starmag;
    }

    /* retrieve magnitude of the star for given filter_name */
    starmag = cpl_table_get_double( table, filter_name, 0, NULL );
    cpl_table_delete(table);

    if (!cpl_errorstate_is_equal(okstate)){
        cpl_error_set_where(cpl_func);
        return starmag;
    }

    if (starmag <= 0.0){
        cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND, "starfile=%s."
                              "target_name=%s. filter_name=%s. %u row(s). Star "
                              "Mag =%g <= 0.", filename, target_name,
                              filter_name, (unsigned)nrow, starmag);
    }

    return starmag;
}
#endif



static const char * sph_phot_read_arm_filter(
								const cpl_propertylist* pl,
								const sph_arm arm){

    const char* filter_key = NULL;

	if(arm == SPH_ARM_IRDIS){
		filter_key =  SPH_PHOT_FILTER_IRD_KEY;
	} else if (arm == SPH_ARM_ZIMPOL1){
		filter_key = SPH_STREHL_FILTER_ZPL1;
	} else if(arm == SPH_ARM_ZIMPOL2){
		filter_key = SPH_STREHL_FILTER_ZPL2;
	} else {
		cpl_msg_warning(cpl_func, "Filter key unknown");
		return NULL;
	}

    const char * strval = cpl_propertylist_get_string( pl, filter_key);
    if (strval == NULL) (void)cpl_error_set_where(cpl_func);
    return strval;
}

static cpl_error_code sph_phot_read_airmass(const cpl_propertylist* pl_strehl,
		double * ms){

    cpl_ensure_code(pl_strehl  != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ms         != NULL, CPL_ERROR_NULL_INPUT);

    *ms = 0.0;

    const double start =  cpl_propertylist_get_double( pl_strehl,
    		SPH_COMMON_KEYWORD_AIRMASS_START);

    if (cpl_error_get_code()){
         return cpl_error_set_message(cpl_func, cpl_error_get_code(), "The airmass keyword %s was not found..",
        		 SPH_COMMON_KEYWORD_AIRMASS_START);
    }

    const double end =  cpl_propertylist_get_double( pl_strehl,
    		SPH_COMMON_KEYWORD_AIRMASS_END);

    if (cpl_error_get_code()){
         return cpl_error_set_message(cpl_func, cpl_error_get_code(), "The airmass keyword %s was not found..",
        		 SPH_COMMON_KEYWORD_AIRMASS_END);
    }

    *ms = 0.5 * (start + end);

    return CPL_ERROR_NONE;
}


static cpl_error_code sph_phot_compute_zero_point(const double flux,
												  const double starmag,
												  const double dit,
                                                  double * zmag)
{
    cpl_ensure_code(flux    > 0.0,     CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(starmag > 0.0,     CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(dit     > 0.0,     CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(zmag    != NULL,   CPL_ERROR_NULL_INPUT);

    *zmag = starmag + 2.5 * log10(flux/dit);

    return CPL_ERROR_NONE;
}




/*Returns ND filter or NULL if OPEN of if unavailable*/
static const char*
sph_phot_read_nd_header(const cpl_propertylist * pl,
						const sph_arm            arm){

	cpl_ensure(pl   != NULL, CPL_ERROR_NULL_INPUT, NULL);

	const char * nd_filter = get_nd_filter_key(arm);
	double nd_value = 0;

	if(nd_filter == NULL || !cpl_propertylist_has(pl, nd_filter)){
		cpl_msg_warning(cpl_func, "Assuming zero-value for missing FITS key %s",
				nd_filter ? nd_filter : "NULL");
		return NULL;
	}

	const char * nd_string = cpl_propertylist_get_string( pl, nd_filter);
	if (cpl_error_get_code()){
		cpl_error_set_where(cpl_func);
		return NULL;
	}

	const cpl_boolean
		has_nd_filter = strcmp(nd_string, "OPEN") != 0;

	if(!has_nd_filter){
		cpl_msg_info(cpl_func, "Neutral Density filter (%s) set to OPEN",
				nd_string);
		return NULL;
	} else if (sscanf(nd_string, "ND_%lg", &nd_value) != 1) {
		/*If it is NOT open and not in the form ND_xxx, the file is wrong*/
		cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
									 "FITS key %s has unexpected "
									 "format of value: %s",
									 nd_filter, nd_string);
		return NULL;
	}
	cpl_msg_info(cpl_func, "Neutral Density filter (%s) value: %g",
				 nd_string, nd_value);

	return nd_string;
}


static const char *
get_nd_filter_key(const sph_arm arm){

	if(arm == SPH_ARM_IRDIS)
		return SPH_PHOT_ND_FILTER_IRDIS_KEY;

	return SPH_PHOT_ND_FILTER_ZIMPOL_KEY;
}


static
cpl_error_code sph_phot_read_header(const cpl_propertylist * pl,
									const sph_arm            arm,
									const char             * dit_key,
									const char            ** tfilter_key,
									const char            ** tstdstar_key,
									double                 * dit,
									double                 * ra,
									double                 * dec ) {

	cpl_ensure_code(dit_key      != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(tfilter_key != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(pl           != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(tstdstar_key != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(ra           != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(dec          != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(dit          != NULL, CPL_ERROR_NULL_INPUT);

	const char * strval;

	/*  get image filter, CAMERA1 */
	*tfilter_key = sph_phot_read_arm_filter(pl, arm);

	if(*tfilter_key == NULL ||  cpl_error_get_code())
		 return cpl_error_set(cpl_func, CPL_ERROR_DATA_NOT_FOUND);

	/* get DIT */
	double val = cpl_propertylist_get_double( pl, dit_key );
	if (cpl_error_get_code()){
		 return cpl_error_set_where(cpl_func);
	}
	*dit = val;

	if (*dit <= 0.0){
		return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
									 "Wrong DIT = %g <= 0.", *dit);
	}
	const char * stdstar_key = SPH_COMMON_TARGET_NAME_KEY;

	/* get target name */
	strval = cpl_propertylist_get_string( pl, stdstar_key );
	if (strval == NULL) return cpl_error_set_where(cpl_func);
	*tstdstar_key = strval;

	/* get RA */
	if (cpl_propertylist_has(pl, SPH_COMMON_KEYWORD_WCS_RA) &&
		cpl_propertylist_get_type(pl, SPH_COMMON_KEYWORD_WCS_RA) ==
		CPL_TYPE_DOUBLE) {
		val = cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_WCS_RA );
		*ra = val;

		if (*ra < 0.0) {
			return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
										 "Wrong RA = %g < 0.", *ra);
		}
	} else {
		*ra = -1.0; /* Negative indicates absence */
	}


	/* get DEC */
	if (cpl_propertylist_has(pl, SPH_COMMON_KEYWORD_WCS_DEC) &&
		cpl_propertylist_get_type(pl, SPH_COMMON_KEYWORD_WCS_DEC) ==
		CPL_TYPE_DOUBLE) {
		val = cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_WCS_DEC );
		*dec = val;
	} else {
		*dec = 0.0;
	}

	/* Check if no CORO */
	// Not implemented. Needed?

	return CPL_ERROR_NONE;
}

static cpl_error_code
sph_phot_correct_for_airmass(const char * filter_key,
						     const sph_arm arm,
						     const double airmass,
							 double * zmag){

	cpl_ensure_code(filter_key !=NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(zmag       !=NULL, CPL_ERROR_NULL_INPUT);

	if(arm == SPH_ARM_IRDIS) return CPL_ERROR_NONE;

	double coeff = 0.0;
	if(!strcmp(filter_key, SPH_ZPL_V_FILTER)){
		coeff = 0.133;
	} else if(!strcmp(filter_key, SPH_ZPL_I_FILTER)){
		coeff = 0.050;
	} else if(!strcmp(filter_key, SPH_ZPL_R_FILTER)){
		coeff = 0.087;
	}
	else{
        cpl_msg_warning(cpl_func, "Cannot correct for airmass for "
        		"the filter %s", filter_key);
		return CPL_ERROR_NONE;
	}

	const double corr = airmass * coeff;
	cpl_msg_info(cpl_func, "Airmass correction: subtracting %f from %f", corr, *zmag);

	*zmag = *zmag - corr;
	return CPL_ERROR_NONE;
}

cpl_error_code sph_phot_irdis(cpl_propertylist* phot_list_left,
							  cpl_propertylist* phot_list_right,
							  const double strehl_flux_left,
							  const double strehl_flux_right,
                              const cpl_propertylist* pl_frame,
                              const cpl_frame* inframe,
                              const cpl_frame* starframe,
                              const cpl_frame* filterframe){

	sph_phot_zeropoint zp_left = {0, 0};
	sph_phot_zeropoint zp_right = {0, 0};

	if(sph_phot_core(strehl_flux_left,
			pl_frame, inframe, starframe, filterframe,
			SPH_ARM_IRDIS, SPH_COMMON_KEYWORD_SEQ1DIT, &zp_left))
		return cpl_error_set_where(cpl_func);

	if(sph_phot_core(strehl_flux_right,
			pl_frame, inframe, starframe, filterframe,
			SPH_ARM_IRDIS, SPH_COMMON_KEYWORD_SEQ1DIT, &zp_right))
		return cpl_error_set_where(cpl_func);


	sph_phot_write_qc(phot_list_left, &zp_left,
				SPH_PHOT_QC_IRD_LEFT, SPH_PHOT_QC_IRD_LEFT_COMMENT,
				SPH_PHOT_QC_IRD_LEFTCORR, SPH_PHOT_QC_IRD_LEFTCORR_COMMENT);

	sph_phot_write_qc(phot_list_right, &zp_right,
				SPH_PHOT_QC_IRD_RIGHT, SPH_PHOT_QC_IRD_RIGHT_COMMENT,
				SPH_PHOT_QC_IRD_RIGHTCORR, SPH_PHOT_QC_IRD_RIGHTCORR_COMMENT);

	return CPL_ERROR_NONE;
}


cpl_propertylist* sph_phot_zimpol(const double strehl_flux_cam,
									const cpl_boolean is_arm1,
                                    const cpl_frame* inframe,
                                    const cpl_frame* starframe,
								    const cpl_frame* filterframe){

	sph_phot_zeropoint zp = {0, 0};


	const sph_arm arm = is_arm1 ? SPH_ARM_ZIMPOL1 : SPH_ARM_ZIMPOL2;
	const cpl_error_code fail = sph_phot_core(strehl_flux_cam, NULL,
			inframe, starframe, filterframe, arm,
			SPH_COMMON_KEYWORD_DIT, &zp);

	if(fail) return NULL;


	cpl_propertylist * pl = cpl_propertylist_new();

	sph_phot_write_qc(pl, &zp,
			SPH_PHOT_QC_ZPL_CAMERA, SPH_PHOT_QC_ZPL_CAMERA_COMMENT,
			SPH_PHOT_QC_ZPL_CAMERACORR, SPH_PHOT_QC_ZPL_CAMERACORR_COMMENT);

	return pl;
}

static void
sph_phot_write_qc(cpl_propertylist * plist, const sph_phot_zeropoint * zp,
			const char * zp_key, const char * zp_comment,
			const char * zp_corr_key, const char * zp_corr_comment){

	cpl_propertylist_update_double(plist, zp_key, zp->zpoint);
	cpl_propertylist_set_comment(plist, zp_key,  zp_comment);

	cpl_propertylist_update_double(plist, zp_corr_key, zp->zpoint_corrected);
	cpl_propertylist_set_comment(plist, zp_corr_key,  zp_corr_comment);
}

static cpl_error_code
sph_phot_core(const double flux,
   			  const cpl_propertylist* pl_frame,
			  const cpl_frame* inframe,
			  const cpl_frame* starframe,
			  const cpl_frame* filterframe,
			  const sph_arm    arm,
			  const char * dit_key,
			  sph_phot_zeropoint * zp_calculated){

	cpl_ensure_code(zp_calculated != NULL, CPL_ERROR_NULL_INPUT);

	cpl_propertylist* rawpl   = NULL;
	const cpl_propertylist* readpl = NULL;

	cpl_ensure_code((inframe  !=NULL) || (pl_frame != NULL), CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(starframe !=NULL, CPL_ERROR_NULL_INPUT);

	if (pl_frame != NULL) {
		readpl = pl_frame;
	} else {
		const char * fname = cpl_frame_get_filename(inframe);
		rawpl = cpl_propertylist_load(fname, 0);
		readpl = rawpl;
	}

	cpl_error_code fail = sph_phot_no_list_allocation(flux, readpl, starframe,
			filterframe, arm, dit_key, zp_calculated);

	cpl_propertylist_delete(rawpl);

	return fail;
}

static cpl_error_code
sph_phot_no_list_allocation(double flux,
		const cpl_propertylist* readpl,
		const cpl_frame* starframe,
		const cpl_frame* filterframe,
		const sph_arm    arm,
	    const char * dit_key,
		sph_phot_zeropoint * zp_calculated){

	double dit        = 0.0;
	double starmag    = 0.0;
	double ra         = 0.0;
	double dec        = 0.0;
	double airmass    = 0.0;

	const char * tfilter_key  = NULL;
	const char * tstdstar     = NULL;

	double zpoint     = 0.0;
	double zpoint_corrected = 0.0;

	if ( readpl == NULL ) {
		return cpl_error_set_where(cpl_func);
	} else if ( sph_phot_read_airmass( readpl, &airmass) ){
		return cpl_error_set_where(cpl_func);
	} else if (sph_phot_read_header(readpl, arm, dit_key,
									&tfilter_key, &tstdstar,
									&dit, &ra, &dec)) {
		return cpl_error_set_where(cpl_func);
	} else if (sph_phot_read_star(starframe, tstdstar, tfilter_key, &starmag,
								  ra, dec)) {
		return cpl_error_set_where(cpl_func);
	} else if (sph_phot_compute_zero_point( flux, starmag, dit,
											 &zpoint)) {
		return cpl_error_set_where(cpl_func);
	} else if (sph_phot_correct_for_airmass(tfilter_key, arm, airmass,
											&zpoint)){
		return cpl_error_set_where(cpl_func);
	} else {


		const char * nd_string = sph_phot_read_nd_header(readpl, arm);

		if(sph_phot_correct_nd(zpoint, tfilter_key, nd_string,
							filterframe, &zpoint_corrected))
			return cpl_error_set_where(cpl_func);

		*zp_calculated = (sph_phot_zeropoint){zpoint, zpoint_corrected};
	}
	return CPL_ERROR_NONE;
}

static cpl_error_code
sph_phot_correct_nd(double zmag,
					const char* tfilter_key,
					const char* nd_string,
					const cpl_frame* filterframe,
					double * out_zmag_corrected){

	/*nd_string == NULL is acceptable*/
	cpl_ensure_code(tfilter_key != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(filterframe != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(out_zmag_corrected != NULL, CPL_ERROR_NULL_INPUT);

    double zpcorr = 0.0;

    cpl_error_code fail = sph_phot_extract_nd_corr(nd_string, filterframe,
    		tfilter_key, &zpcorr);

    if(fail){
        cpl_msg_warning(cpl_func, "Extraction of the ND filter correction for "
        		"Zeropoint failed, aborting correction.");
        return fail;
    }

    *out_zmag_corrected = zmag - zpcorr;

    return CPL_ERROR_NONE;
}


static cpl_error_code
sph_phot_extract_nd_corr(const char * nd_string,
		const cpl_frame * filterframe,
		const char * tfilter_key, double *zpcorr){

	if(!nd_string){
		*zpcorr = 0.0;
		cpl_msg_info(cpl_func, "ND-filter not set, assuming 0.0 as correction");
		return CPL_ERROR_NONE;
	}

	cpl_error_code fail = CPL_ERROR_NONE;

	const char* filename = cpl_frame_get_filename(filterframe);
	const cpl_size iext = cpl_fits_find_extension(filename, nd_string);

	if (iext < 0) {
		return cpl_error_set_where(cpl_func);
	} else if (iext == 0) {
		cpl_msg_warning(cpl_func, "Cannot correct zero-points for Neutral "
						"Density filter for unknown ND filter: %s",
						nd_string);
		return cpl_error_set_where(cpl_func);
	} else {

		cpl_table*     ndtab  = cpl_table_load(filename, iext, 0);
		const cpl_size length = cpl_table_get_nrow(ndtab);
		const cpl_type coltype = cpl_table_get_column_type(ndtab,
														   tfilter_key);

		if (length < 1) {
			fail = cpl_error_set(cpl_func, CPL_ERROR_DATA_NOT_FOUND);
		} else if (length > 1) {
			fail = cpl_error_set(cpl_func, CPL_ERROR_BAD_FILE_FORMAT);
		} else if (coltype != CPL_TYPE_DOUBLE) {
			fail = cpl_error_set(cpl_func, CPL_ERROR_UNSUPPORTED_MODE);
		} else {

			*zpcorr = cpl_table_get_double(ndtab, tfilter_key, 0, NULL);

			cpl_msg_info(cpl_func, "ND-filter %s via filter %s has reduced "
						 "zero-point by: %g < 0.0", nd_string, tfilter_key,
						 *zpcorr);
		}

		cpl_table_delete(ndtab);
	}
	return fail;
}
