/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */
#include "sph_pixel_description_table.h"
#include "sph_ifs_lenslet_model.h"
#include "sph_error.h"
#include "sph_spectrum.h"
#include "sph_common_keywords.h"
#include "sph_cube.h"
#include "sph_fits.h"
#include "sph_spectral_region.h"
#include "sph_keyword_manager.h"
#include "sph_utils.h"
#include "sph_version.h"

#include <gsl/gsl_errno.h>
#include <gsl/gsl_spline.h>
#include <cpl_error.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <math.h>
#include <limits.h>
#include <float.h>
#include <assert.h>

sph_error_code
sph_pixel_description_table_collapse_line_total(
        const sph_pixel_description_table* self,
        const sph_master_frame* mframe,
        int yy, int minx, int maxx,
        double* value,
        int* bpix,
        double* rms,
        double* weight,
        double* delta_lambda )
{
    int xx = 0;
    double        totlam = 0.0;
    int cc = 0;
    int    ibpix = 0;
    int    iibpix = 0;
    int count1 = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    if ( !self )
        return SPH_ERROR_RAISE_ERR(CPL_ERROR_NULL_INPUT, "Null input" );

    for (xx = minx; xx <= maxx; ++xx) {
        const sph_pixel_descriptor* descr
            = sph_pixel_description_table_get_descriptor_const(self, xx, yy);
        if ( descr ) {
            if ( !mframe ) {
                if ( value ) *value += descr->value;
                totlam += descr->wavelength;
                count1++;
            } else {
                ibpix = 0;
                iibpix = cpl_image_get( mframe->badpixelmap, xx + 1, yy + 1,
                                        &ibpix );
                if ( value && ( ibpix == 0 ) && ( iibpix == 0 ) ) {
                    *value +=
                        cpl_image_get( mframe->image, xx + 1, yy + 1, &ibpix );
                    count1++;
                    if ( weight ) *weight +=
                        cpl_image_get(mframe->ncombmap, xx + 1, yy + 1, &ibpix);
                } else {
                    cpl_error_reset();
                    if ( value ) *bpix += 1;
                }
            }
        }
    }

    if ( count1 ) *value /= count1;

    if ( (rms || delta_lambda) && count1 ) {
        cc = 0;
        for (xx = minx; xx <= maxx; ++xx) {
            const sph_pixel_descriptor* descr
                = sph_pixel_description_table_get_descriptor_const(self, xx, yy);
            if ( descr ) {
                if ( delta_lambda ) *delta_lambda += ( descr->wavelength - totlam ) * ( descr->wavelength - totlam );
                if ( rms && value ) {
                    if ( mframe ) {
                        ibpix = 0;
                        iibpix = cpl_image_get( mframe->badpixelmap, xx + 1, yy + 1, &ibpix );
                        if ( ( ibpix == 0 ) && ( iibpix  == 0 ) )
                            *rms += ( cpl_image_get( mframe->image, xx + 1, yy + 1, &ibpix ) - *value ) *
                                        ( cpl_image_get( mframe->image, xx + 1, yy + 1, &ibpix ) - *value );
                        else cc--;
                    }
                    else {
                        *rms += ( descr->value - *value ) * ( descr->value - *value );
                    }
                }
                cc++;
            }
        }
        if ( delta_lambda) {
            *delta_lambda = sqrt(*delta_lambda);
            if ( cc ) *delta_lambda /= cc;
        }
        if ( rms ) {
            if ( cc ) *rms /= cc;
            *rms = sqrt(*rms);
        }
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_pixel_description_table Pixel Description Table
 *
 * @par Synopsis:
 * @code
typedef struct _sph_pixel_description_table_
{
    int                     nx;  // number pixel x
    int                     ny;  // number pixel y
    int                     maxlength; // maximum spectral length
    int                     idlongest; // id of spectrum with max length
    double                  offx;   // the offset in x (pixels)
    double                  offy; // the offset in y (pixels)
    sph_pixel_descriptor*   arr; // the pixel descrption array
} sph_pixel_description_table;
 * @endcode
 * @par Descirption:
 * This module provides functions to define and operate on a pixel
 * description table.
 * @par
 * The pixel description table (PDT) is of central importance for the wavelength
 * calibration procedure for both IFS and IRDIS.
 * @par
 * The pixel desciption table contains information for every pixel on the detector.
 * In the table, there is one row for every pixel, and columns giving:
 * <table bgcolor="#EEF3F5">
 * <tr>
 * <td>Pixel-x</td><td>Pixel-y</td><td>lensid</td>
 * <td>illumination</td><td>wavelength</td><td>dlambda</td><td>ddlambda</td></tr>
 * <tr>
 * <td>pixel x coordinate</td><td>pixel y coordinate</td><td>ID of lenslet</td>
 * <td>ID of spectrum</td>
 * <td>illumination fraction of pixel</td><td>central wavelength at pixel</td>
 * <td>wavelength width of pixel</td><td>second derivate</td></tr>
 * </tr>
 * </table>
 * @par
 * This table can be manipulated with the functions provided as part of this
 * module. The functions allow extractor of pixel descriptors at specific
 * pixel locations, for example.
 * @par
 * Note that within the SPHERE data reduction pipeline, every pixel description
 * table is for one particular detector offset position. This positions is saved
 * in the structure of the PDT in the variables offx and offy. PDTs with different
 * detector offsets can in principle be completely different -- but the assumption
 * within SPHERE wavelength calibration reductions is that the pattern of locations
 * and wavelengths associations is rigid enough to allow a single pattern to be
 * used. This pattern can then be shifted and used to create PDT for different
 * offset positions. The routine sph_pdt_new_copy allows the creation of new
 * pixel description tables by copying an input PDT to another detector
 * offset position.
 */
/*----------------------------------------------------------------------------*/
/**@{*/
sph_error_code SPH_PDT_GENERAL              = SPH_PDT_ERR_START + 0;
sph_error_code SPH_PDT_SPECTRA_NONE_FOUND   = SPH_PDT_ERR_START + 1;
sph_error_code SPH_PDT_REGION_LENSLET_MISMATCH = SPH_PDT_ERR_START + 2;

/*-----------------------------------------------------------------------------
 Function definitions
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new pixel description table thats empty.
 *
 * @param nx   The number of pixels in x
 * @param ny   The number of pixels in y
 * @param offx The detector offset value for this table
 * @param offy The detector offset value for this table
 *
 * @return Pointer to new table or null in case of error.
 *
 * A new pixel description table is created with nx * ny entries.
 * The table is created for the given offsets -- this does not affect the
 * creation algorithm directly, but is rather a label to allow later
 * combinations and comparisions with pixel tables created with different
 * detector dithering offsets.
 *
 */
/*----------------------------------------------------------------------------*/
sph_pixel_description_table* sph_pixel_description_table_new(int nx,
                                                             int ny,
                                                             double offx,
                                                             double offy)
{
    sph_pixel_description_table*    self   = NULL;
    int                             xx      = 0;
    int                             yy      = 0;

    if ( nx < 0 || ny < 0 ) {
        sph_error_raise(CPL_ERROR_ILLEGAL_INPUT, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create pixel"
                                " description table with"
                                " negative number of pixels.\n");
        return NULL;
    }

    self = cpl_calloc( 1, sizeof(sph_pixel_description_table) );

    if ( !self ) {
        sph_error_raise(SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create pixel"
                                " description table "
                                " -- cant allocate space.\n");
        return NULL;
    }

    self->arr = cpl_calloc( nx* ny, sizeof(sph_pixel_descriptor) );
    if ( !self->arr ) {
        sph_error_raise(SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create pixel"
                                " description table "
                                " -- cant allocate space.\n");
        cpl_free(self);
        return NULL;
    }

    for ( yy = 0; yy < ny; ++yy ) {
        for ( xx = 0; xx < nx; ++xx ) {
            self->arr[yy*nx + xx].x = xx;
            self->arr[yy*nx + xx].y = yy;
        }
    }

    self->nx = nx;
    self->ny = ny;
    self->offx = offx;
    self->offy = offy;
    self->maxlength = -1;
    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new pixel description table from an image
 *
 * @param image The image containing the spectra
 * @param offx The detector offset value for this table
 * @param offy The detector offset value for this table
 * @param threshold The threshold above which a pixel is labeled
 *
 * @return Pointer to new table or null in case of error.
 *
 * A new pixel description table is created with nx * ny entries, where
 * nx and ny are the pixel size of the image in x and y respectively.
 * The table is created for the given offsets -- this does not affect the
 * creation algorithm directly, but is rather a label to allow later
 * combinations and comparisions with pixel tables created with different
 * detector dithering offsets.
 *
 */
/*----------------------------------------------------------------------------*/
sph_pixel_description_table*
sph_pixel_description_table_new_from_image( cpl_image* image,
                                        double offx,
                                        double offy,
                                        double threshold )
{
    sph_pixel_description_table*    self        = NULL;
    cpl_image*                      labelimage  = NULL;
    cpl_mask*                       mask        = NULL;
    int                             xx          = 0;
    int                             yy          = 0;
    double                          value       = 0.0;
    sph_pixel_descriptor*           descr       = NULL;
    cpl_size                             nspectra    = 0;
    int                             badpix      = 0;

    if ( !image ) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create pixel"
                                " description table with"
                                " null pointer to cpl_image.\n");
        return NULL;
    }


    mask = cpl_mask_threshold_image_create(image, -FLT_MAX, threshold);

    cpl_mask_not(mask);

    labelimage = cpl_image_labelise_mask_create(mask, &nspectra);

    if ( nspectra ) {
        self = sph_pixel_description_table_new_from_labels(labelimage,
                                                       offx,
                                                       offy);
    }
    else {
        self = sph_pixel_description_table_new(
                                               cpl_image_get_size_x(image),
                                               cpl_image_get_size_y(image),
                                               offx,
                                               offy);
    }

    if ( !self ) {
        sph_error_raise(SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create pixel"
                                " description table ");
        return NULL;
    }

    for ( yy = 0; yy < self->ny; ++yy ) {
        for ( xx = 0; xx < self->ny; ++xx ) {
            value = cpl_image_get(image, xx + 1, yy + 1, &badpix);
            descr = sph_pixel_description_table_get_descriptor(self, xx, yy);
            if ( descr->specid > 0 ) {
                descr->value = value;
            }
            if ( descr->specid > self->nregions ) {
                self->nregions = descr->specid;
            }

        }
    }

    if ( self->nregions != nspectra ) {
        sph_error_raise( SPH_PDT_GENERAL,
                         __FILE__, __func__,
                         __LINE__,
                         SPH_ERROR_WARNING,
                         "Mismatch in number of spectra"
                         " regions and lensids detected: "
                         "%d vs. %d.",
                         self->nregions, (int)nspectra );
    }

    sph_pixel_description_table_find_maxlength( self );
    cpl_image_delete(labelimage); labelimage = NULL;
    cpl_mask_delete(mask); mask = NULL;
    return self;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Set up the regions  for the PDT

 @param self        the PDT

 @return error code

 Description:
 This function sets up the spectal regions of the PDT.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_pixel_description_table_setup_regions( sph_pixel_description_table* self ) {
    int        current_region        = 0;
    short    bad                    = 0;
    int        xx                    = 0;
    int        yy                    = 0;
    sph_pixel_descriptor* pdescr    = NULL;

    if ( !self ) {
        SPH_NO_SELF;
        return CPL_ERROR_NULL_INPUT;
    }
    if ( self->regions ) {
        for ( current_region = 0; current_region < self->nregions; ++current_region ) {
            sph_spectral_region_delete( self->regions[ current_region ] );
            self->regions[current_region] = NULL;
        }
        cpl_free(  self->regions );
    }
    self->regions = cpl_calloc( self->nregions, sizeof( sph_spectral_region* ) );
    for ( current_region = 0; current_region < self->nregions; ++current_region) {
        self->regions[ current_region ] = sph_spectral_region_new( );
        if ( !self->regions[ current_region ] ) {
            bad = 1;
        }
        else {
            self->regions[ current_region ]->minx = self->nx;
            self->regions[ current_region ]->miny = self->ny;
            self->regions[ current_region ]->maxx = -1;
            self->regions[ current_region ]->maxy = -1;
        }
    }
    if ( bad ) {
        for ( current_region = 0; current_region < self->nregions; ++current_region ) {
            if ( self->regions[current_region] ) {
                sph_spectral_region_delete( self->regions[ current_region ] );
                self->regions[current_region] = NULL;
            }
        }
        SPH_ERROR_RAISE_ERR(SPH_ERROR_GENERAL,"Could not create PDT");
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    for (yy = 0; yy < self->ny; ++yy) {
        for (xx = 0; xx < self->nx; ++xx) {
            pdescr = sph_pixel_description_table_get_descriptor( self, xx, yy );
            if ( pdescr->specid > 0 ) {
                if ( self->regions[pdescr->specid - 1]->specid &&
                        self->regions[pdescr->specid - 1]->specid != pdescr->specid ) {
                    SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                            "There was a problem in creating the PDT. "
                            "A region got assigned two different spectra ids"
                            " (%d,%d). This should not happen."
                            "Pixel coords were: %d, %d",
                            self->regions[pdescr->specid - 1]->specid,
                            pdescr->specid, xx, yy );
                    return CPL_ERROR_ILLEGAL_INPUT;
                }
                if ( self->regions[pdescr->specid - 1]->lensid &&
                        self->regions[pdescr->specid - 1]->lensid != pdescr->lensid ) {
                    SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                            "There was a problem in creating the PDT. "
                            "The region %d got assigned two different lensids ids"
                            " (%d,%d). This should not happen."
                            "Pixel coords were: %d, %d.",
                            pdescr->specid,
                            self->regions[pdescr->specid - 1]->lensid,
                            pdescr->lensid, xx ,yy );
                    return CPL_ERROR_ILLEGAL_INPUT;
                }
                self->regions[pdescr->specid - 1 ]->specid = pdescr->specid;
                self->regions[pdescr->specid - 1 ]->lensid = pdescr->lensid;
                if ( xx < self->regions[pdescr->specid - 1 ]->minx ) {
                    self->regions[pdescr->specid - 1 ]->minx = xx;
                }
                if ( yy < self->regions[pdescr->specid - 1 ]->miny ) {
                    self->regions[pdescr->specid - 1 ]->miny = yy;
                }
                if ( xx > self->regions[pdescr->specid - 1 ]->maxx ) {
                    self->regions[pdescr->specid - 1 ]->maxx = xx;
                }
                if ( yy > self->regions[pdescr->specid - 1 ]->maxy ) {
                    self->regions[pdescr->specid - 1 ]->maxy = yy;
                }
            }
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return the spectral region for a given spectrum ID

 @param self            the PDT
 @param specid            the spectrum ID

 @return region    pointer to a sph_spectral_region for the region

 Description:

 */
/*----------------------------------------------------------------------------*/
sph_spectral_region*
sph_pixel_description_table_get_region( sph_pixel_description_table* self,
        int specid ) {
    sph_error_code                rerr    = CPL_ERROR_NONE;

    if ( !self ) {
        SPH_NULL_ERROR;
        return NULL;
    }
    if ( specid < 1 || specid > self->nregions ) {
        return NULL;
    }
    if ( !self->regions ) {
        rerr = sph_pixel_description_table_setup_regions( self );
        if ( rerr != CPL_ERROR_NONE ) {
            sph_error_raise( CPL_ERROR_ILLEGAL_INPUT, __FILE__,
                    __func__, __LINE__,
                    SPH_ERROR_ERROR, "Error in setting up regions.");
            return NULL;
        }
    }
    return self->regions[specid - 1];
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new pixel description table from a label image
 *
 * @param image The image containing the labelised regions of spectra
 * @param offx The detector offset value for this table
 * @param offy The detector offset value for this table
 *
 * @return Pointer to new table or null in case of error.
 *
 * A new pixel description table is created with nx * ny entries, where
 * nx and ny are the pixel size of the image in x and y respectively.
 * The table is created for the given offsets -- this does not affect the
 * creation algorithm directly, but is rather a label to allow later
 * combinations and comparisions with pixel tables created with different
 * detector dithering offsets.
 *
 */
/*----------------------------------------------------------------------------*/
sph_pixel_description_table*
sph_pixel_description_table_new_from_labels( cpl_image* image,
                                        double offx,
                                        double offy)
{
    sph_pixel_description_table*    self   = NULL;
    int                             xx      = 0;
    int                             yy      = 0;
    int                             nx      = 0;
    int                             ny      = 0;
    int                             label   = 0;
    int                             badpix  = 0;

    if ( !image ) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create pixel"
                                " description table with"
                                " null pointer to cpl_image.\n");
        return NULL;
    }

    nx = cpl_image_get_size_x(image);
    ny = cpl_image_get_size_y(image);

    self = sph_pixel_description_table_new(nx,ny,offx,offy);

    if ( !self ) {
        sph_error_raise(SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create pixel"
                                " description table ");
        return NULL;
    }

    for ( yy = 0; yy < ny; ++yy ) {
        for ( xx = 0; xx < nx; ++xx ) {
            self->arr[yy*nx + xx].x = xx;
            self->arr[yy*nx + xx].y = yy;
            label = cpl_image_get(image, xx + 1, yy + 1, &badpix);
            if ( label > 0 ) {
                self->arr[yy*nx + xx].specid = label;
                self->arr[yy*nx + xx].lensid = 0;
                if ( label > self->nregions ) {
                    self->nregions = label;
                }
            }
        }
    }

    sph_pixel_description_table_find_maxlength( self );

    return self;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new pixel description table from an existing one.
 *
 * @param pdt   The input pixel description table to copy
 * @param offx The detector offset value for the new table
 * @param offy The detector offset value for the new table
 *
 * @return Pointer to new table or null in case of error.
 *
 * A new pixel description table is created as a copy of the input
 * existing PDT. The new pixel desciption table is shifted to a new
 * position assuming a rigid shift.
 *
 */
/*----------------------------------------------------------------------------*/
sph_pixel_description_table* sph_pixel_description_table_new_shift( sph_pixel_description_table* pdt,
                                                             sph_ifs_lenslet_model* model,
                                                             double offx,
                                                             double offy)
{
    sph_pixel_description_table*    self   = NULL;
    int                                current_region = 0;
    cpl_vector*                        wavs = NULL;
    cpl_vector*                        dwavs = NULL;
    int                                old_lid = 0;
    int                                ii = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    cpl_ensure(model,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(pdt,CPL_ERROR_NULL_INPUT,NULL);


    self = sph_pixel_description_table_new_from_model( model, pdt->offx + offx, pdt->offy + offy );
    cpl_ensure(self,CPL_ERROR_ILLEGAL_OUTPUT,NULL);
    for ( current_region = 0; current_region < self->nregions; ++current_region ) {
        wavs = sph_pixel_description_table_get_wavs(self,current_region+1);
        cpl_vector_fill(wavs,0.0);
        if ( cpl_error_get_code() == CPL_ERROR_INCOMPATIBLE_INPUT ) {
            SPH_ERROR_RAISE_WARNING( CPL_ERROR_INCOMPATIBLE_INPUT,
                    "Could not set wavelengths for spectrum id %d "
                    "since the number of wavelength "
                    "points were incorrect.",current_region+1);
            SPH_RAISE_CPL_RESET;
        }
        cpl_vector_delete(wavs); wavs = NULL;
    }

    for ( current_region = 0; current_region < pdt->nregions; ++current_region ) {
        SPH_RAISE_CPL_RESET
        if ( current_region < self->nregions ) {
            old_lid = sph_pixel_description_table_get_region(pdt,current_region+1)->lensid;
            wavs = sph_pixel_description_table_get_wavs(pdt,current_region+1);
            if ( wavs ) {
                SPH_RAISE_CPL_RESET
                dwavs = cpl_vector_new(cpl_vector_get_size(wavs));
                for (ii = 1; ii < cpl_vector_get_size(wavs); ++ii) {
                    cpl_vector_set(dwavs,ii,cpl_vector_get(wavs,ii)-cpl_vector_get(wavs,ii-1));
                }
                if ( cpl_vector_get_size(dwavs) > 1 ) cpl_vector_set(dwavs,0,cpl_vector_get(dwavs,1));
                SPH_RAISE_CPL_RESET
                sph_pixel_description_table_set_wavs_lensid(self,old_lid,wavs);
                if ( cpl_error_get_code() == CPL_ERROR_ILLEGAL_INPUT ) {
                    SPH_ERROR_RAISE_WARNING( CPL_ERROR_ILLEGAL_INPUT,
                            "Could not set wavelengths for spectra id %d "
                            "since the number of wavelength "
                            "points were incorrect.",current_region+1);
                    SPH_RAISE_CPL_RESET;
                }
                cpl_vector_delete(wavs); wavs = NULL;
                cpl_vector_delete(dwavs); dwavs = NULL;
            }
            else {
                SPH_ERROR_RAISE_WARNING( CPL_ERROR_INCOMPATIBLE_INPUT,
                        "Could not get wavelengths for spec id %d "
                        "since the number of wavelength "
                        "points were incorrect.",current_region+1);
            }
        }
    }

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return self;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Return a pointer to a pixel descriptor.
 *
 * @param self The pixel description table containing the descriptor
 * @param xx   The x-position of the pixel
 * @param yy   The y-position of the pixel
 * @return pointer to the pixel descriptor in the table or NULL on error.
 *
 * Returns a pointer to the pixel descriptor in the table at xx, yy.
 */
/*----------------------------------------------------------------------------*/
const sph_pixel_descriptor* sph_pixel_description_table_get_descriptor_const
(const sph_pixel_description_table* self,
 int xx,
 int yy)
{
    return (xx < 0 || yy < 0 || xx >=self->nx || yy>= self->ny)
        ? NULL :  &self->arr[yy*self->nx + xx];
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Return a pointer to a pixel descriptor.
 *
 * @param self The pixel description table containing the descriptor
 * @param xx   The x-position of the pixel
 * @param yy   The y-position of the pixel
 * @return pointer to the pixel descriptor in the table or NULL on error.
 *
 * Returns a pointer to the pixel descriptor in the table at xx, yy.
 */
/*----------------------------------------------------------------------------*/
sph_pixel_descriptor*
sph_pixel_description_table_get_descriptor( sph_pixel_description_table* self,
                                            int xx,
                                            int yy
                                          )
{
    return (xx < 0 || yy < 0 || xx >=self->nx || yy>= self->ny)
        ? NULL :  &self->arr[yy*self->nx + xx];
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return a pointer to an array of pixel descriptors.
 *
 * @param self The pixel description table containing the descriptor array
 * @return pointer to the pixel descriptor in the table
 *
 */
/*----------------------------------------------------------------------------*/
sph_pixel_descriptor*
sph_pixel_description_table_get_array(sph_pixel_description_table* self)
{
    return self->arr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return a pointer to an array of pixel descriptors.
 *
 * @param self The pixel description table containing the descriptor array
 * @return pointer to the pixel descriptor in the table
 *
 */
/*----------------------------------------------------------------------------*/
const sph_pixel_descriptor* sph_pixel_description_table_get_array_const
(const sph_pixel_description_table* self)
{
    return self->arr;
}

sph_pixel_description_table*
sph_pixel_description_table_new_from_model(sph_ifs_lenslet_model* self,
                                           double offx,
                                           double offy )
{
    sph_pixel_description_table* pdt;
    int                          u   = 0;
    int                          v   = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    pdt = sph_pixel_description_table_new(self->detsize_pixels,
                                          self->detsize_pixels,
                                          offx, offy);

    /* Can only fail on NULL input */
    (void)sph_ifs_lenslet_model_get_first(self, &u, &v);

    do {
        sph_polygon* poly = 
            sph_ifs_lenslet_model_get_spec_region_pixel_coords(self,u,v,
                                                               offx,offy);
        double dmidx, dmidy;
        int maxx, maxy, minx, miny;

        sph_polygon_get_midxy_(poly, &dmidx, &dmidy);

        minx  = (int)dmidx - (int)(self->specwidth_pixels / 2.0);
        maxx  = minx + (int)(self->specwidth_pixels) - 2;
        miny  = (int)dmidy - (int)(self->speclength_pixels / 2.0);
        maxy  = miny + (int)(self->speclength_pixels) - 1;

        pdt->nregions++;

        for (int yy = miny; yy <= maxy; ++yy) {
            for (int xx = minx; xx <= maxx; ++xx) {
                sph_pixel_descriptor* pdescr =
                    sph_pixel_description_table_get_descriptor(pdt, xx, yy);
                if (pdescr) {
                    const int imodel =
                        sph_ifs_lenslet_model_get_index(self, u, v);

                    pdescr->specid   = imodel;
                    pdescr->lensid   = imodel;
                    pdescr->ddlambda = 0.0;
                    pdescr->dlambda  = 0.0;
                    pdescr->value    = 0.0;
                    pdescr->illumination = 0.0;
                    pdescr->wavelength =
                        sph_ifs_lenslet_model_get_lambda(self, poly, u, v,
                                                         (double)yy + 0.5);
                }
            }
        }
        if ( maxy - miny > pdt->maxlength ) pdt->maxlength = maxy - miny;
    } while (!sph_ifs_lenslet_model_get_next(self, &u, &v));


    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return pdt;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Returns a mask where all pixels belonging to a spectrum
 *        are set to 1 and the others to 0.
 *
 * @param self The pixel description table containing the spectrum
 * @return Pointer to a new cpl_mask or NULL if an error occurred.
 *
 * The routine returns a mask where all pixels that belong to a spectrum (
 * with a spectrum id that is larger than 0) are set to 1 and all other
 * pixels are set to zero. Note that to use this mask to reject all pixels
 * but those beloning to the spectra, the mask has to be inverted first
 * using the cpl_mask_not operation berfore the cpl_image_reject_from_mask
 * routine can be applied.
 */
/*----------------------------------------------------------------------------*/
cpl_mask*
sph_pixel_description_table_get_mask(sph_pixel_description_table* self)
{
    int                 rerr        = CPL_ERROR_NONE;
    int                 xx          = 0;
    int                 yy          = 0;
    sph_pixel_descriptor*   descr   = NULL;
    cpl_mask*           nmask       = NULL;

    if ( !self ) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create mask"
                                " from description table "
                                " -- no table pointer provided.\n");
        return NULL;
    }

    nmask = cpl_mask_new( self->nx, self->ny );

    if ( !nmask ) {
        sph_error_raise( cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create mask"
                                " from description table "
                                " -- could not create cpl_mask.\n"
                                "CPL says: %s\n", cpl_error_get_message() );
        return NULL;
    }

    for (yy = 0; yy < self->ny; ++yy) {
        for (xx = 0; xx < self->nx; ++xx) {
            descr = sph_pixel_description_table_get_descriptor(self, xx, yy);
            if ( descr->specid ) {
               rerr |= cpl_mask_set( nmask, xx + 1, yy + 1, 1 );
            }
            else {
                rerr |= cpl_mask_set( nmask, xx + 1, yy + 1, 0 );
            }
        }
    }

    if ( rerr ) {
        sph_error_raise( cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant get the mask"
                                " from description table "
                                " -- CPL says: %s.\n", cpl_error_get_message() );
        return NULL;
    }


    return nmask;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Returns an image of the pixel-spectra correspondence
 *
 * @param self The pixel description table containing the spectrum
 * @return Pointer to a new cpl_image or NULL if an error occurred.
 *
 * The routine returns an image where all pixels belonging to a spectrum
 * are set to value corresponding to the specid.
 * Pixels that are not covered by a spectrum
 * get a zero assigned.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_pixel_description_table_get_specid_image(sph_pixel_description_table* self)
{
    int                 rerr        = CPL_ERROR_NONE;
    int                 xx          = 0;
    int                 yy          = 0;
    sph_pixel_descriptor*   descr   = NULL;
    cpl_image*            wavimage    = NULL;

    if ( !self ) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create specimage"
                                " from description table "
                                " -- no table pointer provided.\n");
        return NULL;
    }


    wavimage = cpl_image_new( self->nx, self->ny, CPL_TYPE_DOUBLE);
    if ( !wavimage ) {
        sph_error_raise( cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create image"
                                " from description table "
                                " -- could not create cpl_image.\n"
                                "CPL says: %s\n", cpl_error_get_message() );
        return NULL;
    }

    for (yy = 0; yy < self->ny; ++yy) {
        for (xx = 0; xx < self->nx; ++xx) {
            descr = sph_pixel_description_table_get_descriptor(self, xx, yy);
            if ( descr->specid ) {
               rerr |= cpl_image_set( wavimage, xx + 1, yy + 1, descr->specid );
            }
        }
    }

    if ( rerr ) {
        sph_error_raise( cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant get the specidimage"
                                " from description table "
                                " -- CPL says: %s.\n", cpl_error_get_message() );
        return NULL;
    }


    return wavimage;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Apply non-linear correction to pixel description table
 *
 * @param self The pixel description table containing the spectrum
 * @return erro code
 *
 * This routine applies a non-linear correction to the pixel description
 * table as discovered as necessary by Dino Mesa et al. in 2014-2015
 * 
 * The correction is
 * lambda' = lambda + square_term * lambda^2 + lin_term * lambda + offset
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_pixel_description_table_correct_non_linear(sph_pixel_description_table* self, 
        double square_term, double lin_term, double offset)
{
    int                 xx          = 0;
    int                 yy          = 0;
    sph_pixel_descriptor*   descr   = NULL;

    if ( !self ) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create wavimage"
                                " from description table "
                                " -- no table pointer provided.\n");
        return CPL_ERROR_NULL_INPUT;
    }


    for (yy = 0; yy < self->ny; ++yy) {
        for (xx = 0; xx < self->nx; ++xx) {
            descr = sph_pixel_description_table_get_descriptor(self, xx, yy);
            if ( descr->specid ) {
           double dl = square_term * descr->wavelength * descr->wavelength;
           dl = dl + lin_term * descr->wavelength;
           dl = dl + offset;
               descr->wavelength = descr->wavelength + dl;
            }
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Returns an image of the wavelength-pixel correspondence
 *
 * @param self The pixel description table containing the spectrum
 * @return Pointer to a new cpl_mask or NULL if an error occurred.
 *
 * The routine returns an image where all pixels belonging to a spectrum
 * are set to value corresponding to the wavelength of the
 * incident light. Pixels that are not covered by a spectrum
 * get a wavelength of zero assigned.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_pixel_description_table_get_wavimage(sph_pixel_description_table* self)
{
    int                 rerr        = CPL_ERROR_NONE;
    int                 xx          = 0;
    int                 yy          = 0;
    sph_pixel_descriptor*   descr   = NULL;
    cpl_image*            wavimage    = NULL;

    if ( !self ) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create wavimage"
                                " from description table "
                                " -- no table pointer provided.\n");
        return NULL;
    }


    wavimage = cpl_image_new( self->nx, self->ny, CPL_TYPE_DOUBLE);
    if ( !wavimage ) {
        sph_error_raise( cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create image"
                                " from description table "
                                " -- could not create cpl_image.\n"
                                "CPL says: %s\n", cpl_error_get_message() );
        return NULL;
    }

    for (yy = 0; yy < self->ny; ++yy) {
        for (xx = 0; xx < self->nx; ++xx) {
            descr = sph_pixel_description_table_get_descriptor(self, xx, yy);
            if ( descr->specid ) {
               rerr |= cpl_image_set( wavimage, xx + 1, yy + 1, descr->wavelength );
            }
        }
    }

    if ( rerr ) {
        sph_error_raise( cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant get the wavimage"
                                " from description table "
                                " -- CPL says: %s.\n", cpl_error_get_message() );
        return NULL;
    }


    return wavimage;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Returns an image of the dispersion
 *
 * @param self The pixel description table containing the spectrum
 * @return Pointer to a new cpl_mask or NULL if an error occurred.
 *
 * The routine returns an image where all pixels belonging to a spectrum
 * are set to value corresponding to the dispersion of the
 * incident light at that point. Pixels that are not covered by a spectrum
 * get a value of zero assigned.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_pixel_description_table_get_dispimage(sph_pixel_description_table* self)
{
    int                 rerr        = CPL_ERROR_NONE;
    int                 xx          = 0;
    int                 yy          = 0;
    sph_pixel_descriptor*   descr   = NULL;
    cpl_image*            wavimage    = NULL;

    if ( !self ) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create wavimage"
                                " from description table "
                                " -- no table pointer provided.\n");
        return NULL;
    }


    wavimage = cpl_image_new( self->nx, self->ny, CPL_TYPE_DOUBLE);
    if ( !wavimage ) {
        sph_error_raise( cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create image"
                                " from description table "
                                " -- could not create cpl_image.\n"
                                "CPL says: %s\n", cpl_error_get_message() );
        return NULL;
    }

    for (yy = 0; yy < self->ny; ++yy) {
        for (xx = 0; xx < self->nx; ++xx) {
            descr = sph_pixel_description_table_get_descriptor(self, xx, yy);
            if ( descr->specid ) {
               rerr |= cpl_image_set( wavimage, xx + 1, yy + 1, descr->dlambda );
            }
            else {
                cpl_image_reject(wavimage,xx+1,yy+1);
            }
        }
    }

    if ( rerr ) {
        sph_error_raise( cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant get the wavimage"
                                " from description table "
                                " -- CPL says: %s.\n", cpl_error_get_message() );
        return NULL;
    }


    return wavimage;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Broadens the wavelength-pixel correspondence
 *
 * @param self The pixel description table containing the spectrum
 *
 * The routine broadens the spectra associated regions by 1 pixel around the
 * regions.
 * Note that the pixel description values (pixdescr->value) are overwritten
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_pixel_description_table_broaden(sph_pixel_description_table* self)
{
    int                 rerr        = CPL_ERROR_NONE;
    int                 xx          = 0;
    int                 yy          = 0;
    int                    iid            = 0;
    double                value        = 0;
    double                count        = 0;
    sph_pixel_descriptor*   descr   = NULL;
    sph_pixel_descriptor*    ddescr    = NULL;

    if ( !self ) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant create wavimage"
                                " from description table "
                                " -- no table pointer provided.\n");
        return CPL_ERROR_NULL_INPUT;
    }

    for (yy = 0; yy < self->ny; ++yy) {
        for (xx = 0; xx < self->nx; ++xx) {
            descr = sph_pixel_description_table_get_descriptor(self, xx, yy);
            value = 0.0;
            if ( descr->specid <= 0 ) {
                count = 0;
                descr->value = 0.0;
                if ( xx > 0 ) {
                    ddescr = sph_pixel_description_table_get_descriptor(self, xx - 1, yy);
                    if ( ddescr->specid > 0 && descr->value != -99.0 ) {
                        value = value + ddescr->wavelength;
                        iid = ddescr->specid;
                        count++;
                    }
                }
                if ( yy > 0 ) {
                    ddescr = sph_pixel_description_table_get_descriptor(self, xx, yy - 1);
                    if ( ddescr->specid > 0 && descr->value != -99.0 ) {
                        value = value + ddescr->wavelength;
                        iid = ddescr->specid;
                        count++;
                    }
                }
                if ( yy < self->ny - 1) {
                    ddescr = sph_pixel_description_table_get_descriptor(self, xx, yy + 1);
                    if ( ddescr->specid > 0 && descr->value != -99.0 ) {
                        value = value + ddescr->wavelength;
                        iid = ddescr->specid;
                        count++;
                    }
                }
                if ( xx < self->nx -1 ) {
                    ddescr = sph_pixel_description_table_get_descriptor(self, xx + 1, yy );
                    if ( ddescr->specid > 0 && descr->value != -99.0 ) {
                        value = value + ddescr->wavelength;
                        iid = ddescr->specid;
                        count++;
                    }
                }
                if ( count == 1 ) {
                    descr->wavelength = value;
                    descr->dlambda = ddescr->dlambda;
                    descr->ddlambda = ddescr->ddlambda;
                    descr->illumination = 0.0;
                    descr->value = iid;
                }
            }
        }
    }
    for (yy = 0; yy < self->ny; ++yy) {
        for (xx = 0; xx < self->nx; ++xx) {
            descr = sph_pixel_description_table_get_descriptor(self, xx, yy);
            if ( descr->specid <=0 ) {
                if ( descr->value > 0 ) {
                    descr->specid = (int)descr->value;
                }
            }
        }
    }
    if ( rerr ) {
        sph_error_raise( cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant get the wavimage"
                                " from description table "
                                " -- CPL says: %s.\n", cpl_error_get_message() );
        return cpl_error_get_code();
    }

    sph_pixel_description_table_setup_regions(self);
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Return the wavelength points in region as vector
 * @param self      the pixel description table
 * @param specid    the id of the region
 *
 * @return new cpl_vector or NULL
 *
 * This returns the wavelength points in the spectrum as a new
 * vector.
 */
/*----------------------------------------------------------------------------*/
cpl_vector*
sph_pixel_description_table_get_wavs( sph_pixel_description_table* self, int specid)
{
    int minx = -1,maxx = -1 ,miny = -1,maxy = -1;
    sph_spectral_region* reg = NULL;
    cpl_vector*            wavs = NULL;
    sph_pixel_descriptor*    descr = NULL;
    int    xx = 0,yy = 0;
    double val = 0.0;
    int cc = 0;
    if ( !self ) {
        SPH_NO_SELF;
        return NULL;
    }
    if ( specid < 1 || specid > self->nregions ) {
        SPH_ERR("Access out of range.");
        return NULL;
    }
    reg = sph_pixel_description_table_get_region(self,specid);
    if ( !reg ) {
        SPH_ERR("Could not get region.");
        return NULL;
    }

    miny = reg->miny; maxy = reg->maxy;minx = reg->minx;maxx = reg->maxx;

    wavs = cpl_vector_new( maxy - miny + 1 );
    for (yy = 0; yy < cpl_vector_get_size(wavs); ++yy) {
        val = 0.0;cc = 0;
        for (xx = minx; xx <= maxx; ++xx) {
            descr = sph_pixel_description_table_get_descriptor(self,xx,yy + miny);
            val += descr->wavelength;
            cc++;
        }
        if (cc) val=val/cc;
        cpl_vector_set(wavs,yy,val);
    }
    return wavs;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Extract a one dimensional spectrum belonging to a spectrum.
 *
 * @param self The pixel description table containing the spectrum
 * @param lensid The id of the spectrum to extract
 * @param otherimage A possible different master frame to use (maybe NULL)
 * @param wavs    A vector giving the wavelengths to evaluate
 * @return spec  Pointer to the new spectrum structure that was filled.
 *
 * Returns a new sph_spectrum structure filled with values corresponding to the
 * spectrum with the given spectrum id.
 */
/*----------------------------------------------------------------------------*/
sph_spectrum*
sph_pixel_description_table_extract_spectrum_mframe(sph_pixel_description_table* self,
                                            int specid,
                                            const sph_master_frame* otherimage,
                                            const cpl_vector* wavs)
{
    int                 rerr        = CPL_ERROR_NONE;
    int                 yy          = 0;
    int                 maxx        = -1;
    int                 maxy        = -1;
    int                 minx        = __INT_MAX__;
    int                 miny        = __INT_MAX__;
    int                 bpix        = 0;
    double              val         = 0.0;
    int                 ind         = -1;
    cpl_vector*         wavstmp     = NULL;
    const cpl_vector*   wavsuse     = wavs;
    sph_spectrum*        spec       = NULL;
    sph_spectral_region*    reg     = NULL;
    double                rms = 0.0;
    double                ncomb = 0.0;
    int                 nx = 0;
    int                 ny = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    cpl_ensure( self, CPL_ERROR_NULL_INPUT, NULL);
    if ( otherimage ) {
        nx = cpl_image_get_size_x( otherimage->image );
        ny = cpl_image_get_size_y( otherimage->image );
        cpl_ensure( self->nx == nx, CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
        cpl_ensure( self->ny == ny, CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
    }
    reg = sph_pixel_description_table_get_region( self,specid );
    cpl_ensure( reg, CPL_ERROR_ILLEGAL_INPUT, NULL);
    if ( wavs ) {
        if ( reg->maxy-reg->miny+1 > cpl_vector_get_size(wavs) ) {
            return NULL;
        }
    }
    if ( reg->maxx-reg->minx < 1 )
    {
        return NULL;
    }
    if ( reg->maxy-reg->miny < 8 )
    {
        return NULL;
    }

    if ( !wavs ) {
        wavstmp = sph_pixel_description_table_get_wavs(self, specid);
        if ( cpl_vector_get_max(wavstmp) == cpl_vector_get_min(wavstmp) ) {
            cpl_vector_delete(wavstmp); wavstmp = NULL;
            return NULL;
        }
        wavsuse = wavstmp;
    }
    assert( wavsuse != NULL );

    miny = reg->miny; maxy = reg->maxy;minx = reg->minx;maxx = reg->maxx;

    spec = sph_spectrum_new( maxy - miny + 1, cpl_vector_get_min(wavsuse),
                             cpl_vector_get_max(wavsuse) );

    if ( !spec ) {
        SPH_ERR( "Cant extract spectra from description table "
                 " -- empty spectrum could not be created.");
        cpl_vector_delete(wavstmp); wavstmp = NULL;
        return NULL;
    }

    cpl_vector_add_scalar( spec->bads, 1.0 );

    for (yy = miny; yy <= maxy; ++yy)
    {
        bpix = 0;val = rms = ncomb = 0.0;
        if ( sph_pixel_description_table_collapse_line_total(self,
                otherimage,yy,minx,maxx,&val,&bpix,&rms,&ncomb,NULL) != CPL_ERROR_NONE)
                    break;
        ind = yy - miny;
        if ( bpix == 0 ) {
            rerr |= cpl_vector_set( spec->spec, ind, val );
            rerr |= cpl_vector_set( spec->bads, ind, 0 );
            rerr |= cpl_vector_set( spec->rms, ind, rms );
            rerr |= cpl_vector_set( spec->ncomb, ind, ncomb );
            rerr |= cpl_vector_set( spec->wavelengths, ind,
                                    cpl_vector_get(wavsuse,ind) );
        }
        else {
            rerr |= cpl_vector_set( spec->spec, ind, val );
            rerr |= cpl_vector_set( spec->bads, ind, 1 );
            rerr |= cpl_vector_set( spec->rms, ind, rms );
            rerr |= cpl_vector_set( spec->ncomb, ind, ncomb );
            rerr |= cpl_vector_set( spec->wavelengths, ind,
                                    cpl_vector_get(wavsuse,ind) );
        }
        if ( rerr != CPL_ERROR_NONE ) {
            sph_error_raise( SPH_ERROR_GENERAL,
                    __FILE__, __func__, __LINE__,
                    SPH_ERROR_WARNING, "Problem when extracting"
                            " 1D spectral information from "
                             "spectrum with id: %d (%d)", reg->specid, specid );
        }
    }
    spec->specid = reg->specid;
    spec->lensid = reg->lensid;
    cpl_vector_delete(wavstmp); wavstmp = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return spec;
}
sph_pixel_description_table*
sph_pixel_description_table_new_from_images(
        cpl_image* image_id, cpl_image* image_sid, cpl_image* image_wav,
        cpl_image* image_dw, cpl_image* image_ddw, cpl_image* image_ill,
        double offx,
        double offy )
{
    sph_pixel_description_table*    self        = NULL;

    int                             xx      = 0;
    int                             yy      = 0;
    int                             nx      = 0;
    int                             ny      = 0;
    int                             badpix  = 0;
    cpl_size                             nregs   = 0;
    sph_pixel_descriptor*           descr   = NULL;
    cpl_image*                        specidim = NULL;


    cpl_ensure(image_sid,CPL_ERROR_NULL_INPUT,NULL);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    if ( !image_wav || !image_id ) {
        sph_error_raise( SPH_ERROR_INCONSISTENT_INPUT,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Need at least wavelengths and lensids.");
        return NULL;
    }


    nx = cpl_image_get_size_x( image_wav );
    ny = cpl_image_get_size_y( image_wav );


    self = sph_pixel_description_table_new(nx,ny, offx, offy);

    if ( !self ) {
        sph_error_raise(SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Cant create pixel"
                " description table ");
        return NULL;
    }

    for (yy = 0; yy < ny; ++yy) {
        for (xx = 0; xx < nx; ++xx) {

            descr = sph_pixel_description_table_get_descriptor(self, xx, yy);

            if ( !descr ) {
                sph_error_raise(SPH_PDT_SPECTRA_NONE_FOUND,
                        __FILE__, __func__,
                        __LINE__, SPH_ERROR_ERROR,
                        "Could not get descriptor for pixel at %d, %d",
                        xx, yy );
            }
            else
            {
                descr->x = xx;
                descr->y = yy;
                descr->specid = (int)(cpl_image_get( image_sid, xx + 1, yy + 1,  &badpix) + 0.0001);
                if ( descr->specid > nregs ) {
                    nregs = descr->specid;
                }
                descr->lensid = (int)(cpl_image_get( image_id, xx + 1, yy + 1,  &badpix)+0.0001);
                descr->wavelength = cpl_image_get( image_wav, xx + 1, yy + 1, &badpix );
                if ( image_ill ) {
                    descr->illumination = cpl_image_get( image_ill, xx + 1, yy + 1, &badpix );
                }
                if ( image_dw ) {
                    descr->dlambda = cpl_image_get( image_dw, xx + 1, yy + 1, &badpix );
                }
                if ( image_ddw ) {
                    descr->ddlambda = cpl_image_get( image_ddw, xx + 1, yy + 1, &badpix );
                }
            }
        }
    }
    self->nregions = nregs;

    sph_pixel_description_table_find_maxlength( self );
    cpl_image_delete(specidim); specidim = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return self;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Load the pixel description table from image FITS file.
 *
 * @param fitsfilename The filename to load from
 *
 * @return The pixel description table to load or NULL on error.
 *
 * Loads the PDT from a FITS file where each extension is an image with the
 * one of the table columns.
 */
/*----------------------------------------------------------------------------*/
sph_pixel_description_table*
sph_pixel_description_table_load( const char* fitsfilename )
{
    sph_pixel_description_table*    self        = NULL;
    cpl_image*                      image_id    = NULL;
    cpl_image*                      image_sid    = NULL;
    cpl_image*                      image_wav   = NULL;
    cpl_image*                      image_dw    = NULL;
    cpl_image*                      image_ddw   = NULL;
    cpl_image*                      image_ill   = NULL;
    cpl_propertylist*               plist   = NULL;
    double                          offx    = 0.0;
    double                          offy    = 0.0;

    image_wav = cpl_image_load( fitsfilename, CPL_TYPE_FLOAT, 0, 0 );
    image_sid = cpl_image_load( fitsfilename, CPL_TYPE_INT, 0, 1);
    image_id = cpl_image_load( fitsfilename, CPL_TYPE_INT, 0, 2);
    image_dw = cpl_image_load( fitsfilename, CPL_TYPE_FLOAT, 0, 3 );
    image_ddw = cpl_image_load( fitsfilename, CPL_TYPE_FLOAT, 0, 4 );
    image_ill = cpl_image_load( fitsfilename, CPL_TYPE_FLOAT, 0, 5 );

    if ( !image_wav || !image_sid || !image_id ||
         !image_dw  || !image_ddw || !image_ill ) {
        sph_error_raise( SPH_ERROR_INCONSISTENT_INPUT,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Could not load all the planes for the PDT.");
        cpl_image_delete( image_wav ); image_wav = NULL;
        cpl_image_delete( image_sid ); image_sid = NULL;
        cpl_image_delete( image_ill ); image_ill = NULL;
        cpl_image_delete( image_id ); image_id = NULL;
        cpl_image_delete( image_ddw ); image_ddw = NULL;
        cpl_image_delete( image_dw ); image_dw = NULL;
        return NULL;
    }

    plist = sph_keyword_manager_load_properties(fitsfilename,0);

    offx = cpl_propertylist_get_double(plist, SPH_COMMON_KEYWORD_SPH_DITHERX);
    offy = cpl_propertylist_get_double(plist, SPH_COMMON_KEYWORD_SPH_DITHERY);

    if ( cpl_error_get_code() ) {
        sph_error_raise(cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_WARNING,
                        "Could not read the offset header keywords from %s\n"
                        " Set the offset to zero and continued.\n",
                        fitsfilename);
        offx = 0;
        offy = 0;
        cpl_error_reset();
    }

    self = sph_pixel_description_table_new_from_images(
            image_id,
            image_sid,
            image_wav,
            image_dw,
            image_ddw,
            image_ill,
            offx, offy );
    if ( !self ) {
        SPH_ERROR_RAISE_ERR(cpl_error_get_code(),
                "Could not load the PDT.");
    }
    cpl_image_delete( image_wav ); image_wav = NULL;
    cpl_image_delete( image_sid ); image_sid = NULL;
    cpl_image_delete( image_ill ); image_ill = NULL;
    cpl_image_delete( image_id ); image_id = NULL;
    cpl_image_delete( image_ddw ); image_ddw = NULL;
    cpl_image_delete( image_dw ); image_dw = NULL;
    cpl_propertylist_delete( plist ); plist = NULL;
    return self;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Save the pixel description table.
 *
 * @param self The pixel description table to save
 * @param fitsfilename The filename to save to
 * @return Error code.
 *
 * Saves the table in FITS image format. The routine creates a FITS file with
 * several extensions, corresponding to the different columns in the PDT,
 * with the planes being:
 * <ol>
 * <li>the wavelength</li>
 * <li>the lenslet id</li>
 * <li>the wavelength width</li>
 * <li>the second derivative<li>
 * <li>the illumination fraction</li>
 * </ol>
 *
 * The file is created ( if it
 * does not yet exist) or overwritten (if it does).
 */
/*----------------------------------------------------------------------------*/
int
sph_pixel_description_table_save(sph_pixel_description_table* self,
                                 const char* fitsfilename, sph_ifs_lenslet_model* model )
{
    int                 rerr        = CPL_ERROR_NONE;
    cpl_image*          image_id    = NULL;
    cpl_image*          image_sid    = NULL;
    cpl_image*          image_wav   = NULL;
    cpl_image*          image_dw    = NULL;
    cpl_image*          image_ddw   = NULL;
    cpl_image*          image_ill   = NULL;
    cpl_propertylist*    pl            = NULL;
    int                 xx          = 0;
    int                 yy          = 0;
    sph_pixel_descriptor*   descr   = NULL;

    if ( !self ) {
        SPH_NULL_ERROR
        return CPL_ERROR_NULL_INPUT;
    }

    if ( model ) {
        pl = sph_ifs_lenslet_model_get_as_propertylist( model );
    }
    else {
        pl = cpl_propertylist_new();
    }
    if ( pl ) {
        cpl_propertylist_update_int(pl,SPH_COMMON_KEYWORD_CALIB_PRISM_ID,0);
        cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_SPH_DITHERX,self->offx);
        cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_SPH_DITHERY,self->offy);
    }

    image_id    = cpl_image_new( self->nx, self->ny, CPL_TYPE_INT);
    image_sid    = cpl_image_new( self->nx, self->ny, CPL_TYPE_INT);
    image_wav   = cpl_image_new( self->nx, self->ny, CPL_TYPE_FLOAT);
    image_dw    = cpl_image_new( self->nx, self->ny, CPL_TYPE_FLOAT);
    image_ddw   = cpl_image_new( self->nx, self->ny, CPL_TYPE_FLOAT);
    image_ill   = cpl_image_new( self->nx, self->ny, CPL_TYPE_FLOAT);
    if ( !image_id || !image_wav || !image_dw || !image_ddw || !image_ill ) {
        return SPH_ERROR_MEMORY_FAIL;
    }
    for (yy = 0; yy < self->ny; ++yy) {
        for (xx = 0; xx < self->nx; ++xx) {
            descr = sph_pixel_description_table_get_descriptor(self, xx, yy);
            rerr |= cpl_image_set( image_sid, xx + 1, yy + 1, descr->specid );
            rerr |= cpl_image_set( image_id, xx + 1, yy + 1, descr->lensid );
            rerr |= cpl_image_set( image_wav, xx + 1, yy + 1, descr->wavelength );
            rerr |= cpl_image_set( image_dw, xx + 1, yy + 1, descr->dlambda );
            rerr |= cpl_image_set( image_ddw, xx + 1, yy + 1, descr->ddlambda );
            rerr |= cpl_image_set( image_ill, xx + 1, yy + 1, descr->illumination );
        }
    }

    if ( rerr ) {
        sph_error_raise( (int)cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Problem saving table. "
                                "CPL says: %s", cpl_error_get_message() );
        return (int)cpl_error_get_code();
    }

    if ( rerr ) {
        sph_error_raise( (int)cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Problem saving table. "
                                "CPL says: %s", cpl_error_get_message() );
        cpl_image_delete( image_sid );
        cpl_image_delete( image_id );
        cpl_image_delete( image_dw );
        cpl_image_delete( image_ddw );
        cpl_image_delete( image_wav );
        cpl_image_delete( image_ill );

        return (int)cpl_error_get_code();
    }

    unlink(fitsfilename);
    rerr |= sph_cube_append_image( fitsfilename, image_wav, pl,0 );
    rerr |= sph_cube_append_image( fitsfilename, image_sid, pl,1 );
    rerr |= sph_cube_append_image( fitsfilename, image_id, pl,2 );
    rerr |= sph_cube_append_image( fitsfilename, image_dw, pl,3 );
    rerr |= sph_cube_append_image( fitsfilename, image_ddw, pl,4 );
    rerr |= sph_cube_append_image( fitsfilename, image_ill, pl,5 );

    cpl_image_delete( image_sid );
    cpl_image_delete( image_id );
    cpl_image_delete( image_dw );
    cpl_image_delete( image_ddw );
    cpl_image_delete( image_wav );
    cpl_image_delete( image_ill );

    sph_cube_finalise_file(fitsfilename);
    cpl_propertylist_delete(pl);pl = NULL;
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Save the pixel description table.
 *
 * @param self The pixel description table to save
 * @param fitsfilename The filename to save to
 * @return Error code.
 *
 * Saves the table in FITS image format. The routine creates a FITS file with
 * several extensions, corresponding to the different columns in the PDT,
 * with the planes being:
 * <ol>
 * <li>the wavelength</li>
 * <li>the lenslet id</li>
 * <li>the wavelength width</li>
 * <li>the second derivative<li>
 * <li>the illumination fraction</li>
 * </ol>
 *
 * The file is created ( if it
 * does not yet exist) or overwritten (if it does).
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_pixel_description_table_save_dfs(
        sph_pixel_description_table* self,
        const char* fitsfilename,
        sph_ifs_lenslet_model* model,
        cpl_frameset* allframes,
        cpl_frame* template_frame,
        cpl_parameterlist* params,
        const char* tag,
        const char* recipe,
        const char* pipename,
        const cpl_propertylist* plist )
{
    int                 rerr        = CPL_ERROR_NONE;
    const size_t        nxy         = self ? (size_t)(self->nx * self->ny) : 0;
    cpl_image*          image_id    = NULL;
    cpl_image*          image_sid   = NULL;
    cpl_image*          image_wav   = NULL;
    cpl_image*          image_dw    = NULL;
    cpl_image*          image_ddw   = NULL;
    cpl_image*          image_ill   = NULL;
    int*                pimg_id;
    int*                pimg_sid;
    float*              pimg_wav;
    float*              pimg_dw;
    float*              pimg_ddw;
    float*              pimg_ill;
    cpl_propertylist*    pl            = NULL;
    cpl_propertylist*        tpli            = NULL;
    cpl_propertylist*        modprops            = NULL;
    cpl_propertylist*        extpl           = NULL;
    const sph_pixel_descriptor* darray = self ?
        sph_pixel_description_table_get_array_const(self) : NULL;

    cpl_ensure_code(!cpl_error_get_code(), cpl_error_set_where(cpl_func));
    cpl_ensure_code(darray != NULL, CPL_ERROR_NULL_INPUT);

    pimg_id  = cpl_malloc(nxy * sizeof(*pimg_id));
    pimg_sid = cpl_malloc(nxy * sizeof(*pimg_sid));
    pimg_wav = cpl_malloc(nxy * sizeof(*pimg_wav));
    pimg_dw  = cpl_malloc(nxy * sizeof(*pimg_dw));
    pimg_ddw = cpl_malloc(nxy * sizeof(*pimg_ddw));
    pimg_ill = cpl_malloc(nxy * sizeof(*pimg_ill));

    image_id    = cpl_image_wrap_int( self->nx, self->ny, pimg_id);
    image_sid   = cpl_image_wrap_int( self->nx, self->ny, pimg_sid);
    image_wav   = cpl_image_wrap_float( self->nx, self->ny, pimg_wav);
    image_dw    = cpl_image_wrap_float( self->nx, self->ny, pimg_dw);
    image_ddw   = cpl_image_wrap_float( self->nx, self->ny, pimg_ddw);
    image_ill   = cpl_image_wrap_float( self->nx, self->ny, pimg_ill);

    for (size_t i = 0; i < nxy; i++) {
        pimg_sid[i] = darray[i].specid;
        pimg_id [i] = darray[i].lensid;
        pimg_wav[i] = darray[i].wavelength;
        pimg_dw [i] = darray[i].dlambda;
        pimg_ddw[i] = darray[i].ddlambda;
        pimg_ill[i] = darray[i].illumination;
    }

    pl = cpl_propertylist_new();
    if ( model ) {
        modprops = sph_ifs_lenslet_model_get_as_propertylist(model);
        cpl_propertylist_append(pl, modprops);
    }
    if ( plist ) {
        cpl_propertylist_append(pl, plist);
    }
    if ( rerr ) {
        sph_error_raise( (int)cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Problem saving table. "
                                "CPL says: %s", cpl_error_get_message() );
        return (int)cpl_error_get_code();
    }
    cpl_propertylist_append_double(pl,SPH_COMMON_KEYWORD_SPH_DITHERX,self->offx);
    cpl_propertylist_append_double(pl,SPH_COMMON_KEYWORD_SPH_DITHERY,self->offy);
    rerr = cpl_propertylist_update_string( pl, SPH_COMMON_KEYWORD_PRO_CATG, tag );

    if ( pl ) {
        tpli = sph_keyword_manager_trans_keywords( pl );
    }

    cpl_propertylist_update_string(pl,SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_PDT_WAVELENGTH_EXTNAME);
    cpl_propertylist_append_string( pl, SPH_COMMON_KEYWORD_QC_COMPNAME, SPH_QC_COMPNAME );
    cpl_propertylist_append_string( pl, SPH_COMMON_KEYWORD_QC_COMPVERN, SPH_QC_COMPVERN );
    cpl_propertylist_append_string( pl, SPH_COMMON_KEYWORD_QC_COMPREVN, SPH_QC_COMPREVN );
    cpl_propertylist_append_string( pl, SPH_COMMON_KEYWORD_QC_COMPDATE, SPH_QC_COMPDATE );

    sph_utils_remove_wcs_3d(pl);
    /* FIXME: Set WCS to dummy (pixel) value for now */
    sph_utils_reset_wcs_12d(pl);

    rerr = cpl_dfs_save_image( allframes, NULL, params,
            allframes,
            template_frame,
            image_wav,
            CPL_TYPE_DOUBLE, recipe, pl, NULL, pipename, fitsfilename );

    extpl = cpl_propertylist_new();
    cpl_propertylist_update_string(extpl,SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_PDT_SPECID_EXTNAME);
    cpl_image_save(image_sid,fitsfilename,CPL_BPP_32_SIGNED,extpl,CPL_IO_EXTEND);
    cpl_propertylist_update_string(extpl,SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_PDT_LENSID_EXTNAME);
    cpl_image_save(image_id,fitsfilename,CPL_BPP_32_SIGNED,extpl,CPL_IO_EXTEND);
    cpl_propertylist_update_string(extpl,SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_PDT_DLAMBDA_EXTNAME);
    cpl_image_save(image_dw,fitsfilename,CPL_TYPE_DOUBLE,extpl,CPL_IO_EXTEND);
    cpl_propertylist_update_string(extpl,SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_PDT_DDLAMBDA_EXTNAME);
    cpl_image_save(image_ddw,fitsfilename,CPL_TYPE_DOUBLE,extpl,CPL_IO_EXTEND);
    cpl_propertylist_update_string(extpl,SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_PDT_ILLUMINATION_EXTNAME);
    cpl_image_save(image_ill,fitsfilename,CPL_TYPE_DOUBLE,extpl,CPL_IO_EXTEND);


    cpl_image_delete( image_sid );
    cpl_image_delete( image_id );
    cpl_image_delete( image_dw );
    cpl_image_delete( image_ddw );
    cpl_image_delete( image_wav );
    cpl_image_delete( image_ill );

    cpl_propertylist_delete(extpl); extpl = NULL;
    cpl_propertylist_delete(tpli); tpli = NULL;
    cpl_propertylist_delete(modprops); modprops = NULL;
    cpl_propertylist_delete(pl); pl = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Retrun the maximal spectral length in pixels in the PDT
 *
 * @param self The pixel description table
 *
 * @return the maximal spectral length.
 *
 * Returns the length of the longest connection spectral region
 * in pixels.
 */
/*----------------------------------------------------------------------------*/
int
sph_pixel_description_table_get_maxlength( sph_pixel_description_table* self )
{
    int             result      = 0;
    if ( !self ) {
        SPH_NO_SELF;
        return -1;
    }

    result = self->maxlength;

    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Set the wavelength points of a spectral region
 * @param self the pixel description table
 * @param specid     the id of the region
 * @param distp        distances to lower edge of points
 * @param wavs    the wavelength vector to set
 * @param dwavs    the delta lambda vector to set (NULL allowed)
 *
 * @return error code of the operation
 *
 * Sets the wavelength points to those given in the vector.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_pixel_description_table_set_wavelengths(
        sph_pixel_description_table* self,
        int specid,
        const cpl_vector* distp,
        const cpl_vector* iwavs )
{
    sph_spectral_region*        reg = NULL;
    sph_pixel_descriptor*        descr = NULL;
    int                xx = 0;
    int                yy = 0;
    double             val = 0.0;
    double             dval = 0.0;
    double             x = 0.0;
    gsl_interp_accel*     acc = NULL;
    int                    np = 0;
    gsl_spline*         spline = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    if ( !self || !iwavs ) {
        SPH_NO_SELF;
        return CPL_ERROR_NULL_INPUT;
    }
    if ( specid < 1 || specid > self->nregions ) {
        SPH_ERR("Out of bounds spec id.");
        return CPL_ERROR_ACCESS_OUT_OF_RANGE;
    }

    np = cpl_vector_get_size(distp);
    reg = sph_pixel_description_table_get_region( self, specid );
    if ( reg && np > 4 ) {
        gsl_set_error_handler_off();

        acc = gsl_interp_accel_alloc ();
        spline = gsl_spline_alloc (gsl_interp_cspline, np );

        if ( gsl_spline_init (spline, cpl_vector_get_data_const(distp),
                cpl_vector_get_data_const(iwavs), np) == GSL_SUCCESS )
        {
            for (yy = (int)reg->miny; yy <= (int)reg->maxy; ++yy) {
                x = yy - reg->miny;
        if ( x > cpl_vector_get_max(distp) ) x = cpl_vector_get_max(distp);
        if ( x < cpl_vector_get_min(distp) ) x = cpl_vector_get_min(distp);
                val = gsl_spline_eval(spline,x,acc);
                dval = gsl_spline_eval_deriv(spline,x,acc);
                for ( xx = (int)reg->minx; xx <= (int)reg->maxx; ++xx ) {
                    descr = sph_pixel_description_table_get_descriptor(self,xx,yy);
                    if ( descr ) {
                        descr->lensid = reg->lensid;
                        descr->specid = reg->specid;
                        descr->wavelength = val;
                        descr->dlambda = dval;
                        descr->ddlambda = 0.0;
                    }
                }
            }
        }
        else {
            SPH_ERR("Error interpolating the wavelengths.");
        }
        gsl_spline_free(spline);spline = NULL;
        gsl_interp_accel_free(acc);acc = NULL;
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Set the wavelength points of a spectral region
 * @param self the pixel description table
 * @param specid     the id of the region
 * @param wavs    the wavelength vector to set
 * @param dwavs    the delta lambda vector to set (NULL allowed)
 *
 * @return error code of the operation
 *
 * Sets the wavelength points to those given in the vector.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_pixel_description_table_set_spectra_wavelengths(
        sph_pixel_description_table* self,
        int specid,
        const cpl_vector* iwavs,
        short int no_spline_interpol)
{
    cpl_vector*        distp = NULL;
    sph_spectral_region*        reg = NULL;
    double                        dx = 0.0;
    int                            vv = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    if ( !self || !iwavs ) {
        SPH_NO_SELF;
        return CPL_ERROR_NULL_INPUT;
    }
    if ( specid < 1 || specid > self->nregions ) {
        SPH_ERR("Out of bounds spec id.");
        return CPL_ERROR_ACCESS_OUT_OF_RANGE;
    }

    reg = sph_pixel_description_table_get_region( self, specid );
    if ( reg ) {
        distp = cpl_vector_new(cpl_vector_get_size(iwavs));
        dx = ( reg->maxy - reg->miny ) / cpl_vector_get_size(iwavs);
        if (no_spline_interpol)
            dx=1.0;
        for (vv = 0; vv < cpl_vector_get_size(distp); ++vv) {
            cpl_vector_set(distp,vv,dx*vv);
        }
        sph_pixel_description_table_set_wavelengths(self,
                specid,distp,iwavs);
        cpl_vector_delete(distp);
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Set the wavelength points of a spectral region by lensid
 * @param self the pixel description table
 * @param specid     the lens id of the region
 * @param wavs    the wavelength vector to set (NULL allowed)
 * @param dwavs    the delata lambda vector to set (NULL allowed)
 *
 * @return error code of the operation
 *
 * Sets the wavelength points to those given in the vector.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_pixel_description_table_set_wavs_lensid(
        sph_pixel_description_table* self,
        int lid,
        cpl_vector* wavs )
{
    int specid = 0;
    for (specid = 0; specid < self->nregions; ++specid) {
        if ( self->regions[specid]->lensid == lid ) break;
    }
    if ( specid >= self->nregions ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    if ( self->regions[specid]->lensid != lid ) {
        SPH_ERROR_RAISE_WARNING(CPL_ERROR_ILLEGAL_INPUT,
                "Did not find spectrum region with lensid %d specified", lid);
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    return sph_pixel_description_table_set_spectra_wavelengths(self,specid+1,wavs,0);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Find the maximal spectral length in pixels in the PDT
 *
 * @param self The pixel description table
 *
 * @return the maximal spectral length.
 *
 * Returns the length of the longest connection spectral region
 * in pixels.
 */
/*----------------------------------------------------------------------------*/
int
sph_pixel_description_table_find_maxlength( sph_pixel_description_table* self )
{
    int             result      = 0;
    int             imaxv       = 0;
    int*            maxy        = NULL;
    int*            miny        = NULL;
    int             label       = 0;
    int             xx          = 0;
    int             yy          = 0;
    int             size        = 0;
    sph_pixel_descriptor*   pdescr = NULL;
    if ( !self ) {
        SPH_NO_SELF;
        return -1;
    }

    imaxv = self->nregions;
    maxy = cpl_calloc( imaxv, sizeof(int) );
    miny = cpl_calloc( imaxv, sizeof(int) );

    for ( yy = 0; yy < imaxv; yy++ ) {
        miny[yy] = __INT_MAX__;
    }

    for ( yy = 0; yy < self->ny; ++yy ) {
        for ( xx = 0; xx < self->ny; ++xx ) {
            pdescr = sph_pixel_description_table_get_descriptor( self, xx, yy );
            label = pdescr->specid;
            if ( label > 0 ) {
                size = 0;
                if ( yy > maxy[label-1] ) {
                    maxy[label-1] = yy;
                    size = maxy[label-1] - miny[label -1];
                }
                if ( yy < miny[label-1] ) {
                    miny[label-1] = yy;
                    size = maxy[label-1] - miny[label -1];
                }
                if ( size > self->maxlength ) {
                    self->maxlength = size;
                    self->idlongest = label;
                }
            }
        }
    }

    cpl_free(maxy);
    cpl_free(miny);

    result = self->maxlength;

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the pixel description table and free memory.
 *
 * @param self The table to delete
 *
 * @return Error code.
 *
 * The pixel description table is deleted, as are all associated data
 * strctures. All memory is freed.
 */
/*----------------------------------------------------------------------------*/
void sph_pixel_description_table_delete(sph_pixel_description_table* self) {
    int            ii        = 0;

    if ( !self ) return;

    if ( self->arr ) {
        cpl_free(self->arr);
    }
    if ( self->regions ) {
        for (ii = 0; ii < self->nregions; ++ii) {
            sph_spectral_region_delete( self->regions[ii] );
            self->regions[ii] = NULL;
        }
        cpl_free( self->regions);
    }
    cpl_free( self );
    return;
}
/**@}*/
